/**
 * Session Manager
 *
 * Manages multiple SandScript sessions with persistence.
 * Sessions are gzipped memory dumps stored in ~/.sandscript/sessions/
 */

import { createSession as createFuelSession } from './session.js';

// =============================================================================
// Session Name Generation
//
// Three-word names evoking sand, math, and the poetic.
// Examples: "dune-euler-spiral", "grain-fourier-wave", "desert-gauss-curve"
// =============================================================================

// Sand & Desert imagery
const SAND = [
  'dune', 'grain', 'desert', 'oasis', 'mirage', 'drift', 'shore', 'beach',
  'delta', 'sediment', 'silica', 'quartz', 'grit', 'dust', 'erosion', 'ripple',
  'hourglass', 'sandstone', 'mesa', 'canyon', 'arid', 'barren', 'windswept',
  'sunbaked', 'amber', 'ochre', 'golden', 'tawny',
];

// Mathematicians & Thinkers
const MATHEMATICIANS = [
  'euler', 'gauss', 'fourier', 'laplace', 'newton', 'leibniz', 'riemann',
  'cantor', 'godel', 'turing', 'lovelace', 'noether', 'hilbert', 'fermat',
  'pascal', 'descartes', 'archimedes', 'euclid', 'pythagoras', 'fibonacci',
  'mandelbrot', 'ramanujan', 'erdos', 'hypatia', 'poincare', 'kolmogorov',
  'bayes', 'markov',
];

// Mathematical objects & concepts
const MATH_OBJECTS = [
  'spiral', 'wave', 'curve', 'field', 'space', 'series', 'prime', 'root',
  'orbit', 'fractal', 'tensor', 'vector', 'matrix', 'proof', 'theorem',
  'axiom', 'lemma', 'set', 'graph', 'lattice', 'manifold', 'surface',
  'integral', 'limit', 'sum', 'ratio', 'sequence', 'function',
];

/**
 * Pick a random element from an array.
 */
function pick(array) {
  return array[Math.floor(Math.random() * array.length)];
}

/**
 * Generate a session name like "dune-euler-spiral"
 */
function generateName() {
  return `${pick(SAND)}-${pick(MATHEMATICIANS)}-${pick(MATH_OBJECTS)}`;
}

/**
 * Get sessions directory path
 */
function getSessionsDir() {
  const home = Deno.env.get('HOME') || Deno.env.get('USERPROFILE') || '.';
  return `${home}/.sandscript/sessions`;
}

/**
 * Session Manager - handles multiple named sessions with persistence
 */
/**
 * Session entry structure:
 * {
 *   session: FuelSession,
 *   dirty: boolean,        // Has unsaved changes
 *   fromDisk: boolean,     // Was loaded from disk (read-only by default)
 * }
 *
 * Save behavior:
 * - New sessions (fromDisk=false): Auto-save on exit
 * - Loaded sessions (fromDisk=true): Read-only, changes are ephemeral
 * - Use 'save' action to explicitly persist loaded sessions
 * - Use 'save-as' to fork to a new name
 */
export class SessionManager {
  constructor() {
    this.sessions = new Map(); // name -> { session, dirty, fromDisk }
    this.currentName = null;
    this.currentOwned = false; // Whether we own the current session (vs read-only)
    this.sessionsDir = getSessionsDir();
  }

  /**
   * Initialize the manager - load existing sessions list and restore current session
   */
  async init() {
    try {
      await Deno.mkdir(this.sessionsDir, { recursive: true });
    } catch (err) {
      if (!(err instanceof Deno.errors.AlreadyExists)) {
        throw err;
      }
    }

    // Load state file to restore current session name
    await this.loadState();

    // If we have a current session name, load it
    if (this.currentName) {
      const loaded = await this.loadSession(this.currentName);
      if (loaded) {
        // Only mark as owned if it was our implicit session
        if (this.currentOwned) {
          this.sessions.get(this.currentName).fromDisk = false;
        }
      } else {
        // Session file was deleted, reset to no current
        this.currentName = null;
        this.currentOwned = false;
      }
    }
  }

  /**
   * Load state file (~/.sandscript/state.json)
   */
  async loadState() {
    const statePath = `${this.sessionsDir}/../state.json`;
    try {
      const data = await Deno.readTextFile(statePath);
      const state = JSON.parse(data);
      this.currentName = state.current || null;
      this.currentOwned = state.owned || false;
    } catch {
      // State file doesn't exist or is invalid
      this.currentName = null;
      this.currentOwned = false;
    }
  }

  /**
   * Save state file
   */
  async saveState() {
    const statePath = `${this.sessionsDir}/../state.json`;
    const state = {
      current: this.currentName,
      owned: this.currentOwned,
    };
    await Deno.writeTextFile(statePath, JSON.stringify(state, null, 2) + '\n');
  }

  /**
   * Get or create the current session
   */
  async current() {
    if (!this.currentName) {
      // Create default session (new, not from disk)
      this.currentName = generateName();
      this.currentOwned = true; // We created it, we own it
      const session = await createFuelSession();
      // Mark as dirty so it gets saved
      this.sessions.set(this.currentName, { session, dirty: true, fromDisk: false });
      // Save state so next invocation knows the current session
      await this.saveState();
    }
    return this.sessions.get(this.currentName);
  }

  /**
   * Create a new session and switch to it
   */
  async newSession(name) {
    // Auto-save current session if it's new (not loaded from disk)
    await this.autoSaveCurrent();

    const sessionName = name || generateName();
    const session = await createFuelSession();
    // Mark new session as dirty so it gets saved
    this.sessions.set(sessionName, { session, dirty: true, fromDisk: false });
    this.currentName = sessionName;
    this.currentOwned = true; // We created it, we own it
    await this.saveState();

    return { name: sessionName };
  }

  /**
   * Switch to an existing session or create if it doesn't exist
   */
  async useSession(name) {
    // Auto-save current session if it's new (not loaded from disk)
    if (this.currentName && this.currentName !== name) {
      await this.autoSaveCurrent();
    }

    if (this.sessions.has(name)) {
      // Already loaded - keep existing fromDisk status
      this.currentName = name;
      this.currentOwned = !this.sessions.get(name).fromDisk;
      await this.saveState();
      return { name };
    }

    // Try to load from disk
    const loaded = await this.loadSession(name);
    if (loaded) {
      this.currentName = name;
      this.currentOwned = false; // Explicitly loaded = read-only
      await this.saveState();
      return { name };
    }

    // Create new
    const session = await createFuelSession();
    // Mark as dirty so it gets saved
    this.sessions.set(name, { session, dirty: true, fromDisk: false });
    this.currentName = name;
    this.currentOwned = true; // We created it
    await this.saveState();
    return { name, created: true };
  }

  /**
   * List all sessions (in memory + on disk)
   */
  async listSessions() {
    const names = new Set(this.sessions.keys());

    // Add sessions from disk
    try {
      for await (const entry of Deno.readDir(this.sessionsDir)) {
        if (entry.isFile && entry.name.endsWith('.sand')) {
          names.add(entry.name.slice(0, -5)); // remove .sand
        }
      }
    } catch {
      // Directory might not exist yet
    }

    return [...names].map(name => ({
      name,
      current: name === this.currentName,
      loaded: this.sessions.has(name),
    }));
  }

  /**
   * Mark current session as dirty (needs saving)
   */
  markDirty() {
    if (this.currentName && this.sessions.has(this.currentName)) {
      this.sessions.get(this.currentName).dirty = true;
    }
  }

  /**
   * Save a session to disk (using CompressionStream for gzip)
   */
  async saveSession(name) {
    const entry = this.sessions.get(name);
    if (!entry) return false;

    const { session } = entry;
    const mem = session.mem;

    // Get the raw memory bytes
    const bytes = new Uint8Array(mem.memory.buffer, mem.baseOffset, mem.segmentSize);

    // Compress using CompressionStream
    const stream = new Blob([bytes]).stream().pipeThrough(new CompressionStream('gzip'));
    const compressed = new Uint8Array(await new Response(stream).arrayBuffer());

    // Write to file
    const path = `${this.sessionsDir}/${name}.sand`;
    await Deno.writeFile(path, compressed);

    entry.dirty = false;
    return true;
  }

  /**
   * Load a session from disk (read-only by default)
   */
  async loadSession(name) {
    const path = `${this.sessionsDir}/${name}.sand`;

    try {
      const compressed = await Deno.readFile(path);

      // Decompress using DecompressionStream
      const stream = new Blob([compressed]).stream().pipeThrough(new DecompressionStream('gzip'));
      const bytes = new Uint8Array(await new Response(stream).arrayBuffer());

      // Create a new session with the restored memory
      // Mark as fromDisk=true so changes are ephemeral by default
      const session = await createFuelSession({ fromBytes: bytes });
      this.sessions.set(name, { session, dirty: false, fromDisk: true });
      return true;
    } catch {
      return false;
    }
  }

  /**
   * Auto-save current session if it's new (not loaded from disk)
   */
  async autoSaveCurrent() {
    if (!this.currentName) return;
    const entry = this.sessions.get(this.currentName);
    if (entry && entry.dirty && !entry.fromDisk) {
      await this.saveSession(this.currentName);
    }
  }

  /**
   * Save all dirty NEW sessions (not loaded from disk)
   * Loaded sessions require explicit save
   */
  async saveAll() {
    for (const [name, entry] of this.sessions) {
      if (entry.dirty && !entry.fromDisk) {
        await this.saveSession(name);
      }
    }
  }

  /**
   * Explicitly save current session (even if loaded from disk)
   */
  async saveCurrentSession() {
    if (!this.currentName) {
      throw new Error('No current session to save');
    }
    const entry = this.sessions.get(this.currentName);
    if (!entry) {
      throw new Error('Current session not found');
    }
    await this.saveSession(this.currentName);
    // After explicit save, it's no longer "from disk" in the read-only sense
    entry.fromDisk = false;
    return { name: this.currentName, saved: true };
  }

  /**
   * Save current session to a new name (fork)
   */
  async saveCurrentSessionAs(newName) {
    if (!this.currentName) {
      throw new Error('No current session to save');
    }
    if (!newName) {
      throw new Error('New name required');
    }

    // Check if name already exists
    const existingPath = `${this.sessionsDir}/${newName}.sand`;
    try {
      await Deno.stat(existingPath);
      throw new Error(`Session "${newName}" already exists`);
    } catch (err) {
      if (!(err instanceof Deno.errors.NotFound)) {
        throw err;
      }
    }

    const entry = this.sessions.get(this.currentName);
    if (!entry) {
      throw new Error('Current session not found');
    }

    // Save to new name
    const { session } = entry;
    const mem = session.mem;
    const bytes = new Uint8Array(mem.memory.buffer, mem.baseOffset, mem.segmentSize);
    const stream = new Blob([bytes]).stream().pipeThrough(new CompressionStream('gzip'));
    const compressed = new Uint8Array(await new Response(stream).arrayBuffer());
    const path = `${this.sessionsDir}/${newName}.sand`;
    await Deno.writeFile(path, compressed);

    // Switch to the new session (reload it fresh)
    await this.loadSession(newName);
    // Mark the new one as not from disk (we just created it, we own it)
    this.sessions.get(newName).fromDisk = false;
    this.currentName = newName;
    this.currentOwned = true; // We created it via save-as
    await this.saveState();

    return { name: newName, saved: true };
  }

  /**
   * Get current session name
   */
  getCurrentName() {
    return this.currentName;
  }

  /**
   * Rename the current session
   */
  async renameSession(newName) {
    if (!this.currentName) {
      throw new Error('No current session to rename');
    }
    if (!newName) {
      throw new Error('New name required');
    }
    if (newName === this.currentName) {
      return { name: newName };
    }

    // Check if new name already exists
    const existingPath = `${this.sessionsDir}/${newName}.sand`;
    try {
      await Deno.stat(existingPath);
      throw new Error(`Session "${newName}" already exists`);
    } catch (err) {
      if (!(err instanceof Deno.errors.NotFound)) {
        throw err;
      }
    }

    const oldName = this.currentName;
    const entry = this.sessions.get(oldName);

    // Update in-memory map
    this.sessions.delete(oldName);
    this.sessions.set(newName, entry);
    this.currentName = newName;

    // Rename file on disk if it exists
    const oldPath = `${this.sessionsDir}/${oldName}.sand`;
    const newPath = `${this.sessionsDir}/${newName}.sand`;
    try {
      await Deno.rename(oldPath, newPath);
    } catch {
      // File might not exist yet (not saved)
    }

    await this.saveState();
    return { name: newName };
  }

  /**
   * Delete a session
   */
  async deleteSession(name) {
    if (!name) {
      throw new Error('Session name required');
    }

    // Can't delete current session
    if (name === this.currentName) {
      throw new Error('Cannot delete current session');
    }

    // Remove from memory
    this.sessions.delete(name);

    // Remove from disk
    const path = `${this.sessionsDir}/${name}.sand`;
    try {
      await Deno.remove(path);
      return { deleted: true };
    } catch (err) {
      if (err instanceof Deno.errors.NotFound) {
        return { deleted: false };
      }
      throw err;
    }
  }

  /**
   * Export a session to a file
   */
  async exportSession(name, destPath) {
    if (!name) {
      throw new Error('Session name required');
    }
    if (!destPath) {
      throw new Error('Destination path required');
    }

    // Expand ~ to home directory
    if (destPath.startsWith('~/')) {
      const home = Deno.env.get('HOME') || Deno.env.get('USERPROFILE') || '.';
      destPath = `${home}${destPath.slice(1)}`;
    }

    // If session is in memory, save it first
    if (this.sessions.has(name)) {
      await this.saveSession(name);
    }

    // Copy the session file
    const srcPath = `${this.sessionsDir}/${name}.sand`;
    try {
      await Deno.copyFile(srcPath, destPath);
      return { path: destPath };
    } catch (err) {
      if (err instanceof Deno.errors.NotFound) {
        throw new Error(`Session "${name}" not found`);
      }
      throw err;
    }
  }
}
