/**
 * MCP Session
 *
 * Wraps the SandScript fuel interpreter with MCP-friendly methods.
 * Provides fuel control, status handling, and verbose error messages.
 */

import { MemoryManipulator } from '../fuel/manipulator.js';
import { Orchestrator } from '../fuel/orchestrator.js';
import { Parser } from '../fuel/parser.js';
import { instantiate } from '../fuel/interpreter.wasm.js';
import {
  STATUS_RUNNING,
  STATUS_DONE,
  STATUS_PAUSED_FUEL,
  STATUS_FFI_REQUEST,
  STATUS_ERROR,
  STATUS_THROW,
} from '../fuel/constants.js';

/**
 * Create an MCP session with fuel-controlled execution.
 *
 * @param {Object} options
 * @param {Uint8Array} options.fromBytes - Restore session from saved bytes
 */
export async function createSession(options = {}) {
  const { fromBytes } = options;

  let memory;
  let mem;

  if (fromBytes) {
    // Restore from saved bytes
    const pages = Math.ceil(fromBytes.length / 65536);
    memory = new WebAssembly.Memory({ initial: pages });
    const target = new Uint8Array(memory.buffer);
    target.set(fromBytes);
    mem = new MemoryManipulator(memory, 0, fromBytes.length);
  } else {
    // Fresh session
    memory = new WebAssembly.Memory({ initial: 16 }); // 1MB
    mem = new MemoryManipulator(memory, 0);
  }

  const wasm = await instantiate(memory);
  mem.setWasmInstance(wasm);

  if (!fromBytes) {
    mem.initialize();
  } else {
    // Restore WASM globals from saved memory state
    wasm.exports.init_regions();
  }

  const orchestrator = new Orchestrator(mem, wasm);
  const parser = new Parser(mem);

  // Code block - allocated once, appended to on each eval
  // When restoring from bytes, code block already exists in memory
  let codeBlock = fromBytes ? mem.getCodeBlock() : null;

  // Track current status for continue validation
  // Derive initial status from memory when restoring
  let currentStatus = mapStatus(mem.getStatus());

  /**
   * Map internal status to MCP status string
   */
  function mapStatus(internalStatus) {
    switch (internalStatus) {
      case STATUS_DONE:
        return 'done';
      case STATUS_PAUSED_FUEL:
        return 'paused';
      case STATUS_ERROR:
      case STATUS_THROW:
        return 'runtime_error';
      default:
        return 'done';
    }
  }

  /**
   * Run with fuel control, handling FFI requests
   */
  function runWithFuel(fuel) {
    let totalFuelUsed = 0;
    const startFuel = mem.getFuel();

    while (true) {
      wasm.exports.run(fuel);
      const status = mem.getStatus();

      if (status === STATUS_DONE) {
        totalFuelUsed = startFuel + fuel - mem.getFuel();
        return { internalStatus: status, fuelUsed: totalFuelUsed };
      }

      if (status === STATUS_PAUSED_FUEL) {
        totalFuelUsed = startFuel + fuel - mem.getFuel();
        return { internalStatus: status, fuelUsed: totalFuelUsed };
      }

      if (status === STATUS_FFI_REQUEST) {
        orchestrator.handleFFIRequest();
        continue;
      }

      if (status === STATUS_ERROR || status === STATUS_THROW) {
        totalFuelUsed = startFuel + fuel - mem.getFuel();
        return { internalStatus: status, fuelUsed: totalFuelUsed };
      }

      throw new Error(`Unknown status: ${status}`);
    }
  }

  /**
   * Run to completion (keep adding fuel until done or error)
   */
  function runToCompletion() {
    let totalFuelUsed = 0;
    const fuelPerStep = 10000;

    while (true) {
      const { internalStatus, fuelUsed } = runWithFuel(fuelPerStep);
      totalFuelUsed += fuelUsed;

      if (internalStatus !== STATUS_PAUSED_FUEL) {
        return { internalStatus, fuelUsed: totalFuelUsed };
      }
    }
  }

  /**
   * Build error info from interpreter state
   */
  function buildErrorInfo() {
    const errorInfo = mem.getErrorInfo();
    const instructionIndex = mem.getInstructionIndex();

    // TODO: enhance with source position mapping
    return {
      message: `Runtime error: code=${errorInfo.code}, detail=${errorInfo.detail}`,
      instructionIndex,
    };
  }

  return {
    /**
     * Parse and run code.
     *
     * @param {string} code - SandScript code
     * @param {number} [fuel] - Fuel limit. Omit to run to completion. 0 to load only.
     * @returns {object} { status, fuelUsed, ... }
     */
    async eval(code, fuel) {
      // Allocate code block on first eval, then append
      if (codeBlock === null) {
        codeBlock = mem.allocateCodeBlock();
      }

      // Start execution from where new code begins
      const startIndex = mem.codeBlockInstructionCount(codeBlock);
      mem.setInstructionIndex(startIndex);

      // Parse
      try {
        parser.parse(code, codeBlock);
        parser.complete();
      } catch (err) {
        currentStatus = 'syntax_error';
        // Extract line/column if available
        const match = err.message.match(/line (\d+)/i);
        const line = match ? parseInt(match[1], 10) : 1;
        return {
          status: 'syntax_error',
          message: err.message,
          line,
          column: 1,
        };
      }

      // If fuel is 0, just load (don't run)
      if (fuel === 0) {
        currentStatus = 'paused';
        mem.setStatus(STATUS_PAUSED_FUEL);
        return {
          status: 'paused',
          fuelUsed: 0,
        };
      }

      // Run
      mem.setStatus(STATUS_RUNNING);
      const { internalStatus, fuelUsed } = fuel === undefined
        ? runToCompletion()
        : runWithFuel(fuel);

      currentStatus = mapStatus(internalStatus);

      if (currentStatus === 'runtime_error') {
        const errorInfo = buildErrorInfo();
        return {
          status: 'runtime_error',
          fuelUsed,
          ...errorInfo,
        };
      }

      return {
        status: currentStatus,
        fuelUsed,
      };
    },

    /**
     * Resume paused execution.
     *
     * @param {number} [fuel] - Fuel to add. Omit to run to completion.
     * @returns {object} { status, fuelUsed, ... }
     */
    async continue(fuel) {
      if (currentStatus !== 'paused') {
        throw new Error(`Cannot continue: status is "${currentStatus}", not "paused"`);
      }

      mem.setStatus(STATUS_RUNNING);
      const { internalStatus, fuelUsed } = fuel === undefined
        ? runToCompletion()
        : runWithFuel(fuel);

      currentStatus = mapStatus(internalStatus);

      if (currentStatus === 'runtime_error') {
        const errorInfo = buildErrorInfo();
        return {
          status: 'runtime_error',
          fuelUsed,
          ...errorInfo,
        };
      }

      return {
        status: currentStatus,
        fuelUsed,
      };
    },

    /**
     * Read a variable from scope.
     *
     * @param {string} name - Variable name
     * @returns {*} Value or null if not found
     */
    async readVar(name) {
      const value = orchestrator.getVariable(name);
      return value === undefined ? null : value;
    },

    /**
     * List user-defined variables in current scope.
     * Excludes builtins (Math, Array, etc).
     *
     * @returns {string[]} Variable names
     */
    async listVars() {
      const builtins = new Set([
        'Math', 'Array', 'Object', 'String', 'Number',
        'parseInt', 'parseFloat', 'isNaN', 'isFinite',
        'Boolean', 'Infinity', 'NaN', 'undefined',
      ]);
      const scope = mem.getScope();
      const keys = mem.scopeKeys(scope);
      return keys.filter(k => !builtins.has(k));
    },

    /**
     * Inspect interpreter state.
     *
     * @param {string} what - "state", "stack", "scope", or "memory"
     * @param {object} options - { address, length } for memory inspection
     * @returns {object} Inspection result
     */
    async inspect(what = 'state', options = {}) {
      switch (what) {
        case 'state': {
          const instructionIndex = mem.getInstructionIndex();
          const codeBlock = mem.getCodeBlock();
          const instructionCount = mem.codeBlockInstructionCount(codeBlock);
          return {
            status: currentStatus,
            instructionIndex,
            instructionCount,
            callStackDepth: mem.getCallStackDepth(),
            fuel: mem.getFuel(),
          };
        }

        case 'stack': {
          return {
            pending: orchestrator.getPendingStack(),
            callStackDepth: mem.getCallStackDepth(),
          };
        }

        case 'scope': {
          const builtins = new Set([
            'Math', 'Array', 'Object', 'String', 'Number',
            'parseInt', 'parseFloat', 'isNaN', 'isFinite',
            'Boolean', 'Infinity', 'NaN', 'undefined',
          ]);
          const all = orchestrator.getAllVariables();
          const variables = {};
          for (const [k, v] of Object.entries(all)) {
            if (!builtins.has(k)) {
              variables[k] = v;
            }
          }
          return { variables };
        }

        case 'memory': {
          const address = options.address ?? 0;
          const length = options.length ?? 64;
          const bytes = new Uint8Array(mem.memory.buffer, address, length);
          // Format as hex dump
          const hex = Array.from(bytes).map(b => b.toString(16).padStart(2, '0')).join(' ');
          return {
            address,
            length,
            hex,
          };
        }

        default:
          throw new Error(`Unknown inspection target: ${what}`);
      }
    },

    // Expose internals for debugging
    get mem() { return mem; },
    get orchestrator() { return orchestrator; },
  };
}
