/**
 * SandScript MCP Server
 *
 * Exposes SandScript as tools for LLMs via Model Context Protocol.
 * Communicates over stdio using JSON-RPC 2.0.
 */

import { SessionManager } from './sessions.js';

const SERVER_INFO = {
  name: 'sand',
  version: '0.1.0',
};

const SERVER_INSTRUCTIONS = `Sand is a sandboxed JavaScript calculator. Use for ALL arithmetic - never attempt mental math.

CRITICAL RULES:
1. ONE statement per script call. Never combine multiple statements.
2. NO comments in code. Explain your reasoning in plain text to the user instead.
3. NO newlines or \\n in script strings.
4. Use 'get' to retrieve final results.

CORRECT EXAMPLE:
  "I'll calculate compound interest step by step."
  script: "let principal = 10000"
  script: "let rate = 0.05"
  script: "let years = 10"
  script: "let total = principal * Math.pow(1 + rate, years)"
  get: "total"
  "The result is 16288.95"

WRONG - never do this:
  script: "let x = 5 // comment\\nlet y = 10"

SYNTAX:
- Variables: let x = 1
- Math.*: sqrt, pow, sin, cos, tan, log, abs, floor, ceil, round, min, max, PI, E
- Operators: + - * / % ** === !== < > <= >= && || ! ?: ??
- Arrays: [1, 2, 3], arr[0], arr.length
- Objects: {a: 1}, obj.a
- Functions: function add(a,b) { return a+b } or (x) => x*2
- Control flow: if/else, while, for (use separate calls for each statement)`;

// Supported protocol versions (newest first)
const SUPPORTED_VERSIONS = ['2025-11-25', '2025-03-26', '2024-11-05'];
const DEFAULT_VERSION = '2024-11-05';

// Tool definitions
const TOOLS = [
  {
    name: 'script',
    description: `Execute ONE JavaScript statement. Variables persist across calls.

RULES:
- One statement per call (no newlines, no semicolon-separated statements)
- No comments in code - explain reasoning to user in plain text instead
- Call multiple times to build up calculations step by step

Example sequence:
  script("let width = 10")
  script("let height = 5")
  script("let area = width * height")
  get("area") -> 50`,
    inputSchema: {
      type: 'object',
      properties: {
        script: {
          type: 'string',
          description: 'JavaScript code to execute',
        },
      },
      required: ['script'],
    },
  },
  {
    name: 'get',
    description: 'Get a calculated result by variable name, or list all variables if no name given.',
    inputSchema: {
      type: 'object',
      properties: {
        name: {
          type: 'string',
          description: 'Variable name (omit to list all)',
        },
      },
    },
  },
  {
    name: 'step',
    description: 'Debug calculations step-by-step. Load code without running (fuel=0), or resume paused execution.',
    inputSchema: {
      type: 'object',
      properties: {
        code: {
          type: 'string',
          description: 'Code to load (omit to resume)',
        },
        fuel: {
          type: 'integer',
          description: 'Fuel to add (omit to run to completion)',
        },
      },
    },
  },
  {
    name: 'session',
    description: 'Manage sessions. Actions: "new", "use", "list", "rename", "delete", "export", "save", "save-as". Loaded sessions are read-only by default - use "save" to persist changes.',
    inputSchema: {
      type: 'object',
      properties: {
        action: {
          type: 'string',
          enum: ['new', 'use', 'list', 'rename', 'delete', 'export', 'save', 'save-as'],
          description: 'Action to perform',
        },
        name: {
          type: 'string',
          description: 'Session name (for new/use/rename/delete/export/save-as)',
        },
        path: {
          type: 'string',
          description: 'Destination path (for export)',
        },
      },
      required: ['action'],
    },
  },
  {
    name: 'inspect',
    description: 'Inspect interpreter state for debugging. What: "state" (default), "stack", "scope", "memory".',
    inputSchema: {
      type: 'object',
      properties: {
        what: {
          type: 'string',
          enum: ['state', 'stack', 'scope', 'memory'],
          description: 'What to inspect (default: state)',
        },
        address: {
          type: 'integer',
          description: 'Memory address (for memory inspection)',
        },
        length: {
          type: 'integer',
          description: 'Bytes to read (for memory inspection, default: 64)',
        },
      },
    },
  },
];

/**
 * MCP Server - handles JSON-RPC messages over stdio
 */
class McpServer {
  constructor() {
    this.sessions = new SessionManager();
    this.decoder = new TextDecoder();
    this.encoder = new TextEncoder();
  }

  /**
   * Send a JSON-RPC response
   */
  send(message) {
    const json = JSON.stringify(message);
    Deno.stdout.writeSync(this.encoder.encode(json + '\n'));
  }

  /**
   * Send a successful response
   */
  respond(id, result) {
    this.send({ jsonrpc: '2.0', id, result });
  }

  /**
   * Send an error response
   */
  respondError(id, code, message, data) {
    const error = { code, message };
    if (data !== undefined) {
      error.data = data;
    }
    this.send({ jsonrpc: '2.0', id, error });
  }

  /**
   * Handle incoming JSON-RPC request
   */
  async handleRequest(request) {
    const { id, method, params } = request;

    try {
      switch (method) {
        case 'initialize':
          return this.handleInitialize(id, params);

        case 'notifications/initialized':
          // No response needed for notifications
          return;

        case 'tools/list':
          return this.handleToolsList(id, params);

        case 'tools/call':
          return this.handleToolsCall(id, params);

        case 'ping':
          return this.respond(id, {});

        default:
          return this.respondError(id, -32601, `Method not found: ${method}`);
      }
    } catch (err) {
      return this.respondError(id, -32603, err.message);
    }
  }

  /**
   * Handle initialize request
   */
  async handleInitialize(id, params) {
    await this.sessions.init();

    // Negotiate protocol version - use client's version if we support it
    const clientVersion = params?.protocolVersion;
    let protocolVersion = DEFAULT_VERSION;
    if (clientVersion && SUPPORTED_VERSIONS.includes(clientVersion)) {
      protocolVersion = clientVersion;
    }

    this.respond(id, {
      protocolVersion,
      capabilities: {
        tools: {},
      },
      serverInfo: SERVER_INFO,
      instructions: SERVER_INSTRUCTIONS,
    });
  }

  /**
   * Handle tools/list request
   */
  handleToolsList(id, params) {
    this.respond(id, { tools: TOOLS });
  }

  /**
   * Handle tools/call request
   */
  async handleToolsCall(id, params) {
    const { name, arguments: args } = params;

    try {
      switch (name) {
        case 'script': {
          const { session } = await this.sessions.current();
          const result = await session.eval(args.script);
          this.sessions.markDirty();

          if (result.status === 'done') {
            return this.respond(id, {
              content: [{ type: 'text', text: 'ok' }],
            });
          }
          return this.respond(id, {
            content: [{ type: 'text', text: result.message || result.status }],
            isError: result.status !== 'paused',
          });
        }

        case 'get': {
          const { session } = await this.sessions.current();
          if (args.name) {
            const value = await session.readVar(args.name);
            const text = value === null ? 'undefined' : String(value);
            return this.respond(id, {
              content: [{ type: 'text', text }],
            });
          } else {
            const vars = await session.listVars();
            const text = vars.length ? vars.join(', ') : '(none)';
            return this.respond(id, {
              content: [{ type: 'text', text }],
            });
          }
        }

        case 'step': {
          const { session } = await this.sessions.current();
          let result;
          if (args.code) {
            result = await session.eval(args.code, args.fuel ?? 0);
          } else {
            result = await session.continue(args.fuel);
          }
          this.sessions.markDirty();
          return this.respond(id, {
            content: [{ type: 'text', text: result.status }],
            structuredContent: result,
          });
        }

        case 'session': {
          switch (args.action) {
            case 'new': {
              const result = await this.sessions.newSession(args.name);
              return this.respond(id, {
                content: [{ type: 'text', text: result.name }],
              });
            }
            case 'use': {
              if (!args.name) {
                return this.respond(id, {
                  content: [{ type: 'text', text: 'name required' }],
                  isError: true,
                });
              }
              const result = await this.sessions.useSession(args.name);
              const text = result.created ? `${result.name} (created)` : result.name;
              return this.respond(id, {
                content: [{ type: 'text', text }],
              });
            }
            case 'list': {
              const list = await this.sessions.listSessions();
              const text = list.length
                ? list.map(s => s.current ? `*${s.name}` : s.name).join(', ')
                : '(none)';
              return this.respond(id, {
                content: [{ type: 'text', text }],
              });
            }
            case 'rename': {
              if (!args.name) {
                return this.respond(id, {
                  content: [{ type: 'text', text: 'name required' }],
                  isError: true,
                });
              }
              const result = await this.sessions.renameSession(args.name);
              return this.respond(id, {
                content: [{ type: 'text', text: result.name }],
              });
            }
            case 'delete': {
              if (!args.name) {
                return this.respond(id, {
                  content: [{ type: 'text', text: 'name required' }],
                  isError: true,
                });
              }
              const result = await this.sessions.deleteSession(args.name);
              const text = result.deleted ? 'deleted' : 'not found';
              return this.respond(id, {
                content: [{ type: 'text', text }],
              });
            }
            case 'export': {
              if (!args.name) {
                return this.respond(id, {
                  content: [{ type: 'text', text: 'name required' }],
                  isError: true,
                });
              }
              if (!args.path) {
                return this.respond(id, {
                  content: [{ type: 'text', text: 'path required' }],
                  isError: true,
                });
              }
              const result = await this.sessions.exportSession(args.name, args.path);
              return this.respond(id, {
                content: [{ type: 'text', text: result.path }],
              });
            }
            case 'save': {
              const result = await this.sessions.saveCurrentSession();
              return this.respond(id, {
                content: [{ type: 'text', text: `saved ${result.name}` }],
              });
            }
            case 'save-as': {
              if (!args.name) {
                return this.respond(id, {
                  content: [{ type: 'text', text: 'name required' }],
                  isError: true,
                });
              }
              const result = await this.sessions.saveCurrentSessionAs(args.name);
              return this.respond(id, {
                content: [{ type: 'text', text: `saved as ${result.name}` }],
              });
            }
            default:
              return this.respond(id, {
                content: [{ type: 'text', text: 'unknown action' }],
                isError: true,
              });
          }
        }

        case 'inspect': {
          const { session } = await this.sessions.current();
          const what = args.what || 'state';
          const result = await session.inspect(what, {
            address: args.address,
            length: args.length,
          });
          // Format result as readable text
          let text;
          switch (what) {
            case 'state':
              text = `status: ${result.status}, instruction: ${result.instructionIndex}/${result.instructionCount}, callStack: ${result.callStackDepth}, fuel: ${result.fuel}`;
              break;
            case 'stack':
              text = `pending: [${result.pending.map(v => JSON.stringify(v)).join(', ')}], callStackDepth: ${result.callStackDepth}`;
              break;
            case 'scope':
              text = Object.entries(result.variables)
                .map(([k, v]) => `${k}: ${JSON.stringify(v)}`)
                .join('\n') || '(empty)';
              break;
            case 'memory':
              text = `@${result.address} (${result.length} bytes):\n${result.hex}`;
              break;
            default:
              text = JSON.stringify(result);
          }
          return this.respond(id, {
            content: [{ type: 'text', text }],
          });
        }

        default:
          return this.respondError(id, -32602, `Unknown tool: ${name}`);
      }
    } catch (err) {
      this.respond(id, {
        content: [{ type: 'text', text: err.message }],
        isError: true,
      });
    }
  }

  /**
   * Run the server - read from stdin, process messages
   */
  async run() {
    const reader = Deno.stdin.readable.getReader();
    let buffer = '';

    while (true) {
      const { done, value } = await reader.read();
      if (done) {
        // Save all sessions before exit
        await this.sessions.saveAll();
        break;
      }

      buffer += this.decoder.decode(value, { stream: true });

      // Process complete lines
      let newlineIndex;
      while ((newlineIndex = buffer.indexOf('\n')) !== -1) {
        const line = buffer.slice(0, newlineIndex).trim();
        buffer = buffer.slice(newlineIndex + 1);

        if (line) {
          try {
            const request = JSON.parse(line);
            await this.handleRequest(request);
          } catch (err) {
            // JSON parse error
            this.respondError(null, -32700, 'Parse error');
          }
        }
      }
    }
  }
}

// =============================================================================
// CLI: --install, --uninstall
// =============================================================================

const CALC_SKILL = `---
trigger: User asks to calculate, compute, or figure out any arithmetic, percentages, conversions, taxes, tips, interest, or multi-step math. Also triggers on "/calc".
---

# Calculator Mode

Use the sand MCP tools for this calculation. Do NOT attempt mental math.

## Steps

1. Break down the problem into variables
2. Use \`script\` for each statement separately:
   - \`let income = 85000\`
   - \`let deductions = 12000\`
   - \`let taxable = income - deductions\`
   - \`let tax = taxable * 0.22\`
3. Use \`get\` to retrieve the final result
4. Present the answer clearly

## Available

**Arithmetic:** \`+\`, \`-\`, \`*\`, \`/\`, \`%\`, \`**\`

**Math:** abs, floor, ceil, round, trunc, sign, min, max, pow, sqrt, cbrt, hypot, sin, cos, tan, asin, acos, atan, atan2, sinh, cosh, tanh, asinh, acosh, atanh, log, log10, log2, log1p, exp, expm1, clz32, imul, fround

**Math constants:** PI, E, LN2, LN10, LOG2E, LOG10E, SQRT2, SQRT1_2

**Arrays:** push, pop, shift, unshift, slice, concat, join, reverse, indexOf, includes, map, filter, reduce, forEach, find, findIndex, some, every, Array.isArray

**Strings:** charAt, charCodeAt, indexOf, includes, slice, substring, split, trim, toLowerCase, toUpperCase, startsWith, endsWith, repeat, padStart, padEnd, replace, String.fromCharCode, String.fromCodePoint

**Objects:** Object.keys, Object.values, Object.entries, Object.assign

**Numbers:** Number.isNaN, Number.isFinite, Number.isInteger, Number.parseInt, Number.parseFloat, MAX_VALUE, MIN_VALUE, MAX_SAFE_INTEGER, MIN_SAFE_INTEGER, EPSILON

**Globals:** parseInt, parseFloat, isNaN, isFinite, Boolean, Infinity, NaN, undefined

**Variables persist across calls.**

## Examples

"What's the monthly payment on a $450k mortgage at 6.5% for 30 years?"
→ Break into steps: principal, monthly rate, number of payments, then the amortization formula

"If I invest $500/month for 25 years at 7% annual return, what do I end up with?"
→ Future value of annuity calculation with compounding

"I have 3 invoices: $1,847.33, $923.17, $2,104.89. Apply 2.5% early payment discount to each, then add 8.25% tax to the total."
→ Multiple steps where rounding errors accumulate

"What's the great circle distance between two lat/long coordinates?"
→ Haversine formula with trig functions

"Convert this recipe from 6 servings to 17 servings: 2.5 cups flour, 1.75 cups sugar, 0.875 cups butter"
→ Awkward fractions that multiply into ugly decimals

"My portfolio is 40% stocks (up 12.3%), 35% bonds (down 2.1%), 25% crypto (up 47.8%). What's my overall return?"
→ Weighted average with mixed positive/negative percentages

"What angle should I cut a board to make a regular octagon?"
→ Interior angle formula: (n-2) * 180 / n, then halve for miter cut

"If a project is 34% complete after 47 days, and the deadline is in 89 more days, will we make it at this pace?"
→ Rate extrapolation with percentages and days
`;

// IDE configuration definitions
const IDE_CONFIGS = {
  'claude-code': {
    configPath: '~/.claude/mcp.json',
    key: 'mcpServers',
    format: 'json-standard',
    skillPath: '~/.claude/skills/sand-calc/SKILL.md',
  },
  'cursor': {
    configPath: '~/.cursor/mcp.json',
    key: 'mcpServers',
    format: 'json-standard',
  },
  'claude-desktop': {
    configPath: '~/Library/Application Support/Claude/claude_desktop_config.json',
    key: 'mcpServers',
    format: 'json-standard',
  },
  'windsurf': {
    configPath: '~/.codeium/windsurf/mcp_config.json',
    key: 'mcpServers',
    format: 'json-standard',
    skillPath: '~/.codeium/windsurf/skills/sand-calc/SKILL.md',
  },
  'vscode': {
    configPath: '~/.vscode/mcp.json',
    key: 'servers',
    format: 'json-vscode',
  },
  'lm-studio': {
    configPath: '~/Library/Application Support/LM Studio/mcp.json',
    key: 'mcpServers',
    format: 'json-standard',
  },
  'opencode': {
    configPath: '~/.config/opencode/opencode.json',
    key: 'mcp',
    format: 'json-opencode',
  },
  'codex': {
    configPath: '~/.codex/config.toml',
    key: 'mcp_servers',
    format: 'toml',
  },
  'zed': {
    configPath: '~/.config/zed/settings.json',
    key: 'context_servers',
    format: 'json-zed',
  },
  'copilot': {
    configPath: '~/.copilot/mcp-config.json',
    key: 'mcpServers',
    format: 'json-standard',
  },
};

/**
 * Expand ~ to home directory
 */
function expandPath(path) {
  const home = Deno.env.get('HOME') || Deno.env.get('USERPROFILE') || '.';
  if (path.startsWith('~/')) {
    return home + path.slice(1);
  }
  return path;
}

/**
 * Get directory from path
 */
function dirname(path) {
  const lastSlash = path.lastIndexOf('/');
  return lastSlash > 0 ? path.slice(0, lastSlash) : '.';
}

/**
 * Prompt user with Y/n (Enter defaults to Yes)
 * Reads from /dev/tty to work when stdin is piped (curl | sh)
 */
async function promptYesNo(question) {
  // Write prompt to stderr (stdout may be redirected)
  const encoder = new TextEncoder();
  await Deno.stderr.write(encoder.encode(`${question} [Y/n] `));

  // Try to read from /dev/tty for interactive input even when piped
  let answer = '';
  try {
    const tty = await Deno.open('/dev/tty', { read: true });
    const buf = new Uint8Array(100);
    const n = await tty.read(buf);
    tty.close();
    if (n === null) return false;
    answer = new TextDecoder().decode(buf.subarray(0, n)).trim();
  } catch {
    // No TTY available (non-interactive), default to yes
    await Deno.stderr.write(encoder.encode('(auto: y)\n'));
    return true;
  }

  // Empty or y/Y = yes
  return answer === '' || answer.toLowerCase() === 'y';
}

/**
 * Detect which IDEs are installed by checking for config directories or CLI
 */
async function detectInstalledIDEs() {
  const detected = [];

  for (const [name, ide] of Object.entries(IDE_CONFIGS)) {
    // Claude Code: check if CLI exists
    if (name === 'claude-code') {
      try {
        const cmd = new Deno.Command('claude', {
          args: ['--version'],
          stdout: 'null',
          stderr: 'null',
        });
        const result = await cmd.output();
        if (result.success) {
          detected.push(name);
        }
      } catch {
        // CLI not found
      }
      continue;
    }

    // Other IDEs: check config directory
    const configPath = expandPath(ide.configPath);
    const configDir = dirname(configPath);

    try {
      await Deno.stat(configDir);
      detected.push(name);
    } catch {
      // Directory doesn't exist
    }
  }

  return detected;
}

/**
 * Write standard mcpServers format
 */
function writeStandardConfig(config, serverName, serverConfig) {
  if (!config.mcpServers) config.mcpServers = {};
  config.mcpServers[serverName] = serverConfig;
  return config;
}

/**
 * Write VS Code format (servers key)
 */
function writeVSCodeConfig(config, serverName, serverConfig) {
  if (!config.servers) config.servers = {};
  config.servers[serverName] = serverConfig;
  return config;
}

/**
 * Write OpenCode format (mcp key, type field)
 */
function writeOpenCodeConfig(config, serverName, binPath) {
  if (!config.mcp) config.mcp = {};
  config.mcp[serverName] = {
    type: 'local',
    command: [binPath, '--mcp'],
    enabled: true,
  };
  return config;
}

/**
 * Write Zed format (context_servers key, nested command)
 */
function writeZedConfig(config, serverName, binPath) {
  if (!config.context_servers) config.context_servers = {};
  config.context_servers[serverName] = {
    command: {
      path: binPath,
      args: ['--mcp'],
    },
  };
  return config;
}

/**
 * Install to a specific IDE
 */
async function installToIDE(target, binPath) {
  const ide = IDE_CONFIGS[target];
  if (!ide) {
    throw new Error(`Unknown target: ${target}`);
  }

  // Claude Code uses its own CLI for MCP management
  if (target === 'claude-code') {
    return await installToClaudeCode(binPath);
  }

  const configPath = expandPath(ide.configPath);

  // Handle TOML (Codex) separately
  if (ide.format === 'toml') {
    return await installToCodex(binPath, configPath);
  }

  // Create directory if needed
  await Deno.mkdir(dirname(configPath), { recursive: true });

  // Read existing config or create empty
  let config = {};
  try {
    const content = await Deno.readTextFile(configPath);
    config = JSON.parse(content);
  } catch {
    // File doesn't exist or invalid JSON
  }

  // Add sand config based on format
  const serverConfig = {
    command: binPath,
    args: ['--mcp'],
  };

  switch (ide.format) {
    case 'json-standard':
      config = writeStandardConfig(config, 'sand', serverConfig);
      break;
    case 'json-vscode':
      config = writeVSCodeConfig(config, 'sand', serverConfig);
      break;
    case 'json-opencode':
      config = writeOpenCodeConfig(config, 'sand', binPath);
      break;
    case 'json-zed':
      config = writeZedConfig(config, 'sand', binPath);
      break;
  }

  // Write config
  await Deno.writeTextFile(configPath, JSON.stringify(config, null, 2) + '\n');

  return { target, configPath };
}

/**
 * Install to Claude Code using its CLI
 */
async function installToClaudeCode(binPath) {
  // First remove any existing sand server
  const removeCmd = new Deno.Command('claude', {
    args: ['mcp', 'remove', 'sand'],
    stdout: 'null',
    stderr: 'null',
  });
  await removeCmd.output(); // Ignore errors if not present

  // Add the server using claude CLI
  const addCmd = new Deno.Command('claude', {
    args: ['mcp', 'add', '--transport', 'stdio', '--scope', 'user', 'sand', '--', binPath, '--mcp'],
    stdout: 'piped',
    stderr: 'piped',
  });
  const result = await addCmd.output();

  if (!result.success) {
    const stderr = new TextDecoder().decode(result.stderr);
    throw new Error(`claude mcp add failed: ${stderr}`);
  }

  return { target: 'claude-code', configPath: '~/.claude.json' };
}

/**
 * Install to Codex (TOML format - append only)
 */
async function installToCodex(binPath, configPath) {
  // Create directory if needed
  await Deno.mkdir(dirname(configPath), { recursive: true });

  let content = '';
  try {
    content = await Deno.readTextFile(configPath);
  } catch {
    // File doesn't exist
  }

  // Remove existing section if present (for updates)
  content = content.replace(/\n?\[mcp_servers\.sand\][^\[]*/, '');

  // Append config
  const section = `
[mcp_servers.sand]
command = ["${binPath}", "--mcp"]
`;

  await Deno.writeTextFile(configPath, content.trimEnd() + '\n' + section);
  return { target: 'codex', configPath };
}

/**
 * Install skill to a path
 */
async function installSkill(skillPath) {
  const fullPath = expandPath(skillPath);
  await Deno.mkdir(dirname(fullPath), { recursive: true });
  await Deno.writeTextFile(fullPath, CALC_SKILL);
  return { skillPath: fullPath };
}

/**
 * Main install function
 */
async function install(explicitTarget) {
  const binPath = Deno.execPath();
  const errors = [];
  const installed = [];

  let targets;
  if (explicitTarget) {
    // Explicit target - no prompts
    if (!IDE_CONFIGS[explicitTarget]) {
      console.error(`Unknown target: ${explicitTarget}`);
      console.error(`Available: ${Object.keys(IDE_CONFIGS).join(', ')}`);
      Deno.exit(1);
    }
    targets = [explicitTarget];
  } else {
    // Auto-detect and prompt
    const detected = await detectInstalledIDEs();
    if (detected.length === 0) {
      console.log('No supported IDEs detected.');
      return;
    }

    console.log('Detected IDEs:\n');
    targets = [];
    for (const ide of detected) {
      if (await promptYesNo(`Install to ${ide}?`)) {
        targets.push(ide);
      } else {
        console.log('  - skipped\n');
      }
    }
  }

  if (targets.length === 0) {
    console.log('\nNo IDEs selected.');
    return;
  }

  // Install to each target
  for (const target of targets) {
    try {
      const result = await installToIDE(target, binPath);
      console.log(`  ✓ ${target} (${result.configPath})`);
      installed.push(target);

      // Install skill if supported
      const ide = IDE_CONFIGS[target];
      if (ide.skillPath) {
        const skillResult = await installSkill(ide.skillPath);
        console.log(`  ✓ skill: /calc (${skillResult.skillPath})`);
      }
      console.log('');
    } catch (err) {
      errors.push({ target, error: err.message });
      console.log(`  ✗ ${target}: ${err.message}\n`);
    }
  }

  // Report errors at end
  if (errors.length > 0) {
    console.log('Errors:');
    for (const { target, error } of errors) {
      console.log(`  ✗ ${target}: ${error}`);
    }
    console.log('');
  }

  if (installed.length > 0) {
    console.log('Done. Restart your IDEs to use sand.');
  }
}

/**
 * Uninstall from a specific IDE
 */
async function uninstallFromIDE(target) {
  const ide = IDE_CONFIGS[target];
  if (!ide) {
    throw new Error(`Unknown target: ${target}`);
  }

  // Claude Code uses its own CLI
  if (target === 'claude-code') {
    const cmd = new Deno.Command('claude', {
      args: ['mcp', 'remove', 'sand'],
      stdout: 'piped',
      stderr: 'piped',
    });
    const result = await cmd.output();
    return { target, configPath: '~/.claude.json', removed: result.success };
  }

  const configPath = expandPath(ide.configPath);

  // Handle TOML (Codex) - print manual instructions
  if (ide.format === 'toml') {
    console.log(`  To uninstall from ${target}, remove the [mcp_servers.sand] section from ${configPath}`);
    return { target, manual: true };
  }

  // Read existing config
  let config;
  try {
    const content = await Deno.readTextFile(configPath);
    config = JSON.parse(content);
  } catch {
    return { target, notFound: true };
  }

  // Remove sand from appropriate key
  let removed = false;
  switch (ide.format) {
    case 'json-standard':
      if (config.mcpServers?.sand) {
        delete config.mcpServers.sand;
        removed = true;
      }
      break;
    case 'json-vscode':
      if (config.servers?.sand) {
        delete config.servers.sand;
        removed = true;
      }
      break;
    case 'json-opencode':
      if (config.mcp?.sand) {
        delete config.mcp.sand;
        removed = true;
      }
      break;
    case 'json-zed':
      if (config.context_servers?.sand) {
        delete config.context_servers.sand;
        removed = true;
      }
      break;
  }

  if (removed) {
    await Deno.writeTextFile(configPath, JSON.stringify(config, null, 2) + '\n');
  }

  // Remove skill if present
  if (ide.skillPath) {
    const skillDir = dirname(expandPath(ide.skillPath));
    try {
      await Deno.remove(skillDir, { recursive: true });
    } catch {
      // Skill not found
    }
  }

  return { target, removed, configPath };
}

/**
 * Main uninstall function
 */
async function uninstall(explicitTarget) {
  const errors = [];

  let targets;
  if (explicitTarget) {
    // Explicit target
    if (!IDE_CONFIGS[explicitTarget]) {
      console.error(`Unknown target: ${explicitTarget}`);
      console.error(`Available: ${Object.keys(IDE_CONFIGS).join(', ')}`);
      Deno.exit(1);
    }
    targets = [explicitTarget];
  } else {
    // Auto-detect and prompt
    const detected = await detectInstalledIDEs();
    if (detected.length === 0) {
      console.log('No supported IDEs detected.');
      return;
    }

    console.log('Detected IDEs:\n');
    targets = [];
    for (const ide of detected) {
      if (await promptYesNo(`Uninstall from ${ide}?`)) {
        targets.push(ide);
      } else {
        console.log('  - skipped\n');
      }
    }
  }

  if (targets.length === 0) {
    console.log('\nNo IDEs selected.');
    return;
  }

  // Uninstall from each target
  for (const target of targets) {
    try {
      const result = await uninstallFromIDE(target);
      if (result.manual) {
        // Already printed instructions
      } else if (result.notFound) {
        console.log(`  - ${target}: config not found`);
      } else if (result.removed) {
        console.log(`  ✓ ${target} (${result.configPath})`);
      } else {
        console.log(`  - ${target}: sand not found in config`);
      }
      console.log('');
    } catch (err) {
      errors.push({ target, error: err.message });
      console.log(`  ✗ ${target}: ${err.message}\n`);
    }
  }

  // Note about sessions and binary
  const home = Deno.env.get('HOME') || Deno.env.get('USERPROFILE');
  const sessionsDir = `${home}/.sandscript`;
  const binPath = Deno.execPath();
  console.log(`Session data remains at ${sessionsDir}`);
  console.log(`Binary remains at ${binPath}`);
  console.log('Remove manually if no longer needed.');
}

/**
 * Run the MCP server (exported for CLI integration)
 */
export async function runMcpServer() {
  const server = new McpServer();
  await server.run();
}

// Also export install functions for CLI
export { install, uninstall };
