/**
 * Sand CLI
 *
 * Multi-modal CLI for SandScript. Docker-style subcommands.
 * REPL is default, --mcp for MCP server mode.
 */

import { runMcpServer, install, uninstall } from './mcp.js';
import { SessionManager } from './sessions.js';

const VERSION = '0.1.0';

// =============================================================================
// Shared session manager (persists across commands in same process)
// =============================================================================

let sessionManager = null;

async function getSessionManager() {
  if (!sessionManager) {
    sessionManager = new SessionManager();
    await sessionManager.init();
  }
  return sessionManager;
}

// =============================================================================
// Commands
// =============================================================================

async function scriptCommand(args, flags) {
  const code = args[0];
  if (!code) {
    console.error('Usage: sand script "code"');
    Deno.exit(1);
  }

  const sessions = await getSessionManager();
  const { session } = await sessions.current();
  const result = await session.eval(code);

  if (result.status === 'syntax_error' || result.status === 'runtime_error') {
    console.error(result.message);
    Deno.exit(1);
  }

  sessions.markDirty();

  if (flags.verbose) {
    await printScope(session);
  }

  if (flags.save) {
    await sessions.saveCurrentSession();
  }
}

async function getCommand(args, flags) {
  const sessions = await getSessionManager();
  const { session } = await sessions.current();

  if (args.length === 0) {
    // List all variables
    const vars = await session.listVars();
    if (vars.length === 0) {
      console.log('(no variables)');
    } else {
      console.log(vars.join(', '));
    }
  } else {
    // Get specific variable
    const name = args[0];
    const value = await session.readVar(name);
    if (value === null) {
      console.error(`undefined: ${name}`);
      Deno.exit(1);
    }
    console.log(formatValue(value));
  }
}

async function stepCommand(args, flags) {
  const sessions = await getSessionManager();
  const { session } = await sessions.current();

  if (args.length > 0 && isNaN(parseInt(args[0]))) {
    // First arg is code to load
    const code = args[0];
    const result = await session.eval(code, 0); // fuel=0 means load only

    if (result.status === 'syntax_error') {
      console.error(result.message);
      Deno.exit(1);
    }

    console.log('loaded');
    sessions.markDirty();
  } else {
    // Resume execution
    const fuel = args.length > 0 ? parseInt(args[0]) : undefined;
    const result = await session.continue(fuel);

    console.log(JSON.stringify({ status: result.status, fuelUsed: result.fuelUsed }));
    sessions.markDirty();
  }
}

async function inspectCommand(args, flags) {
  const sessions = await getSessionManager();
  const { session } = await sessions.current();

  const what = args[0] || 'state';
  const options = {};

  if (what === 'memory' && args[1]) {
    options.address = parseInt(args[1], 16) || parseInt(args[1]);
    options.length = args[2] ? parseInt(args[2]) : 64;
  }

  const result = await session.inspect(what, options);

  switch (what) {
    case 'state':
      console.log(`status: ${result.status}`);
      console.log(`instruction: ${result.instructionIndex}/${result.instructionCount}`);
      console.log(`callStack: ${result.callStackDepth}`);
      console.log(`fuel: ${result.fuel}`);
      break;

    case 'scope':
      await printScope(session);
      break;

    case 'stack':
      console.log(`pending: [${result.pending.map(v => JSON.stringify(v)).join(', ')}]`);
      console.log(`callStackDepth: ${result.callStackDepth}`);
      break;

    case 'memory':
      console.log(`@${result.address} (${result.length} bytes):`);
      console.log(result.hex);
      break;

    default:
      console.log(JSON.stringify(result));
  }
}

async function sessionCommand(args, flags) {
  const [subcommand, ...rest] = args;
  const sessions = await getSessionManager();

  switch (subcommand) {
    case 'list': {
      const list = await sessions.listSessions();
      if (list.length === 0) {
        console.log('(no sessions)');
      } else {
        for (const s of list) {
          const marker = s.current ? '* ' : '  ';
          console.log(`${marker}${s.name}`);
        }
      }
      break;
    }

    case 'use': {
      const name = rest[0];
      if (!name) {
        console.error('Usage: sand session use <name>');
        Deno.exit(1);
      }
      const result = await sessions.useSession(name);
      if (result.created) {
        console.log(`created ${result.name}`);
      } else {
        console.log(`switched to ${result.name}`);
      }
      break;
    }

    case 'new': {
      const name = rest[0];
      const result = await sessions.newSession(name);
      console.log(`created ${result.name}`);
      break;
    }

    case 'save': {
      const result = await sessions.saveCurrentSession();
      console.log(`saved ${result.name}`);
      break;
    }

    case 'save-as': {
      const name = rest[0];
      if (!name) {
        console.error('Usage: sand session save-as <name>');
        Deno.exit(1);
      }
      const result = await sessions.saveCurrentSessionAs(name);
      console.log(`saved as ${result.name}`);
      break;
    }

    case 'rename': {
      const name = rest[0];
      if (!name) {
        console.error('Usage: sand session rename <name>');
        Deno.exit(1);
      }
      const result = await sessions.renameSession(name);
      console.log(`renamed to ${result.name}`);
      break;
    }

    case 'delete': {
      const name = rest[0];
      if (!name) {
        console.error('Usage: sand session delete <name>');
        Deno.exit(1);
      }
      const result = await sessions.deleteSession(name);
      if (result.deleted) {
        console.log(`deleted ${name}`);
      } else {
        console.log(`not found: ${name}`);
      }
      break;
    }

    case 'export': {
      const path = rest[0];
      if (!path) {
        console.error('Usage: sand session export <path>');
        Deno.exit(1);
      }
      const name = sessions.getCurrentName();
      const result = await sessions.exportSession(name, path);
      console.log(`exported to ${result.path}`);
      break;
    }

    default:
      console.error('Usage: sand session <list|use|new|save|save-as|rename|delete|export>');
      Deno.exit(1);
  }
}

// =============================================================================
// REPL
// =============================================================================

async function runRepl() {
  const sessions = await getSessionManager();

  console.log('sand REPL (Ctrl+D or /exit to quit, /help for commands)');

  while (true) {
    const line = prompt('sand>');
    if (line === null) {
      // EOF (Ctrl+D)
      await sessions.saveAll();
      break;
    }

    const trimmed = line.trim();
    if (trimmed === '') continue;

    try {
      if (trimmed.startsWith('/')) {
        // Command mode
        const cmdLine = trimmed.slice(1);

        // Handle REPL-specific commands
        if (cmdLine === 'exit' || cmdLine === 'quit' || cmdLine === 'q') {
          await sessions.saveAll();
          break;
        }

        if (cmdLine === 'help' || cmdLine === 'h' || cmdLine === '?') {
          printReplHelp();
          continue;
        }

        const args = parseCommandLine(cmdLine);
        await runCommand(args);
      } else {
        // Implicit script
        await runCommand(['script', trimmed]);
      }
    } catch (err) {
      console.error(err.message);
    }
  }
}

function printReplHelp() {
  console.log(`
REPL Commands (prefix with /):
  /get [name]           Get variable value (or list all)
  /inspect [what]       Inspect state/scope/stack/memory
  /step ["code"]        Load code without running
  /step [fuel]          Resume with fuel limit
  /session list         List sessions
  /session use <name>   Switch session
  /session new [name]   New session
  /session save         Save current session
  /session save-as <n>  Fork to new name
  /exit, /quit, /q      Exit REPL
  /help, /h, /?         This help

Code runs directly (no / prefix):
  let x = 10
  let y = x * 2
`);
}

// =============================================================================
// File runner
// =============================================================================

async function runFile(path, flags) {
  let code = await Deno.readTextFile(path);

  // Strip shebang if present
  if (code.startsWith('#!')) {
    const newlineIndex = code.indexOf('\n');
    if (newlineIndex !== -1) {
      code = code.slice(newlineIndex + 1);
    }
  }

  const sessions = await getSessionManager();
  const { session } = await sessions.current();
  const result = await session.eval(code);

  if (result.status === 'syntax_error' || result.status === 'runtime_error') {
    console.error(result.message);
    Deno.exit(1);
  }

  sessions.markDirty();

  if (flags.verbose) {
    await printScope(session);
  }
}

// =============================================================================
// Helpers
// =============================================================================

function formatValue(value) {
  if (value === null || value === undefined) {
    return String(value);
  }
  if (typeof value === 'object') {
    return JSON.stringify(value);
  }
  return String(value);
}

async function printScope(session) {
  const vars = await session.listVars();
  for (const name of vars) {
    const value = await session.readVar(name);
    console.log(`${name} = ${formatValue(value)}`);
  }
}

function parseCommandLine(line) {
  // Simple arg parser: split on spaces, respecting quotes
  const args = [];
  let current = '';
  let inQuote = false;
  let quoteChar = '';

  for (const char of line) {
    if (inQuote) {
      if (char === quoteChar) {
        inQuote = false;
      } else {
        current += char;
      }
    } else {
      if (char === '"' || char === "'") {
        inQuote = true;
        quoteChar = char;
      } else if (char === ' ') {
        if (current) {
          args.push(current);
          current = '';
        }
      } else {
        current += char;
      }
    }
  }

  if (current) {
    args.push(current);
  }

  return args;
}

function parseFlags(args) {
  const flags = {};
  const positional = [];

  for (const arg of args) {
    if (arg === '--verbose' || arg === '-v') {
      flags.verbose = true;
    } else if (arg === '--save') {
      flags.save = true;
    } else if (arg.startsWith('--')) {
      // Unknown flag, ignore for now
    } else if (arg.startsWith('-')) {
      // Unknown short flag, ignore for now
    } else {
      positional.push(arg);
    }
  }

  return { flags, positional };
}

// =============================================================================
// Command dispatch
// =============================================================================

async function runCommand(args) {
  const [command, ...rest] = args;
  const { flags, positional } = parseFlags(rest);

  switch (command) {
    case 'script':
      await scriptCommand(positional, flags);
      break;

    case 'get':
      await getCommand(positional, flags);
      break;

    case 'step':
      await stepCommand(positional, flags);
      break;

    case 'inspect':
      await inspectCommand(positional, flags);
      break;

    case 'session':
      await sessionCommand(positional, flags);
      break;

    default:
      // Check if it's a file
      if (command && (command.endsWith('.ss') || command.endsWith('.sand') || await fileExists(command))) {
        await runFile(command, flags);
      } else {
        console.error(`Unknown command: ${command}`);
        console.error('Run "sand --help" for usage');
        Deno.exit(1);
      }
  }
}

async function fileExists(path) {
  try {
    await Deno.stat(path);
    return true;
  } catch {
    return false;
  }
}

// =============================================================================
// Help
// =============================================================================

function printHelp() {
  console.log(`sand - SandScript CLI

Usage:
  sand                              REPL mode (default)
  sand script "code"                Run code
  sand get [name]                   Get variable (or list all)
  sand step ["code"]                Load code or resume execution
  sand step [fuel]                  Resume with fuel limit
  sand inspect [what]               Inspect state/scope/stack/memory
  sand session <cmd>                Manage sessions
  sand <file.ss>                    Run file

Session commands:
  sand session list                 List all sessions
  sand session use <name>           Switch to session (creates if needed)
  sand session new [name]           Create new session
  sand session save                 Save current session
  sand session save-as <name>       Fork to new name
  sand session rename <name>        Rename current session
  sand session delete <name>        Delete a session
  sand session export <path>        Export to file

Flags:
  --verbose, -v                     Print scope after execution
  --save                            Save session after execution
  --mcp                             Run as MCP server
  --install [target]                Install MCP server (auto-detect or specify target)
  --uninstall [target]              Remove MCP config (auto-detect or specify target)
  --version                         Show version
  --help, -h                        Show this help

Install targets:
  claude-code, cursor, claude-desktop, windsurf, vscode,
  lm-studio, opencode, codex, zed, copilot

Examples:

  Quick calculation (implicit session, auto-persists):
    $ sand script "let income = 85000"
    $ sand script "let tax = income * 0.22"
    $ sand get tax
    18700

  Named session for a project:
    $ sand session new taxes-2024
    $ sand script "let income = 120000"
    $ sand script "let deductions = 15000"
    $ sand script "let taxable = income - deductions"
    $ sand script "let rate = 0.24"
    $ sand script "let owed = taxable * rate"
    $ sand get owed
    25200
    $ sand session save-as taxes-2024-final

  Load and modify a saved session (read-only by default):
    $ sand session use taxes-2024
    $ sand script "let scenario = taxable * 0.30"  # Temporary, not saved
    $ sand get scenario
    31500
    $ sand session save                            # Explicitly save if wanted

  Inspect interpreter state:
    $ sand inspect              # status, instruction pointer, fuel
    $ sand inspect scope        # all variables and values
    $ sand inspect stack        # pending stack (during execution)
    $ sand inspect memory 0x100 # raw memory dump

  Step-by-step debugging:
    $ sand step "let a = 1; let b = 2; let c = a + b"
    loaded
    $ sand inspect
    status: paused, instruction: 0/8, fuel: 0
    $ sand step 3               # execute 3 instructions
    {"status":"paused","fuelUsed":3}
    $ sand step                 # run to completion
    {"status":"done","fuelUsed":...}
    $ sand get c
    3

  Run a file:
    $ sand calc.ss --verbose    # run file, show all variables after

REPL:
  In REPL, code runs directly. Use / to escape to commands.
  Exit with Ctrl+D or /exit.

    $ sand
    sand> let x = 10
    sand> let y = x * 2
    sand> /get y
    20
    sand> /inspect scope
    x = 10
    y = 20
    sand> /session save-as mywork
    sand> /exit
`);
}

function printVersion() {
  console.log(`sand ${VERSION}`);
}

// =============================================================================
// Main
// =============================================================================

const args = Deno.args;

if (args.includes('--mcp')) {
  await runMcpServer();
} else if (args.includes('--install')) {
  // Get explicit target if provided (e.g., --install cursor)
  const installIndex = args.indexOf('--install');
  const target = args[installIndex + 1];
  // Don't treat flags as targets
  const explicitTarget = target && !target.startsWith('-') ? target : undefined;
  await install(explicitTarget);
} else if (args.includes('--uninstall')) {
  // Get explicit target if provided (e.g., --uninstall cursor)
  const uninstallIndex = args.indexOf('--uninstall');
  const target = args[uninstallIndex + 1];
  const explicitTarget = target && !target.startsWith('-') ? target : undefined;
  await uninstall(explicitTarget);
} else if (args.includes('--version')) {
  printVersion();
} else if (args.includes('--help') || args.includes('-h')) {
  printHelp();
} else if (args.length === 0) {
  await runRepl();
} else {
  await runCommand(args);
  // Auto-save new sessions
  if (sessionManager) {
    await sessionManager.saveAll();
  }
}
