/**
 * SandScript Fuel Session API
 *
 * Debug-oriented session wrapper with explicit FFI control.
 * Provides stepwise execution via fuel budgeting.
 */

import { MemoryManipulator } from './manipulator.js';
import { Orchestrator } from './orchestrator.js';
import { Parser } from './parser.js';
import { Collector } from './collector.js';
import { instantiate } from './interpreter.wasm.js';
import {
  STATUS_RUNNING,
  STATUS_DONE,
  STATUS_FFI_REQUEST,
  STATUS_ERROR,
  statusToString,
  errorCodeToString,
  opcodeToString,
} from './constants.js';

/**
 * Create a new SandScript session.
 *
 * @param {Object} options
 * @param {WebAssembly.Memory} options.memory - Existing memory to attach to (segment model)
 * @param {number} options.offset - Offset within memory for this segment
 * @param {number} options.memorySize - Size of memory to create if not provided (default 1MB)
 * @param {Object} options.ffi - Initial FFI bindings { name: { method: fn, ... }, ... }
 * @returns {Promise<Object>} Session object
 */
export async function createSession(options = {}) {
  const {
    memory = null,
    offset = 0,
    memorySize = 1024 * 1024,
    ffi = {},
  } = options;

  const mem = memory
    ? new MemoryManipulator(memory, offset)
    : new MemoryManipulator(
        new WebAssembly.Memory({ initial: Math.ceil(memorySize / 65536) }),
        0
      );

  const wasm = await instantiate(mem.memory);
  mem.setWasmInstance(wasm);
  mem.initialize();

  const orchestrator = new Orchestrator(mem, wasm);
  const parser = new Parser(mem);
  const collector = new Collector(mem);

  // Track FFI bindings for reset
  const ffiBindings = { ...ffi };
  for (const [name, impl] of Object.entries(ffi)) {
    orchestrator.bindFFI(name, impl);
  }

  // Track current code block (reused across parse calls)
  let codeBlock = null;
  let hasParsed = false;

  return {
    /**
     * Parse source code and prepare for execution.
     * Appends to existing code block to preserve closure instruction indices.
     *
     * @param {string} source - SandScript source code
     * @returns {Object} { instructions, complete, error, startIndex }
     */
    parse(source) {
      // Allocate code block on first parse, then append
      if (codeBlock === null) {
        codeBlock = mem.allocateCodeBlock();
      }

      // Remember where new code starts (for running just this snippet)
      const startIndex = mem.codeBlockInstructionCount(codeBlock);

      mem.setInstructionIndex(startIndex);
      mem.setStatus(STATUS_RUNNING);

      let error = null;
      try {
        parser.parse(source, codeBlock);
        parser.complete();
        hasParsed = true;
      } catch (e) {
        error = e;
      }

      return {
        instructions: mem.codeBlockInstructionCount(codeBlock),
        startIndex,
        complete: error === null,
        error: error?.message ?? null,
      };
    },

    /**
     * Run until fuel exhausted, done, FFI request, or error.
     * If already done/error, returns current status (no-op).
     *
     * @param {number} fuel - Fuel budget (default 10000)
     * @returns {Object} { status, instruction, fuelRemaining, request?, error? }
     */
    run(fuel = 10000) {
      const status = wasm.exports.run(fuel);
      const result = {
        status: statusToString(status),
        instruction: mem.getInstructionIndex(),
        fuelRemaining: mem.getFuel(),
      };

      if (status === STATUS_FFI_REQUEST) {
        const req = mem.getFFIRequest();
        result.request = {
          refId: req.refId,
          namespace: orchestrator.refNames[req.refId] ?? '',
          method: mem.readString(req.methodOffset),
          args: orchestrator.extractArgs(req.argsPointer, req.argCount)
            .map(arg => arg.value),
        };
      }

      if (status === STATUS_ERROR) {
        const err = mem.getErrorInfo();
        result.error = {
          code: errorCodeToString(err.code),
          detail: err.detail,
        };
      }

      return result;
    },

    /**
     * Resume execution after FFI pause, pushing the return value.
     *
     * @param {*} value - Return value from FFI call
     */
    resumeWithFFIResult(value) {
      orchestrator.pushFFIResult(value);
      mem.setStatus(STATUS_RUNNING);
    },

    /**
     * Resume execution after FFI pause, setting error state.
     *
     * @param {Error} error - Error to throw
     */
    resumeWithFFIError(error) {
      orchestrator.pushFFIError(error);
    },

    /**
     * Get current execution state snapshot.
     *
     * @returns {Object} { instruction, instructionCount, status, scope, pending, callStackDepth }
     */
    state() {
      return {
        instruction: mem.getInstructionIndex(),
        instructionCount: codeBlock ? mem.codeBlockInstructionCount(codeBlock) : 0,
        status: statusToString(mem.getStatus()),
        scope: orchestrator.getAllVariables(),
        pending: orchestrator.getPendingStack(),
        callStackDepth: mem.getCallStackDepth(),
      };
    },

    /**
     * Get details about a specific instruction.
     *
     * @param {number} index - Instruction index
     * @returns {Object} { opcode, operand1, operand2, sourceStart, sourceEnd }
     */
    instruction(index) {
      const instr = mem.codeBlockReadInstruction(codeBlock, index);
      return {
        opcode: opcodeToString(instr.opcode),
        operand1: instr.operand1,
        operand2: instr.operand2,
        sourceStart: instr.sourceStart,
        sourceEnd: instr.sourceEnd,
      };
    },

    /**
     * Find all instructions that map to a source range.
     *
     * @param {number} start - Source start position
     * @param {number} end - Source end position
     * @returns {number[]} Instruction indices whose source overlaps [start, end)
     */
    instructionsForRange(start, end) {
      if (!codeBlock) return [];
      const count = mem.codeBlockInstructionCount(codeBlock);
      const result = [];
      for (let i = 0; i < count; i++) {
        const instr = mem.codeBlockReadInstruction(codeBlock, i);
        if (instr.sourceEnd > start && instr.sourceStart < end) {
          result.push(i);
        }
      }
      return result;
    },

    /**
     * Jump to a specific instruction.
     *
     * @param {number} index - Instruction index
     */
    setInstruction(index) {
      mem.setInstructionIndex(index);
    },

    /**
     * Soft reset: clears execution state but keeps parsed code intact.
     *
     * Resets: instruction pointer, stacks, scope, status.
     * Keeps: parsed code block, FFI bindings, heap allocations.
     */
    reset() {
      mem.setInstructionIndex(0);
      mem.resetPendingStack();
      mem.resetCallStack();
      mem.resetTryStack();

      // Create fresh scope and re-register FFI bindings
      mem.createFreshScope();
      for (const [name, impl] of Object.entries(ffiBindings)) {
        orchestrator.bindFFI(name, impl);
      }

      // Re-initialize builtins in new scope
      mem.initializeBuiltins();

      mem.setStatus(STATUS_RUNNING);
    },

    /**
     * Reset code block for fresh parsing.
     *
     * Call this when you want to discard all previously parsed code
     * and start with an empty code block. Note: this invalidates
     * any closures created from the previous code.
     */
    resetCode() {
      mem.resetCodeBlock();
    },

    /**
     * Get a variable from global scope.
     *
     * @param {string} name - Variable name
     * @returns {*} Variable value or undefined
     */
    get(name) {
      return orchestrator.getVariable(name);
    },

    /**
     * Run garbage collection.
     *
     * @returns {Object} { heapCollected, heapRetained, stringsCollected, stringsRetained }
     */
    gc() {
      return collector.collect();
    },

    /**
     * Register an FFI namespace at runtime.
     *
     * @param {string} name - Namespace name (e.g., 'Timer')
     * @param {Object} impl - Methods { methodName: fn, ... }
     */
    bindFFI(name, impl) {
      ffiBindings[name] = impl;
      orchestrator.bindFFI(name, impl);
    },

    // Expose internals for advanced access
    get mem() { return mem; },
    get orchestrator() { return orchestrator; },
    get parser() { return parser; },
    get collector() { return collector; },
  };
}
