/**
 * SandScript Fuel Streaming Parser
 *
 * Parses loose SandScript source and emits flat instructions to CodeBlocks.
 * This is a Pratt parser (top-down operator precedence) that emits bytecode
 * directly instead of building an AST.
 */

import { Lexer, TokenType } from './lexer.js';
import { OP, CODE_BLOCK_FLAG } from './constants.js';

// Operator precedence levels
const PREC = {
  NONE: 0,
  COMMA: 1,        // ,
  ASSIGNMENT: 2,   // =, +=, -=, etc.
  TERNARY: 3,      // ? :
  NULLISH: 4,      // ??
  OR: 5,           // ||
  AND: 6,          // &&
  BIT_OR: 7,       // |
  BIT_XOR: 8,      // ^
  BIT_AND: 9,      // &
  EQUALITY: 10,    // === !==
  COMPARISON: 11,  // < > <= >=
  SHIFT: 12,       // << >> >>>
  TERM: 13,        // + -
  FACTOR: 14,      // * / %
  POWER: 15,       // **
  UNARY: 16,       // ! - ~ typeof ++ --
  CALL: 17,        // . () [] ?.
  PRIMARY: 18,
};

/**
 * Parser emits instructions directly to CodeBlocks.
 */
export class Parser {
  constructor(mem) {
    this.mem = mem;
    this.lexer = new Lexer();
    this.current = null;
    this.previous = null;
    this.codeBlock = null;
    this.hadError = false;

    // For continuation blocks when we hit nested functions
    this.blockStack = [];

    // For break/continue - stack of { breaks: number[], continues: number[], continueTarget: number | null }
    this.loopStack = [];

    // For method calls - set by dot/index, consumed by call
    // When true, call() emits CALL_METHOD instead of CALL
    this.pendingMethodCall = false;
  }

  /**
   * Parse source and emit instructions to a CodeBlock.
   *
   * For REPL: pass an existing codeBlock to append instructions to it.
   * For batch: pass null to create a new CodeBlock.
   *
   * Streaming: instructions are emitted as parsing proceeds. If the input
   * is incomplete (e.g., unclosed block), partial instructions remain and
   * an "expecting" error is thrown. The caller can continue feeding input.
   *
   * @param {string} source - Source code
   * @param {number|null} codeBlock - Existing CodeBlock to append to, or null to create new
   * @returns {number} - The CodeBlock pointer (same as input if provided)
   */
  parse(source, codeBlock = null) {
    this.lexer.reset(source);
    this.hadError = false;
    this.blockStack = [];

    // Use provided CodeBlock or allocate new one
    if (codeBlock !== null) {
      this.codeBlock = codeBlock;
    } else {
      this.codeBlock = this.mem.allocateCodeBlock();
    }

    this.advance();

    while (!this.check(TokenType.EOF)) {
      this.statement();
    }

    // Don't mark COMPLETE - caller decides when the block is done
    // This allows REPL to keep appending

    return this.codeBlock;
  }

  /**
   * Mark the current CodeBlock as complete.
   * Call this when done adding statements (e.g., end of file/session).
   */
  complete() {
    if (this.codeBlock && !this.hadError) {
      this.mem.codeBlockSetFlags(this.codeBlock, CODE_BLOCK_FLAG.COMPLETE);
    }
  }

  // ===========================================================================
  // Token handling
  // ===========================================================================

  advance() {
    this.previous = this.current;
    this.current = this.lexer.next();
  }

  check(type) {
    return this.current.type === type;
  }

  match(type) {
    if (!this.check(type)) return false;
    this.advance();
    return true;
  }

  consume(type, message) {
    if (this.check(type)) {
      this.advance();
      return this.previous;
    }
    this.error(message);
  }

  error(message) {
    if (this.hadError) return; // suppress cascading errors
    this.hadError = true;
    const token = this.current;
    throw new SyntaxError(`${message} at line ${token.line}, col ${token.col}`);
  }

  // ===========================================================================
  // Instruction emission
  // ===========================================================================

  emit(opcode, operand1 = 0, operand2 = 0) {
    const start = this.previous?.start ?? 0;
    const end = this.previous?.end ?? 0;
    return this.mem.codeBlockAppend(this.codeBlock, opcode, operand1, operand2, start, end);
  }

  emitWithSource(opcode, operand1, operand2, start, end) {
    return this.mem.codeBlockAppend(this.codeBlock, opcode, operand1, operand2, start, end);
  }

  /**
   * Get current instruction index (for backpatching).
   */
  get here() {
    return this.mem.codeBlockInstructionCount(this.codeBlock);
  }

  /**
   * Patch operand1 of an instruction.
   */
  patch(index, value) {
    this.mem.codeBlockPatch(this.codeBlock, index, value);
  }

  /**
   * Intern a string and return its offset.
   */
  internString(str) {
    return this.mem.internString(str);
  }

  // ===========================================================================
  // Statements
  // ===========================================================================

  statement() {
    // Empty statement (standalone semicolon)
    if (this.match(TokenType.SEMICOLON)) {
      return;
    }
    if (this.match(TokenType.LET)) {
      this.letStatement();
    } else if (this.match(TokenType.FUNCTION)) {
      this.functionDeclaration();
    } else if (this.match(TokenType.IF)) {
      this.ifStatement();
    } else if (this.match(TokenType.WHILE)) {
      this.whileStatement();
    } else if (this.match(TokenType.DO)) {
      this.doWhileStatement();
    } else if (this.match(TokenType.FOR)) {
      this.forStatement();
    } else if (this.match(TokenType.BREAK)) {
      this.breakStatement();
    } else if (this.match(TokenType.CONTINUE)) {
      this.continueStatement();
    } else if (this.match(TokenType.RETURN)) {
      this.returnStatement();
    } else if (this.match(TokenType.THROW)) {
      this.throwStatement();
    } else if (this.match(TokenType.TRY)) {
      this.tryStatement();
    } else if (this.match(TokenType.LBRACE)) {
      this.block();
    } else {
      this.expressionStatement();
    }
  }

  /**
   * Parse function declaration statement: function name(params) { body }
   * Binds the function to the name in the current scope.
   */
  functionDeclaration() {
    // Require function name
    this.consume(TokenType.IDENTIFIER, "Expected function name");
    const nameStart = this.previous.start;
    const nameEnd = this.previous.end;
    const name = this.internString(this.previous.value);

    // Parse parameters
    this.consume(TokenType.LPAREN, "Expected '(' after function name");
    const params = [];
    if (!this.check(TokenType.RPAREN)) {
      do {
        this.consume(TokenType.IDENTIFIER, "Expected parameter name");
        params.push(this.previous.value);
      } while (this.match(TokenType.COMMA));
    }
    this.consume(TokenType.RPAREN, "Expected ')' after parameters");

    // Parse body
    this.consume(TokenType.LBRACE, "Expected '{' before function body");
    this.emitFunctionBody(params, false);

    // Bind function to name in current scope
    this.emitWithSource(OP.LET_VAR, name, 0, nameStart, nameEnd);
  }

  letStatement() {
    this.consume(TokenType.IDENTIFIER, 'Expected variable name');
    const varStart = this.previous.start;
    const varEnd = this.previous.end;
    const name = this.internString(this.previous.value);

    if (this.match(TokenType.ASSIGN)) {
      this.expression();
    } else {
      this.emit(OP.LIT_UNDEFINED);
    }

    // Point LET_VAR to the variable name, not the value
    this.emitWithSource(OP.LET_VAR, name, 0, varStart, varEnd);
    this.match(TokenType.SEMICOLON); // optional semicolon
  }

  ifStatement() {
    this.consume(TokenType.LPAREN, "Expected '(' after 'if'");
    this.expression();
    this.consume(TokenType.RPAREN, "Expected ')' after condition");

    const jumpIfFalse = this.emit(OP.JUMP_IF_FALSE, 0); // placeholder

    this.statement(); // then branch

    if (this.match(TokenType.ELSE)) {
      const jumpOver = this.emit(OP.JUMP, 0); // placeholder
      this.patch(jumpIfFalse, this.here);
      this.statement(); // else branch
      this.patch(jumpOver, this.here);
    } else {
      this.patch(jumpIfFalse, this.here);
    }
  }

  whileStatement() {
    const loopStart = this.here;

    this.consume(TokenType.LPAREN, "Expected '(' after 'while'");
    this.expression();
    this.consume(TokenType.RPAREN, "Expected ')' after condition");

    const jumpIfFalse = this.emit(OP.JUMP_IF_FALSE, 0); // placeholder

    // Push loop context - continue jumps to condition (loopStart)
    this.loopStack.push({ breaks: [], continues: [], continueTarget: loopStart });

    this.statement(); // body

    this.emit(OP.JUMP, loopStart);
    this.patch(jumpIfFalse, this.here);

    // Patch all breaks to exit point
    const loop = this.loopStack.pop();
    for (const breakJump of loop.breaks) {
      this.patch(breakJump, this.here);
    }
  }

  doWhileStatement() {
    const loopStart = this.here;

    // Continue target unknown until we reach condition - will be patched
    this.loopStack.push({ breaks: [], continues: [], continueTarget: null });

    this.statement(); // body

    // Condition starts here - patch any continues
    const conditionStart = this.here;
    const loop = this.loopStack[this.loopStack.length - 1];
    for (const continueJump of loop.continues) {
      this.patch(continueJump, conditionStart);
    }

    this.consume(TokenType.WHILE, "Expected 'while' after do body");
    this.consume(TokenType.LPAREN, "Expected '(' after 'while'");
    this.expression();
    this.consume(TokenType.RPAREN, "Expected ')' after condition");
    this.match(TokenType.SEMICOLON);

    this.emit(OP.JUMP_IF_TRUE, loopStart);

    // Patch breaks to exit point
    for (const breakJump of loop.breaks) {
      this.patch(breakJump, this.here);
    }
    this.loopStack.pop();
  }

  forStatement() {
    // Syntax: for (init; condition; update) body
    // Bytecode layout:
    //   SCOPE_PUSH
    //   <init>
    //   JUMP condition_start
    //   update_start:
    //     <update>
    //     POP
    //   condition_start:
    //     <condition>
    //     JUMP_IF_FALSE exit
    //     <body>
    //     JUMP update_start
    //   exit:
    //   SCOPE_POP

    this.consume(TokenType.LPAREN, "Expected '(' after 'for'");
    this.emit(OP.SCOPE_PUSH);

    // Init clause - ends with ;
    if (this.match(TokenType.SEMICOLON)) {
      // empty init
    } else if (this.match(TokenType.LET)) {
      this.letStatement(); // handles semicolon
    } else {
      this.expression();
      this.emit(OP.POP);
      this.consume(TokenType.SEMICOLON, "Expected ';' after for init");
    }

    // Jump over update to condition (update emitted first but runs after body)
    const jumpToCondition = this.emit(OP.JUMP, 0);

    // We need to parse condition and update in source order (condition; update)
    // but emit them as (update then condition)
    // So we need to defer emission. Simplest: just emit in execution order.

    // Actually, let's restructure: emit condition first, then update code will be
    // at the jump-back point.
    //
    // New layout:
    //   SCOPE_PUSH
    //   <init>
    //   loop_start:        <- condition is here
    //     <condition>
    //     JUMP_IF_FALSE exit
    //     <body>
    //     <update>
    //     POP
    //     JUMP loop_start
    //   exit:
    //   SCOPE_POP
    //
    // This means continue jumps to just before update. But we don't know that
    // position yet when parsing body. Let's use deferred patching like do-while.

    // Parse condition (between first ; and second ;)
    const loopStart = this.here;
    this.patch(jumpToCondition, loopStart);

    let exitJump = -1;
    if (!this.check(TokenType.SEMICOLON)) {
      this.expression();
      exitJump = this.emit(OP.JUMP_IF_FALSE, 0);
    }
    this.consume(TokenType.SEMICOLON, "Expected ';' after for condition");

    // Parse update expression but don't emit yet - save the tokens? No, that's complex.
    // Instead, just note that we'll emit body, then update, and continue jumps to update.

    // Actually, simplest approach: parse and emit update here, jump over it,
    // then jump back to it from end of body.

    // Jump over update to body
    const jumpToBody = this.emit(OP.JUMP, 0);

    // Update clause
    const updateStart = this.here;
    if (!this.check(TokenType.RPAREN)) {
      this.expression();
      this.emit(OP.POP);
    }
    this.consume(TokenType.RPAREN, "Expected ')' after for clauses");

    // After update, jump back to condition
    this.emit(OP.JUMP, loopStart);

    // Body starts here
    this.patch(jumpToBody, this.here);

    // Push loop context - continue jumps to update
    this.loopStack.push({ breaks: [], continues: [], continueTarget: updateStart });

    // Body
    this.statement();

    // Jump to update (which then jumps to condition)
    this.emit(OP.JUMP, updateStart);

    // Exit point (breaks jump here, before SCOPE_POP)
    const exitPoint = this.here;
    if (exitJump !== -1) {
      this.patch(exitJump, exitPoint);
    }

    // Patch breaks
    const loop = this.loopStack.pop();
    for (const breakJump of loop.breaks) {
      this.patch(breakJump, exitPoint);
    }

    this.emit(OP.SCOPE_POP);
  }

  breakStatement() {
    if (this.loopStack.length === 0) {
      this.error("'break' outside of loop");
    }
    const breakJump = this.emit(OP.JUMP, 0); // placeholder - patched at loop end
    this.loopStack[this.loopStack.length - 1].breaks.push(breakJump);
    this.match(TokenType.SEMICOLON);
  }

  continueStatement() {
    if (this.loopStack.length === 0) {
      this.error("'continue' outside of loop");
    }
    const loop = this.loopStack[this.loopStack.length - 1];
    if (loop.continueTarget !== null) {
      // Target known (while, for)
      this.emit(OP.JUMP, loop.continueTarget);
    } else {
      // Target not yet known (do-while body) - defer patching
      const continueJump = this.emit(OP.JUMP, 0);
      loop.continues.push(continueJump);
    }
    this.match(TokenType.SEMICOLON);
  }

  returnStatement() {
    if (this.check(TokenType.SEMICOLON) || this.check(TokenType.RBRACE) || this.check(TokenType.EOF)) {
      this.emit(OP.RETURN_UNDEFINED);
    } else {
      this.expression();
      this.emit(OP.RETURN);
    }
    this.match(TokenType.SEMICOLON);
  }

  throwStatement() {
    this.expression();
    this.emit(OP.THROW);
    this.match(TokenType.SEMICOLON);
  }

  tryStatement() {
    // Parse: try { ... } catch (e) { ... } finally { ... }
    // At least one of catch or finally is required

    // Emit TRY_PUSH with placeholder operands
    const tryPushIndex = this.emit(OP.TRY_PUSH, 0, 0);

    // Parse try block
    this.consume(TokenType.LBRACE, "Expected '{' after 'try'");
    this.emit(OP.SCOPE_PUSH);
    while (!this.check(TokenType.RBRACE) && !this.check(TokenType.EOF)) {
      this.statement();
    }
    this.consume(TokenType.RBRACE, "Expected '}' after try block");
    this.emit(OP.SCOPE_POP);

    // Emit TRY_POP (normal exit from try)
    this.emit(OP.TRY_POP);

    // Jump over catch to finally (or end)
    const jumpAfterTryIndex = this.emit(OP.JUMP, 0);

    // Track catch and finally positions
    let catchIndex = 0;
    let finallyIndex = 0;
    let catchParamName = null;

    // Parse catch clause (optional)
    if (this.match(TokenType.CATCH)) {
      catchIndex = this.here;

      // Parse parameter
      this.consume(TokenType.LPAREN, "Expected '(' after 'catch'");
      this.consume(TokenType.IDENTIFIER, "Expected catch parameter name");
      catchParamName = this.internString(this.previous.value);
      this.consume(TokenType.RPAREN, "Expected ')' after catch parameter");

      // Catch body with scope for the error variable
      this.consume(TokenType.LBRACE, "Expected '{' after catch parameter");
      this.emit(OP.SCOPE_PUSH);

      // Bind exception to parameter (exception is on pending stack from THROW)
      this.emit(OP.LET_VAR, catchParamName, 0);

      while (!this.check(TokenType.RBRACE) && !this.check(TokenType.EOF)) {
        this.statement();
      }
      this.consume(TokenType.RBRACE, "Expected '}' after catch block");
      this.emit(OP.SCOPE_POP);
    }

    // Parse finally clause (optional)
    if (this.match(TokenType.FINALLY)) {
      finallyIndex = this.here;

      this.consume(TokenType.LBRACE, "Expected '{' after 'finally'");
      this.emit(OP.SCOPE_PUSH);
      while (!this.check(TokenType.RBRACE) && !this.check(TokenType.EOF)) {
        this.statement();
      }
      this.consume(TokenType.RBRACE, "Expected '}' after finally block");
      this.emit(OP.SCOPE_POP);

      // FINALLY_END dispatches based on completion type
      this.emit(OP.FINALLY_END);
    }

    // Must have at least catch or finally
    if (catchIndex === 0 && finallyIndex === 0) {
      this.error("Expected 'catch' or 'finally' after try block");
    }

    // Patch TRY_PUSH with actual catch and finally indices
    this.patch(tryPushIndex, catchIndex);
    // Patch operand2 - need a different approach since patch only does operand1
    // We'll need to re-emit with both operands
    // Actually, looking at the emit function, we need to patch operand2 separately
    // Let's modify the instruction directly
    const codeStart = this.mem.getCodeStart();
    const instrOffset = codeStart - (tryPushIndex + 1) * 16;
    this.mem.view.setUint32(this.mem.abs(instrOffset + 12), finallyIndex, true);

    // Patch jump after try
    // If we have finally, jump to finally
    // If we only have catch, jump to end
    if (finallyIndex !== 0) {
      this.patch(jumpAfterTryIndex, finallyIndex);
    } else {
      this.patch(jumpAfterTryIndex, this.here);
    }
  }

  block() {
    this.emit(OP.SCOPE_PUSH);
    while (!this.check(TokenType.RBRACE) && !this.check(TokenType.EOF)) {
      this.statement();
    }
    this.consume(TokenType.RBRACE, "Expected '}' after block");
    this.emit(OP.SCOPE_POP);
  }

  expressionStatement() {
    this.expression();
    this.emit(OP.POP); // discard result
    this.match(TokenType.SEMICOLON);
  }

  // ===========================================================================
  // Expressions (Pratt parser)
  // ===========================================================================

  expression() {
    this.parsePrecedence(PREC.ASSIGNMENT);
  }

  parsePrecedence(precedence) {
    this.advance();

    // Prefix rule
    const prefixRule = this.getPrefixRule(this.previous.type);
    if (!prefixRule) {
      this.error(`Unexpected token '${this.previous.value || this.previous.type}'`);
      return;
    }

    const canAssign = precedence <= PREC.ASSIGNMENT;
    prefixRule.call(this, canAssign);

    // Infix rules
    while (precedence <= this.getInfixPrecedence(this.current.type)) {
      this.advance();
      const infixRule = this.getInfixRule(this.previous.type);
      infixRule.call(this, canAssign);
    }
  }

  // ===========================================================================
  // Prefix rules
  // ===========================================================================

  getPrefixRule(type) {
    switch (type) {
      case TokenType.NUMBER: return this.number;
      case TokenType.STRING: return this.string;
      case TokenType.TRUE: return this.literal;
      case TokenType.FALSE: return this.literal;
      case TokenType.NULL: return this.literal;
      case TokenType.UNDEFINED: return this.literal;
      case TokenType.IDENTIFIER: return this.variable;
      case TokenType.THIS: return this.thisKeyword;
      case TokenType.FUNCTION: return this.functionExpression;
      case TokenType.LPAREN: return this.grouping;
      case TokenType.LBRACKET: return this.arrayLiteral;
      case TokenType.LBRACE: return this.objectLiteral;
      case TokenType.MINUS: return this.unary;
      case TokenType.NOT: return this.unary;
      case TokenType.TYPEOF: return this.unary;
      case TokenType.TILDE: return this.unary;
      case TokenType.PLUS_PLUS: return this.prefixIncrement;
      case TokenType.MINUS_MINUS: return this.prefixIncrement;
      case TokenType.NEW: return this.newExpression;
      default: return null;
    }
  }

  number() {
    // All numbers are f64 (JS semantics)
    const value = this.previous.value;
    const buffer = new ArrayBuffer(8);
    const f64 = new Float64Array(buffer);
    const u32 = new Uint32Array(buffer);
    f64[0] = value;
    this.emit(OP.LIT_FLOAT, u32[0], u32[1]);
  }

  string() {
    const offset = this.internString(this.previous.value);
    this.emit(OP.LIT_STRING, offset);
  }

  literal() {
    switch (this.previous.type) {
      case TokenType.TRUE: this.emit(OP.LIT_TRUE); break;
      case TokenType.FALSE: this.emit(OP.LIT_FALSE); break;
      case TokenType.NULL: this.emit(OP.LIT_NULL); break;
      case TokenType.UNDEFINED: this.emit(OP.LIT_UNDEFINED); break;
    }
  }

  variable(canAssign) {
    const varStart = this.previous.start;
    const varEnd = this.previous.end;
    const nameValue = this.previous.value;  // Don't intern yet - might be arrow param

    // Check for arrow function: x => ...
    // Only allow in assignment contexts (matches JS grammar restriction)
    if (canAssign && this.check(TokenType.ARROW)) {
      this.arrowFunction([nameValue], varStart);
      return;
    }

    // Now intern the name since it's definitely a variable
    const name = this.internString(nameValue);

    if (canAssign && this.match(TokenType.ASSIGN)) {
      this.expression();
      // Point SET_VAR to the variable name, not the value
      this.emitWithSource(OP.SET_VAR, name, 0, varStart, varEnd);
    } else if (canAssign && this.matchCompoundAssign()) {
      // Compound assignment: x += expr  -->  GET x, <expr>, OP, SET x
      const compoundOp = this.previous.type;
      this.emit(OP.GET_VAR, name);
      this.expression();
      this.emitCompoundOp(compoundOp);
      this.emitWithSource(OP.SET_VAR, name, 0, varStart, varEnd);
    } else if (this.match(TokenType.PLUS_PLUS) || this.match(TokenType.MINUS_MINUS)) {
      // Postfix increment/decrement: i++, i--
      // Returns OLD value, stores NEW value
      const isIncrement = this.previous.type === TokenType.PLUS_PLUS;
      // Stack needs: [old] at end
      // Strategy: GET, GET, LIT 1, ADD/SUB, SET, POP
      this.emit(OP.GET_VAR, name);               // [old]
      this.emit(OP.GET_VAR, name);               // [old, old]
      this.emitLitFloat(1);                      // [old, old, 1]
      this.emit(isIncrement ? OP.ADD : OP.SUB); // [old, new]
      this.emitWithSource(OP.SET_VAR, name, 0, varStart, varEnd); // [old, new]
      this.emit(OP.POP);                         // [old]
    } else {
      this.emit(OP.GET_VAR, name);
    }
  }

  /**
   * Handle `this` keyword - emits GET_VAR with interned "this" string.
   */
  thisKeyword() {
    const thisName = this.internString('this');
    this.emit(OP.GET_VAR, thisName);
  }

  /**
   * Handle `function` keyword for function expressions.
   * Syntax: function(params) { body } or function name(params) { body }
   * Emits MAKE_CLOSURE (not MAKE_ARROW_CLOSURE - binds this).
   */
  functionExpression() {
    // Parse optional function name (for named function expressions)
    // For now we ignore the name - just skip it
    if (this.check(TokenType.IDENTIFIER)) {
      this.advance();
    }

    // Parse parameters
    this.consume(TokenType.LPAREN, "Expected '(' after 'function'");
    const params = [];
    if (!this.check(TokenType.RPAREN)) {
      do {
        this.consume(TokenType.IDENTIFIER, "Expected parameter name");
        params.push(this.previous.value);
      } while (this.match(TokenType.COMMA));
    }
    this.consume(TokenType.RPAREN, "Expected ')' after parameters");

    // Parse body - must be a block
    this.consume(TokenType.LBRACE, "Expected '{' before function body");
    this.emitFunctionBody(params, false);
  }

  /**
   * Match any compound assignment token.
   */
  matchCompoundAssign() {
    if (this.check(TokenType.PLUS_ASSIGN) ||
        this.check(TokenType.MINUS_ASSIGN) ||
        this.check(TokenType.STAR_ASSIGN) ||
        this.check(TokenType.SLASH_ASSIGN) ||
        this.check(TokenType.PERCENT_ASSIGN) ||
        this.check(TokenType.STAR_STAR_ASSIGN) ||
        this.check(TokenType.AMPERSAND_ASSIGN) ||
        this.check(TokenType.PIPE_ASSIGN) ||
        this.check(TokenType.CARET_ASSIGN) ||
        this.check(TokenType.LSHIFT_ASSIGN) ||
        this.check(TokenType.RSHIFT_ASSIGN) ||
        this.check(TokenType.URSHIFT_ASSIGN)) {
      this.advance();
      return true;
    }
    return false;
  }

  /**
   * Emit the binary operation for compound assignment.
   */
  emitCompoundOp(tokenType) {
    switch (tokenType) {
      case TokenType.PLUS_ASSIGN: this.emit(OP.ADD); break;
      case TokenType.MINUS_ASSIGN: this.emit(OP.SUB); break;
      case TokenType.STAR_ASSIGN: this.emit(OP.MUL); break;
      case TokenType.SLASH_ASSIGN: this.emit(OP.DIV); break;
      case TokenType.PERCENT_ASSIGN: this.emit(OP.MOD); break;
      case TokenType.STAR_STAR_ASSIGN: this.emit(OP.POW); break;
      case TokenType.AMPERSAND_ASSIGN: this.emit(OP.BAND); break;
      case TokenType.PIPE_ASSIGN: this.emit(OP.BOR); break;
      case TokenType.CARET_ASSIGN: this.emit(OP.BXOR); break;
      case TokenType.LSHIFT_ASSIGN: this.emit(OP.SHL); break;
      case TokenType.RSHIFT_ASSIGN: this.emit(OP.SHR); break;
      case TokenType.URSHIFT_ASSIGN: this.emit(OP.USHR); break;
    }
  }

  grouping() {
    // Could be: (expr) or (params) => body
    // We need to look ahead to check for arrow

    // Save state in case we need to reparse as arrow function
    const startPos = this.previous.start;

    // Check if this looks like arrow function params
    // Collect potential parameter names as we go
    const params = [];
    let isArrowParams = true;

    // If first token after ( is ), then it's either () or () =>
    if (this.check(TokenType.RPAREN)) {
      this.advance(); // consume )
      if (this.check(TokenType.ARROW)) {
        // () => body - no params arrow function
        this.arrowFunction(params, startPos);
        return;
      }
      // () is not valid as expression, but let's be lenient and return undefined
      this.emit(OP.LIT_UNDEFINED);
      return;
    }

    // Check if first token is identifier
    if (this.check(TokenType.IDENTIFIER)) {
      // Could be (x) or (x, y) => or (x + y)
      // We need to peek ahead

      // Collect first identifier
      this.advance();
      const firstParam = this.previous.value;

      // Check what follows
      if (this.check(TokenType.RPAREN)) {
        // (x) - could be (x) => or just (x)
        this.advance(); // consume )
        if (this.check(TokenType.ARROW)) {
          // (x) => body
          params.push(firstParam);
          this.arrowFunction(params, startPos);
          return;
        }
        // Just (x) - push the variable
        const name = this.internString(firstParam);
        this.emit(OP.GET_VAR, name);
        return;
      }

      if (this.check(TokenType.COMMA)) {
        // (x, ...) => - definitely arrow params
        params.push(firstParam);
        while (this.match(TokenType.COMMA)) {
          this.consume(TokenType.IDENTIFIER, 'Expected parameter name');
          params.push(this.previous.value);
        }
        this.consume(TokenType.RPAREN, "Expected ')' after parameters");
        if (!this.check(TokenType.ARROW)) {
          this.error("Expected '=>' after arrow function parameters");
        }
        this.arrowFunction(params, startPos);
        return;
      }

      // (x + ...) or (x.foo) or (x = ...) or (x += ...) - this is an expression
      // We need to continue parsing the expression with x already consumed
      const name = this.internString(firstParam);

      // Check for assignment: (x = expr)
      if (this.check(TokenType.ASSIGN)) {
        this.advance(); // consume =
        this.expression();
        this.emit(OP.SET_VAR, name);
        // Continue with infix parsing (to handle comma after assignment)
        while (PREC.COMMA <= this.getInfixPrecedence(this.current.type)) {
          this.advance();
          const infixRule = this.getInfixRule(this.previous.type);
          infixRule.call(this, true);
        }
        this.consume(TokenType.RPAREN, "Expected ')' after expression");
        return;
      }

      // Check for compound assignment: (x += expr)
      if (this.matchCompoundAssign()) {
        const compoundOp = this.previous.type;
        this.emit(OP.GET_VAR, name);
        this.expression();
        this.emitCompoundOp(compoundOp);
        this.emit(OP.SET_VAR, name);
        // Continue with infix parsing (to handle comma after assignment)
        while (PREC.COMMA <= this.getInfixPrecedence(this.current.type)) {
          this.advance();
          const infixRule = this.getInfixRule(this.previous.type);
          infixRule.call(this, true);
        }
        this.consume(TokenType.RPAREN, "Expected ')' after expression");
        return;
      }

      // Push GET_VAR for x, then continue with infix parsing
      this.emit(OP.GET_VAR, name);

      // Continue parsing infix operators (allow comma inside parens)
      while (PREC.COMMA <= this.getInfixPrecedence(this.current.type)) {
        this.advance();
        const infixRule = this.getInfixRule(this.previous.type);
        infixRule.call(this, true);
      }

      this.consume(TokenType.RPAREN, "Expected ')' after expression");
      return;
    }

    // Not an identifier first - must be regular grouping expression
    // Allow comma operator inside parentheses
    this.parsePrecedence(PREC.COMMA);
    this.consume(TokenType.RPAREN, "Expected ')' after expression");
  }

  /**
   * Parse arrow function body and emit MAKE_ARROW_CLOSURE.
   *
   * Bytecode layout:
   *   [MAKE_ARROW_CLOSURE start, end]
   *   [JUMP past_end]
   *   [function body...]  <- start points here
   *   [RETURN]
   *   ... <- end points here, past_end too
   *
   * @param {string[]} params - Parameter names (not interned yet)
   * @param {number} startPos - Source position of arrow function start
   */
  arrowFunction(params, startPos) {
    this.consume(TokenType.ARROW, "Expected '=>'");

    // Parse body
    if (this.check(TokenType.LBRACE)) {
      // Block body: { statements } - use shared helper
      this.advance(); // consume {
      this.emitFunctionBody(params, true);
    } else {
      // Expression body: expr (implicit return)
      // Can't use helper since there's no closing brace
      const closureInstr = this.emit(OP.MAKE_ARROW_CLOSURE, 0, 0);
      const jumpInstr = this.emit(OP.JUMP, 0);

      const bodyStart = this.here;

      // Bind parameters in reverse order
      for (let i = params.length - 1; i >= 0; i--) {
        const name = this.internString(params[i]);
        this.emit(OP.LET_VAR, name);
      }

      this.expression();
      this.emit(OP.RETURN);

      const bodyEnd = this.here;

      this.patch(closureInstr, bodyStart);
      this.patchOperand2(closureInstr, bodyEnd);
      this.patch(jumpInstr, bodyEnd);
    }
  }

  /**
   * Patch operand2 of an instruction.
   */
  patchOperand2(index, value) {
    // Instructions grow downward: instr_address = codeStart - (index + 1) * INSTRUCTION_SIZE
    const codeStart = this.mem.getCodeStart();
    const instrOffset = codeStart - (index + 1) * 16; // INSTRUCTION_SIZE = 16
    this.mem.view.setUint32(this.mem.abs(instrOffset + 12), value, true);
  }

  /**
   * Emit function body bytecode (shared by declarations, expressions, methods, arrows).
   *
   * Expects: opening brace already consumed (for block bodies)
   * Emits: MAKE_CLOSURE/MAKE_ARROW_CLOSURE, JUMP, body, RETURN, patches operands
   *
   * @param {string[]} params - Parameter names (not interned yet)
   * @param {boolean} isArrow - If true, emit MAKE_ARROW_CLOSURE (no `this` binding)
   */
  emitFunctionBody(params, isArrow = false) {
    const closureOp = isArrow ? OP.MAKE_ARROW_CLOSURE : OP.MAKE_CLOSURE;
    const closureInstr = this.emit(closureOp, 0, 0);
    const jumpInstr = this.emit(OP.JUMP, 0);

    const bodyStart = this.here;

    // Bind parameters in reverse order (stack has args in order, pop gives reverse)
    for (let i = params.length - 1; i >= 0; i--) {
      const paramName = this.internString(params[i]);
      this.emit(OP.LET_VAR, paramName);
    }

    // Parse body statements
    while (!this.check(TokenType.RBRACE) && !this.check(TokenType.EOF)) {
      this.statement();
    }
    this.consume(TokenType.RBRACE, "Expected '}' after function body");

    // Implicit return undefined if needed
    const lastInstr = this.here > bodyStart ?
      this.mem.codeBlockReadInstruction(this.codeBlock, this.here - 1) : null;
    if (!lastInstr || (lastInstr.opcode !== OP.RETURN && lastInstr.opcode !== OP.RETURN_UNDEFINED)) {
      this.emit(OP.RETURN_UNDEFINED);
    }

    const bodyEnd = this.here;

    // Patch operands
    this.patch(closureInstr, bodyStart);
    this.patchOperand2(closureInstr, bodyEnd);
    this.patch(jumpInstr, bodyEnd);
  }

  arrayLiteral() {
    let count = 0;
    if (!this.check(TokenType.RBRACKET)) {
      do {
        this.expression();
        count++;
      } while (this.match(TokenType.COMMA));
    }
    this.consume(TokenType.RBRACKET, "Expected ']' after array elements");
    this.emit(OP.MAKE_ARRAY, count);
  }

  objectLiteral() {
    let count = 0;
    if (!this.check(TokenType.RBRACE)) {
      do {
        // Key
        let keyName = null;
        if (this.match(TokenType.IDENTIFIER)) {
          keyName = this.previous.value;
          const key = this.internString(keyName);
          this.emit(OP.LIT_STRING, key);
        } else if (this.match(TokenType.STRING)) {
          const key = this.internString(this.previous.value);
          this.emit(OP.LIT_STRING, key);
        } else {
          this.error('Expected property name');
        }

        // Check for method shorthand: { foo() { ... } }
        if (keyName && this.check(TokenType.LPAREN)) {
          // Method shorthand - emit MAKE_CLOSURE (not arrow, so it binds `this`)
          this.methodShorthand();
        } else {
          // Regular property: { foo: value }
          this.consume(TokenType.COLON, "Expected ':' after property name");
          this.expression();
        }
        count++;
      } while (this.match(TokenType.COMMA));
    }
    this.consume(TokenType.RBRACE, "Expected '}' after object properties");
    this.emit(OP.MAKE_OBJECT, count);
  }

  /**
   * Parse method shorthand in object literal: { foo() { body } }
   * Emits MAKE_CLOSURE (binds `this`, unlike arrow functions).
   */
  methodShorthand() {
    this.consume(TokenType.LPAREN, "Expected '(' after method name");

    // Parse parameters
    const params = [];
    if (!this.check(TokenType.RPAREN)) {
      do {
        this.consume(TokenType.IDENTIFIER, "Expected parameter name");
        params.push(this.previous.value);
      } while (this.match(TokenType.COMMA));
    }
    this.consume(TokenType.RPAREN, "Expected ')' after parameters");

    // Parse body
    this.consume(TokenType.LBRACE, "Expected '{' before method body");
    this.emitFunctionBody(params, false);
  }

  unary() {
    const op = this.previous.type;
    this.parsePrecedence(PREC.UNARY);

    switch (op) {
      case TokenType.MINUS: this.emit(OP.NEG); break;
      case TokenType.NOT: this.emit(OP.NOT); break;
      case TokenType.TYPEOF: this.emit(OP.TYPEOF); break;
      case TokenType.TILDE: this.emit(OP.BNOT); break;
    }
  }

  /**
   * Handle prefix ++i, --i, ++obj.x, --obj.x, ++arr[i], --arr[i]
   * Returns the NEW value on the stack
   */
  prefixIncrement() {
    const op = this.previous.type;
    const isIncrement = op === TokenType.PLUS_PLUS;

    // Must start with an identifier
    if (!this.match(TokenType.IDENTIFIER)) {
      this.error("Expected identifier after increment/decrement operator");
      return;
    }

    const baseName = this.internString(this.previous.value);
    const varStart = this.previous.start;
    const varEnd = this.previous.end;

    // Check if followed by property access or index access
    if (this.match(TokenType.DOT)) {
      // ++obj.prop
      if (!this.match(TokenType.IDENTIFIER)) {
        this.error("Expected property name after '.'");
        return;
      }
      const propName = this.internString(this.previous.value);

      // Emit: [obj, obj] -> GET_PROP -> [obj, old] -> LIT 1 -> ADD/SUB -> [obj, new] -> SET_PROP -> [new]
      this.emit(OP.GET_VAR, baseName);   // [obj]
      this.emit(OP.DUP);                  // [obj, obj]
      this.emit(OP.GET_PROP, propName);   // [obj, old]
      this.emitLitFloat(1);               // [obj, old, 1]
      this.emit(isIncrement ? OP.ADD : OP.SUB);  // [obj, new]
      this.emit(OP.SET_PROP, propName);   // [new]
    } else if (this.match(TokenType.LBRACKET)) {
      // ++arr[expr]
      const indexStart = this.here;
      this.expression();  // Parse the index expression
      const indexEnd = this.here;
      this.consume(TokenType.RBRACKET, "Expected ']' after index");

      // Strategy for any index expression:
      // 1. We have [indexExpr] on stack from parsing
      // 2. Emit: GET_VAR arr, SWAP -> [arr, index]
      // 3. Copy index instructions, GET_VAR arr, copy index instructions -> [arr, index, arr, index]
      // 4. GET_INDEX -> [arr, index, old]
      // 5. LIT 1, ADD/SUB -> [arr, index, new]
      // 6. SET_INDEX -> [new]

      this.emit(OP.GET_VAR, baseName);           // [index, arr]
      this.emit(OP.SWAP);                         // [arr, index]
      this.emit(OP.GET_VAR, baseName);           // [arr, index, arr]
      // Re-emit the index expression
      for (let i = indexStart; i < indexEnd; i++) {
        const instr = this.mem.codeBlockReadInstruction(this.codeBlock, i);
        this.emit(instr.opcode, instr.operand1, instr.operand2);
      }
      this.emit(OP.GET_INDEX);                    // [arr, index, old]
      this.emitLitFloat(1);                       // [arr, index, old, 1]
      this.emit(isIncrement ? OP.ADD : OP.SUB);  // [arr, index, new]
      this.emit(OP.SET_INDEX);                    // [new]
    } else {
      // Simple identifier: ++i
      // Emit: GET, LIT 1, ADD/SUB, SET (leaves NEW value on stack)
      this.emit(OP.GET_VAR, baseName);
      this.emitLitFloat(1);
      this.emit(isIncrement ? OP.ADD : OP.SUB);
      this.emitWithSource(OP.SET_VAR, baseName, 0, varStart, varEnd);
    }
  }

  /**
   * Emit a literal float value.
   */
  emitLitFloat(value) {
    const buffer = new ArrayBuffer(8);
    const f64 = new Float64Array(buffer);
    const u32 = new Uint32Array(buffer);
    f64[0] = value;
    this.emit(OP.LIT_FLOAT, u32[0], u32[1]);
  }

  // ===========================================================================
  // Infix rules
  // ===========================================================================

  getInfixPrecedence(type) {
    switch (type) {
      case TokenType.COMMA: return PREC.COMMA;
      case TokenType.QUESTION: return PREC.TERNARY;
      case TokenType.QUESTION_QUESTION: return PREC.NULLISH;
      case TokenType.OR: return PREC.OR;
      case TokenType.AND: return PREC.AND;
      case TokenType.PIPE: return PREC.BIT_OR;
      case TokenType.CARET: return PREC.BIT_XOR;
      case TokenType.AMPERSAND: return PREC.BIT_AND;
      case TokenType.EQ:
      case TokenType.NEQ: return PREC.EQUALITY;
      case TokenType.LT:
      case TokenType.GT:
      case TokenType.LTE:
      case TokenType.GTE: return PREC.COMPARISON;
      case TokenType.LSHIFT:
      case TokenType.RSHIFT:
      case TokenType.URSHIFT: return PREC.SHIFT;
      case TokenType.PLUS:
      case TokenType.MINUS: return PREC.TERM;
      case TokenType.STAR:
      case TokenType.SLASH:
      case TokenType.PERCENT: return PREC.FACTOR;
      case TokenType.STAR_STAR: return PREC.POWER;
      case TokenType.LPAREN:
      case TokenType.DOT:
      case TokenType.LBRACKET:
      case TokenType.QUESTION_DOT: return PREC.CALL;
      default: return PREC.NONE;
    }
  }

  getInfixRule(type) {
    switch (type) {
      case TokenType.PLUS:
      case TokenType.MINUS:
      case TokenType.STAR:
      case TokenType.SLASH:
      case TokenType.PERCENT:
      case TokenType.STAR_STAR:
      case TokenType.EQ:
      case TokenType.NEQ:
      case TokenType.LT:
      case TokenType.GT:
      case TokenType.LTE:
      case TokenType.GTE:
      case TokenType.AMPERSAND:
      case TokenType.PIPE:
      case TokenType.CARET:
      case TokenType.LSHIFT:
      case TokenType.RSHIFT:
      case TokenType.URSHIFT:
        return this.binary;
      case TokenType.AND: return this.and;
      case TokenType.OR: return this.or;
      case TokenType.QUESTION_QUESTION: return this.nullish;
      case TokenType.QUESTION: return this.ternary;
      case TokenType.COMMA: return this.comma;
      case TokenType.LPAREN: return this.call;
      case TokenType.DOT: return this.dot;
      case TokenType.QUESTION_DOT: return this.optionalChain;
      case TokenType.LBRACKET: return this.index;
      default: return null;
    }
  }

  binary() {
    const op = this.previous.type;
    const opStart = this.previous.start;
    const opEnd = this.previous.end;
    const prec = this.getInfixPrecedence(op);
    // Right associative for ** (power)
    this.parsePrecedence(op === TokenType.STAR_STAR ? prec : prec + 1);

    let opcode;
    switch (op) {
      case TokenType.PLUS: opcode = OP.ADD; break;
      case TokenType.MINUS: opcode = OP.SUB; break;
      case TokenType.STAR: opcode = OP.MUL; break;
      case TokenType.SLASH: opcode = OP.DIV; break;
      case TokenType.PERCENT: opcode = OP.MOD; break;
      case TokenType.STAR_STAR: opcode = OP.POW; break;
      case TokenType.EQ: opcode = OP.EQ; break;
      case TokenType.NEQ: opcode = OP.NEQ; break;
      case TokenType.LT: opcode = OP.LT; break;
      case TokenType.GT: opcode = OP.GT; break;
      case TokenType.LTE: opcode = OP.LTE; break;
      case TokenType.GTE: opcode = OP.GTE; break;
      case TokenType.AMPERSAND: opcode = OP.BAND; break;
      case TokenType.PIPE: opcode = OP.BOR; break;
      case TokenType.CARET: opcode = OP.BXOR; break;
      case TokenType.LSHIFT: opcode = OP.SHL; break;
      case TokenType.RSHIFT: opcode = OP.SHR; break;
      case TokenType.URSHIFT: opcode = OP.USHR; break;
    }
    this.emitWithSource(opcode, 0, 0, opStart, opEnd);
  }

  and() {
    // Short-circuit: if left is falsy, jump over right (keep left as result)
    const jumpIndex = this.emit(OP.AND, 0); // placeholder
    this.parsePrecedence(PREC.AND + 1);
    this.patch(jumpIndex, this.here);
  }

  or() {
    // Short-circuit: if left is truthy, jump over right (keep left as result)
    const jumpIndex = this.emit(OP.OR, 0); // placeholder
    this.parsePrecedence(PREC.OR + 1);
    this.patch(jumpIndex, this.here);
  }

  nullish() {
    // Short-circuit: if left is NOT nullish, keep it and jump; else pop and evaluate right
    const jumpIndex = this.emit(OP.NULLISH, 0); // placeholder
    this.parsePrecedence(PREC.NULLISH + 1);
    this.patch(jumpIndex, this.here);
  }

  ternary() {
    // Condition already on stack
    const jumpIfFalse = this.emit(OP.JUMP_IF_FALSE, 0);

    this.parsePrecedence(PREC.TERNARY); // consequent (right-associative)
    this.consume(TokenType.COLON, "Expected ':' in ternary expression");

    const jumpOver = this.emit(OP.JUMP, 0);
    this.patch(jumpIfFalse, this.here);

    this.parsePrecedence(PREC.TERNARY); // alternate (right-associative)
    this.patch(jumpOver, this.here);
  }

  comma() {
    // Left value already on stack - discard it
    this.emit(OP.POP);
    // Clear pending method call - comma separates expressions
    this.pendingMethodCall = false;
    // Parse right side (left-associative)
    this.parsePrecedence(PREC.COMMA + 1);
  }

  call() {
    // Check if this is a method call (preceded by . or [])
    const isMethodCall = this.pendingMethodCall;
    this.pendingMethodCall = false;

    let argCount = 0;
    if (!this.check(TokenType.RPAREN)) {
      do {
        this.expression();
        argCount++;
      } while (this.match(TokenType.COMMA));
    }
    this.consume(TokenType.RPAREN, "Expected ')' after arguments");

    if (isMethodCall) {
      // Method call: receiver is still on stack below the function value
      // CALL_METHOD will bind `this` to receiver
      this.emit(OP.CALL_METHOD, argCount);
    } else {
      // Regular call: no receiver binding
      this.emit(OP.CALL, argCount);
    }
  }

  /**
   * Handle `new` expression: new Foo(args)
   * Parses the constructor and arguments, emits OP.NEW
   */
  newExpression() {
    // Parse ONLY the primary expression (identifier, grouping, etc.)
    // We handle member access manually after to exclude call parens
    this.parsePrecedence(PREC.PRIMARY);

    // Now handle member access manually (DOT and LBRACKET only, not LPAREN)
    while (true) {
      if (this.match(TokenType.DOT)) {
        this.dot();
      } else if (this.match(TokenType.LBRACKET)) {
        this.index();
      } else {
        break;
      }
    }

    // Check for arguments
    let argCount = 0;
    if (this.match(TokenType.LPAREN)) {
      if (!this.check(TokenType.RPAREN)) {
        do {
          this.expression();
          argCount++;
        } while (this.match(TokenType.COMMA));
      }
      this.consume(TokenType.RPAREN, "Expected ')' after constructor arguments");
    }
    // Note: `new Foo` without parens is valid JS (0 args)

    this.emit(OP.NEW, argCount);
  }

  /**
   * Optional chaining: obj?.prop, obj?.[expr], obj?.()
   * If obj is nullish, short-circuit to undefined; else continue normally.
   */
  optionalChain() {
    // Stack has [obj]
    // Emit: DUP, check nullish, if nullish: POP both and push undefined, jump to end
    //       else: POP the dup, continue with normal access

    // We use NULLISH opcode which: if nullish, pops and jumps; else keeps value
    // But we need different behavior: if nullish, replace with undefined and skip access

    // Strategy:
    //   DUP                    ; [obj, obj]
    //   NULLISH jump_to_end    ; if nullish: [obj] and jump; else [obj, obj]
    //   POP                    ; [obj] - remove the duplicate
    //   <access>               ; [result]
    //   JUMP past_undefined
    // end:
    //   POP                    ; [] - remove the nullish value
    //   LIT_UNDEFINED          ; [undefined]
    // past_undefined:

    this.emit(OP.DUP);
    const nullishJump = this.emit(OP.NULLISH, 0); // if NOT nullish, jumps over
    // If we get here, it was nullish - pop the original obj and push undefined
    this.emit(OP.POP);
    this.emit(OP.LIT_UNDEFINED);
    const skipAccess = this.emit(OP.JUMP, 0);

    // Not nullish path
    this.patch(nullishJump, this.here);
    this.emit(OP.POP); // remove the DUP'd copy

    // Now do the actual access based on what follows
    if (this.match(TokenType.IDENTIFIER)) {
      // obj?.prop
      const name = this.internString(this.previous.value);
      this.emit(OP.GET_PROP, name);
    } else if (this.match(TokenType.LBRACKET)) {
      // obj?.[expr]
      this.expression();
      this.consume(TokenType.RBRACKET, "Expected ']' after optional index");
      this.emit(OP.GET_INDEX);
    } else if (this.match(TokenType.LPAREN)) {
      // obj?.()
      let argCount = 0;
      if (!this.check(TokenType.RPAREN)) {
        do {
          this.expression();
          argCount++;
        } while (this.match(TokenType.COMMA));
      }
      this.consume(TokenType.RPAREN, "Expected ')' after optional call arguments");
      this.emit(OP.CALL, argCount);
    } else {
      this.error("Expected property name, '[', or '(' after '?.'");
    }

    this.patch(skipAccess, this.here);
  }

  dot(canAssign) {
    this.consume(TokenType.IDENTIFIER, 'Expected property name after "."');
    const name = this.internString(this.previous.value);

    if (canAssign && this.match(TokenType.ASSIGN)) {
      this.expression();
      this.emit(OP.SET_PROP, name);
    } else if (canAssign && this.matchCompoundAssign()) {
      // Compound assignment: obj.x += expr
      // Stack has [obj] from before dot() was called
      // We need: [obj, old], then [obj, new], then SET_PROP
      // Strategy: DUP obj, GET_PROP, expr, OP, SET_PROP
      const compoundOp = this.previous.type;
      this.emit(OP.DUP);              // [obj, obj]
      this.emit(OP.GET_PROP, name);   // [obj, old]
      this.expression();               // [obj, old, expr]
      this.emitCompoundOp(compoundOp); // [obj, new]
      this.emit(OP.SET_PROP, name);   // [new]
    } else if (this.match(TokenType.PLUS_PLUS) || this.match(TokenType.MINUS_MINUS)) {
      // Postfix: obj.x++ returns old value, stores new
      // Stack at entry: [obj]
      // Need to re-emit obj to have it twice for GET and SET
      const isIncrement = this.previous.type === TokenType.PLUS_PLUS;
      const here = this.here;
      const objInstr = this.mem.codeBlockReadInstruction(this.codeBlock, here - 1);

      if (objInstr.opcode === OP.GET_VAR) {
        // Simple case: obj is a variable
        this.emit(OP.GET_PROP, name);              // [old]
        this.emit(OP.GET_VAR, objInstr.operand1);  // [old, obj]
        this.emit(OP.GET_VAR, objInstr.operand1);  // [old, obj, obj]
        this.emit(OP.GET_PROP, name);              // [old, obj, old]
        this.emitLitFloat(1);                      // [old, obj, old, 1]
        this.emit(isIncrement ? OP.ADD : OP.SUB); // [old, obj, new]
        this.emit(OP.SET_PROP, name);             // [old, new]
        this.emit(OP.POP);                         // [old]
      } else {
        this.error("Postfix increment on complex property access not supported");
      }
    } else {
      // Property access - check if followed by call (method call pattern)
      if (this.check(TokenType.LPAREN)) {
        // Method call: obj.method() - need receiver for CALL_METHOD
        // Stack: [obj] → DUP → [obj, obj] → GET_PROP → [obj, method]
        this.emit(OP.DUP);
        this.emit(OP.GET_PROP, name);
        this.pendingMethodCall = true;
      } else {
        // Plain property access: obj.prop
        this.emit(OP.GET_PROP, name);
      }
    }
  }

  index(canAssign) {
    // At entry, stack has [object]
    // We need to emit DUP before parsing index if compound assignment is coming
    // But we don't know yet! So we emit first, then handle compound at the end.

    // Track where the object instruction is (for re-emission)
    const objInstrIndex = this.here - 1;

    // Track where the index expression starts
    const indexStart = this.here;
    this.expression();  // Stack: [object, index]
    const indexEnd = this.here;
    this.consume(TokenType.RBRACKET, "Expected ']' after index");

    if (canAssign && this.match(TokenType.ASSIGN)) {
      this.expression();
      this.emit(OP.SET_INDEX);
    } else if (canAssign && this.matchCompoundAssign()) {
      // Compound assignment: arr[expr] += val
      // Stack: [arr, index]
      // Strategy: re-emit obj and index, then GET_INDEX, expr, OP, SET_INDEX
      const compoundOp = this.previous.type;
      const objInstr = this.mem.codeBlockReadInstruction(this.codeBlock, objInstrIndex);

      if (objInstr.opcode === OP.GET_VAR) {
        // Re-emit object and index to get [arr, index, arr, index]
        this.emit(OP.GET_VAR, objInstr.operand1);
        for (let i = indexStart; i < indexEnd; i++) {
          const instr = this.mem.codeBlockReadInstruction(this.codeBlock, i);
          this.emit(instr.opcode, instr.operand1, instr.operand2);
        }
        this.emit(OP.GET_INDEX);                     // [arr, index, old]
        this.expression();                            // [arr, index, old, expr]
        this.emitCompoundOp(compoundOp);             // [arr, index, new]
        this.emit(OP.SET_INDEX);                     // [new]
      } else {
        this.error("Compound assignment on complex object expressions not supported");
      }
    } else if (this.match(TokenType.PLUS_PLUS) || this.match(TokenType.MINUS_MINUS)) {
      // Postfix increment/decrement on index: arr[expr]++ or obj[expr]--
      // Stack: [arr, index]
      // Want: return old value, store old+1
      const isIncrement = this.previous.type === TokenType.PLUS_PLUS;
      const objInstr = this.mem.codeBlockReadInstruction(this.codeBlock, objInstrIndex);

      if (objInstr.opcode === OP.GET_VAR) {
        // Stack: [arr, index]
        this.emit(OP.GET_INDEX);                      // [old]
        // Re-emit arr, index twice: [old, arr, index, arr, index]
        this.emit(OP.GET_VAR, objInstr.operand1);
        for (let i = indexStart; i < indexEnd; i++) {
          const instr = this.mem.codeBlockReadInstruction(this.codeBlock, i);
          this.emit(instr.opcode, instr.operand1, instr.operand2);
        }
        this.emit(OP.GET_VAR, objInstr.operand1);
        for (let i = indexStart; i < indexEnd; i++) {
          const instr = this.mem.codeBlockReadInstruction(this.codeBlock, i);
          this.emit(instr.opcode, instr.operand1, instr.operand2);
        }
        this.emit(OP.GET_INDEX);                      // [old, arr, index, old]
        this.emitLitFloat(1);                         // [old, arr, index, old, 1]
        this.emit(isIncrement ? OP.ADD : OP.SUB);    // [old, arr, index, new]
        this.emit(OP.SET_INDEX);                      // [old, new]
        this.emit(OP.POP);                            // [old]
      } else {
        this.error("Postfix increment on complex object expressions not supported");
      }
    } else {
      // Index access - check if followed by call (method call pattern)
      if (this.check(TokenType.LPAREN)) {
        // Method call: obj[key]() - need receiver for CALL_METHOD
        // Stack currently: [obj, key]
        // We need: [obj, method] for CALL_METHOD
        // Strategy: re-emit obj after GET_INDEX, then SWAP to get [obj, method]
        const here = this.here;
        const objInstr = this.mem.codeBlockReadInstruction(this.codeBlock, here - 2);

        if (objInstr.opcode === OP.GET_VAR) {
          // Simple case: obj is a variable - re-emit it
          // Stack: [obj, key]
          this.emit(OP.GET_INDEX);                    // [method]
          this.emit(OP.GET_VAR, objInstr.operand1);   // [method, obj]
          this.emit(OP.SWAP);                         // [obj, method]
          this.pendingMethodCall = true;
        } else {
          // Complex case: obj is an expression - fall back to regular call
          this.emit(OP.GET_INDEX);
        }
      } else {
        // Plain index access
        this.emit(OP.GET_INDEX);
      }
    }
  }
}
