/**
 * SandScript Fuel-Based Interpreter - Orchestrator
 *
 * Manages execution of SandScript code with FFI support.
 * Wraps MemoryManipulator and WASM instance to provide high-level API.
 */

import {
  STATUS_RUNNING,
  STATUS_DONE,
  STATUS_PAUSED_FUEL,
  STATUS_FFI_REQUEST,
  STATUS_ERROR,
  STATUS_CALLBACK_DONE,
  STATUS_THROW,
  FRAME_FLAG_CALLBACK,
  TYPE,
  LAYOUT,
  STATE,
  ERR_FFI_UNKNOWN,
  ERR_FFI_FAILED,
  ERR_USER_THROW,
} from './constants.js';

import { Collector } from './collector.js';

/**
 * Orchestrator manages SandScript execution with FFI support.
 */
export class Orchestrator {
  /**
   * @param {MemoryManipulator} mem - Memory manipulator instance
   * @param {WebAssembly.Instance} wasm - WASM instance
   */
  constructor(mem, wasm) {
    this.mem = mem;
    this.wasm = wasm;

    // FFI ref table: refId -> JS object
    this.refTable = {};
    // Debug names: refId -> string
    this.refNames = {};
    // Next ref ID
    this.nextRefId = 1;

    // Closure handle table: handleId -> closurePointer
    // Wrappers store handleId instead of raw pointer, so pointers can be updated after GC
    this.closureHandles = new Map();
    this.nextHandleId = 1;

    // When JS garbage collects a wrapper function, remove its handle
    // so the closure becomes collectible on the next SS GC
    this.closureRegistry = new FinalizationRegistry((handleId) => {
      this.closureHandles.delete(handleId);
    });

    // Garbage collector
    this.collector = new Collector(mem);
  }

  // ===========================================================================
  // FFI Binding
  // ===========================================================================

  /**
   * Bind an FFI namespace to global scope.
   *
   * @param {string} name - Namespace name (e.g., 'MyAPI')
   * @param {object} implementation - JS object with methods
   */
  bindFFI(name, implementation) {
    const refId = this.nextRefId++;
    this.refTable[refId] = implementation;
    this.refNames[refId] = name;

    // Intern the name
    const nameOffset = this.mem.internString(name);

    // Write FFI ref to scratch area
    this.mem.writeFFIRef(LAYOUT.SCRATCH, refId);

    // Bind to global scope
    const globalScope = this.mem.getScope();
    this.mem.scopeDefine(globalScope, nameOffset, LAYOUT.SCRATCH);
  }

  // ===========================================================================
  // Execution
  // ===========================================================================

  /**
   * Run until completion or error.
   *
   * @param {number} fuelPerStep - Fuel to add per step (default 1000)
   * @returns {*} - Result value (JS)
   * @throws {Error} - On runtime error
   */
  run(fuelPerStep = 1000) {
    while (true) {
      this.wasm.exports.run(fuelPerStep);

      const status = this.mem.getStatus();

      if (status === STATUS_DONE) {
        return this.extractResult();
      }

      if (status === STATUS_PAUSED_FUEL) {
        continue;
      }

      if (status === STATUS_FFI_REQUEST) {
        this.handleFFIRequest();
        continue;
      }

      if (status === STATUS_ERROR) {
        throw this.buildError();
      }

      throw new Error(`Unknown status: ${status}`);
    }
  }

  /**
   * Handle an FFI request.
   */
  handleFFIRequest() {
    const req = this.mem.getFFIRequest();
    const method = this.mem.readString(req.methodOffset);

    // Look up the implementation
    const impl = this.refTable[req.refId];
    if (!impl) {
      this.mem.setError(ERR_FFI_UNKNOWN, req.refId);
      throw new Error(`Unknown FFI ref: ${req.refId}`);
    }

    const fn = impl[method];
    if (typeof fn !== 'function') {
      this.mem.setError(ERR_FFI_UNKNOWN, req.methodOffset);
      throw new Error(`Unknown FFI method: ${method} on ref ${req.refId}`);
    }

    // Extract arguments
    const args = this.extractArgs(req.argsPointer, req.argCount);

    // Wrap any closures
    const wrappedArgs = args.map(arg => {
      if (arg.type === TYPE.CLOSURE) {
        return this.wrapClosure(arg.pointer);
      }
      return arg.value;
    });

    // Calculate receiver position for stack cleanup
    // Stack layout: [receiver, method, arg1, ..., argN]
    // argsPointer = method + VALUE_SIZE
    // receiver = argsPointer - 2 * VALUE_SIZE
    const receiverPtr = req.argsPointer - 32; // 2 * 16

    // Call the function
    try {
      const result = fn.apply(impl, wrappedArgs);
      // Clean up stack by resetting to receiver position
      this.mem.setPendingPointer(receiverPtr);
      // Inject result (pushes to stack)
      this.injectResult(result);
      // Reset status so interpreter continues
      this.mem.setStatus(STATUS_RUNNING);
    } catch (e) {
      // Clean up stack even on error
      this.mem.setPendingPointer(receiverPtr);
      // Push exception value and set STATUS_THROW
      // The interpreter will handle it like a regular throw
      const errorMessage = e && e.message ? e.message : String(e);
      const errorPtr = this.writeString(errorMessage);
      this.mem.pushPending(errorPtr);
      this.mem.setStatus(STATUS_THROW);
    }
  }

  // ===========================================================================
  // Value Marshalling: SS -> JS
  // ===========================================================================

  /**
   * Extract arguments from pending stack.
   *
   * @param {number} argsPointer - Pointer to first arg on pending stack
   * @param {number} argCount - Number of arguments
   * @returns {Array} - Array of { type, value, pointer }
   */
  extractArgs(argsPointer, argCount) {
    const args = [];
    const seen = new Map();

    for (let i = 0; i < argCount; i++) {
      const ptr = argsPointer + i * 16;
      args.push(this.extractValueAt(ptr, seen));
    }

    return args;
  }

  /**
   * Extract a value at an address.
   *
   * @param {number} addr - Segment-relative address
   * @param {Map} seen - Map of seen pointers (for cycle detection)
   * @returns {object} - { type, value, pointer }
   */
  extractValueAt(addr, seen = new Map()) {
    const type = this.mem.getValueType(addr);
    const { payload } = this.mem.getValue(addr);

    switch (type) {
      case TYPE.NULL:
        return { type, value: null };

      case TYPE.UNDEFINED:
        return { type, value: undefined };

      case TYPE.BOOLEAN:
        return { type, value: payload !== 0n };

      case TYPE.INTEGER:
        return { type, value: Number(payload) };

      case TYPE.FLOAT:
        return { type, value: payload };

      case TYPE.STRING: {
        const offset = Number(payload);
        return { type, value: this.mem.readString(offset) };
      }

      case TYPE.ARRAY: {
        const ptr = Number(payload);
        if (seen.has(ptr)) {
          return { type, value: seen.get(ptr) };
        }
        const arr = [];
        seen.set(ptr, arr);
        this.extractArrayInto(ptr, arr, seen);
        return { type, value: arr };
      }

      case TYPE.OBJECT: {
        const ptr = Number(payload);
        if (seen.has(ptr)) {
          return { type, value: seen.get(ptr) };
        }
        const obj = {};
        seen.set(ptr, obj);
        this.extractObjectInto(ptr, obj, seen);
        return { type, value: obj };
      }

      case TYPE.CLOSURE:
        return { type, pointer: Number(payload) };

      case TYPE.FFI_REF: {
        const refId = Number(payload);
        return { type, value: this.refTable[refId] };
      }

      case TYPE.MSGPACK_REF: {
        // payload is [addr:u32][length:u32]
        const addr = Number(payload & 0xffffffffn);
        const length = Number(payload >> 32n);
        return this.extractMsgpackValue(addr, length, seen);
      }

      default:
        return { type, value: undefined };
    }
  }

  /**
   * Extract a value from msgpack bytes.
   *
   * @param {number} addr - Absolute address of msgpack data
   * @param {number} length - Byte length of msgpack data
   * @param {Map} seen - Seen map for cycle detection
   * @returns {object} - { type, value }
   */
  extractMsgpackValue(addr, length, seen = new Map()) {
    const u8 = new Uint8Array(this.mem.memory.buffer);
    const view = new DataView(this.mem.memory.buffer);
    const endAddr = addr + length;

    if (addr >= endAddr) {
      return { type: TYPE.UNDEFINED, value: undefined };
    }

    const byte = u8[addr];

    // Positive fixint (0x00-0x7f): value = byte
    if (byte <= 0x7f) {
      return { type: TYPE.FLOAT, value: byte };
    }

    // Negative fixint (0xe0-0xff): value = byte as signed
    if (byte >= 0xe0) {
      return { type: TYPE.FLOAT, value: (byte | 0xffffff00) >> 0 };
    }

    // Fixmap (0x80-0x8f)
    if (byte >= 0x80 && byte <= 0x8f) {
      const count = byte & 0x0f;
      return this.extractMsgpackMap(addr + 1, endAddr, count, seen);
    }

    // Fixarray (0x90-0x9f)
    if (byte >= 0x90 && byte <= 0x9f) {
      const count = byte & 0x0f;
      return this.extractMsgpackArray(addr + 1, endAddr, count, seen);
    }

    // Fixstr (0xa0-0xbf)
    if (byte >= 0xa0 && byte <= 0xbf) {
      const len = byte & 0x1f;
      const str = new TextDecoder().decode(u8.slice(addr + 1, addr + 1 + len));
      return { type: TYPE.STRING, value: str };
    }

    // nil (0xc0)
    if (byte === 0xc0) {
      return { type: TYPE.NULL, value: null };
    }

    // false (0xc2)
    if (byte === 0xc2) {
      return { type: TYPE.BOOLEAN, value: false };
    }

    // true (0xc3)
    if (byte === 0xc3) {
      return { type: TYPE.BOOLEAN, value: true };
    }

    // float32 (0xca)
    if (byte === 0xca) {
      const val = view.getFloat32(addr + 1, false);  // big-endian
      return { type: TYPE.FLOAT, value: val };
    }

    // float64 (0xcb)
    if (byte === 0xcb) {
      const val = view.getFloat64(addr + 1, false);  // big-endian
      return { type: TYPE.FLOAT, value: val };
    }

    // uint8 (0xcc)
    if (byte === 0xcc) {
      return { type: TYPE.FLOAT, value: u8[addr + 1] };
    }

    // uint16 (0xcd)
    if (byte === 0xcd) {
      const val = view.getUint16(addr + 1, false);  // big-endian
      return { type: TYPE.FLOAT, value: val };
    }

    // uint32 (0xce)
    if (byte === 0xce) {
      const val = view.getUint32(addr + 1, false);  // big-endian
      return { type: TYPE.FLOAT, value: val };
    }

    // uint64 (0xcf)
    if (byte === 0xcf) {
      const hi = view.getUint32(addr + 1, false);
      const lo = view.getUint32(addr + 5, false);
      return { type: TYPE.FLOAT, value: hi * 4294967296 + lo };
    }

    // int8 (0xd0)
    if (byte === 0xd0) {
      const val = view.getInt8(addr + 1);
      return { type: TYPE.FLOAT, value: val };
    }

    // int16 (0xd1)
    if (byte === 0xd1) {
      const val = view.getInt16(addr + 1, false);  // big-endian
      return { type: TYPE.FLOAT, value: val };
    }

    // int32 (0xd2)
    if (byte === 0xd2) {
      const val = view.getInt32(addr + 1, false);  // big-endian
      return { type: TYPE.FLOAT, value: val };
    }

    // int64 (0xd3)
    if (byte === 0xd3) {
      const hi = view.getInt32(addr + 1, false);
      const lo = view.getUint32(addr + 5, false);
      return { type: TYPE.FLOAT, value: hi * 4294967296 + lo };
    }

    // str8 (0xd9)
    if (byte === 0xd9) {
      const len = u8[addr + 1];
      const str = new TextDecoder().decode(u8.slice(addr + 2, addr + 2 + len));
      return { type: TYPE.STRING, value: str };
    }

    // str16 (0xda)
    if (byte === 0xda) {
      const len = view.getUint16(addr + 1, false);
      const str = new TextDecoder().decode(u8.slice(addr + 3, addr + 3 + len));
      return { type: TYPE.STRING, value: str };
    }

    // str32 (0xdb)
    if (byte === 0xdb) {
      const len = view.getUint32(addr + 1, false);
      const str = new TextDecoder().decode(u8.slice(addr + 5, addr + 5 + len));
      return { type: TYPE.STRING, value: str };
    }

    // array16 (0xdc)
    if (byte === 0xdc) {
      const count = view.getUint16(addr + 1, false);
      return this.extractMsgpackArray(addr + 3, endAddr, count, seen);
    }

    // array32 (0xdd)
    if (byte === 0xdd) {
      const count = view.getUint32(addr + 1, false);
      return this.extractMsgpackArray(addr + 5, endAddr, count, seen);
    }

    // map16 (0xde)
    if (byte === 0xde) {
      const count = view.getUint16(addr + 1, false);
      return this.extractMsgpackMap(addr + 3, endAddr, count, seen);
    }

    // map32 (0xdf)
    if (byte === 0xdf) {
      const count = view.getUint32(addr + 1, false);
      return this.extractMsgpackMap(addr + 5, endAddr, count, seen);
    }

    // Unsupported type
    return { type: TYPE.UNDEFINED, value: undefined };
  }

  /**
   * Extract a msgpack array.
   */
  extractMsgpackArray(addr, endAddr, count, seen) {
    const arr = [];
    let pos = addr;
    for (let i = 0; i < count && pos < endAddr; i++) {
      const { value, nextAddr } = this.extractMsgpackValueWithLength(pos, endAddr, seen);
      arr.push(value);
      pos = nextAddr;
    }
    return { type: TYPE.ARRAY, value: arr };
  }

  /**
   * Extract a msgpack map.
   */
  extractMsgpackMap(addr, endAddr, count, seen) {
    const obj = {};
    let pos = addr;
    for (let i = 0; i < count && pos < endAddr; i++) {
      const { value: key, nextAddr: keyEnd } = this.extractMsgpackValueWithLength(pos, endAddr, seen);
      const { value: val, nextAddr: valEnd } = this.extractMsgpackValueWithLength(keyEnd, endAddr, seen);
      if (typeof key === 'string') {
        obj[key] = val;
      }
      pos = valEnd;
    }
    return { type: TYPE.OBJECT, value: obj };
  }

  /**
   * Extract a msgpack value and return the next address.
   */
  extractMsgpackValueWithLength(addr, endAddr, seen) {
    const u8 = new Uint8Array(this.mem.memory.buffer);
    const view = new DataView(this.mem.memory.buffer);

    if (addr >= endAddr) {
      return { value: undefined, nextAddr: addr };
    }

    const byte = u8[addr];

    // Positive fixint
    if (byte <= 0x7f) {
      return { value: byte, nextAddr: addr + 1 };
    }

    // Negative fixint
    if (byte >= 0xe0) {
      return { value: (byte | 0xffffff00) >> 0, nextAddr: addr + 1 };
    }

    // Fixmap
    if (byte >= 0x80 && byte <= 0x8f) {
      const count = byte & 0x0f;
      const { value, nextAddr } = this.extractMsgpackMapWithLength(addr + 1, endAddr, count, seen);
      return { value, nextAddr };
    }

    // Fixarray
    if (byte >= 0x90 && byte <= 0x9f) {
      const count = byte & 0x0f;
      const { value, nextAddr } = this.extractMsgpackArrayWithLength(addr + 1, endAddr, count, seen);
      return { value, nextAddr };
    }

    // Fixstr
    if (byte >= 0xa0 && byte <= 0xbf) {
      const len = byte & 0x1f;
      const str = new TextDecoder().decode(u8.slice(addr + 1, addr + 1 + len));
      return { value: str, nextAddr: addr + 1 + len };
    }

    // nil
    if (byte === 0xc0) return { value: null, nextAddr: addr + 1 };
    // false
    if (byte === 0xc2) return { value: false, nextAddr: addr + 1 };
    // true
    if (byte === 0xc3) return { value: true, nextAddr: addr + 1 };

    // float32
    if (byte === 0xca) {
      return { value: view.getFloat32(addr + 1, false), nextAddr: addr + 5 };
    }
    // float64
    if (byte === 0xcb) {
      return { value: view.getFloat64(addr + 1, false), nextAddr: addr + 9 };
    }

    // uint8
    if (byte === 0xcc) return { value: u8[addr + 1], nextAddr: addr + 2 };
    // uint16
    if (byte === 0xcd) return { value: view.getUint16(addr + 1, false), nextAddr: addr + 3 };
    // uint32
    if (byte === 0xce) return { value: view.getUint32(addr + 1, false), nextAddr: addr + 5 };
    // uint64
    if (byte === 0xcf) {
      const hi = view.getUint32(addr + 1, false);
      const lo = view.getUint32(addr + 5, false);
      return { value: hi * 4294967296 + lo, nextAddr: addr + 9 };
    }

    // int8
    if (byte === 0xd0) return { value: view.getInt8(addr + 1), nextAddr: addr + 2 };
    // int16
    if (byte === 0xd1) return { value: view.getInt16(addr + 1, false), nextAddr: addr + 3 };
    // int32
    if (byte === 0xd2) return { value: view.getInt32(addr + 1, false), nextAddr: addr + 5 };
    // int64
    if (byte === 0xd3) {
      const hi = view.getInt32(addr + 1, false);
      const lo = view.getUint32(addr + 5, false);
      return { value: hi * 4294967296 + lo, nextAddr: addr + 9 };
    }

    // str8
    if (byte === 0xd9) {
      const len = u8[addr + 1];
      const str = new TextDecoder().decode(u8.slice(addr + 2, addr + 2 + len));
      return { value: str, nextAddr: addr + 2 + len };
    }
    // str16
    if (byte === 0xda) {
      const len = view.getUint16(addr + 1, false);
      const str = new TextDecoder().decode(u8.slice(addr + 3, addr + 3 + len));
      return { value: str, nextAddr: addr + 3 + len };
    }
    // str32
    if (byte === 0xdb) {
      const len = view.getUint32(addr + 1, false);
      const str = new TextDecoder().decode(u8.slice(addr + 5, addr + 5 + len));
      return { value: str, nextAddr: addr + 5 + len };
    }

    // array16
    if (byte === 0xdc) {
      const count = view.getUint16(addr + 1, false);
      return this.extractMsgpackArrayWithLength(addr + 3, endAddr, count, seen);
    }
    // array32
    if (byte === 0xdd) {
      const count = view.getUint32(addr + 1, false);
      return this.extractMsgpackArrayWithLength(addr + 5, endAddr, count, seen);
    }

    // map16
    if (byte === 0xde) {
      const count = view.getUint16(addr + 1, false);
      return this.extractMsgpackMapWithLength(addr + 3, endAddr, count, seen);
    }
    // map32
    if (byte === 0xdf) {
      const count = view.getUint32(addr + 1, false);
      return this.extractMsgpackMapWithLength(addr + 5, endAddr, count, seen);
    }

    return { value: undefined, nextAddr: addr + 1 };
  }

  extractMsgpackArrayWithLength(addr, endAddr, count, seen) {
    const arr = [];
    let pos = addr;
    for (let i = 0; i < count && pos < endAddr; i++) {
      const { value, nextAddr } = this.extractMsgpackValueWithLength(pos, endAddr, seen);
      arr.push(value);
      pos = nextAddr;
    }
    return { value: arr, nextAddr: pos };
  }

  extractMsgpackMapWithLength(addr, endAddr, count, seen) {
    const obj = {};
    let pos = addr;
    for (let i = 0; i < count && pos < endAddr; i++) {
      const { value: key, nextAddr: keyEnd } = this.extractMsgpackValueWithLength(pos, endAddr, seen);
      const { value: val, nextAddr: valEnd } = this.extractMsgpackValueWithLength(keyEnd, endAddr, seen);
      if (typeof key === 'string') {
        obj[key] = val;
      }
      pos = valEnd;
    }
    return { value: obj, nextAddr: pos };
  }

  /**
   * Extract array elements into existing array.
   *
   * @param {number} arrayPtr - Segment-relative pointer to array header
   * @param {Array} arr - Array to fill
   * @param {Map} seen - Seen map
   */
  extractArrayInto(arrayPtr, arr, seen) {
    const absPtr = this.mem.abs(arrayPtr);
    // Array header: [GC:8][length:4][capacity:4][data_ptr:4]
    const length = this.mem.view.getUint32(absPtr + 8, true);
    const dataPtr = this.mem.view.getUint32(absPtr + 16, true);

    // Data block starts with GC header (8 bytes), then elements
    const dataStart = dataPtr + 8;

    for (let i = 0; i < length; i++) {
      const elemAddr = dataStart + i * 16;
      const { value } = this.extractValueAt(elemAddr, seen);
      arr.push(value);
    }
  }

  /**
   * Extract object properties into existing object.
   *
   * @param {number} objPtr - Segment-relative pointer to object header
   * @param {object} obj - Object to fill
   * @param {Map} seen - Seen map
   */
  extractObjectInto(objPtr, obj, seen) {
    const absPtr = this.mem.abs(objPtr);
    // Object header: [GC:8][count:4][capacity:4][entries_ptr:4]
    const count = this.mem.view.getUint32(absPtr + 8, true);
    const entriesPtr = this.mem.view.getUint32(absPtr + 16, true);

    // Entries start after GC header (8 bytes)
    const entriesStart = entriesPtr + 8;

    for (let i = 0; i < count; i++) {
      // Entry: [key_offset:4][type:4][flags:4][data_lo:4][data_hi:4]
      const entryAddr = entriesStart + i * 20;
      const absEntry = this.mem.abs(entryAddr);

      const keyOffset = this.mem.view.getUint32(absEntry, true);
      const key = this.mem.readString(keyOffset);

      // Build a temporary value slot to extract from
      // Entry has type at +4, so we need to read from there
      const entryType = this.mem.view.getUint32(absEntry + 4, true);
      const dataLo = this.mem.view.getUint32(absEntry + 12, true);
      const dataHi = this.mem.view.getUint32(absEntry + 16, true);

      // Create inline extraction
      const { value } = this.extractValueFromParts(entryType, dataLo, dataHi, seen);
      obj[key] = value;
    }
  }

  /**
   * Extract value from parts.
   *
   * @param {number} type - Type tag
   * @param {number} dataLo - Low 32 bits
   * @param {number} dataHi - High 32 bits
   * @param {Map} seen - Seen map
   * @returns {object} - { value }
   */
  extractValueFromParts(type, dataLo, dataHi, seen) {
    switch (type) {
      case TYPE.NULL:
        return { value: null };

      case TYPE.UNDEFINED:
        return { value: undefined };

      case TYPE.BOOLEAN:
        return { value: dataLo !== 0 };

      case TYPE.INTEGER: {
        // Reconstruct i64 from two u32s
        const value = dataLo + dataHi * 0x100000000;
        return { value };
      }

      case TYPE.FLOAT: {
        // Reconstruct f64 from two u32s
        const buffer = new ArrayBuffer(8);
        const u32 = new Uint32Array(buffer);
        const f64 = new Float64Array(buffer);
        u32[0] = dataLo;
        u32[1] = dataHi;
        return { value: f64[0] };
      }

      case TYPE.STRING:
        return { value: this.mem.readString(dataLo) };

      case TYPE.ARRAY: {
        if (seen.has(dataLo)) {
          return { value: seen.get(dataLo) };
        }
        const arr = [];
        seen.set(dataLo, arr);
        this.extractArrayInto(dataLo, arr, seen);
        return { value: arr };
      }

      case TYPE.OBJECT: {
        if (seen.has(dataLo)) {
          return { value: seen.get(dataLo) };
        }
        const obj = {};
        seen.set(dataLo, obj);
        this.extractObjectInto(dataLo, obj, seen);
        return { value: obj };
      }

      case TYPE.FFI_REF:
        return { value: this.refTable[dataLo] };

      default:
        return { value: undefined };
    }
  }

  /**
   * Extract result from pending stack.
   *
   * @returns {*} - JS value
   */
  extractResult() {
    const pendingDepth = this.mem.getPendingDepth();
    if (pendingDepth === 0) {
      return undefined;
    }
    // Top of stack is at pending pointer - 16 bytes
    const ptr = this.mem.getPendingPointer() - 16;
    const { value } = this.extractValueAt(ptr);
    return value;
  }

  /**
   * Get a variable from the global scope by name.
   *
   * @param {string} name - Variable name
   * @returns {*} - JS value or undefined if not found
   */
  getVariable(name) {
    const scope = this.mem.getScope();
    const nameOffset = this.mem.internString(name);
    const valuePtr = this.mem.scopeLookup(scope, nameOffset);
    if (!valuePtr) {
      return undefined;
    }
    const { value } = this.extractValueAt(valuePtr);
    return value;
  }

  // ===========================================================================
  // Value Marshalling: JS -> SS
  // ===========================================================================

  /**
   * Inject a JS value as result.
   *
   * @param {*} value - JS value
   */
  injectResult(value) {
    const ptr = this.writeValue(value);
    this.mem.pushPending(ptr);
  }

  /**
   * Write a JS value to heap and return pointer.
   *
   * @param {*} value - JS value
   * @param {Map} seen - Map of seen objects (for cycle detection)
   * @returns {number} - Segment-relative pointer
   */
  writeValue(value, seen = new Map()) {
    if (value === null) {
      return this.writeNull();
    }

    if (value === undefined) {
      return this.writeUndefined();
    }

    if (typeof value === 'boolean') {
      return this.writeBoolean(value);
    }

    if (typeof value === 'number') {
      // Always use float for FFI values - the interpreter's arithmetic
      // operations expect floats (integers are only used internally)
      return this.writeFloat(value);
    }

    if (typeof value === 'string') {
      return this.writeString(value);
    }

    if (Array.isArray(value)) {
      if (seen.has(value)) {
        return seen.get(value);
      }
      return this.writeArray(value, seen);
    }

    if (typeof value === 'object') {
      if (seen.has(value)) {
        return seen.get(value);
      }
      // Check if it's a plain object that can be fully marshalled
      if (value.constructor === Object) {
        // Check if all values are marshallable (no functions)
        const hasUnmarshallable = Object.values(value).some(
          v => typeof v === 'function'
        );
        if (!hasUnmarshallable) {
          return this.writeObject(value, seen);
        }
      }
      // Object with functions or non-plain object becomes FFI ref
      return this.writeFFIRefValue(value);
    }

    if (typeof value === 'function') {
      // Functions become FFI refs (callable from SS via wrapClosure)
      return this.writeFFIRefValue(value);
    }

    // Unknown type becomes FFI ref
    return this.writeFFIRefValue(value);
  }

  writeNull() {
    const ptr = this.mem.allocate(16, 0);
    this.mem.writeNull(ptr);
    return ptr;
  }

  writeUndefined() {
    const ptr = this.mem.allocate(16, 0);
    this.mem.writeUndefined(ptr);
    return ptr;
  }

  writeBoolean(value) {
    const ptr = this.mem.allocate(16, 0);
    this.mem.writeBoolean(ptr, value);
    return ptr;
  }

  writeInteger(value) {
    const ptr = this.mem.allocate(16, 0);
    this.mem.writeInteger(ptr, value);
    return ptr;
  }

  writeFloat(value) {
    const ptr = this.mem.allocate(16, 0);
    this.mem.writeFloat(ptr, value);
    return ptr;
  }

  writeString(value) {
    const ptr = this.mem.allocate(16, 0);
    const offset = this.mem.internString(value);
    this.mem.writeString(ptr, offset);
    return ptr;
  }

  writeArray(value, seen) {
    // Allocate array with capacity (heap object)
    const arrayHeapPtr = this.mem.allocateArrayWithCapacity(value.length || 1);

    // Allocate a VALUE slot and write TYPE_ARRAY with the heap pointer
    const ptr = this.mem.allocate(16, 0);
    this.mem.writeArray(ptr, arrayHeapPtr);

    // Register in seen map using the VALUE slot pointer
    seen.set(value, ptr);

    // Write elements
    for (let i = 0; i < value.length; i++) {
      const elemPtr = this.writeValue(value[i], seen);
      this.mem.arraySetFromPointer(arrayHeapPtr, i, elemPtr);
    }

    return ptr;
  }

  writeObject(value, seen) {
    const keys = Object.keys(value);
    // Allocate object on heap
    const objHeapPtr = this.mem.allocateObject(keys.length || 1);

    // Allocate a VALUE slot and write TYPE_OBJECT with the heap pointer
    const ptr = this.mem.allocate(16, 0);
    this.mem.writeObject(ptr, objHeapPtr);

    // Register in seen map using the VALUE slot pointer
    seen.set(value, ptr);

    for (const key of keys) {
      const keyOffset = this.mem.internString(key);
      const valuePtr = this.writeValue(value[key], seen);
      this.mem.objectSetFromPointer(objHeapPtr, keyOffset, valuePtr);
    }

    return ptr;
  }

  writeFFIRefValue(value) {
    const refId = this.nextRefId++;
    this.refTable[refId] = value;

    const ptr = this.mem.allocate(16, 0);
    this.mem.writeFFIRef(ptr, refId);
    return ptr;
  }

  /**
   * Copy a 16-byte value from src to dest.
   *
   * @param {number} srcAddr - Source address (segment-relative)
   * @param {number} destAddr - Destination address (segment-relative)
   */
  copyValue(srcAddr, destAddr) {
    const absSrc = this.mem.abs(srcAddr);
    const absDest = this.mem.abs(destAddr);

    for (let i = 0; i < 16; i++) {
      this.mem.u8[absDest + i] = this.mem.u8[absSrc + i];
    }
  }

  // ===========================================================================
  // Callback Support
  // ===========================================================================

  /**
   * Wrap a closure as a callable JS function.
   *
   * Uses a handle table so the closure survives GC:
   * - Wrapper stores a stable handleId, not a raw pointer
   * - Handle table maps handleId -> current pointer
   * - After GC compaction, orchestrator updates pointers in handle table
   * - FinalizationRegistry cleans up handle when JS collects the wrapper
   *
   * @param {number} closurePointer - Segment-relative pointer to closure
   * @returns {Function} - JS function that invokes the closure
   */
  wrapClosure(closurePointer) {
    const handleId = this.nextHandleId++;
    this.closureHandles.set(handleId, closurePointer);

    const wrapper = (...args) => {
      const currentPointer = this.closureHandles.get(handleId);
      if (currentPointer === undefined) {
        throw new Error('Callback has been released');
      }
      return this.invokeCallback(currentPointer, args);
    };

    // When JS collects the wrapper, remove the handle
    this.closureRegistry.register(wrapper, handleId);

    return wrapper;
  }

  /**
   * Invoke a callback (SS closure).
   *
   * @param {number} closurePointer - Segment-relative pointer to closure
   * @param {Array} args - JS arguments
   * @returns {*} - JS result
   */
  invokeCallback(closurePointer, args) {
    const closure = this.mem.getClosure(closurePointer);

    // Save current execution state (so we can restore after callback)
    const savedInstructionIndex = this.mem.getInstructionIndex();
    const savedScope = this.mem.getScope();
    const savedStackPointer = this.mem.getStackPointer();

    // Create scope with closure's captured environment as parent
    const callScope = this.mem.createScope(closure.scope);

    // Push args to pending stack
    for (const arg of args) {
      const ptr = this.writeValue(arg);
      this.mem.pushPending(ptr);
    }

    // Push frame with FRAME_FLAG_CALLBACK
    // The frame stores the return address (where to resume after callback)
    this.mem.pushFrameWithFlags(
      this.mem.getCodeBlock(),
      savedInstructionIndex,  // Return to where we were
      savedScope,              // Restore caller's scope
      FRAME_FLAG_CALLBACK
    );

    // Set up execution to start at the closure's body
    this.mem.setInstructionIndex(closure.startInstr);
    this.mem.setScope(callScope);

    // Set status to RUNNING (we may be in FFI_REQUEST state)
    this.mem.setStatus(STATUS_RUNNING);

    // Run until callback completes
    while (true) {
      this.wasm.exports.run(1000);

      const status = this.mem.getStatus();

      if (status === STATUS_CALLBACK_DONE) {
        return this.extractResult();
      }

      if (status === STATUS_PAUSED_FUEL) {
        continue;
      }

      if (status === STATUS_FFI_REQUEST) {
        this.handleFFIRequest();
        continue;
      }

      if (status === STATUS_ERROR) {
        // Restore execution state before throwing
        // The error might be caught by JS, allowing execution to continue
        this.mem.setInstructionIndex(savedInstructionIndex);
        this.mem.setScope(savedScope);
        this.mem.setStackPointer(savedStackPointer);
        throw this.buildError();
      }
    }
  }

  // ===========================================================================
  // Error Handling
  // ===========================================================================

  /**
   * Build an error from the error info region.
   * Also clears the completion state so it doesn't affect subsequent runs.
   *
   * @returns {Error}
   */
  buildError() {
    const info = this.mem.getErrorInfo();

    // Clear completion state (JS is consuming this error)
    this.mem.view.setUint32(this.mem.abs(STATE.COMPLETION_TYPE), 0, true); // COMPLETION_NORMAL

    // Clear error state
    this.mem.setError(0, 0); // ERR_NONE

    // For user throws, extract the thrown value from completion slot
    if (info.code === ERR_USER_THROW) {
      const completionValuePtr = this.mem.view.getUint32(
        this.mem.abs(STATE.COMPLETION_VALUE), true
      );
      if (completionValuePtr) {
        const { value } = this.extractValueAt(completionValuePtr);
        return new Error(String(value));
      }
    }

    return new Error(`SandScript error: code=${info.code}, detail=${info.detail}`);
  }

  // ===========================================================================
  // Garbage Collection
  // ===========================================================================

  /**
   * Run garbage collection.
   *
   * Passes closure handles as external roots to prevent collection,
   * then updates the handle table with new pointers after compaction.
   *
   * @returns {object} - Collection stats from collector
   */
  gc() {
    // Gather all closure pointers we need to keep alive
    const externalRoots = [...this.closureHandles.values()];

    // Run collection with external roots
    const result = this.collector.collect(externalRoots);

    // Update handle table with forwarded pointers
    if (result.forwarding) {
      for (const [handleId, oldPtr] of this.closureHandles) {
        const newPtr = result.forwarding.get(oldPtr);
        if (newPtr !== undefined && newPtr !== oldPtr) {
          this.closureHandles.set(handleId, newPtr);
        }
      }
    }

    return result;
  }

  /**
   * Get heap usage in bytes.
   *
   * @returns {number} - Bytes used on heap
   */
  getHeapUsed() {
    return this.mem.getHeapPointer() - this.mem.getHeapStart();
  }

  // ===========================================================================
  // State Inspection (for debug API)
  // ===========================================================================

  /**
   * Get all variables from current scope and parent chain.
   *
   * @returns {object} - { name: value, ... }
   */
  getAllVariables() {
    const result = {};
    let scopePointer = this.mem.getScope();

    while (scopePointer !== 0) {
      const absPointer = this.mem.abs(scopePointer);
      const count = this.mem.view.getUint32(absPointer + 4, true);
      const entriesPointer = this.mem.view.getUint32(absPointer + 12, true);

      // Entry data starts after GC header (8 bytes)
      const entryDataStart = entriesPointer + 8;

      for (let i = 0; i < count; i++) {
        // Entry is 20 bytes: [name_ptr:4][type:4][flags:4][data_lo:4][data_hi:4]
        const entryOffset = entryDataStart + i * 20;
        const absEntryPointer = this.mem.abs(entryOffset);
        const nameOffset = this.mem.view.getUint32(absEntryPointer, true);
        const name = this.mem.readString(nameOffset);

        // Skip if already defined in child scope (shadowed)
        if (name in result) continue;

        // Extract value from entry (value starts at offset +4)
        const valuePointer = entryOffset + 4;
        const { value } = this.extractValueAt(valuePointer);
        result[name] = value;
      }

      // Move to parent scope
      scopePointer = this.mem.view.getUint32(absPointer, true);
    }

    return result;
  }

  /**
   * Get the pending stack as an array of values with types.
   *
   * @returns {Array<{type: string, value: *}>}
   */
  getPendingStack() {
    const result = [];
    const pendingBase = this.mem.getPendingBase();
    const pendingPointer = this.mem.getPendingPointer();
    const depth = (pendingPointer - pendingBase) / 16;

    for (let i = 0; i < depth; i++) {
      const ptr = pendingBase + i * 16;
      const type = this.mem.getValueType(ptr);
      const { value } = this.extractValueAt(ptr);

      // Convert type number to string
      const typeNames = {
        [TYPE.NULL]: 'null',
        [TYPE.UNDEFINED]: 'undefined',
        [TYPE.BOOLEAN]: 'boolean',
        [TYPE.INTEGER]: 'integer',
        [TYPE.FLOAT]: 'float',
        [TYPE.STRING]: 'string',
        [TYPE.ARRAY]: 'array',
        [TYPE.OBJECT]: 'object',
        [TYPE.CLOSURE]: 'closure',
        [TYPE.FFI_REF]: 'ffi_ref',
      };

      result.push({
        type: typeNames[type] ?? `type_${type}`,
        value,
      });
    }

    return result;
  }

  /**
   * Push an FFI result value to the pending stack.
   * Used after handling an FFI request.
   *
   * @param {*} value - JS value to push
   */
  pushFFIResult(value) {
    const ptr = this.writeValue(value);
    this.mem.pushPending(ptr);
  }

  /**
   * Set error state after FFI denial.
   *
   * @param {Error} error - Error to set
   */
  pushFFIError(error) {
    const errorMessage = error && error.message ? error.message : String(error);
    const errorPtr = this.writeString(errorMessage);
    this.mem.pushPending(errorPtr);
    this.mem.setStatus(STATUS_THROW);
  }

}
