/**
 * SandScript Fuel-Based Interpreter - Memory Manipulator
 *
 * Provides typed access to interpreter state in linear memory.
 * Supports segment model (multiple interpreters in one WebAssembly.Memory).
 */

import {
  HEADER,
  HEADER_SIZE,
  VERSION,
  STATE,
  STATE_SIZE,
  LAYOUT,
  REGION_SIZE,
  FRAME,
  FRAME_SIZE,
  VALUE_SIZE,
  TYPE,
  OBJ,
  GC_HEADER_SIZE,
  STATUS_RUNNING,
  SCRATCH_SIZE,
  CALL_STACK_SIZE,
  PENDING_STACK_SIZE,
  TRY_STACK_SIZE,
  TRY_ENTRY,
  TRY_ENTRY_SIZE,
  INSTRUCTION_SIZE,
  CODE_BLOCK_FLAG,
  BUILTIN_NAME,
  CLOSURE_FLAG_ARROW,
  METHOD,
  INVOCATION,
  INVOCATION_SIZE,
  MAX_INPUT_SLOTS,
  MAX_OUTPUT_SLOTS,
  SLOT,
} from './constants.js';

// Hash table size for string interning (must match interpreter.wat)
const HASH_TABLE_SIZE = 2048 * 8; // 16KB

/**
 * MemoryManipulator provides typed access to interpreter state.
 */
export class MemoryManipulator {
  /**
   * @param {WebAssembly.Memory} memory - WASM memory instance
   * @param {number} baseOffset - Start of this interpreter's segment (default 0)
   * @param {number|null} segmentSize - Size of segment (default: entire memory)
   */
  constructor(memory, baseOffset = 0, segmentSize = null) {
    this.memory = memory;
    this.baseOffset = baseOffset;
    this._refreshViews();
    this.segmentSize = segmentSize ?? (this.buffer.byteLength - baseOffset);

    // WASM instance (set by setWasmInstance after instantiation)
    this.wasm = null;
    this.encoder = new TextEncoder();
    this.decoder = new TextDecoder();
  }

  /**
   * Refresh typed array views after memory growth.
   */
  _refreshViews() {
    this.buffer = this.memory.buffer;
    this.view = new DataView(this.buffer);
    this.u8 = new Uint8Array(this.buffer);
  }

  /**
   * Set the WASM instance for string interning.
   *
   * @param {WebAssembly.Instance} instance - WASM instance
   */
  setWasmInstance(instance) {
    this.wasm = instance;
    this.wasm.exports.init_segment(this.baseOffset);
  }

  /**
   * Convert segment-relative offset to absolute memory offset.
   *
   * @param {number} offset - Segment-relative offset
   * @returns {number} - Absolute memory offset
   */
  abs(offset) {
    return this.baseOffset + offset;
  }

  // ===========================================================================
  // SANDFUEL Header
  // ===========================================================================

  /**
   * Get the magic bytes as a string.
   * @returns {string} - "SANDFUEL" if valid
   */
  getMagic() {
    const bytes = this.u8.slice(this.abs(HEADER.MAGIC), this.abs(HEADER.MAGIC) + 8);
    return this.decoder.decode(bytes);
  }

  /**
   * Write the magic bytes.
   */
  writeMagic() {
    const magic = this.encoder.encode('SANDFUEL');
    this.u8.set(magic, this.abs(HEADER.MAGIC));
  }

  /**
   * Validate the magic bytes.
   * @throws {Error} - If magic is invalid
   */
  validateMagic() {
    const magic = this.getMagic();
    if (magic !== 'SANDFUEL') {
      throw new Error(`Invalid magic: expected "SANDFUEL", got "${magic}"`);
    }
  }

  /**
   * Get layout version.
   * @returns {number}
   */
  getLayoutVersion() {
    return this.view.getUint16(this.abs(HEADER.LAYOUT_VERSION), true);
  }

  /**
   * Set layout version.
   * @param {number} v
   */
  setLayoutVersion(v) {
    this.view.setUint16(this.abs(HEADER.LAYOUT_VERSION), v, true);
  }

  /**
   * Get bytecode version.
   * @returns {number}
   */
  getBytecodeVersion() {
    return this.view.getUint16(this.abs(HEADER.BYTECODE_VERSION), true);
  }

  /**
   * Set bytecode version.
   * @param {number} v
   */
  setBytecodeVersion(v) {
    this.view.setUint16(this.abs(HEADER.BYTECODE_VERSION), v, true);
  }

  /**
   * Get type version.
   * @returns {number}
   */
  getTypeVersion() {
    return this.view.getUint16(this.abs(HEADER.TYPE_VERSION), true);
  }

  /**
   * Set type version.
   * @param {number} v
   */
  setTypeVersion(v) {
    this.view.setUint16(this.abs(HEADER.TYPE_VERSION), v, true);
  }

  /**
   * Get builtin version.
   * @returns {number}
   */
  getBuiltinVersion() {
    return this.view.getUint16(this.abs(HEADER.BUILTIN_VERSION), true);
  }

  /**
   * Set builtin version.
   * @param {number} v
   */
  setBuiltinVersion(v) {
    this.view.setUint16(this.abs(HEADER.BUILTIN_VERSION), v, true);
  }

  /**
   * Get all version fields.
   * @returns {object} - { layout, bytecode, type, builtin }
   */
  getVersions() {
    return {
      layout: this.getLayoutVersion(),
      bytecode: this.getBytecodeVersion(),
      type: this.getTypeVersion(),
      builtin: this.getBuiltinVersion(),
    };
  }

  // ===========================================================================
  // Initialization
  // ===========================================================================

  /**
   * Initialize memory layout for this segment.
   *
   * @param {object} options
   * @param {number} options.stringTableSize - Size reserved for strings (default 256KB)
   * @param {number} options.ffiRequestSize - FFI request region size
   * @param {number} options.errorInfoSize - Error info region size
   * @param {number} options.scratchSize - Scratch region size
   * @param {number} options.callStackSize - Call stack size
   * @param {number} options.pendingStackSize - Pending stack size
   * @param {number} options.tryStackSize - Try stack size
   * @param {number} options.builtinsSize - Builtins region size
   */
  initialize(options = {}) {
    const {
      stringTableSize = 256 * 1024,
      ffiRequestSize = REGION_SIZE.FFI_REQUEST,
      errorInfoSize = REGION_SIZE.ERROR_INFO,
      scratchSize = REGION_SIZE.SCRATCH,
      callStackSize = REGION_SIZE.CALL_STACK,
      pendingStackSize = REGION_SIZE.PENDING_STACK,
      tryStackSize = REGION_SIZE.TRY_STACK,
      builtinsSize = REGION_SIZE.BUILTINS,
    } = options;

    // Write SANDFUEL header
    this.writeMagic();
    this.setLayoutVersion(VERSION.LAYOUT);
    this.setBytecodeVersion(VERSION.BYTECODE);
    this.setTypeVersion(VERSION.TYPE);
    this.setBuiltinVersion(VERSION.BUILTIN);

    // Compute region offsets sequentially after STATE header
    let offset = HEADER_SIZE + STATE_SIZE;

    const ffiRequestBase = offset;
    offset += ffiRequestSize;

    const errorInfoBase = offset;
    offset += errorInfoSize;

    // INVOCATION region is at fixed offset (not configurable size)
    // Skip over it - it's initialized separately
    offset += INVOCATION_SIZE;

    const scratchBase = offset;
    offset += scratchSize;

    const callStackBase = offset;
    offset += callStackSize;

    const pendingStackBase = offset;
    offset += pendingStackSize;

    const tryStackBase = offset;
    offset += tryStackSize;

    const builtinsBase = offset;
    offset += builtinsSize;

    const heapStart = offset;

    // Calculate string table boundaries
    const stringStart = this.segmentSize - stringTableSize;
    const heapEnd = stringStart;

    // Validate regions fit in segment
    if (heapStart > heapEnd) {
      throw new Error(
        `Region layout exceeds segment size: regions need ${heapStart} bytes, ` +
        `but only ${heapEnd} available (segment ${this.segmentSize}, strings ${stringTableSize})`
      );
    }

    // String pointer starts after hash table
    const stringPointer = stringStart + HASH_TABLE_SIZE;

    // Write region base offsets to STATE
    this.view.setUint32(this.abs(STATE.FFI_REQUEST_BASE), ffiRequestBase, true);
    this.view.setUint32(this.abs(STATE.ERROR_INFO_BASE), errorInfoBase, true);
    this.view.setUint32(this.abs(STATE.SCRATCH_BASE), scratchBase, true);
    this.view.setUint32(this.abs(STATE.CALL_STACK_BASE), callStackBase, true);
    this.view.setUint32(this.abs(STATE.PENDING_STACK_BASE), pendingStackBase, true);
    this.view.setUint32(this.abs(STATE.TRY_STACK_BASE), tryStackBase, true);
    this.view.setUint32(this.abs(STATE.BUILTINS_BASE), builtinsBase, true);

    // Tell WASM to read region bases from STATE
    if (this.wasm) {
      this.wasm.exports.init_regions();
    }

    // Write state header
    this.view.setUint32(this.abs(STATE.HEAP_POINTER), heapStart, true);
    this.view.setUint32(this.abs(STATE.STRING_POINTER), stringPointer, true);
    this.view.setUint32(this.abs(STATE.SCOPE), 0, true);  // will be set after createScope
    this.view.setUint32(this.abs(STATE.RESULT_POINTER), 0, true);
    this.view.setUint32(this.abs(STATE.HEAP_START), heapStart, true);
    this.view.setUint32(this.abs(STATE.HEAP_END), heapEnd, true);
    this.view.setUint32(this.abs(STATE.STRING_START), stringStart, true);
    this.view.setUint32(this.abs(STATE.SEGMENT_SIZE), this.segmentSize, true);
    this.view.setUint32(this.abs(STATE.FUEL), 0, true);
    this.view.setUint32(this.abs(STATE.STATUS), STATUS_RUNNING, true);
    this.view.setUint32(this.abs(STATE.STACK_POINTER), callStackBase, true);
    this.view.setUint32(this.abs(STATE.STACK_BASE), callStackBase, true);
    this.view.setUint32(this.abs(STATE.PENDING_POINTER), pendingStackBase, true);
    this.view.setUint32(this.abs(STATE.PENDING_BASE), pendingStackBase, true);
    this.view.setUint32(this.abs(STATE.BASE_OFFSET), this.baseOffset, true);
    this.view.setUint32(this.abs(STATE.INSTRUCTION_INDEX), 0, true);
    this.view.setUint32(this.abs(STATE.TRY_POINTER), tryStackBase, true);
    this.view.setUint32(this.abs(STATE.TRY_BASE), tryStackBase, true);
    // Completion state (for finally blocks)
    this.view.setUint32(this.abs(STATE.COMPLETION_TYPE), 0, true); // COMPLETION_NORMAL
    this.view.setUint32(this.abs(STATE.COMPLETION_VALUE), pendingStackBase - 16, true); // reserved slot below pending stack

    // Code pointer starts just below STRING_START (room for header)
    // Layout: [instructions grow downward ← ][header 16B] STRING_START
    // Header is: [count 4B][flags 4B][GC header 8B]
    const codeStart = stringStart - 16; // 16 bytes for header
    this.view.setUint32(this.abs(STATE.CODE_POINTER), codeStart, true);
    this.view.setUint32(this.abs(STATE.CODE_BLOCK), codeStart, true); // WAT needs this for the run() check

    // Initialize the code block header at fixed position
    // GC header at codeStart + 8: [size | (type << 24)], [flags]
    // The size will be updated as instructions are added
    this.view.setUint32(this.abs(codeStart + 8), (OBJ.CODE_BLOCK << 24) | 16, true); // initial size 16
    this.view.setUint32(this.abs(codeStart + 12), 0, true); // GC forwarding pointer
    // Code block header: [count 4B][flags 4B]
    this.view.setUint32(this.abs(codeStart), 0, true); // instruction count
    this.view.setUint32(this.abs(codeStart + 4), 0, true); // flags

    // Initialize INVOCATION region (fixed offset, not configurable)
    this.view.setUint32(this.abs(LAYOUT.INVOCATION + INVOCATION.INPUT_COUNT), 0, true);
    this.view.setUint32(this.abs(LAYOUT.INVOCATION + INVOCATION.OUTPUT_COUNT), 0, true);

    // Initialize WASM string table if available
    if (this.wasm) {
      this.wasm.exports.init_string_table();

      // Pre-intern built-in names into BUILTINS segment
      this.internBuiltinNames();
    }

    // Create root scope (parent == 0 means this is the global/root scope)
    // Capacity stored in scope header, so changing this default has no versioning implications
    const rootScope = this.createScope(0, 128);
    this.view.setUint32(this.abs(STATE.SCOPE), rootScope, true);

    // Initialize built-in objects and global functions
    this.initializeBuiltins();
  }

  // ===========================================================================
  // State Header
  // ===========================================================================

  getFuel() {
    return this.view.getUint32(this.abs(STATE.FUEL), true);
  }

  setFuel(n) {
    this.view.setUint32(this.abs(STATE.FUEL), n, true);
  }

  getStatus() {
    return this.view.getUint32(this.abs(STATE.STATUS), true);
  }

  setStatus(s) {
    this.view.setUint32(this.abs(STATE.STATUS), s, true);
  }

  getHeapPointer() {
    return this.view.getUint32(this.abs(STATE.HEAP_POINTER), true);
  }

  setHeapPointer(v) {
    this.view.setUint32(this.abs(STATE.HEAP_POINTER), v, true);
  }

  getHeapStart() {
    return this.view.getUint32(this.abs(STATE.HEAP_START), true);
  }

  getHeapEnd() {
    return this.view.getUint32(this.abs(STATE.HEAP_END), true);
  }

  getStringPointer() {
    return this.view.getUint32(this.abs(STATE.STRING_POINTER), true);
  }

  setStringPointer(v) {
    this.view.setUint32(this.abs(STATE.STRING_POINTER), v, true);
  }

  getStringStart() {
    return this.view.getUint32(this.abs(STATE.STRING_START), true);
  }

  getStringEnd() {
    return this.segmentSize;
  }

  getScope() {
    return this.view.getUint32(this.abs(STATE.SCOPE), true);
  }

  setScope(v) {
    this.view.setUint32(this.abs(STATE.SCOPE), v, true);
  }

  getResultPointer() {
    return this.view.getUint32(this.abs(STATE.RESULT_POINTER), true);
  }

  setResultPointer(v) {
    this.view.setUint32(this.abs(STATE.RESULT_POINTER), v, true);
  }

  getStackPointer() {
    return this.view.getUint32(this.abs(STATE.STACK_POINTER), true);
  }

  setStackPointer(v) {
    this.view.setUint32(this.abs(STATE.STACK_POINTER), v, true);
  }

  getStackBase() {
    return this.view.getUint32(this.abs(STATE.STACK_BASE), true);
  }

  getPendingPointer() {
    return this.view.getUint32(this.abs(STATE.PENDING_POINTER), true);
  }

  setPendingPointer(v) {
    this.view.setUint32(this.abs(STATE.PENDING_POINTER), v, true);
  }

  getPendingBase() {
    return this.view.getUint32(this.abs(STATE.PENDING_BASE), true);
  }

  getCodePointer() {
    return this.view.getUint32(this.abs(STATE.CODE_POINTER), true);
  }

  setCodePointer(v) {
    this.view.setUint32(this.abs(STATE.CODE_POINTER), v, true);
  }

  // ===========================================================================
  // Region Base Offsets
  // ===========================================================================

  getFfiRequestBase() {
    return this.view.getUint32(this.abs(STATE.FFI_REQUEST_BASE), true);
  }

  getErrorInfoBase() {
    return this.view.getUint32(this.abs(STATE.ERROR_INFO_BASE), true);
  }

  getScratchBase() {
    return this.view.getUint32(this.abs(STATE.SCRATCH_BASE), true);
  }

  getCallStackBase() {
    return this.view.getUint32(this.abs(STATE.CALL_STACK_BASE), true);
  }

  getPendingStackBase() {
    return this.view.getUint32(this.abs(STATE.PENDING_STACK_BASE), true);
  }

  getTryStackBase() {
    return this.view.getUint32(this.abs(STATE.TRY_STACK_BASE), true);
  }

  getBuiltinsBase() {
    return this.view.getUint32(this.abs(STATE.BUILTINS_BASE), true);
  }

  // ===========================================================================
  // Call Stack
  // ===========================================================================

  /**
   * Push a new frame onto the call stack.
   *
   * @param {number} codeBlock - Pointer to CodeBlock
   * @param {number} instructionIndex - Return instruction index
   * @param {number} scopePointer - Pointer to scope
   * @param {number} sourceStart - Source start position
   * @param {number} sourceEnd - Source end position
   * @returns {number} - Frame index
   */
  pushFrame(codeBlock, instructionIndex, scopePointer, sourceStart, sourceEnd) {
    const stackPointer = this.getStackPointer();
    const callStackBase = this.getCallStackBase();
    const stackEnd = callStackBase + CALL_STACK_SIZE;

    if (stackPointer + FRAME_SIZE > stackEnd) {
      throw new Error('Call stack overflow');
    }

    const framePointer = this.abs(stackPointer);
    const pendingPointer = this.getPendingPointer();

    this.view.setUint32(framePointer + FRAME.CODE_BLOCK, codeBlock, true);
    this.view.setUint32(framePointer + FRAME.INSTRUCTION_INDEX, instructionIndex, true);
    this.view.setUint32(framePointer + FRAME.SCOPE_POINTER, scopePointer, true);
    this.view.setUint32(framePointer + FRAME.PENDING_BASE, pendingPointer, true);
    this.view.setUint32(framePointer + FRAME.PENDING_COUNT, 0, true);
    this.view.setUint32(framePointer + FRAME.FLAGS, 0, true);
    this.view.setUint32(framePointer + FRAME.SOURCE_START, sourceStart, true);
    this.view.setUint32(framePointer + FRAME.SOURCE_END, sourceEnd, true);

    this.setStackPointer(stackPointer + FRAME_SIZE);

    return (stackPointer - callStackBase) / FRAME_SIZE;
  }

  /**
   * Pop the top frame from the call stack.
   *
   * @returns {object} - Frame data
   */
  popFrame() {
    const stackPointer = this.getStackPointer();
    const stackBase = this.getStackBase();

    if (stackPointer <= stackBase) {
      throw new Error('Call stack underflow');
    }

    const newStackPointer = stackPointer - FRAME_SIZE;
    this.setStackPointer(newStackPointer);

    return this.getFrame((newStackPointer - this.getCallStackBase()) / FRAME_SIZE);
  }

  /**
   * Get frame at index.
   *
   * @param {number} index - Frame index
   * @returns {object} - Frame data
   */
  getFrame(index) {
    const frameOffset = this.getCallStackBase() + index * FRAME_SIZE;
    const framePointer = this.abs(frameOffset);

    return {
      codeBlock: this.view.getUint32(framePointer + FRAME.CODE_BLOCK, true),
      instructionIndex: this.view.getUint32(framePointer + FRAME.INSTRUCTION_INDEX, true),
      scopePointer: this.view.getUint32(framePointer + FRAME.SCOPE_POINTER, true),
      pendingBase: this.view.getUint32(framePointer + FRAME.PENDING_BASE, true),
      pendingCount: this.view.getUint32(framePointer + FRAME.PENDING_COUNT, true),
      flags: this.view.getUint32(framePointer + FRAME.FLAGS, true),
      sourceStart: this.view.getUint32(framePointer + FRAME.SOURCE_START, true),
      sourceEnd: this.view.getUint32(framePointer + FRAME.SOURCE_END, true),
    };
  }

  /**
   * Get current (top) frame, or null if stack is empty.
   *
   * @returns {object|null} - Frame data or null
   */
  getCurrentFrame() {
    const stackPointer = this.getStackPointer();
    const stackBase = this.getStackBase();

    if (stackPointer <= stackBase) {
      return null;
    }

    const index = (stackPointer - FRAME_SIZE - this.getCallStackBase()) / FRAME_SIZE;
    return this.getFrame(index);
  }

  /**
   * Get all frames on the call stack.
   *
   * @returns {Array<object>} - Array of frame data
   */
  getCallStack() {
    const frames = [];
    const stackPointer = this.getStackPointer();
    const stackBase = this.getStackBase();
    const count = (stackPointer - stackBase) / FRAME_SIZE;

    for (let i = 0; i < count; i++) {
      frames.push(this.getFrame(i));
    }

    return frames;
  }

  /**
   * Set child index for a frame.
   *
   * @param {number} index - Frame index
   * @param {number} value - New child index value
   */
  setFrameChildIndex(index, value) {
    const frameOffset = this.getCallStackBase() + index * FRAME_SIZE;
    const framePointer = this.abs(frameOffset);
    this.view.setUint32(framePointer + FRAME.CHILD_INDEX, value, true);
  }

  /**
   * Get call stack depth (number of frames).
   *
   * @returns {number}
   */
  getCallStackDepth() {
    const stackPointer = this.getStackPointer();
    const stackBase = this.getStackBase();
    return (stackPointer - stackBase) / FRAME_SIZE;
  }

  // ===========================================================================
  // Pending Values Stack
  // ===========================================================================

  /**
   * Push a value onto the pending stack.
   * Copies 16 bytes from the given address.
   *
   * @param {number} valuePointer - Segment-relative pointer to value
   */
  pushPending(valuePointer) {
    const pendingPointer = this.getPendingPointer();
    const pendingEnd = this.getPendingStackBase() + PENDING_STACK_SIZE;

    if (pendingPointer + VALUE_SIZE > pendingEnd) {
      throw new Error('Pending stack overflow');
    }

    // Copy 16 bytes
    const src = this.abs(valuePointer);
    const dst = this.abs(pendingPointer);
    for (let i = 0; i < VALUE_SIZE; i++) {
      this.u8[dst + i] = this.u8[src + i];
    }

    this.setPendingPointer(pendingPointer + VALUE_SIZE);

    // Update current frame's pending count
    const stackPointer = this.getStackPointer();
    if (stackPointer > this.getStackBase()) {
      const framePointer = this.abs(stackPointer - FRAME_SIZE);
      const count = this.view.getUint32(framePointer + FRAME.PENDING_COUNT, true);
      this.view.setUint32(framePointer + FRAME.PENDING_COUNT, count + 1, true);
    }
  }

  /**
   * Pop a value from the pending stack.
   *
   * @returns {number} - Segment-relative pointer to popped value (still in memory)
   */
  popPending() {
    const pendingPointer = this.getPendingPointer();
    const pendingBase = this.getPendingBase();

    if (pendingPointer <= pendingBase) {
      throw new Error('Pending stack underflow');
    }

    const newPendingPointer = pendingPointer - VALUE_SIZE;
    this.setPendingPointer(newPendingPointer);

    // Update current frame's pending count
    const stackPointer = this.getStackPointer();
    if (stackPointer > this.getStackBase()) {
      const framePointer = this.abs(stackPointer - FRAME_SIZE);
      const count = this.view.getUint32(framePointer + FRAME.PENDING_COUNT, true);
      if (count > 0) {
        this.view.setUint32(framePointer + FRAME.PENDING_COUNT, count - 1, true);
      }
    }

    return newPendingPointer;
  }

  /**
   * Get pending stack depth (number of values).
   *
   * @returns {number}
   */
  getPendingDepth() {
    const pendingPointer = this.getPendingPointer();
    const pendingBase = this.getPendingBase();
    return (pendingPointer - pendingBase) / VALUE_SIZE;
  }

  // ===========================================================================
  // Heap Allocation
  // ===========================================================================

  /**
   * Allocate memory on the heap with a GC header.
   *
   * @param {number} dataSize - Size of data (not including header)
   * @param {number} objType - Object type (OBJ.*)
   * @returns {number} - Segment-relative pointer to allocated memory (after header)
   */
  allocate(dataSize, objType) {
    const totalSize = GC_HEADER_SIZE + dataSize;
    const aligned = (totalSize + 15) & ~15; // 16-byte alignment

    const heapPointer = this.getHeapPointer();
    const heapEnd = this.getHeapEnd();

    if (heapPointer + aligned > heapEnd) {
      throw new Error('Heap overflow');
    }

    const absPointer = this.abs(heapPointer);

    // Write GC header
    // Header word: mark bit (1) + type (7) + size (24)
    const headerWord = (objType << 24) | (aligned & 0x00ffffff);
    this.view.setUint32(absPointer, headerWord, true);
    this.view.setUint32(absPointer + 4, 0, true); // forwarding pointer

    this.setHeapPointer(heapPointer + aligned);

    // Return pointer to data (after header)
    return heapPointer + GC_HEADER_SIZE;
  }

  // ===========================================================================
  // CodeBlock Methods (Reverse-Growing)
  //
  // The code block grows backward from STRING_START. Layout:
  //
  //   [heap →→→→→→    ←←← code block][string table]
  //                  ↑               ↑
  //            CODE_POINTER      STRING_START
  //
  // Code block structure (header at high address, instructions grow down):
  //   [instr N]...[instr 1][instr 0][count 4B][flags 4B][GC header 8B]
  //   ↑                             ↑                                ↑
  //   CODE_POINTER              INSTR_0_PTR                    CODE_START
  //
  // CODE_START = STRING_START - 16 (fixed position for header)
  // INSTR_0_PTR = CODE_START - 8 - 8 = STRING_START - 24 (just below header)
  // Actually: header is [count:4][flags:4][gc_type:4][gc_fwd:4] = 16 bytes
  // So INSTR_0_PTR = STRING_START - 16 (first instruction slot)
  // ===========================================================================

  /**
   * Get the fixed code block start (header position).
   * @returns {number} - Segment-relative pointer to code block header
   */
  getCodeStart() {
    return this.getStringStart() - 16;
  }

  /**
   * Get pointer to instruction 0 position (just below header).
   * @returns {number} - Segment-relative pointer
   */
  getInstr0Ptr() {
    return this.getCodeStart(); // instruction 0 is at header - 16, but we store [count][flags] at header
  }

  /**
   * Get current CodeBlock pointer from state.
   * For reverse-growing model, this always returns the fixed code block start.
   * @returns {number} - Segment-relative pointer to CodeBlock data
   */
  getCodeBlock() {
    // Return pointer to [count][flags] at CODE_START
    return this.getCodeStart();
  }

  /**
   * Set current CodeBlock pointer in state.
   * For reverse-growing model, this sets CODE_BLOCK to the fixed code start.
   * @param {number} ptr - Segment-relative pointer (ignored, we use CODE_START)
   */
  setCodeBlock(ptr) {
    // Always use the fixed code start position
    this.view.setUint32(this.abs(STATE.CODE_BLOCK), this.getCodeStart(), true);
  }

  /**
   * Get current instruction index from state.
   * @returns {number} - Instruction index
   */
  getInstructionIndex() {
    return this.view.getUint32(this.abs(STATE.INSTRUCTION_INDEX), true);
  }

  /**
   * Set current instruction index in state.
   * @param {number} index - Instruction index
   */
  setInstructionIndex(index) {
    this.view.setUint32(this.abs(STATE.INSTRUCTION_INDEX), index, true);
  }

  /**
   * Reset the code block for re-parsing.
   * Resets CODE_POINTER to just below header and clears instruction count.
   * Does NOT touch heap - preserves runtime objects (scopes, arrays, etc).
   */
  resetCodeBlock() {
    const codeStart = this.getCodeStart();

    // Reset code pointer to just below header
    this.setCodePointer(codeStart);

    // Reset instruction count and flags
    this.view.setUint32(this.abs(codeStart), 0, true); // count = 0
    this.view.setUint32(this.abs(codeStart + 4), 0, true); // flags = 0

    // Reset GC header size to 16 (header only)
    this.view.setUint32(this.abs(codeStart + 8), (OBJ.CODE_BLOCK << 24) | 16, true);
  }

  /**
   * Allocate a new CodeBlock.
   * For reverse-growing model, this just resets the single code block.
   * @returns {number} - Segment-relative pointer to CodeBlock data
   */
  allocateCodeBlock() {
    this.resetCodeBlock();
    return this.getCodeBlock();
  }

  /**
   * Get instruction count for the code block.
   * @param {number} pointer - Segment-relative pointer (ignored, there's only one)
   * @returns {number} - Number of instructions
   */
  codeBlockInstructionCount(pointer) {
    const codeStart = this.getCodeStart();
    return this.view.getUint32(this.abs(codeStart), true);
  }

  /**
   * Truncate the code block to a specific instruction count.
   * Used for rollback on parse errors.
   * @param {number} pointer - Segment-relative pointer (ignored)
   * @param {number} count - Number of instructions to keep
   */
  codeBlockTruncate(pointer, count) {
    const codeStart = this.getCodeStart();
    const currentCount = this.view.getUint32(this.abs(codeStart), true);

    if (count < currentCount) {
      // Update count
      this.view.setUint32(this.abs(codeStart), count, true);

      // Reclaim space: CODE_POINTER = codeStart - count * INSTRUCTION_SIZE
      const newCodePointer = codeStart - count * INSTRUCTION_SIZE;
      this.setCodePointer(newCodePointer);

      // Update GC header size
      const headerSize = 16 + count * INSTRUCTION_SIZE;
      this.view.setUint32(this.abs(codeStart + 8), (OBJ.CODE_BLOCK << 24) | headerSize, true);
    }
  }

  /**
   * Get flags for the code block.
   * @param {number} pointer - Segment-relative pointer (ignored)
   * @returns {number} - Flags
   */
  codeBlockGetFlags(pointer) {
    const codeStart = this.getCodeStart();
    return this.view.getUint32(this.abs(codeStart + 4), true);
  }

  /**
   * Set flags for the code block.
   * @param {number} pointer - Segment-relative pointer (ignored)
   * @param {number} flags - Flags to set
   */
  codeBlockSetFlags(pointer, flags) {
    const codeStart = this.getCodeStart();
    this.view.setUint32(this.abs(codeStart + 4), flags, true);
  }

  /**
   * Emit an instruction to the code block.
   * Instructions grow downward from CODE_POINTER.
   *
   * @param {number} opcode - Instruction opcode (u8)
   * @param {number} operand1 - First operand (u32)
   * @param {number} operand2 - Second operand (u32)
   * @param {number} sourceStart - Source start position (u16)
   * @param {number} sourceEnd - Source end position (u16)
   * @param {number} flags - Instruction flags (u8)
   * @returns {number} - Instruction index
   */
  emitInstruction(opcode, operand1 = 0, operand2 = 0, sourceStart = 0, sourceEnd = 0, flags = 0) {
    const codePointer = this.getCodePointer();
    const heapPointer = this.getHeapPointer();

    // Collision check
    if (codePointer - INSTRUCTION_SIZE < heapPointer) {
      throw new Error('Out of memory: code block collided with heap');
    }

    // Decrement code pointer
    const newCodePointer = codePointer - INSTRUCTION_SIZE;
    this.setCodePointer(newCodePointer);

    // Write instruction at new code pointer
    const instrPointer = this.abs(newCodePointer);
    this.view.setUint8(instrPointer, opcode);
    this.view.setUint8(instrPointer + 1, flags);
    this.view.setUint16(instrPointer + 2, sourceStart, true);
    this.view.setUint16(instrPointer + 4, sourceEnd, true);
    this.view.setUint16(instrPointer + 6, 0, true); // reserved
    this.view.setUint32(instrPointer + 8, operand1, true);
    this.view.setUint32(instrPointer + 12, operand2, true);

    // Increment instruction count
    const codeStart = this.getCodeStart();
    const count = this.view.getUint32(this.abs(codeStart), true);
    this.view.setUint32(this.abs(codeStart), count + 1, true);

    // Update GC header size
    const headerSize = 16 + (count + 1) * INSTRUCTION_SIZE;
    this.view.setUint32(this.abs(codeStart + 8), (OBJ.CODE_BLOCK << 24) | headerSize, true);

    return count;
  }

  /**
   * Append an instruction to a CodeBlock (legacy API).
   * Delegates to emitInstruction.
   */
  codeBlockAppend(pointer, opcode, operand1 = 0, operand2 = 0, sourceStart = 0, sourceEnd = 0, flags = 0) {
    return this.emitInstruction(opcode, operand1, operand2, sourceStart, sourceEnd, flags);
  }

  /**
   * Patch an instruction's operand1.
   *
   * @param {number} pointer - Segment-relative pointer (ignored)
   * @param {number} instructionIndex - Which instruction to patch
   * @param {number} value - New value for operand1
   */
  codeBlockPatch(pointer, instructionIndex, value) {
    // Instructions grow downward: instr_address = codeStart - (index + 1) * INSTRUCTION_SIZE
    const codeStart = this.getCodeStart();
    const instrOffset = codeStart - (instructionIndex + 1) * INSTRUCTION_SIZE;
    this.view.setUint32(this.abs(instrOffset + 8), value, true);
  }

  /**
   * Read an instruction from the code block.
   *
   * @param {number} pointer - Segment-relative pointer (ignored)
   * @param {number} instructionIndex - Which instruction to read
   * @returns {object} - Instruction data
   */
  codeBlockReadInstruction(pointer, instructionIndex) {
    // Instructions grow downward: instr_address = codeStart - (index + 1) * INSTRUCTION_SIZE
    const codeStart = this.getCodeStart();
    const instrOffset = codeStart - (instructionIndex + 1) * INSTRUCTION_SIZE;
    const instrPointer = this.abs(instrOffset);

    return {
      opcode: this.view.getUint8(instrPointer),
      flags: this.view.getUint8(instrPointer + 1),
      sourceStart: this.view.getUint16(instrPointer + 2, true),
      sourceEnd: this.view.getUint16(instrPointer + 4, true),
      operand1: this.view.getUint32(instrPointer + 8, true),
      operand2: this.view.getUint32(instrPointer + 12, true),
    };
  }

  // ===========================================================================
  // Try Stack Methods
  // ===========================================================================

  /**
   * Get try stack pointer.
   * @returns {number} - Segment-relative pointer
   */
  getTryPointer() {
    return this.view.getUint32(this.abs(STATE.TRY_POINTER), true);
  }

  /**
   * Set try stack pointer.
   * @param {number} ptr - Segment-relative pointer
   */
  setTryPointer(ptr) {
    this.view.setUint32(this.abs(STATE.TRY_POINTER), ptr, true);
  }

  /**
   * Get try stack base.
   * @returns {number} - Segment-relative pointer
   */
  getTryBase() {
    return this.view.getUint32(this.abs(STATE.TRY_BASE), true);
  }

  /**
   * Get completion type.
   * @returns {number} - 0=normal, 1=throw, 2=return
   */
  getCompletionType() {
    return this.view.getUint32(this.abs(STATE.COMPLETION_TYPE), true);
  }

  /**
   * Set completion type.
   * @param {number} type - 0=normal, 1=throw, 2=return
   */
  setCompletionType(type) {
    this.view.setUint32(this.abs(STATE.COMPLETION_TYPE), type, true);
  }

  /**
   * Get completion value pointer.
   * Points to a 16-byte value stored at PENDING_BASE - 16.
   * @returns {number} - Segment-relative pointer to value
   */
  getCompletionValue() {
    return this.view.getUint32(this.abs(STATE.COMPLETION_VALUE), true);
  }

  /**
   * Set completion value pointer.
   * @param {number} ptr - Segment-relative pointer to value
   */
  setCompletionValue(ptr) {
    this.view.setUint32(this.abs(STATE.COMPLETION_VALUE), ptr, true);
  }

  /**
   * Get try stack depth.
   * @returns {number} - Number of entries
   */
  getTryDepth() {
    return (this.getTryPointer() - this.getTryBase()) / TRY_ENTRY_SIZE;
  }

  /**
   * Push a try entry onto the try stack.
   *
   * @param {number} codeBlock - CodeBlock pointer
   * @param {number} catchIndex - Catch handler instruction index (0 if none)
   * @param {number} finallyIndex - Finally handler instruction index (0 if none)
   * @returns {number} - Entry index
   */
  pushTryEntry(codeBlock, catchIndex, finallyIndex) {
    const tryPointer = this.getTryPointer();
    const tryStackBase = this.getTryStackBase();
    const tryEnd = tryStackBase + TRY_STACK_SIZE;

    if (tryPointer + TRY_ENTRY_SIZE > tryEnd) {
      throw new Error('Try stack overflow');
    }

    const entryPointer = this.abs(tryPointer);
    const frameDepth = this.getCallStackDepth();

    this.view.setUint32(entryPointer + TRY_ENTRY.CODE_BLOCK, codeBlock, true);
    this.view.setUint32(entryPointer + TRY_ENTRY.CATCH_INDEX, catchIndex, true);
    this.view.setUint32(entryPointer + TRY_ENTRY.FINALLY_INDEX, finallyIndex, true);
    this.view.setUint32(entryPointer + TRY_ENTRY.FRAME_DEPTH, frameDepth, true);

    this.setTryPointer(tryPointer + TRY_ENTRY_SIZE);

    return (tryPointer - tryStackBase) / TRY_ENTRY_SIZE;
  }

  /**
   * Pop the top try entry from the try stack.
   *
   * @returns {object} - Try entry data
   */
  popTryEntry() {
    const tryPointer = this.getTryPointer();
    const tryBase = this.getTryBase();

    if (tryPointer <= tryBase) {
      throw new Error('Try stack underflow');
    }

    const newTryPointer = tryPointer - TRY_ENTRY_SIZE;
    this.setTryPointer(newTryPointer);

    return this.getTryEntry((newTryPointer - this.getTryStackBase()) / TRY_ENTRY_SIZE);
  }

  /**
   * Peek the top try entry without popping.
   *
   * @returns {object|null} - Try entry data or null if empty
   */
  peekTryEntry() {
    const depth = this.getTryDepth();
    if (depth === 0) return null;
    return this.getTryEntry(depth - 1);
  }

  /**
   * Get try entry at index.
   *
   * @param {number} index - Entry index
   * @returns {object} - Try entry data
   */
  getTryEntry(index) {
    const entryOffset = this.getTryStackBase() + index * TRY_ENTRY_SIZE;
    const entryPointer = this.abs(entryOffset);

    return {
      codeBlock: this.view.getUint32(entryPointer + TRY_ENTRY.CODE_BLOCK, true),
      catchIndex: this.view.getUint32(entryPointer + TRY_ENTRY.CATCH_INDEX, true),
      finallyIndex: this.view.getUint32(entryPointer + TRY_ENTRY.FINALLY_INDEX, true),
      frameDepth: this.view.getUint32(entryPointer + TRY_ENTRY.FRAME_DEPTH, true),
    };
  }

  // ===========================================================================
  // Values
  // ===========================================================================

  /**
   * Write a null value.
   *
   * @param {number} addr - Segment-relative address
   */
  writeNull(addr) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.NULL, true);
    this.view.setUint32(absAddr + 4, 0, true);
    this.view.setBigInt64(absAddr + 8, 0n, true);
  }

  /**
   * Write an undefined value.
   *
   * @param {number} addr - Segment-relative address
   */
  writeUndefined(addr) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.UNDEFINED, true);
    this.view.setUint32(absAddr + 4, 0, true);
    this.view.setBigInt64(absAddr + 8, 0n, true);
  }

  /**
   * Write a boolean value.
   *
   * @param {number} addr - Segment-relative address
   * @param {boolean} v - Boolean value
   */
  writeBoolean(addr, v) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.BOOLEAN, true);
    this.view.setUint32(absAddr + 4, 0, true);
    this.view.setBigInt64(absAddr + 8, v ? 1n : 0n, true);
  }

  /**
   * Write an integer value.
   *
   * @param {number} addr - Segment-relative address
   * @param {number|bigint} v - Integer value
   */
  writeInteger(addr, v) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.INTEGER, true);
    this.view.setUint32(absAddr + 4, 0, true);
    this.view.setBigInt64(absAddr + 8, BigInt(v), true);
  }

  /**
   * Write a float value.
   *
   * @param {number} addr - Segment-relative address
   * @param {number} v - Float value
   */
  writeFloat(addr, v) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.FLOAT, true);
    this.view.setUint32(absAddr + 4, 0, true);
    this.view.setFloat64(absAddr + 8, v, true);
  }

  /**
   * Write a string value.
   *
   * @param {number} addr - Segment-relative address
   * @param {number} stringOffset - String table offset
   */
  writeString(addr, stringOffset) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.STRING, true);
    this.view.setUint32(absAddr + 4, 0, true);
    this.view.setBigInt64(absAddr + 8, BigInt(stringOffset), true);
  }

  /**
   * Write an array value (pointer to heap array).
   *
   * @param {number} addr - Segment-relative address for VALUE slot
   * @param {number} arrayPtr - Segment-relative pointer to heap array
   */
  writeArray(addr, arrayPtr) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.ARRAY, true);
    this.view.setUint32(absAddr + 4, 0, true);
    this.view.setBigInt64(absAddr + 8, BigInt(arrayPtr), true);
  }

  /**
   * Write an object value (pointer to heap object).
   *
   * @param {number} addr - Segment-relative address for VALUE slot
   * @param {number} objPtr - Segment-relative pointer to heap object
   */
  writeObject(addr, objPtr) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.OBJECT, true);
    this.view.setUint32(absAddr + 4, 0, true);
    this.view.setBigInt64(absAddr + 8, BigInt(objPtr), true);
  }

  /**
   * Read a value.
   *
   * @param {number} addr - Segment-relative address
   * @returns {object} - { type, payload }
   */
  getValue(addr) {
    const absAddr = this.abs(addr);
    const type = this.view.getUint32(absAddr, true);

    let payload;
    if (type === TYPE.FLOAT) {
      payload = this.view.getFloat64(absAddr + 8, true);
    } else {
      payload = this.view.getBigInt64(absAddr + 8, true);
    }

    return { type, payload };
  }

  /**
   * Get value type.
   *
   * @param {number} addr - Segment-relative address
   * @returns {number} - Type tag
   */
  getValueType(addr) {
    return this.view.getUint32(this.abs(addr), true);
  }

  /**
   * Get value pointer (for heap-allocated values).
   *
   * @param {number} addr - Segment-relative address
   * @returns {number} - Pointer value
   */
  getValuePointer(addr) {
    return Number(this.view.getBigInt64(this.abs(addr) + 8, true));
  }

  // ===========================================================================
  // Strings
  // ===========================================================================

  /**
   * Intern a string using WASM.
   *
   * @param {string} str - String to intern
   * @returns {number} - Segment-relative offset to string entry
   */
  internString(str) {
    if (!this.wasm) {
      throw new Error('WASM instance not set - call setWasmInstance first');
    }

    const bytes = this.encoder.encode(str);

    if (bytes.length > SCRATCH_SIZE) {
      throw new Error(`String too long for interning: ${bytes.length} bytes (max ${SCRATCH_SIZE})`);
    }

    const scratchPointer = this.abs(this.getScratchBase());
    this.u8.set(bytes, scratchPointer);
    return this.wasm.exports.intern_string(scratchPointer, bytes.length);
  }

  /**
   * Read a string from the string table.
   *
   * @param {number} offset - Segment-relative offset to string entry
   * @returns {string} - The string
   */
  readString(offset) {
    const absOffset = this.abs(offset);
    const length = this.view.getUint32(absOffset, true);
    const bytes = this.u8.slice(absOffset + 4, absOffset + 4 + length);
    return this.decoder.decode(bytes);
  }

  /**
   * Pre-intern all built-in names into the BUILTINS segment.
   * Called during initialization after WASM is ready.
   */
  internBuiltinNames() {
    const builtinsBase = this.getBuiltinsBase();

    // Helper to intern and store at builtin offset
    const intern = (offset, str) => {
      const stringOffset = this.internString(str);
      this.view.setUint32(this.abs(builtinsBase + offset), stringOffset, true);
    };

    // Typeof strings
    intern(BUILTIN_NAME.TYPEOF_UNDEFINED, 'undefined');
    intern(BUILTIN_NAME.TYPEOF_BOOLEAN, 'boolean');
    intern(BUILTIN_NAME.TYPEOF_NUMBER, 'number');
    intern(BUILTIN_NAME.TYPEOF_STRING, 'string');
    intern(BUILTIN_NAME.TYPEOF_OBJECT, 'object');
    intern(BUILTIN_NAME.TYPEOF_FUNCTION, 'function');

    // Special keywords
    intern(BUILTIN_NAME.THIS, 'this');
    intern(BUILTIN_NAME.LENGTH, 'length');

    // Array methods
    intern(BUILTIN_NAME.PUSH, 'push');
    intern(BUILTIN_NAME.POP, 'pop');
    intern(BUILTIN_NAME.SHIFT, 'shift');
    intern(BUILTIN_NAME.UNSHIFT, 'unshift');
    intern(BUILTIN_NAME.SLICE, 'slice');
    intern(BUILTIN_NAME.CONCAT, 'concat');
    intern(BUILTIN_NAME.JOIN, 'join');
    intern(BUILTIN_NAME.REVERSE, 'reverse');
    intern(BUILTIN_NAME.INDEX_OF, 'indexOf');
    intern(BUILTIN_NAME.INCLUDES, 'includes');
    intern(BUILTIN_NAME.MAP, 'map');
    intern(BUILTIN_NAME.FILTER, 'filter');
    intern(BUILTIN_NAME.REDUCE, 'reduce');
    intern(BUILTIN_NAME.FOR_EACH, 'forEach');
    intern(BUILTIN_NAME.FIND, 'find');
    intern(BUILTIN_NAME.FIND_INDEX, 'findIndex');
    intern(BUILTIN_NAME.SOME, 'some');
    intern(BUILTIN_NAME.EVERY, 'every');

    // String methods
    intern(BUILTIN_NAME.CHAR_AT, 'charAt');
    intern(BUILTIN_NAME.CHAR_CODE_AT, 'charCodeAt');
    intern(BUILTIN_NAME.STRING_INDEX_OF, 'indexOf');
    intern(BUILTIN_NAME.STRING_INCLUDES, 'includes');
    intern(BUILTIN_NAME.STRING_SLICE, 'slice');
    intern(BUILTIN_NAME.SUBSTRING, 'substring');
    intern(BUILTIN_NAME.SPLIT, 'split');
    intern(BUILTIN_NAME.TRIM, 'trim');
    intern(BUILTIN_NAME.TO_LOWER_CASE, 'toLowerCase');
    intern(BUILTIN_NAME.TO_UPPER_CASE, 'toUpperCase');
    intern(BUILTIN_NAME.STARTS_WITH, 'startsWith');
    intern(BUILTIN_NAME.ENDS_WITH, 'endsWith');
    intern(BUILTIN_NAME.REPEAT, 'repeat');
    intern(BUILTIN_NAME.PAD_START, 'padStart');
    intern(BUILTIN_NAME.PAD_END, 'padEnd');
    intern(BUILTIN_NAME.REPLACE, 'replace');

    // Literal strings for type conversion
    intern(BUILTIN_NAME.LIT_NULL, 'null');
    intern(BUILTIN_NAME.LIT_TRUE, 'true');
    intern(BUILTIN_NAME.LIT_FALSE, 'false');
    intern(BUILTIN_NAME.LIT_OBJECT_OBJECT, '[object Object]');
    intern(BUILTIN_NAME.LIT_NAN, 'NaN');
    intern(BUILTIN_NAME.LIT_INFINITY, 'Infinity');
    intern(BUILTIN_NAME.LIT_NEG_INFINITY, '-Infinity');
    intern(BUILTIN_NAME.LIT_OBJECT_FUNCTION, '[object Function]');
    intern(BUILTIN_NAME.LIT_EMPTY_STRING, '');
  }

  /**
   * Read a built-in name pointer from the BUILTINS segment.
   *
   * @param {number} offset - Offset within BUILTINS segment (e.g., BUILTIN_NAME.LENGTH)
   * @returns {number} - String table offset for the interned name
   */
  getBuiltinName(offset) {
    return this.view.getUint32(this.abs(this.getBuiltinsBase() + offset), true);
  }

  // ===========================================================================
  // Scopes
  // ===========================================================================

  /**
   * Create a new scope.
   *
   * Scope layout: [GC 8][parent_pointer 4][count 4][capacity 4][entries_pointer 4]
   * Entries: array of [name_offset 4][value_pointer 4] pairs
   *
   * @param {number} parentPointer - Parent scope pointer (0 for global)
   * @returns {number} - Segment-relative pointer to scope
   */
  createScope(parentPointer, capacity = 8) {
    // Each entry is 20 bytes: [name_ptr:4][type:4][flags:4][data_lo:4][data_hi:4]
    const entriesSize = capacity * 20;

    // Allocate entries array (allocate returns DATA pointer, we need HEADER pointer)
    const entriesDataPointer = this.allocate(entriesSize, OBJ.ARRAY_DATA);
    const entriesHeaderPointer = entriesDataPointer - GC_HEADER_SIZE;

    // Allocate scope header
    const scopePointer = this.allocate(16, OBJ.SCOPE);
    const absPointer = this.abs(scopePointer);

    this.view.setUint32(absPointer, parentPointer, true);
    this.view.setUint32(absPointer + 4, 0, true); // count
    this.view.setUint32(absPointer + 8, capacity, true);
    // Store HEADER pointer for entries (WAT expects header pointer and adds GC_HEADER_SIZE to access data)
    this.view.setUint32(absPointer + 12, entriesHeaderPointer, true);

    return scopePointer;
  }

  /**
   * Define a variable in a scope.
   *
   * @param {number} scopePointer - Scope pointer
   * @param {number} nameOffset - String table offset for name
   * @param {number} valuePointer - Pointer to value
   */
  scopeDefine(scopePointer, nameOffset, valuePointer) {
    const absPointer = this.abs(scopePointer);
    const count = this.view.getUint32(absPointer + 4, true);
    const capacity = this.view.getUint32(absPointer + 8, true);
    const entriesPointer = this.view.getUint32(absPointer + 12, true);

    if (count >= capacity) {
      throw new Error('Scope capacity exceeded (TODO: grow)');
    }

    // Entry is 20 bytes: [name_ptr:4][type:4][flags:4][data_lo:4][data_hi:4]
    // Entry data starts after GC header
    const entryOffset = entriesPointer + GC_HEADER_SIZE + count * 20;
    const absEntryPointer = this.abs(entryOffset);

    // Read value from valuePointer and copy inline
    const absValuePointer = this.abs(valuePointer);
    const type = this.view.getUint32(absValuePointer, true);
    const flags = this.view.getUint32(absValuePointer + 4, true);
    const dataLo = this.view.getUint32(absValuePointer + 8, true);
    const dataHi = this.view.getUint32(absValuePointer + 12, true);

    this.view.setUint32(absEntryPointer, nameOffset, true);
    this.view.setUint32(absEntryPointer + 4, type, true);
    this.view.setUint32(absEntryPointer + 8, flags, true);
    this.view.setUint32(absEntryPointer + 12, dataLo, true);
    this.view.setUint32(absEntryPointer + 16, dataHi, true);

    this.view.setUint32(absPointer + 4, count + 1, true);
  }

  /**
   * Look up a variable in a scope (and parent scopes).
   *
   * @param {number} scopePointer - Scope pointer
   * @param {number} nameOffset - String table offset for name
   * @returns {number|null} - Value pointer or null if not found
   */
  scopeLookup(scopePointer, nameOffset) {
    while (scopePointer !== 0) {
      const absPointer = this.abs(scopePointer);
      const count = this.view.getUint32(absPointer + 4, true);
      const entriesPointer = this.view.getUint32(absPointer + 12, true);

      // Entry data starts after GC header
      const entryDataStart = entriesPointer + GC_HEADER_SIZE;

      for (let i = 0; i < count; i++) {
        // Entry is 20 bytes: [name_ptr:4][type:4][flags:4][data_lo:4][data_hi:4]
        const entryOffset = entryDataStart + i * 20;
        const absEntryPointer = this.abs(entryOffset);
        const entryName = this.view.getUint32(absEntryPointer, true);

        if (entryName === nameOffset) {
          // Return pointer to value (starts at offset +4 within entry)
          return entryOffset + 4;
        }
      }

      // Check parent scope
      scopePointer = this.view.getUint32(absPointer, true);
    }

    return null;
  }

  /**
   * Get all variable names in scope (current scope only, not parent chain).
   *
   * @param {number} scopePointer - Segment-relative pointer to scope
   * @returns {string[]} - Variable names
   */
  scopeKeys(scopePointer) {
    const keys = [];
    const absPointer = this.abs(scopePointer);
    const count = this.view.getUint32(absPointer + 4, true);
    const entriesPointer = this.view.getUint32(absPointer + 12, true);
    const entryDataStart = entriesPointer + GC_HEADER_SIZE;

    for (let i = 0; i < count; i++) {
      const entryOffset = entryDataStart + i * 20;
      const absEntryPointer = this.abs(entryOffset);
      const nameOffset = this.view.getUint32(absEntryPointer, true);
      const name = this.readString(nameOffset);
      keys.push(name);
    }

    return keys;
  }

  // ===========================================================================
  // Closures
  // ===========================================================================

  /**
   * Create a closure object.
   *
   * Closure layout: [GC 8][start_instr 4][end_instr 4][scope 4][flags 4]
   *
   * @param {number} startInstr - First instruction index of function body
   * @param {number} endInstr - Instruction index after RETURN (for bounds)
   * @param {number} scopePointer - Captured scope pointer (lexical environment)
   * @param {number} flags - Closure flags (0 = regular, CLOSURE_FLAG_ARROW = arrow function)
   * @returns {number} - Segment-relative pointer to closure
   */
  createClosure(startInstr, endInstr, scopePointer, flags = 0) {
    const closurePointer = this.allocate(16, OBJ.CLOSURE);
    const absPointer = this.abs(closurePointer);

    this.view.setUint32(absPointer, startInstr, true);
    this.view.setUint32(absPointer + 4, endInstr, true);
    this.view.setUint32(absPointer + 8, scopePointer, true);
    this.view.setUint32(absPointer + 12, flags, true);

    return closurePointer;
  }

  /**
   * Read closure data.
   *
   * @param {number} closurePointer - Segment-relative pointer to closure
   * @returns {object} - { startInstr, endInstr, scope, flags }
   */
  getClosure(closurePointer) {
    const absPointer = this.abs(closurePointer);

    return {
      startInstr: this.view.getUint32(absPointer, true),
      endInstr: this.view.getUint32(absPointer + 4, true),
      scope: this.view.getUint32(absPointer + 8, true),
      flags: this.view.getUint32(absPointer + 12, true),
    };
  }

  /**
   * Check if a closure is an arrow function.
   *
   * @param {number} closurePointer - Segment-relative pointer to closure
   * @returns {boolean}
   */
  isArrowClosure(closurePointer) {
    const flags = this.view.getUint32(this.abs(closurePointer) + 12, true);
    return (flags & CLOSURE_FLAG_ARROW) !== 0;
  }

  /**
   * Write a closure value at an address.
   *
   * @param {number} addr - Segment-relative address
   * @param {number} closurePointer - Segment-relative pointer to closure
   */
  writeClosure(addr, closurePointer) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.CLOSURE, true);
    this.view.setUint32(absAddr + 4, 0, true);
    this.view.setBigInt64(absAddr + 8, BigInt(closurePointer), true);
  }

  // ===========================================================================
  // Bound Methods
  // ===========================================================================

  /**
   * Write a bound method value at an address.
   *
   * Bound method value layout (16 bytes):
   *   [type: 4][receiver_ptr: 4][method_id_or_closure: 4][flags: 4]
   *
   * Flags:
   *   0 = built-in method (method_id_or_closure is METHOD.* constant)
   *   1 = user function (method_id_or_closure is closure pointer)
   *
   * @param {number} addr - Segment-relative address
   * @param {number} receiverPointer - Segment-relative pointer to receiver object
   * @param {number} methodIdOrClosure - Method ID (built-in) or closure pointer (user)
   * @param {number} flags - 0 for built-in, 1 for user function
   */
  writeBoundMethod(addr, receiverPointer, methodIdOrClosure, flags = 0) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.BOUND_METHOD, true);
    this.view.setUint32(absAddr + 4, receiverPointer, true);
    this.view.setUint32(absAddr + 8, methodIdOrClosure, true);
    this.view.setUint32(absAddr + 12, flags, true);
  }

  /**
   * Read a bound method value from an address.
   *
   * @param {number} addr - Segment-relative address
   * @returns {object} - { receiver, methodIdOrClosure, isUserFunction }
   */
  readBoundMethod(addr) {
    const absAddr = this.abs(addr);
    return {
      receiver: this.view.getUint32(absAddr + 4, true),
      methodIdOrClosure: this.view.getUint32(absAddr + 8, true),
      isUserFunction: this.view.getUint32(absAddr + 12, true) === 1,
    };
  }

  // ===========================================================================
  // FFI / Error Regions
  // ===========================================================================

  /**
   * Get FFI request data.
   *
   * @returns {object} - { refId, methodOffset, argsPointer, argCount }
   */
  getFFIRequest() {
    const offset = this.abs(this.getFfiRequestBase());
    return {
      refId: this.view.getUint32(offset, true),
      methodOffset: this.view.getUint32(offset + 4, true),
      argsPointer: this.view.getUint32(offset + 8, true),
      argCount: this.view.getUint32(offset + 12, true),
    };
  }

  /**
   * Get error info.
   *
   * @returns {object} - { code, detail }
   */
  getErrorInfo() {
    const offset = this.abs(this.getErrorInfoBase());
    return {
      code: this.view.getUint32(offset, true),
      detail: this.view.getUint32(offset + 4, true),
    };
  }

  /**
   * Set error info.
   *
   * @param {number} code - Error code
   * @param {number} detail - Error detail (optional, defaults to 0)
   */
  setError(code, detail = 0) {
    const offset = this.abs(this.getErrorInfoBase());
    this.view.setUint32(offset, code, true);
    this.view.setUint32(offset + 4, detail, true);
  }

  // ===========================================================================
  // Stats
  // ===========================================================================

  /**
   * Get heap usage statistics.
   *
   * @returns {object} - { used, total, percent }
   */
  heapUsage() {
    const heapPointer = this.getHeapPointer();
    const heapStart = this.getHeapStart();
    const heapEnd = this.getHeapEnd();
    const used = heapPointer - heapStart;
    const total = heapEnd - heapStart;
    const percent = ((used / total) * 100).toFixed(1);

    return { used, total, percent };
  }

  /**
   * Get string table usage statistics.
   *
   * @returns {object} - { used, total, percent }
   */
  stringUsage() {
    const stringPointer = this.getStringPointer();
    const stringStart = this.getStringStart() + HASH_TABLE_SIZE;
    const stringEnd = this.getStringEnd();
    const used = stringPointer - stringStart;
    const total = stringEnd - stringStart;
    const percent = ((used / total) * 100).toFixed(1);

    return { used, total, percent };
  }

  // ===========================================================================
  // Soft Reset Helpers (for session.reset())
  // ===========================================================================

  /**
   * Reset the pending stack to empty state.
   */
  resetPendingStack() {
    this.setPendingPointer(this.getPendingStackBase());
  }

  /**
   * Reset the call stack to empty state.
   */
  resetCallStack() {
    this.setStackPointer(this.getCallStackBase());
  }

  /**
   * Reset the try stack to empty state.
   */
  resetTryStack() {
    this.setTryPointer(this.getTryStackBase());
    // Clear completion state
    this.view.setUint32(this.abs(STATE.COMPLETION_TYPE), 0, true); // COMPLETION_NORMAL
  }

  /**
   * Create a fresh root scope (for soft reset).
   * Creates a new root scope with the same capacity as the original.
   * Does NOT clear the heap - old scope becomes garbage for GC.
   *
   * @param {number} capacity - Initial capacity (default 128 to match initialize())
   * @returns {number} - New scope pointer
   */
  createFreshScope(capacity = 128) {
    const newScope = this.createScope(0, capacity);
    this.setScope(newScope);
    return newScope;
  }

  // ===========================================================================
  // Built-in Object Initialization
  // ===========================================================================

  /**
   * Allocate an empty object on the heap.
   * Object layout: [GC:8][count:4][capacity:4][entries_ptr:4]
   * Entries layout: [GC:8][entries: capacity * 20]
   *
   * @param {number} initialCapacity - Initial entry capacity (default 8)
   * @returns {number} - Segment-relative pointer to object
   */
  allocateObject(initialCapacity = 8) {
    const heapPtr = this.getHeapPointer();

    // Object header: 20 bytes (GC:8 + count:4 + capacity:4 + entries_ptr:4)
    const objPtr = heapPtr;
    const objHeaderSize = 20;

    // Write GC header for object
    const absObjPtr = this.abs(objPtr);
    this.view.setUint32(absObjPtr, objHeaderSize | (OBJ.OBJECT << 24), true);
    this.view.setUint32(absObjPtr + 4, 0, true); // GC flags

    // Entries block: GC:8 + capacity * 20
    const entriesPtr = objPtr + objHeaderSize;
    const entriesDataSize = GC_HEADER_SIZE + initialCapacity * 20;

    // Write GC header for entries (OBJ.ARRAY_DATA is used for object entries too)
    const absEntriesPtr = this.abs(entriesPtr);
    // Note: WASM uses OBJ_OBJECT_DATA (7) for object entries, but constants.js uses ARRAY_DATA (5)
    // Let's use 7 to match WASM
    this.view.setUint32(absEntriesPtr, entriesDataSize | (7 << 24), true);
    this.view.setUint32(absEntriesPtr + 4, 0, true); // GC flags

    // Object header fields
    this.view.setUint32(absObjPtr + 8, 0, true);              // count = 0
    this.view.setUint32(absObjPtr + 12, initialCapacity, true); // capacity
    this.view.setUint32(absObjPtr + 16, entriesPtr, true);    // entries_ptr

    // Update heap pointer
    this.setHeapPointer(entriesPtr + entriesDataSize);

    return objPtr;
  }

  /**
   * Set a property on an object.
   * Entry layout: [key_ptr:4][type:4][flags:4][data_lo:4][data_hi:4]
   *
   * @param {number} objPtr - Object pointer
   * @param {number} keyOffset - String table offset for key
   * @param {number} type - Value type
   * @param {number} dataLo - Lower 32 bits of value payload
   * @param {number} dataHi - Upper 32 bits of value payload
   */
  objectSetRaw(objPtr, keyOffset, type, dataLo, dataHi) {
    const absObjPtr = this.abs(objPtr);
    const count = this.view.getUint32(absObjPtr + 8, true);
    const capacity = this.view.getUint32(absObjPtr + 12, true);
    const entriesPtr = this.view.getUint32(absObjPtr + 16, true);

    if (count >= capacity) {
      throw new Error(`Object capacity exceeded: ${count} >= ${capacity}`);
    }

    // Entry at index = count
    const entryPtr = entriesPtr + GC_HEADER_SIZE + count * 20;
    const absEntryPtr = this.abs(entryPtr);

    this.view.setUint32(absEntryPtr, keyOffset, true);       // key
    this.view.setUint32(absEntryPtr + 4, type, true);        // type
    this.view.setUint32(absEntryPtr + 8, 0, true);           // flags
    this.view.setUint32(absEntryPtr + 12, dataLo, true);     // data_lo
    this.view.setUint32(absEntryPtr + 16, dataHi, true);     // data_hi

    // Increment count
    this.view.setUint32(absObjPtr + 8, count + 1, true);
  }

  /**
   * Set a float property on an object.
   *
   * @param {number} objPtr - Object pointer
   * @param {number} keyOffset - String table offset for key
   * @param {number} value - Float value
   */
  objectSetFloat(objPtr, keyOffset, value) {
    const buffer = new ArrayBuffer(8);
    const f64 = new Float64Array(buffer);
    const u32 = new Uint32Array(buffer);
    f64[0] = value;
    this.objectSetRaw(objPtr, keyOffset, TYPE.FLOAT, u32[0], u32[1]);
  }

  /**
   * Set a bound method property on an object.
   *
   * @param {number} objPtr - Object pointer
   * @param {number} keyOffset - String table offset for key
   * @param {number} receiverPtr - Receiver object pointer
   * @param {number} methodId - Method ID from METHOD constants
   */
  objectSetBoundMethod(objPtr, keyOffset, receiverPtr, methodId) {
    // TYPE_BOUND_METHOD layout: [type:4][receiver:4][method_id:4][flags:4]
    // For object entries: [key][type=BOUND_METHOD][flags=0][data_lo=receiver][data_hi=methodId<<16]
    // Wait, let me check the actual layout...
    // Actually for pending values: [type:4][padding:4][data_lo:4][data_hi:4]
    // For object entries: [key:4][type:4][flags:4][data_lo:4][data_hi:4]
    // BOUND_METHOD uses data_lo = receiver, data_hi = (flags << 16) | methodId
    // Actually from writeBoundMethod: [type][receiver][methodId][flags]
    // So in object entry: type=BOUND_METHOD, data_lo=receiver, data_hi=methodId
    this.objectSetRaw(objPtr, keyOffset, TYPE.BOUND_METHOD, receiverPtr, methodId);
  }

  /**
   * Set a property on an object from a value pointer.
   *
   * @param {number} objPtr - Object pointer
   * @param {number} keyOffset - String table offset for key
   * @param {number} valuePtr - Pointer to 16-byte value
   */
  objectSetFromPointer(objPtr, keyOffset, valuePtr) {
    const absValuePtr = this.abs(valuePtr);
    const type = this.view.getUint32(absValuePtr, true);
    const dataLo = this.view.getUint32(absValuePtr + 8, true);
    const dataHi = this.view.getUint32(absValuePtr + 12, true);
    this.objectSetRaw(objPtr, keyOffset, type, dataLo, dataHi);
  }

  /**
   * Allocate an empty array on the heap.
   * Array layout: [GC:8][length:4][capacity:4][data_ptr:4]
   * Data layout: [GC:8][elements: capacity * 16]
   *
   * @param {number} initialCapacity - Initial capacity
   * @returns {number} - Segment-relative pointer to array header
   */
  allocateArrayWithCapacity(initialCapacity) {
    const heapPtr = this.getHeapPointer();

    // Array header: 20 bytes (GC:8 + length:4 + capacity:4 + data_ptr:4)
    const arrayPtr = heapPtr;
    const arrayHeaderSize = 20;

    // Write GC header for array
    const absArrayPtr = this.abs(arrayPtr);
    this.view.setUint32(absArrayPtr, arrayHeaderSize | (OBJ.ARRAY << 24), true);
    this.view.setUint32(absArrayPtr + 4, 0, true); // GC flags

    // Data block: GC:8 + capacity * 16
    const dataPtr = arrayPtr + arrayHeaderSize;
    const dataSize = GC_HEADER_SIZE + initialCapacity * 16;

    // Write GC header for data
    const absDataPtr = this.abs(dataPtr);
    this.view.setUint32(absDataPtr, dataSize | (OBJ.ARRAY_DATA << 24), true);
    this.view.setUint32(absDataPtr + 4, 0, true); // GC flags

    // Array header fields
    this.view.setUint32(absArrayPtr + 8, 0, true);              // length = 0
    this.view.setUint32(absArrayPtr + 12, initialCapacity, true); // capacity
    this.view.setUint32(absArrayPtr + 16, dataPtr, true);       // data_ptr

    // Update heap pointer
    this.setHeapPointer(dataPtr + dataSize);

    return arrayPtr;
  }

  /**
   * Set an array element from a value pointer.
   *
   * @param {number} arrayPtr - Array pointer
   * @param {number} index - Element index
   * @param {number} valuePtr - Pointer to 16-byte value
   */
  arraySetFromPointer(arrayPtr, index, valuePtr) {
    const absArrayPtr = this.abs(arrayPtr);
    const length = this.view.getUint32(absArrayPtr + 8, true);
    const capacity = this.view.getUint32(absArrayPtr + 12, true);
    const dataPtr = this.view.getUint32(absArrayPtr + 16, true);

    if (index >= capacity) {
      throw new Error(`Array index out of capacity: ${index} >= ${capacity}`);
    }

    // Copy 16 bytes from valuePtr to array slot
    const destAddr = dataPtr + GC_HEADER_SIZE + index * 16;
    const absValuePtr = this.abs(valuePtr);
    const absDestAddr = this.abs(destAddr);

    for (let i = 0; i < 16; i++) {
      this.u8[absDestAddr + i] = this.u8[absValuePtr + i];
    }

    // Update length if necessary
    if (index >= length) {
      this.view.setUint32(absArrayPtr + 8, index + 1, true);
    }
  }

  /**
   * Write a TYPE_CONSTRUCTOR value.
   * Layout: [type=CONSTRUCTOR][padding][object_ptr][method_id]
   *
   * @param {number} addr - Segment-relative address
   * @param {number} objectPtr - Pointer to object with static methods
   * @param {number} methodId - Method ID for call/new dispatch
   */
  writeConstructor(addr, objectPtr, methodId) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.CONSTRUCTOR, true);
    this.view.setUint32(absAddr + 4, 0, true);  // padding
    this.view.setUint32(absAddr + 8, objectPtr, true);
    this.view.setUint32(absAddr + 12, methodId, true);
  }

  /**
   * Set a value in a scope.
   *
   * @param {number} scopePtr - Scope data pointer (after GC header)
   * @param {number} keyOffset - String table offset for key
   * @param {number} type - Value type
   * @param {number} dataLo - Lower 32 bits of value payload
   * @param {number} dataHi - Upper 32 bits of value payload
   */
  scopeSetRaw(scopePtr, keyOffset, type, dataLo, dataHi) {
    // Scope uses the same entry format as objects
    // Scope data layout (scopePtr points AFTER GC header):
    //   [parent:4][count:4][capacity:4][entries_ptr:4]
    const absScopePtr = this.abs(scopePtr);
    const count = this.view.getUint32(absScopePtr + 4, true);
    const capacity = this.view.getUint32(absScopePtr + 8, true);
    const entriesPtr = this.view.getUint32(absScopePtr + 12, true);

    if (count >= capacity) {
      throw new Error(`Scope capacity exceeded: ${count} >= ${capacity}`);
    }

    // Entry at index = count (entries_ptr points to GC header of entries block)
    const entryPtr = entriesPtr + GC_HEADER_SIZE + count * 20;
    const absEntryPtr = this.abs(entryPtr);

    this.view.setUint32(absEntryPtr, keyOffset, true);       // key
    this.view.setUint32(absEntryPtr + 4, type, true);        // type
    this.view.setUint32(absEntryPtr + 8, 0, true);           // flags
    this.view.setUint32(absEntryPtr + 12, dataLo, true);     // data_lo
    this.view.setUint32(absEntryPtr + 16, dataHi, true);     // data_hi

    // Increment count
    this.view.setUint32(absScopePtr + 4, count + 1, true);
  }

  /**
   * Set a TYPE_OBJECT value in a scope.
   *
   * @param {number} scopePtr - Scope pointer
   * @param {number} keyOffset - String table offset for key
   * @param {number} objectPtr - Object pointer
   */
  scopeSetObject(scopePtr, keyOffset, objectPtr) {
    this.scopeSetRaw(scopePtr, keyOffset, TYPE.OBJECT, objectPtr, 0);
  }

  /**
   * Set a TYPE_CONSTRUCTOR value in a scope.
   *
   * @param {number} scopePtr - Scope pointer
   * @param {number} keyOffset - String table offset for key
   * @param {number} objectPtr - Object with static methods
   * @param {number} methodId - Method ID for call/new
   */
  scopeSetConstructor(scopePtr, keyOffset, objectPtr, methodId) {
    // TYPE_CONSTRUCTOR: data_lo = objectPtr, data_hi = methodId
    this.scopeSetRaw(scopePtr, keyOffset, TYPE.CONSTRUCTOR, objectPtr, methodId);
  }

  /**
   * Set a TYPE_BOUND_METHOD value in a scope.
   *
   * @param {number} scopePtr - Scope pointer
   * @param {number} keyOffset - String table offset for key
   * @param {number} receiverPtr - Receiver (0 for global functions)
   * @param {number} methodId - Method ID
   */
  scopeSetBoundMethod(scopePtr, keyOffset, receiverPtr, methodId) {
    this.scopeSetRaw(scopePtr, keyOffset, TYPE.BOUND_METHOD, receiverPtr, methodId);
  }

  /**
   * Set a float value in scope.
   *
   * @param {number} scopePtr - Scope pointer
   * @param {number} keyOffset - String table offset for variable name
   * @param {number} value - Float value
   */
  scopeSetFloat(scopePtr, keyOffset, value) {
    const buffer = new ArrayBuffer(8);
    const f64 = new Float64Array(buffer);
    const u32 = new Uint32Array(buffer);
    f64[0] = value;
    this.scopeSetRaw(scopePtr, keyOffset, TYPE.FLOAT, u32[0], u32[1]);
  }

  /**
   * Initialize built-in objects (Math, Object, Array, String, Number)
   * and global functions (parseInt, parseFloat, isNaN, isFinite, Boolean).
   *
   * Should be called after initialize() and before parsing/running code.
   */
  initializeBuiltins() {
    const globalScope = this.getScope();

    // =========================================================================
    // Math object (not callable, just properties)
    // =========================================================================
    const mathObj = this.allocateObject(48); // 33 methods + 8 constants + room

    // Math constants
    this.objectSetFloat(mathObj, this.internString('PI'), Math.PI);
    this.objectSetFloat(mathObj, this.internString('E'), Math.E);
    this.objectSetFloat(mathObj, this.internString('LN2'), Math.LN2);
    this.objectSetFloat(mathObj, this.internString('LN10'), Math.LN10);
    this.objectSetFloat(mathObj, this.internString('LOG2E'), Math.LOG2E);
    this.objectSetFloat(mathObj, this.internString('LOG10E'), Math.LOG10E);
    this.objectSetFloat(mathObj, this.internString('SQRT2'), Math.SQRT2);
    this.objectSetFloat(mathObj, this.internString('SQRT1_2'), Math.SQRT1_2);

    // Math methods
    this.objectSetBoundMethod(mathObj, this.internString('abs'), mathObj, METHOD.MATH_ABS);
    this.objectSetBoundMethod(mathObj, this.internString('floor'), mathObj, METHOD.MATH_FLOOR);
    this.objectSetBoundMethod(mathObj, this.internString('ceil'), mathObj, METHOD.MATH_CEIL);
    this.objectSetBoundMethod(mathObj, this.internString('round'), mathObj, METHOD.MATH_ROUND);
    this.objectSetBoundMethod(mathObj, this.internString('trunc'), mathObj, METHOD.MATH_TRUNC);
    this.objectSetBoundMethod(mathObj, this.internString('sign'), mathObj, METHOD.MATH_SIGN);
    this.objectSetBoundMethod(mathObj, this.internString('min'), mathObj, METHOD.MATH_MIN);
    this.objectSetBoundMethod(mathObj, this.internString('max'), mathObj, METHOD.MATH_MAX);
    this.objectSetBoundMethod(mathObj, this.internString('pow'), mathObj, METHOD.MATH_POW);
    this.objectSetBoundMethod(mathObj, this.internString('sqrt'), mathObj, METHOD.MATH_SQRT);
    this.objectSetBoundMethod(mathObj, this.internString('cbrt'), mathObj, METHOD.MATH_CBRT);
    this.objectSetBoundMethod(mathObj, this.internString('hypot'), mathObj, METHOD.MATH_HYPOT);
    this.objectSetBoundMethod(mathObj, this.internString('sin'), mathObj, METHOD.MATH_SIN);
    this.objectSetBoundMethod(mathObj, this.internString('cos'), mathObj, METHOD.MATH_COS);
    this.objectSetBoundMethod(mathObj, this.internString('tan'), mathObj, METHOD.MATH_TAN);
    this.objectSetBoundMethod(mathObj, this.internString('asin'), mathObj, METHOD.MATH_ASIN);
    this.objectSetBoundMethod(mathObj, this.internString('acos'), mathObj, METHOD.MATH_ACOS);
    this.objectSetBoundMethod(mathObj, this.internString('atan'), mathObj, METHOD.MATH_ATAN);
    this.objectSetBoundMethod(mathObj, this.internString('atan2'), mathObj, METHOD.MATH_ATAN2);
    this.objectSetBoundMethod(mathObj, this.internString('sinh'), mathObj, METHOD.MATH_SINH);
    this.objectSetBoundMethod(mathObj, this.internString('cosh'), mathObj, METHOD.MATH_COSH);
    this.objectSetBoundMethod(mathObj, this.internString('tanh'), mathObj, METHOD.MATH_TANH);
    this.objectSetBoundMethod(mathObj, this.internString('asinh'), mathObj, METHOD.MATH_ASINH);
    this.objectSetBoundMethod(mathObj, this.internString('acosh'), mathObj, METHOD.MATH_ACOSH);
    this.objectSetBoundMethod(mathObj, this.internString('atanh'), mathObj, METHOD.MATH_ATANH);
    this.objectSetBoundMethod(mathObj, this.internString('log'), mathObj, METHOD.MATH_LOG);
    this.objectSetBoundMethod(mathObj, this.internString('log10'), mathObj, METHOD.MATH_LOG10);
    this.objectSetBoundMethod(mathObj, this.internString('log2'), mathObj, METHOD.MATH_LOG2);
    this.objectSetBoundMethod(mathObj, this.internString('log1p'), mathObj, METHOD.MATH_LOG1P);
    this.objectSetBoundMethod(mathObj, this.internString('exp'), mathObj, METHOD.MATH_EXP);
    this.objectSetBoundMethod(mathObj, this.internString('expm1'), mathObj, METHOD.MATH_EXPM1);
    this.objectSetBoundMethod(mathObj, this.internString('clz32'), mathObj, METHOD.MATH_CLZ32);
    this.objectSetBoundMethod(mathObj, this.internString('imul'), mathObj, METHOD.MATH_IMUL);
    this.objectSetBoundMethod(mathObj, this.internString('fround'), mathObj, METHOD.MATH_FROUND);

    // Bind Math to global scope as TYPE_OBJECT (not callable)
    this.scopeSetObject(globalScope, this.internString('Math'), mathObj);

    // =========================================================================
    // Array constructor (TYPE_CONSTRUCTOR)
    // =========================================================================
    const arrayObj = this.allocateObject(4);
    this.objectSetBoundMethod(arrayObj, this.internString('isArray'), arrayObj, METHOD.ARRAY_IS_ARRAY);
    this.objectSetBoundMethod(arrayObj, this.internString('from'), arrayObj, METHOD.ARRAY_FROM);

    this.scopeSetConstructor(globalScope, this.internString('Array'),
      arrayObj, METHOD.ARRAY_CONSTRUCTOR);

    // =========================================================================
    // Object constructor (TYPE_CONSTRUCTOR)
    // =========================================================================
    const objectObj = this.allocateObject(8);
    this.objectSetBoundMethod(objectObj, this.internString('keys'), objectObj, METHOD.OBJECT_KEYS);
    this.objectSetBoundMethod(objectObj, this.internString('values'), objectObj, METHOD.OBJECT_VALUES);
    this.objectSetBoundMethod(objectObj, this.internString('entries'), objectObj, METHOD.OBJECT_ENTRIES);
    this.objectSetBoundMethod(objectObj, this.internString('assign'), objectObj, METHOD.OBJECT_ASSIGN);

    this.scopeSetConstructor(globalScope, this.internString('Object'),
      objectObj, METHOD.OBJECT_CONSTRUCTOR);

    // =========================================================================
    // String constructor (TYPE_CONSTRUCTOR)
    // =========================================================================
    const stringObj = this.allocateObject(4);
    this.objectSetBoundMethod(stringObj, this.internString('fromCharCode'), stringObj, METHOD.STRING_FROM_CHAR_CODE);
    this.objectSetBoundMethod(stringObj, this.internString('fromCodePoint'), stringObj, METHOD.STRING_FROM_CODE_POINT);

    this.scopeSetConstructor(globalScope, this.internString('String'),
      stringObj, METHOD.TO_STRING);

    // =========================================================================
    // Number constructor (TYPE_CONSTRUCTOR)
    // =========================================================================
    const numberObj = this.allocateObject(16);

    // Number static methods
    this.objectSetBoundMethod(numberObj, this.internString('isNaN'), numberObj, METHOD.NUMBER_IS_NAN);
    this.objectSetBoundMethod(numberObj, this.internString('isFinite'), numberObj, METHOD.NUMBER_IS_FINITE);
    this.objectSetBoundMethod(numberObj, this.internString('isInteger'), numberObj, METHOD.NUMBER_IS_INTEGER);
    this.objectSetBoundMethod(numberObj, this.internString('parseInt'), numberObj, METHOD.NUMBER_PARSE_INT);
    this.objectSetBoundMethod(numberObj, this.internString('parseFloat'), numberObj, METHOD.NUMBER_PARSE_FLOAT);

    // Number constants
    this.objectSetFloat(numberObj, this.internString('MAX_VALUE'), Number.MAX_VALUE);
    this.objectSetFloat(numberObj, this.internString('MIN_VALUE'), Number.MIN_VALUE);
    this.objectSetFloat(numberObj, this.internString('MAX_SAFE_INTEGER'), Number.MAX_SAFE_INTEGER);
    this.objectSetFloat(numberObj, this.internString('MIN_SAFE_INTEGER'), Number.MIN_SAFE_INTEGER);
    this.objectSetFloat(numberObj, this.internString('POSITIVE_INFINITY'), Infinity);
    this.objectSetFloat(numberObj, this.internString('NEGATIVE_INFINITY'), -Infinity);
    this.objectSetFloat(numberObj, this.internString('NaN'), NaN);
    this.objectSetFloat(numberObj, this.internString('EPSILON'), Number.EPSILON);

    this.scopeSetConstructor(globalScope, this.internString('Number'),
      numberObj, METHOD.TO_NUMBER);

    // =========================================================================
    // Global functions (TYPE_BOUND_METHOD with receiver = 0)
    // =========================================================================
    this.scopeSetBoundMethod(globalScope, this.internString('parseInt'), 0, METHOD.PARSE_INT);
    this.scopeSetBoundMethod(globalScope, this.internString('parseFloat'), 0, METHOD.PARSE_FLOAT);
    this.scopeSetBoundMethod(globalScope, this.internString('isNaN'), 0, METHOD.IS_NAN);
    this.scopeSetBoundMethod(globalScope, this.internString('isFinite'), 0, METHOD.IS_FINITE);
    this.scopeSetBoundMethod(globalScope, this.internString('Boolean'), 0, METHOD.TO_BOOLEAN);

    // =========================================================================
    // Global constants
    // =========================================================================
    this.scopeSetFloat(globalScope, this.internString('Infinity'), Infinity);
    this.scopeSetFloat(globalScope, this.internString('NaN'), NaN);
    this.scopeSetRaw(globalScope, this.internString('undefined'), TYPE.UNDEFINED, 0, 0);

    // Note: String() and Number() are handled via TYPE_CONSTRUCTOR values above
  }

  // ===========================================================================
  // Garbage Collection
  // ===========================================================================

  // ===========================================================================
  // FFI Support
  // ===========================================================================

  /**
   * Write an FFI ref value at an address.
   *
   * @param {number} addr - Segment-relative address
   * @param {number} refId - Opaque handle ID
   */
  writeFFIRef(addr, refId) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.FFI_REF, true);
    this.view.setUint32(absAddr + 4, 0, true);
    this.view.setUint32(absAddr + 8, refId, true);
    this.view.setUint32(absAddr + 12, 0, true);
  }

  /**
   * Write an FFI method value at an address.
   *
   * @param {number} addr - Segment-relative address
   * @param {number} refId - Opaque handle ID
   * @param {number} methodOffset - String table offset for method name
   */
  writeFFIMethod(addr, refId, methodOffset) {
    const absAddr = this.abs(addr);
    this.view.setUint32(absAddr, TYPE.FFI_METHOD, true);
    this.view.setUint32(absAddr + 4, 0, true);
    this.view.setUint32(absAddr + 8, refId, true);
    this.view.setUint32(absAddr + 12, methodOffset, true);
  }

  /**
   * Read an FFI ref value.
   *
   * @param {number} addr - Segment-relative address
   * @returns {number} - Ref ID
   */
  readFFIRef(addr) {
    return this.view.getUint32(this.abs(addr) + 8, true);
  }

  /**
   * Read an FFI method value.
   *
   * @param {number} addr - Segment-relative address
   * @returns {object} - { refId, methodOffset }
   */
  readFFIMethod(addr) {
    const absAddr = this.abs(addr);
    return {
      refId: this.view.getUint32(absAddr + 8, true),
      methodOffset: this.view.getUint32(absAddr + 12, true),
    };
  }

  /**
   * Push a frame with flags for callback invocation.
   *
   * @param {number} codeBlock - Code block pointer
   * @param {number} instructionIndex - Instruction index
   * @param {number} scopePointer - Scope pointer
   * @param {number} flags - Frame flags (e.g., FRAME_FLAG_CALLBACK)
   * @returns {number} - Frame index
   */
  pushFrameWithFlags(codeBlock, instructionIndex, scopePointer, flags) {
    const stackPointer = this.getStackPointer();
    const callStackBase = this.getCallStackBase();
    const stackEnd = callStackBase + CALL_STACK_SIZE;

    if (stackPointer + FRAME_SIZE > stackEnd) {
      throw new Error('Call stack overflow');
    }

    const framePointer = this.abs(stackPointer);
    const pendingPointer = this.getPendingPointer();

    this.view.setUint32(framePointer + FRAME.CODE_BLOCK, codeBlock, true);
    this.view.setUint32(framePointer + FRAME.INSTRUCTION_INDEX, instructionIndex, true);
    this.view.setUint32(framePointer + FRAME.SCOPE_POINTER, scopePointer, true);
    this.view.setUint32(framePointer + FRAME.PENDING_BASE, pendingPointer, true);
    this.view.setUint32(framePointer + FRAME.PENDING_COUNT, 0, true);
    this.view.setUint32(framePointer + FRAME.FLAGS, flags, true);
    this.view.setUint32(framePointer + FRAME.SOURCE_START, 0, true);
    this.view.setUint32(framePointer + FRAME.SOURCE_END, 0, true);

    this.setStackPointer(stackPointer + FRAME_SIZE);

    return (stackPointer - callStackBase) / FRAME_SIZE;
  }

  // ===========================================================================
  // Invocation Slots (for callable images)
  // ===========================================================================

  /**
   * Get the base address of the INVOCATION region.
   * @returns {number} - Segment-relative pointer to INVOCATION region
   */
  getInvocationBase() {
    return LAYOUT.INVOCATION;
  }

  /**
   * Get the number of configured input slots.
   * @returns {number} - Input slot count (0-8)
   */
  getInputCount() {
    return this.view.getUint32(this.abs(LAYOUT.INVOCATION + INVOCATION.INPUT_COUNT), true);
  }

  /**
   * Set the number of configured input slots.
   * @param {number} count - Input slot count (0-8)
   */
  setInputCount(count) {
    this.view.setUint32(this.abs(LAYOUT.INVOCATION + INVOCATION.INPUT_COUNT), count, true);
  }

  /**
   * Get the number of configured output slots.
   * @returns {number} - Output slot count (0-8)
   */
  getOutputCount() {
    return this.view.getUint32(this.abs(LAYOUT.INVOCATION + INVOCATION.OUTPUT_COUNT), true);
  }

  /**
   * Set the number of configured output slots.
   * @param {number} count - Output slot count (0-8)
   */
  setOutputCount(count) {
    this.view.setUint32(this.abs(LAYOUT.INVOCATION + INVOCATION.OUTPUT_COUNT), count, true);
  }

  /**
   * Get an input slot's value pointer.
   * @param {number} index - Slot index (0-7)
   * @returns {number} - Segment-relative pointer to value region in scope entry
   */
  getInputValuePtr(index) {
    const slotAddr = LAYOUT.INVOCATION + INVOCATION.INPUT_SLOTS + index * 8;
    return this.view.getUint32(this.abs(slotAddr + SLOT.VALUE_PTR), true);
  }

  /**
   * Get an output slot's value pointer.
   * @param {number} index - Slot index (0-7)
   * @returns {number} - Segment-relative pointer to value region in scope entry
   */
  getOutputValuePtr(index) {
    const slotAddr = LAYOUT.INVOCATION + INVOCATION.OUTPUT_SLOTS + index * 8;
    return this.view.getUint32(this.abs(slotAddr + SLOT.VALUE_PTR), true);
  }

  /**
   * Define an input slot pointing to a scope variable.
   * Variable must already exist in scope.
   *
   * @param {number} index - Slot index (0-7)
   * @param {string} name - Variable name in scope
   * @throws if index out of range or variable not found
   */
  defineInput(index, name) {
    if (index < 0 || index >= MAX_INPUT_SLOTS) {
      throw new Error(`Input slot index ${index} out of range (0-${MAX_INPUT_SLOTS - 1})`);
    }

    const nameOffset = this.internString(name);
    const scope = this.getScope();
    const valuePtr = this.scopeLookup(scope, nameOffset);

    if (valuePtr === null) {
      throw new Error(`Variable '${name}' not found in scope`);
    }

    const slotAddr = LAYOUT.INVOCATION + INVOCATION.INPUT_SLOTS + index * 8;
    this.view.setUint32(this.abs(slotAddr + SLOT.VALUE_PTR), valuePtr, true);
    this.view.setUint32(this.abs(slotAddr + SLOT.FLAGS), 0, true);

    // Update count if extending
    const currentCount = this.getInputCount();
    if (index >= currentCount) {
      this.setInputCount(index + 1);
    }
  }

  /**
   * Define an output slot pointing to a scope variable.
   * Variable must already exist in scope.
   *
   * @param {number} index - Slot index (0-7)
   * @param {string} name - Variable name in scope
   * @throws if index out of range or variable not found
   */
  defineOutput(index, name) {
    if (index < 0 || index >= MAX_OUTPUT_SLOTS) {
      throw new Error(`Output slot index ${index} out of range (0-${MAX_OUTPUT_SLOTS - 1})`);
    }

    const nameOffset = this.internString(name);
    const scope = this.getScope();
    const valuePtr = this.scopeLookup(scope, nameOffset);

    if (valuePtr === null) {
      throw new Error(`Variable '${name}' not found in scope`);
    }

    const slotAddr = LAYOUT.INVOCATION + INVOCATION.OUTPUT_SLOTS + index * 8;
    this.view.setUint32(this.abs(slotAddr + SLOT.VALUE_PTR), valuePtr, true);
    this.view.setUint32(this.abs(slotAddr + SLOT.FLAGS), 0, true);

    // Update count if extending
    const currentCount = this.getOutputCount();
    if (index >= currentCount) {
      this.setOutputCount(index + 1);
    }
  }

  /**
   * Write a JS value to a 16-byte value slot in memory.
   * For objects/arrays, allocates on heap and writes pointer.
   *
   * @param {number} destPtr - Segment-relative pointer to value slot
   * @param {*} jsValue - JS value to write
   */
  writeValueAt(destPtr, jsValue) {
    const absPtr = this.abs(destPtr);

    if (jsValue === null) {
      this.view.setUint32(absPtr, TYPE.NULL, true);
      this.view.setUint32(absPtr + 4, 0, true);
      this.view.setUint32(absPtr + 8, 0, true);
      this.view.setUint32(absPtr + 12, 0, true);
    } else if (jsValue === undefined) {
      this.view.setUint32(absPtr, TYPE.UNDEFINED, true);
      this.view.setUint32(absPtr + 4, 0, true);
      this.view.setUint32(absPtr + 8, 0, true);
      this.view.setUint32(absPtr + 12, 0, true);
    } else if (typeof jsValue === 'boolean') {
      this.view.setUint32(absPtr, TYPE.BOOLEAN, true);
      this.view.setUint32(absPtr + 4, 0, true);
      this.view.setUint32(absPtr + 8, jsValue ? 1 : 0, true);
      this.view.setUint32(absPtr + 12, 0, true);
    } else if (typeof jsValue === 'number') {
      this.view.setUint32(absPtr, TYPE.FLOAT, true);
      this.view.setUint32(absPtr + 4, 0, true);
      this.view.setFloat64(absPtr + 8, jsValue, true);
    } else if (typeof jsValue === 'string') {
      const offset = this.internString(jsValue);
      this.view.setUint32(absPtr, TYPE.STRING, true);
      this.view.setUint32(absPtr + 4, 0, true);
      this.view.setUint32(absPtr + 8, offset, true);
      this.view.setUint32(absPtr + 12, 0, true);
    } else if (Array.isArray(jsValue)) {
      const heapPtr = this.marshalArray(jsValue);
      this.view.setUint32(absPtr, TYPE.ARRAY, true);
      this.view.setUint32(absPtr + 4, 0, true);
      this.view.setUint32(absPtr + 8, heapPtr, true);
      this.view.setUint32(absPtr + 12, 0, true);
    } else if (typeof jsValue === 'object') {
      const heapPtr = this.marshalObject(jsValue);
      this.view.setUint32(absPtr, TYPE.OBJECT, true);
      this.view.setUint32(absPtr + 4, 0, true);
      this.view.setUint32(absPtr + 8, heapPtr, true);
      this.view.setUint32(absPtr + 12, 0, true);
    } else {
      throw new Error(`Cannot marshal value of type ${typeof jsValue}`);
    }
  }

  /**
   * Write a TYPE_MSGPACK_REF value to a 16-byte value slot.
   * The msgpack bytes must already exist at the given address.
   *
   * @param {number} destPtr - Segment-relative pointer to value slot
   * @param {number} msgpackAddr - Absolute address of msgpack bytes in memory
   * @param {number} length - Byte length of msgpack data
   */
  writeMsgpackRef(destPtr, msgpackAddr, length) {
    const absPtr = this.abs(destPtr);
    this.view.setUint32(absPtr, TYPE.MSGPACK_REF, true);
    this.view.setUint32(absPtr + 4, 0, true);
    this.view.setUint32(absPtr + 8, msgpackAddr, true);  // Absolute address
    this.view.setUint32(absPtr + 12, length, true);
  }

  /**
   * Marshal a JS array to heap, returning the header pointer.
   * Recursively marshals nested values.
   *
   * @param {Array} jsArray - JS array to marshal
   * @returns {number} - Segment-relative header pointer to allocated array
   */
  marshalArray(jsArray) {
    const arrayPtr = this.allocateArrayWithCapacity(jsArray.length);
    for (let i = 0; i < jsArray.length; i++) {
      this.arraySetByMarshal(arrayPtr, i, jsArray[i]);
    }
    return arrayPtr;
  }

  /**
   * Set an array element by marshalling a JS value.
   * @param {number} arrayPtr - Header pointer to array
   * @param {number} index - Array index
   * @param {*} jsValue - JS value to set
   */
  arraySetByMarshal(arrayPtr, index, jsValue) {
    const absArray = this.abs(arrayPtr);
    const count = this.view.getUint32(absArray + GC_HEADER_SIZE, true);
    const capacity = this.view.getUint32(absArray + GC_HEADER_SIZE + 4, true);
    const elementsPtr = this.view.getUint32(absArray + GC_HEADER_SIZE + 8, true);

    if (index >= capacity) {
      throw new Error(`Array index ${index} out of bounds (capacity ${capacity})`);
    }

    // Elements are 16 bytes each, after GC header
    const elementPtr = elementsPtr + GC_HEADER_SIZE + index * VALUE_SIZE;
    this.writeValueAt(elementPtr, jsValue);

    // Update count if needed
    if (index >= count) {
      this.view.setUint32(absArray + GC_HEADER_SIZE, index + 1, true);
    }
  }

  /**
   * Marshal a JS object to heap, returning the header pointer.
   * Recursively marshals nested values.
   *
   * @param {Object} jsObj - JS object to marshal
   * @returns {number} - Segment-relative header pointer to allocated object
   */
  marshalObject(jsObj) {
    const keys = Object.keys(jsObj);
    const objectPtr = this.allocateObject(keys.length);
    for (const key of keys) {
      this.objectSetByMarshal(objectPtr, key, jsObj[key]);
    }
    return objectPtr;
  }

  /**
   * Set an object property by marshalling a JS value.
   * @param {number} objectPtr - Header pointer to object
   * @param {string} key - Property name
   * @param {*} jsValue - JS value to set
   */
  objectSetByMarshal(objectPtr, key, jsValue) {
    const keyOffset = this.internString(key);

    // Determine type and data for the value
    let type, flags = 0, dataLo = 0, dataHi = 0;

    if (jsValue === null) {
      type = TYPE.NULL;
    } else if (jsValue === undefined) {
      type = TYPE.UNDEFINED;
    } else if (typeof jsValue === 'boolean') {
      type = TYPE.BOOLEAN;
      dataLo = jsValue ? 1 : 0;
    } else if (typeof jsValue === 'number') {
      type = TYPE.FLOAT;
      // Store float64 as two u32s
      const tempBuf = new ArrayBuffer(8);
      const tempView = new DataView(tempBuf);
      tempView.setFloat64(0, jsValue, true);
      dataLo = tempView.getUint32(0, true);
      dataHi = tempView.getUint32(4, true);
    } else if (typeof jsValue === 'string') {
      type = TYPE.STRING;
      dataLo = this.internString(jsValue);
    } else if (Array.isArray(jsValue)) {
      type = TYPE.ARRAY;
      dataLo = this.marshalArray(jsValue);
    } else if (typeof jsValue === 'object') {
      type = TYPE.OBJECT;
      dataLo = this.marshalObject(jsValue);
    } else {
      throw new Error(`Cannot marshal value of type ${typeof jsValue}`);
    }

    this.objectSetRaw(objectPtr, keyOffset, type, dataLo, dataHi);
  }

  /**
   * Read a JS value from a 16-byte value slot in memory.
   * For objects/arrays, follows heap pointers and reconstructs.
   *
   * @param {number} srcPtr - Segment-relative pointer to value slot
   * @returns {*} - JS value
   */
  readValueAt(srcPtr) {
    const absPtr = this.abs(srcPtr);
    const type = this.view.getUint32(absPtr, true);
    const dataLo = this.view.getUint32(absPtr + 8, true);
    const dataHi = this.view.getUint32(absPtr + 12, true);

    switch (type) {
      case TYPE.UNDEFINED:
        return undefined;
      case TYPE.NULL:
        return null;
      case TYPE.BOOLEAN:
        return dataLo !== 0;
      case TYPE.INTEGER:
        return dataLo;  // Integer stored in dataLo
      case TYPE.FLOAT:
        return this.view.getFloat64(absPtr + 8, true);
      case TYPE.STRING:
        return this.readString(dataLo);
      case TYPE.ARRAY:
        return this.unmarshalArray(dataLo);
      case TYPE.OBJECT:
        return this.unmarshalObject(dataLo);
      default:
        // For closures, FFI refs, etc. - return a descriptor
        return { _type: type, _dataLo: dataLo, _dataHi: dataHi };
    }
  }

  /**
   * Unmarshal a heap array to a JS array.
   * @param {number} headerPtr - Header pointer to array
   * @returns {Array} - JS array
   */
  unmarshalArray(headerPtr) {
    const absArray = this.abs(headerPtr);
    const count = this.view.getUint32(absArray + GC_HEADER_SIZE, true);
    const elementsPtr = this.view.getUint32(absArray + GC_HEADER_SIZE + 8, true);

    const result = [];
    for (let i = 0; i < count; i++) {
      const elementPtr = elementsPtr + GC_HEADER_SIZE + i * VALUE_SIZE;
      result.push(this.readValueAt(elementPtr));
    }
    return result;
  }

  /**
   * Unmarshal a heap object to a JS object.
   * @param {number} headerPtr - Header pointer to object
   * @returns {Object} - JS object
   */
  unmarshalObject(headerPtr) {
    const absObj = this.abs(headerPtr);
    const count = this.view.getUint32(absObj + GC_HEADER_SIZE, true);
    const entriesPtr = this.view.getUint32(absObj + GC_HEADER_SIZE + 8, true);

    const result = {};
    const absEntries = this.abs(entriesPtr) + GC_HEADER_SIZE;

    for (let i = 0; i < count; i++) {
      // Entry: [nameOffset:4][type:4][flags:4][dataLo:4][dataHi:4]
      const entryAddr = absEntries + i * 20;
      const nameOffset = this.view.getUint32(entryAddr, true);
      const key = this.readString(nameOffset);

      // Read value from entry (type starts at +4)
      const valuePtr = entriesPtr + GC_HEADER_SIZE + i * 20 + 4;
      result[key] = this.readValueAt(valuePtr);
    }
    return result;
  }

  /**
   * Set an input value by marshalling JS value into scope.
   *
   * @param {number} index - Input slot index
   * @param {*} jsValue - JS value to marshal
   * @throws if slot not configured
   */
  setInput(index, jsValue) {
    if (index >= this.getInputCount()) {
      throw new Error(`Input slot ${index} not configured`);
    }

    const valuePtr = this.getInputValuePtr(index);
    this.writeValueAt(valuePtr, jsValue);
  }

  /**
   * Get an output value by marshalling scope value to JS.
   *
   * @param {number} index - Output slot index
   * @returns {*} - JS value
   * @throws if slot not configured
   */
  getOutput(index) {
    if (index >= this.getOutputCount()) {
      throw new Error(`Output slot ${index} not configured`);
    }

    const valuePtr = this.getOutputValuePtr(index);
    return this.readValueAt(valuePtr);
  }

}
