/**
 * SandScript Fuel Lexer
 *
 * Tokenizes loose SandScript source including operators.
 * Unlike the strict lexer, this handles +, -, *, /, ===, etc.
 */

export const TokenType = {
  // Literals
  NUMBER: 'NUMBER',
  STRING: 'STRING',
  IDENTIFIER: 'IDENTIFIER',

  // Keywords
  LET: 'LET',
  RETURN: 'RETURN',
  THROW: 'THROW',
  TRY: 'TRY',
  CATCH: 'CATCH',
  FINALLY: 'FINALLY',
  IF: 'IF',
  ELSE: 'ELSE',
  WHILE: 'WHILE',
  DO: 'DO',
  FOR: 'FOR',
  BREAK: 'BREAK',
  CONTINUE: 'CONTINUE',
  TRUE: 'TRUE',
  FALSE: 'FALSE',
  NULL: 'NULL',
  UNDEFINED: 'UNDEFINED',
  TYPEOF: 'TYPEOF',
  THIS: 'THIS',
  FUNCTION: 'FUNCTION',
  NEW: 'NEW',

  // Punctuation
  LPAREN: 'LPAREN',
  RPAREN: 'RPAREN',
  LBRACE: 'LBRACE',
  RBRACE: 'RBRACE',
  LBRACKET: 'LBRACKET',
  RBRACKET: 'RBRACKET',
  SEMICOLON: 'SEMICOLON',
  COLON: 'COLON',
  COMMA: 'COMMA',
  DOT: 'DOT',
  ARROW: 'ARROW',

  // Assignment
  ASSIGN: 'ASSIGN',

  // Arithmetic operators
  PLUS: 'PLUS',
  MINUS: 'MINUS',
  STAR: 'STAR',
  SLASH: 'SLASH',
  PERCENT: 'PERCENT',
  STAR_STAR: 'STAR_STAR',

  // Compound assignment
  PLUS_ASSIGN: 'PLUS_ASSIGN',       // +=
  MINUS_ASSIGN: 'MINUS_ASSIGN',     // -=
  STAR_ASSIGN: 'STAR_ASSIGN',       // *=
  SLASH_ASSIGN: 'SLASH_ASSIGN',     // /=
  PERCENT_ASSIGN: 'PERCENT_ASSIGN', // %=
  STAR_STAR_ASSIGN: 'STAR_STAR_ASSIGN', // **=

  // Increment/Decrement
  PLUS_PLUS: 'PLUS_PLUS',     // ++
  MINUS_MINUS: 'MINUS_MINUS', // --

  // Comparison operators
  EQ: 'EQ',           // ===
  NEQ: 'NEQ',         // !==
  LT: 'LT',           // <
  GT: 'GT',           // >
  LTE: 'LTE',         // <=
  GTE: 'GTE',         // >=

  // Logical operators
  AND: 'AND',         // &&
  OR: 'OR',           // ||
  NOT: 'NOT',         // !

  // Bitwise operators
  AMPERSAND: 'AMPERSAND',   // &
  PIPE: 'PIPE',             // |
  CARET: 'CARET',           // ^
  TILDE: 'TILDE',           // ~
  LSHIFT: 'LSHIFT',         // <<
  RSHIFT: 'RSHIFT',         // >>
  URSHIFT: 'URSHIFT',       // >>>

  // Bitwise compound assignment
  AMPERSAND_ASSIGN: 'AMPERSAND_ASSIGN', // &=
  PIPE_ASSIGN: 'PIPE_ASSIGN',           // |=
  CARET_ASSIGN: 'CARET_ASSIGN',         // ^=
  LSHIFT_ASSIGN: 'LSHIFT_ASSIGN',       // <<=
  RSHIFT_ASSIGN: 'RSHIFT_ASSIGN',       // >>=
  URSHIFT_ASSIGN: 'URSHIFT_ASSIGN',     // >>>=

  // Ternary
  QUESTION: 'QUESTION', // ?

  // Nullish/Optional
  QUESTION_QUESTION: 'QUESTION_QUESTION', // ??
  QUESTION_DOT: 'QUESTION_DOT',           // ?.

  EOF: 'EOF',
};

const KEYWORDS = {
  'let': TokenType.LET,
  'return': TokenType.RETURN,
  'throw': TokenType.THROW,
  'try': TokenType.TRY,
  'catch': TokenType.CATCH,
  'finally': TokenType.FINALLY,
  'if': TokenType.IF,
  'else': TokenType.ELSE,
  'while': TokenType.WHILE,
  'do': TokenType.DO,
  'for': TokenType.FOR,
  'break': TokenType.BREAK,
  'continue': TokenType.CONTINUE,
  'true': TokenType.TRUE,
  'false': TokenType.FALSE,
  'null': TokenType.NULL,
  'undefined': TokenType.UNDEFINED,
  'typeof': TokenType.TYPEOF,
  'this': TokenType.THIS,
  'function': TokenType.FUNCTION,
  'new': TokenType.NEW,
};

export class Lexer {
  constructor(source = '') {
    this.source = source;
    this.pos = 0;
    this.line = 1;
    this.col = 1;
    this.tokenStart = 0;
    this.tokenLine = 1;
    this.tokenCol = 1;
  }

  reset(source) {
    this.source = source;
    this.pos = 0;
    this.line = 1;
    this.col = 1;
    this.tokenStart = 0;
    this.tokenLine = 1;
    this.tokenCol = 1;
  }

  peek(offset = 0) {
    const idx = this.pos + offset;
    return idx < this.source.length ? this.source.charCodeAt(idx) : 0;
  }

  advance() {
    const ch = this.source.charCodeAt(this.pos++) || 0;
    if (ch === 10) { // \n
      this.line++;
      this.col = 1;
    } else {
      this.col++;
    }
    return ch;
  }

  skipWhitespace() {
    while (true) {
      const ch = this.peek();
      if (ch === 32 || ch === 9 || ch === 13 || ch === 10) { // space, tab, \r, \n
        this.advance();
      } else if (ch === 47 && this.peek(1) === 47) { // //
        while (this.peek() !== 10 && this.peek() !== 0) this.advance();
      } else if (ch === 47 && this.peek(1) === 42) { // /*
        this.advance();
        this.advance();
        while (!(this.peek() === 42 && this.peek(1) === 47) && this.peek() !== 0) {
          this.advance();
        }
        if (this.peek() !== 0) {
          this.advance();
          this.advance();
        }
      } else {
        break;
      }
    }
  }

  error(message) {
    return new SyntaxError(`${message} at line ${this.tokenLine}, col ${this.tokenCol}`);
  }

  makeToken(type, value = null) {
    return {
      type,
      value,
      start: this.tokenStart,
      end: this.pos,
      line: this.tokenLine,
      col: this.tokenCol,
    };
  }

  scanString() {
    const quote = this.advance(); // consume " or '
    let value = '';
    while (this.peek() !== quote && this.peek() !== 0) {
      if (this.peek() === 10) throw this.error('Unterminated string');
      if (this.peek() === 92) { // \
        this.advance();
        const esc = this.advance();
        switch (esc) {
          case 110: value += '\n'; break; // n
          case 116: value += '\t'; break; // t
          case 114: value += '\r'; break; // r
          case 92: value += '\\'; break;  // \
          case 34: value += '"'; break;   // "
          case 39: value += "'"; break;   // '
          default: value += String.fromCharCode(esc);
        }
      } else {
        value += String.fromCharCode(this.advance());
      }
    }
    if (this.peek() === 0) throw this.error('Unterminated string');
    this.advance(); // consume closing quote
    return this.makeToken(TokenType.STRING, value);
  }

  scanNumber() {
    const firstChar = this.advance();

    // Check for base prefix when first char is '0'
    if (firstChar === 48) { // '0'
      const prefix = this.peek();
      if (prefix === 120 || prefix === 88) { // x, X
        this.advance();
        return this.scanBaseNumber(16, 'hexadecimal');
      } else if (prefix === 98 || prefix === 66) { // b, B
        this.advance();
        return this.scanBaseNumber(2, 'binary');
      } else if (prefix === 111 || prefix === 79) { // o, O
        this.advance();
        return this.scanBaseNumber(8, 'octal');
      }
      // Fall through - decimal starting with 0
    }

    // Decimal number
    let value = String.fromCharCode(firstChar);
    value += this.scanDecimalDigits(false);

    // Fractional part
    if (this.peek() === 46 && this.isDigit(this.peek(1))) { // .
      value += String.fromCharCode(this.advance());
      value += this.scanDecimalDigits(true);
    }

    // Exponent
    if (this.peek() === 101 || this.peek() === 69) { // e, E
      value += String.fromCharCode(this.advance());
      if (this.peek() === 43 || this.peek() === 45) { // +, -
        value += String.fromCharCode(this.advance());
      }
      value += this.scanDecimalDigits(true);
    }

    return this.makeToken(TokenType.NUMBER, parseFloat(value));
  }

  /**
   * Scan digits for non-decimal bases (hex, binary, octal).
   * Returns NUMBER token with parsed integer value.
   */
  scanBaseNumber(base, name) {
    // Check for separator immediately after prefix
    if (this.peek() === 95) { // _
      throw this.error(`Numeric separator after ${name} prefix`);
    }

    let value = '';
    let lastWasSeparator = false;

    while (true) {
      const ch = this.peek();

      if (ch === 95) { // _
        if (lastWasSeparator) {
          throw this.error('Consecutive numeric separators');
        }
        if (value === '') {
          throw this.error(`Numeric separator after ${name} prefix`);
        }
        lastWasSeparator = true;
        this.advance();
        continue;
      }

      if (this.isValidDigit(ch, base)) {
        value += String.fromCharCode(this.advance());
        lastWasSeparator = false;
      } else if (this.isDigit(ch) || this.isAlpha(ch)) {
        // Invalid digit for this base
        throw this.error(`Invalid ${name} digit '${String.fromCharCode(ch)}'`);
      } else {
        break;
      }
    }

    if (value === '') {
      throw this.error(`${name.charAt(0).toUpperCase() + name.slice(1)} literal with no digits`);
    }

    if (lastWasSeparator) {
      throw this.error('Trailing numeric separator');
    }

    return this.makeToken(TokenType.NUMBER, parseInt(value, base));
  }

  /**
   * Scan decimal digits with separator support.
   * @param {boolean} required - If true, at least one digit is required
   */
  scanDecimalDigits(required) {
    let value = '';
    let lastWasSeparator = false;

    while (true) {
      const ch = this.peek();

      if (ch === 95) { // _
        if (lastWasSeparator) {
          throw this.error('Consecutive numeric separators');
        }
        if (value === '' && required) {
          throw this.error('Numeric separator at invalid position');
        }
        lastWasSeparator = true;
        this.advance();
        continue;
      }

      if (this.isDigit(ch)) {
        value += String.fromCharCode(this.advance());
        lastWasSeparator = false;
      } else {
        break;
      }
    }

    if (lastWasSeparator) {
      throw this.error('Trailing numeric separator');
    }

    if (required && value === '') {
      throw this.error('Expected digits');
    }

    return value;
  }

  /**
   * Check if character is a valid digit for the given base.
   */
  isValidDigit(ch, base) {
    if (base === 2) {
      return ch === 48 || ch === 49; // 0, 1
    } else if (base === 8) {
      return ch >= 48 && ch <= 55; // 0-7
    } else if (base === 16) {
      return (ch >= 48 && ch <= 57) ||  // 0-9
             (ch >= 65 && ch <= 70) ||  // A-F
             (ch >= 97 && ch <= 102);   // a-f
    }
    return this.isDigit(ch);
  }

  scanIdentifier() {
    let value = '';
    while (this.isAlphaNumeric(this.peek())) {
      value += String.fromCharCode(this.advance());
    }
    const type = KEYWORDS[value] || TokenType.IDENTIFIER;
    return this.makeToken(type, value);
  }

  isDigit(ch) {
    return ch >= 48 && ch <= 57; // 0-9
  }

  isAlpha(ch) {
    return (ch >= 65 && ch <= 90) || (ch >= 97 && ch <= 122) || ch === 95 || ch === 36; // A-Z, a-z, _, $
  }

  isAlphaNumeric(ch) {
    return this.isAlpha(ch) || this.isDigit(ch);
  }

  next() {
    this.skipWhitespace();

    this.tokenStart = this.pos;
    this.tokenLine = this.line;
    this.tokenCol = this.col;

    if (this.pos >= this.source.length) {
      return this.makeToken(TokenType.EOF);
    }

    const ch = this.peek();

    // String
    if (ch === 34 || ch === 39) { // " or '
      return this.scanString();
    }

    // Number
    if (this.isDigit(ch)) {
      return this.scanNumber();
    }

    // Identifier or keyword
    if (this.isAlpha(ch)) {
      return this.scanIdentifier();
    }

    // Single character tokens
    this.advance();

    switch (ch) {
      case 40: return this.makeToken(TokenType.LPAREN);   // (
      case 41: return this.makeToken(TokenType.RPAREN);   // )
      case 123: return this.makeToken(TokenType.LBRACE); // {
      case 125: return this.makeToken(TokenType.RBRACE); // }
      case 91: return this.makeToken(TokenType.LBRACKET); // [
      case 93: return this.makeToken(TokenType.RBRACKET); // ]
      case 59: return this.makeToken(TokenType.SEMICOLON); // ;
      case 58: return this.makeToken(TokenType.COLON);    // :
      case 44: return this.makeToken(TokenType.COMMA);    // ,
      case 46: return this.makeToken(TokenType.DOT);      // .

      case 43: // +
        if (this.peek() === 43) { // ++
          this.advance();
          return this.makeToken(TokenType.PLUS_PLUS);
        }
        if (this.peek() === 61) { // +=
          this.advance();
          return this.makeToken(TokenType.PLUS_ASSIGN);
        }
        return this.makeToken(TokenType.PLUS);
      case 45: // -
        if (this.peek() === 45) { // --
          this.advance();
          return this.makeToken(TokenType.MINUS_MINUS);
        }
        if (this.peek() === 61) { // -=
          this.advance();
          return this.makeToken(TokenType.MINUS_ASSIGN);
        }
        return this.makeToken(TokenType.MINUS);
      case 42: // *
        if (this.peek() === 42) { // **
          this.advance();
          if (this.peek() === 61) { // **=
            this.advance();
            return this.makeToken(TokenType.STAR_STAR_ASSIGN);
          }
          return this.makeToken(TokenType.STAR_STAR);
        }
        if (this.peek() === 61) { // *=
          this.advance();
          return this.makeToken(TokenType.STAR_ASSIGN);
        }
        return this.makeToken(TokenType.STAR);
      case 47: // /
        if (this.peek() === 61) { // /=
          this.advance();
          return this.makeToken(TokenType.SLASH_ASSIGN);
        }
        return this.makeToken(TokenType.SLASH);
      case 37: // %
        if (this.peek() === 61) { // %=
          this.advance();
          return this.makeToken(TokenType.PERCENT_ASSIGN);
        }
        return this.makeToken(TokenType.PERCENT);

      case 60: // <
        if (this.peek() === 60) { // <<
          this.advance();
          if (this.peek() === 61) { // <<=
            this.advance();
            return this.makeToken(TokenType.LSHIFT_ASSIGN);
          }
          return this.makeToken(TokenType.LSHIFT);
        }
        if (this.peek() === 61) { // <=
          this.advance();
          return this.makeToken(TokenType.LTE);
        }
        return this.makeToken(TokenType.LT);
      case 62: // >
        if (this.peek() === 62) { // >>
          this.advance();
          if (this.peek() === 62) { // >>>
            this.advance();
            if (this.peek() === 61) { // >>>=
              this.advance();
              return this.makeToken(TokenType.URSHIFT_ASSIGN);
            }
            return this.makeToken(TokenType.URSHIFT);
          }
          if (this.peek() === 61) { // >>=
            this.advance();
            return this.makeToken(TokenType.RSHIFT_ASSIGN);
          }
          return this.makeToken(TokenType.RSHIFT);
        }
        if (this.peek() === 61) { // >=
          this.advance();
          return this.makeToken(TokenType.GTE);
        }
        return this.makeToken(TokenType.GT);

      case 61: // =
        if (this.peek() === 61 && this.peek(1) === 61) { // ===
          this.advance();
          this.advance();
          return this.makeToken(TokenType.EQ);
        }
        if (this.peek() === 62) { // =>
          this.advance();
          return this.makeToken(TokenType.ARROW);
        }
        return this.makeToken(TokenType.ASSIGN);

      case 33: // !
        if (this.peek() === 61 && this.peek(1) === 61) { // !==
          this.advance();
          this.advance();
          return this.makeToken(TokenType.NEQ);
        }
        return this.makeToken(TokenType.NOT);

      case 38: // &
        if (this.peek() === 38) { // &&
          this.advance();
          return this.makeToken(TokenType.AND);
        }
        if (this.peek() === 61) { // &=
          this.advance();
          return this.makeToken(TokenType.AMPERSAND_ASSIGN);
        }
        return this.makeToken(TokenType.AMPERSAND);

      case 124: // |
        if (this.peek() === 124) { // ||
          this.advance();
          return this.makeToken(TokenType.OR);
        }
        if (this.peek() === 61) { // |=
          this.advance();
          return this.makeToken(TokenType.PIPE_ASSIGN);
        }
        return this.makeToken(TokenType.PIPE);

      case 94: // ^
        if (this.peek() === 61) { // ^=
          this.advance();
          return this.makeToken(TokenType.CARET_ASSIGN);
        }
        return this.makeToken(TokenType.CARET);

      case 126: // ~
        return this.makeToken(TokenType.TILDE);

      case 63: // ?
        if (this.peek() === 63) { // ??
          this.advance();
          return this.makeToken(TokenType.QUESTION_QUESTION);
        }
        if (this.peek() === 46 && !this.isDigit(this.peek(1))) { // ?. (not ?.5 which would be ? .5)
          this.advance();
          return this.makeToken(TokenType.QUESTION_DOT);
        }
        return this.makeToken(TokenType.QUESTION);

      default:
        throw this.error(`Unexpected character '${String.fromCharCode(ch)}'`);
    }
  }

  /**
   * Peek at the next token without consuming it.
   */
  peekToken() {
    const savedPos = this.pos;
    const savedLine = this.line;
    const savedCol = this.col;
    const token = this.next();
    this.pos = savedPos;
    this.line = savedLine;
    this.col = savedCol;
    return token;
  }
}
