(module
  ;; ==========================================================================
  ;; Fuel-Based SandScript Interpreter
  ;;
  ;; Phase 0: String interning only
  ;; Phase 1+: Full interpreter with run(), gc()
  ;; ==========================================================================

  ;; Memory is imported from JS (allows segment model - multiple interpreters
  ;; can share a single WebAssembly.Memory with different base offsets)
  (import "env" "memory" (memory 1))

  ;; ==========================================================================
  ;; Constants
  ;; ==========================================================================

  ;; Hash table: 2048 buckets × 8 bytes = 16384 bytes (0x4000)
  ;; Each bucket: [hash:i32][offset:i32]
  (global $HASH_TABLE_BUCKETS i32 (i32.const 2048))
  (global $HASH_TABLE_SIZE i32 (i32.const 16384))

  ;; FNV-1a constants (32-bit)
  (global $FNV_OFFSET_BASIS i32 (i32.const 0x811c9dc5))
  (global $FNV_PRIME i32 (i32.const 0x01000193))

  ;; Header size (SANDFUEL magic + versions = 16 bytes)
  (global $HEADER_SIZE i32 (i32.const 16))

  ;; State header offsets (must match constants.js, all include HEADER_SIZE)
  (global $STATE_STRING_POINTER i32 (i32.const 20))   ;; 16 + 4
  (global $STATE_STRING_START i32 (i32.const 40))     ;; 16 + 24
  (global $STATE_SEGMENT_SIZE i32 (i32.const 44))     ;; 16 + 28

  ;; ==========================================================================
  ;; Internal: base_offset global (set by init)
  ;; ==========================================================================
  (global $base_offset (mut i32) (i32.const 0))

  ;; ==========================================================================
  ;; init_segment(base_offset)
  ;;
  ;; Initialize the interpreter for a memory segment.
  ;; Must be called before any other functions.
  ;; ==========================================================================
  (func $init_segment (export "init_segment")
    (param $base i32)
    (global.set $base_offset (local.get $base))
  )

  ;; ==========================================================================
  ;; init_regions()
  ;;
  ;; Read region base offsets from STATE into WASM globals.
  ;; Must be called after JS has written region bases to STATE.
  ;; ==========================================================================
  (func $init_regions (export "init_regions")
    (global.set $ffi_request_base (call $read_state (global.get $STATE_FFI_REQUEST_BASE)))
    (global.set $error_info_base (call $read_state (global.get $STATE_ERROR_INFO_BASE)))
    (global.set $call_stack_base (call $read_state (global.get $STATE_CALL_STACK_BASE)))
    (global.set $pending_stack_base (call $read_state (global.get $STATE_PENDING_STACK_BASE)))
    (global.set $try_stack_base (call $read_state (global.get $STATE_TRY_STACK_BASE)))
    (global.set $builtins_base (call $read_state (global.get $STATE_BUILTINS_BASE)))
  )

  ;; ==========================================================================
  ;; init_string_table()
  ;;
  ;; Zero out the hash table region. Call after memory is set up.
  ;; Hash table starts at string_start (from state header).
  ;; ==========================================================================
  (func $init_string_table (export "init_string_table")
    (local $hash_table_start i32)
    (local $i i32)

    ;; hash_table_start = base_offset + string_start
    (local.set $hash_table_start
      (i32.add
        (global.get $base_offset)
        (i32.load (i32.add (global.get $base_offset) (global.get $STATE_STRING_START)))
      )
    )

    ;; Zero out 16KB hash table (2048 buckets × 8 bytes)
    (local.set $i (i32.const 0))
    (block $done
      (loop $loop
        (br_if $done (i32.ge_u (local.get $i) (global.get $HASH_TABLE_SIZE)))
        (i32.store
          (i32.add (local.get $hash_table_start) (local.get $i))
          (i32.const 0)
        )
        (local.set $i (i32.add (local.get $i) (i32.const 4)))
        (br $loop)
      )
    )
  )

  ;; ==========================================================================
  ;; fnv1a(pointer, len) → hash
  ;;
  ;; Compute FNV-1a hash of bytes at pointer..pointer+len
  ;; ==========================================================================
  (func $fnv1a (param $pointer i32) (param $len i32) (result i32)
    (local $hash i32)
    (local $end i32)
    (local $byte i32)

    (local.set $hash (global.get $FNV_OFFSET_BASIS))
    (local.set $end (i32.add (local.get $pointer) (local.get $len)))

    (block $done
      (loop $loop
        (br_if $done (i32.ge_u (local.get $pointer) (local.get $end)))

        ;; hash ^= byte
        (local.set $byte (i32.load8_u (local.get $pointer)))
        (local.set $hash (i32.xor (local.get $hash) (local.get $byte)))

        ;; hash *= FNV_PRIME
        (local.set $hash (i32.mul (local.get $hash) (global.get $FNV_PRIME)))

        (local.set $pointer (i32.add (local.get $pointer) (i32.const 1)))
        (br $loop)
      )
    )

    (local.get $hash)
  )

  ;; ==========================================================================
  ;; strings_equal(pointer1, len1, pointer2, len2) → bool
  ;;
  ;; Compare two byte sequences for equality.
  ;; ==========================================================================
  (func $strings_equal
    (param $pointer1 i32) (param $len1 i32)
    (param $pointer2 i32) (param $len2 i32)
    (result i32)
    (local $end i32)

    ;; Different lengths → not equal
    (if (i32.ne (local.get $len1) (local.get $len2))
      (then (return (i32.const 0)))
    )

    ;; Compare bytes
    (local.set $end (i32.add (local.get $pointer1) (local.get $len1)))
    (block $done
      (loop $loop
        (br_if $done (i32.ge_u (local.get $pointer1) (local.get $end)))

        (if (i32.ne
              (i32.load8_u (local.get $pointer1))
              (i32.load8_u (local.get $pointer2)))
          (then (return (i32.const 0)))
        )

        (local.set $pointer1 (i32.add (local.get $pointer1) (i32.const 1)))
        (local.set $pointer2 (i32.add (local.get $pointer2) (i32.const 1)))
        (br $loop)
      )
    )

    (i32.const 1)
  )

  ;; ==========================================================================
  ;; intern_string(pointer, len) → string_offset
  ;;
  ;; Intern a string. pointer/len point to UTF-8 bytes in memory.
  ;; Returns segment-relative offset to string entry.
  ;; If string already exists, returns existing offset (deduplication).
  ;;
  ;; String entry format: [length:i32][bytes...]
  ;; Entries are 4-byte aligned.
  ;; ==========================================================================
  (func $intern_string (export "intern_string")
    (param $pointer i32) (param $len i32)
    (result i32)
    (local $hash i32)
    (local $bucket_index i32)
    (local $hash_table_start i32)
    (local $bucket_pointer i32)
    (local $stored_hash i32)
    (local $stored_offset i32)
    (local $abs_stored_pointer i32)
    (local $stored_len i32)
    (local $probe i32)
    (local $string_pointer i32)
    (local $abs_string_pointer i32)
    (local $entry_size i32)
    (local $aligned_size i32)
    (local $string_start i32)

    ;; Compute hash
    (local.set $hash (call $fnv1a (local.get $pointer) (local.get $len)))

    ;; bucket_index = hash % HASH_TABLE_BUCKETS
    (local.set $bucket_index
      (i32.rem_u (local.get $hash) (global.get $HASH_TABLE_BUCKETS))
    )

    ;; hash_table_start = base_offset + string_start (from state header)
    (local.set $string_start
      (i32.load (i32.add (global.get $base_offset) (global.get $STATE_STRING_START)))
    )
    (local.set $hash_table_start
      (i32.add (global.get $base_offset) (local.get $string_start))
    )

    ;; Linear probing
    (local.set $probe (i32.const 0))
    (block $found
      (block $not_found
        (loop $probe_loop
          ;; Give up after checking all buckets
          (br_if $not_found
            (i32.ge_u (local.get $probe) (global.get $HASH_TABLE_BUCKETS))
          )

          ;; bucket_pointer = hash_table_start + ((bucket_index + probe) % buckets) * 8
          (local.set $bucket_pointer
            (i32.add
              (local.get $hash_table_start)
              (i32.mul
                (i32.rem_u
                  (i32.add (local.get $bucket_index) (local.get $probe))
                  (global.get $HASH_TABLE_BUCKETS)
                )
                (i32.const 8)
              )
            )
          )

          ;; Read bucket
          (local.set $stored_hash (i32.load (local.get $bucket_pointer)))
          (local.set $stored_offset (i32.load (i32.add (local.get $bucket_pointer) (i32.const 4))))

          ;; Empty bucket → insert here
          (if (i32.eqz (local.get $stored_offset))
            (then (br $not_found))
          )

          ;; Check if same string
          (if (i32.eq (local.get $stored_hash) (local.get $hash))
            (then
              ;; Compare actual bytes
              (local.set $abs_stored_pointer
                (i32.add (global.get $base_offset) (local.get $stored_offset))
              )
              (local.set $stored_len (i32.load (local.get $abs_stored_pointer)))

              (if (call $strings_equal
                    (local.get $pointer)
                    (local.get $len)
                    (i32.add (local.get $abs_stored_pointer) (i32.const 4))
                    (local.get $stored_len))
                (then
                  ;; Found existing string
                  (return (local.get $stored_offset))
                )
              )
            )
          )

          ;; Try next bucket
          (local.set $probe (i32.add (local.get $probe) (i32.const 1)))
          (br $probe_loop)
        )
      )

      ;; Not found - allocate new string entry

      ;; Read current string_pointer from state header
      (local.set $string_pointer
        (i32.load (i32.add (global.get $base_offset) (global.get $STATE_STRING_POINTER)))
      )

      ;; entry_size = 4 (length) + len (bytes)
      ;; aligned_size = (entry_size + 3) & ~3
      (local.set $entry_size (i32.add (i32.const 4) (local.get $len)))
      (local.set $aligned_size
        (i32.and
          (i32.add (local.get $entry_size) (i32.const 3))
          (i32.const 0xfffffffc)
        )
      )

      ;; Write string entry at string_pointer
      (local.set $abs_string_pointer
        (i32.add (global.get $base_offset) (local.get $string_pointer))
      )

      ;; Write length
      (i32.store (local.get $abs_string_pointer) (local.get $len))

      ;; Copy bytes
      (memory.copy
        (i32.add (local.get $abs_string_pointer) (i32.const 4))
        (local.get $pointer)
        (local.get $len)
      )

      ;; Update bucket
      (i32.store (local.get $bucket_pointer) (local.get $hash))
      (i32.store (i32.add (local.get $bucket_pointer) (i32.const 4)) (local.get $string_pointer))

      ;; Update string_pointer in state header
      (i32.store
        (i32.add (global.get $base_offset) (global.get $STATE_STRING_POINTER))
        (i32.add (local.get $string_pointer) (local.get $aligned_size))
      )

      ;; Return the offset we just allocated
      (return (local.get $string_pointer))
    )

    ;; Unreachable
    (unreachable)
  )

  ;; ==========================================================================
  ;; read_string_len(offset) → length
  ;;
  ;; Read the byte length of a string at the given segment-relative offset.
  ;; ==========================================================================
  (func $read_string_len (export "read_string_len")
    (param $offset i32) (result i32)
    (i32.load (i32.add (global.get $base_offset) (local.get $offset)))
  )

  ;; ==========================================================================
  ;; get_string_pointer(offset) → absolute_pointer
  ;;
  ;; Get absolute pointer to string bytes (after length prefix).
  ;; ==========================================================================
  (func $get_string_pointer (export "get_string_ptr")
    (param $offset i32) (result i32)
    (i32.add
      (i32.add (global.get $base_offset) (local.get $offset))
      (i32.const 4)
    )
  )

  ;; ==========================================================================
  ;; string_concat(offset1, offset2) → new_string_offset
  ;;
  ;; Concatenate two strings and return the interned result.
  ;; Uses heap space beyond heap_pointer as scratch (doesn't move pointer).
  ;; ==========================================================================
  (func $string_concat (param $offset1 i32) (param $offset2 i32) (result i32)
    (local $len1 i32)
    (local $len2 i32)
    (local $total_len i32)
    (local $scratch_ptr i32)
    (local $src1_ptr i32)
    (local $src2_ptr i32)

    ;; Get lengths
    (local.set $len1 (call $read_string_len (local.get $offset1)))
    (local.set $len2 (call $read_string_len (local.get $offset2)))
    (local.set $total_len (i32.add (local.get $len1) (local.get $len2)))

    ;; Use space beyond heap pointer as scratch (absolute address)
    (local.set $scratch_ptr
      (call $abs (call $read_state (global.get $STATE_HEAP_POINTER))))

    ;; Get source pointers (absolute)
    (local.set $src1_ptr (call $get_string_pointer (local.get $offset1)))
    (local.set $src2_ptr (call $get_string_pointer (local.get $offset2)))

    ;; Copy first string
    (memory.copy
      (local.get $scratch_ptr)
      (local.get $src1_ptr)
      (local.get $len1))

    ;; Copy second string
    (memory.copy
      (i32.add (local.get $scratch_ptr) (local.get $len1))
      (local.get $src2_ptr)
      (local.get $len2))

    ;; Intern the concatenated string (scratch_ptr is absolute, intern expects absolute)
    (call $intern_string (local.get $scratch_ptr) (local.get $total_len))
  )

  ;; ==========================================================================
  ;; value_to_string(value_ptr) → string_offset
  ;;
  ;; Convert any value to its string representation.
  ;; Returns a string table offset.
  ;; ==========================================================================
  (func $value_to_string (param $val_ptr i32) (result i32)
    (local $type i32)
    (local $data_lo i32)

    (local.set $type (call $value_type (local.get $val_ptr)))
    (local.set $data_lo (call $value_data_lo (local.get $val_ptr)))

    ;; String - already a string offset
    (if (i32.eq (local.get $type) (global.get $TYPE_STRING))
      (then (return (local.get $data_lo)))
    )

    ;; Float - use f64_to_string
    (if (i32.eq (local.get $type) (global.get $TYPE_FLOAT))
      (then (return (call $f64_to_string (call $read_f64 (local.get $val_ptr)))))
    )

    ;; Boolean - "true" or "false"
    (if (i32.eq (local.get $type) (global.get $TYPE_BOOLEAN))
      (then
        (if (local.get $data_lo)
          (then (return (call $read_builtin (global.get $BUILTIN_LIT_TRUE))))
          (else (return (call $read_builtin (global.get $BUILTIN_LIT_FALSE))))
        )
      )
    )

    ;; Null - "null"
    (if (i32.eq (local.get $type) (global.get $TYPE_NULL))
      (then (return (call $read_builtin (global.get $BUILTIN_LIT_NULL))))
    )

    ;; Undefined - "undefined"
    (if (i32.eq (local.get $type) (global.get $TYPE_UNDEFINED))
      (then (return (call $read_builtin (global.get $BUILTIN_TYPEOF_UNDEFINED))))
    )

    ;; Object - "[object Object]"
    (if (i32.eq (local.get $type) (global.get $TYPE_OBJECT))
      (then (return (call $read_builtin (global.get $BUILTIN_LIT_OBJECT_OBJECT))))
    )

    ;; Array - TODO: would need to join elements, for now return placeholder
    ;; For now, return empty string (will need proper implementation)
    (call $read_builtin (global.get $BUILTIN_EMPTY_STRING))
  )

  ;; ==========================================================================
  ;; Constants for interpreter
  ;; ==========================================================================

  ;; State header offsets (must match constants.js STATE)
  ;; All offsets include HEADER_SIZE (16 bytes)
  (global $STATE_HEAP_POINTER i32 (i32.const 0x10))      ;; 16 + 0x00
  ;; $STATE_STRING_POINTER already defined above as 20 (16 + 4)
  (global $STATE_SCOPE i32 (i32.const 0x18))             ;; 16 + 0x08
  (global $STATE_RESULT_POINTER i32 (i32.const 0x1C))    ;; 16 + 0x0C
  (global $STATE_HEAP_START i32 (i32.const 0x20))        ;; 16 + 0x10
  (global $STATE_HEAP_END i32 (i32.const 0x24))          ;; 16 + 0x14
  ;; $STATE_STRING_START already defined above as 40 (16 + 24)
  ;; $STATE_SEGMENT_SIZE already defined above as 44 (16 + 28)
  (global $STATE_FUEL i32 (i32.const 0x30))              ;; 16 + 0x20
  (global $STATE_STATUS i32 (i32.const 0x34))            ;; 16 + 0x24
  (global $STATE_STACK_POINTER i32 (i32.const 0x38))     ;; 16 + 0x28
  (global $STATE_STACK_BASE i32 (i32.const 0x3C))        ;; 16 + 0x2C
  (global $STATE_PENDING_POINTER i32 (i32.const 0x40))   ;; 16 + 0x30
  (global $STATE_PENDING_BASE i32 (i32.const 0x44))      ;; 16 + 0x34
  (global $STATE_CODE_BLOCK i32 (i32.const 0x50))        ;; 16 + 0x40
  (global $STATE_INSTRUCTION_INDEX i32 (i32.const 0x54)) ;; 16 + 0x44
  (global $STATE_CODE_POINTER i32 (i32.const 0x4C))      ;; 16 + 0x3C
  (global $STATE_TRY_POINTER i32 (i32.const 0x58))       ;; 16 + 0x48
  (global $STATE_TRY_BASE i32 (i32.const 0x5C))          ;; 16 + 0x4C
  (global $STATE_COMPLETION_TYPE i32 (i32.const 0x60))   ;; 16 + 0x50
  (global $STATE_COMPLETION_VALUE i32 (i32.const 0x64))  ;; 16 + 0x54
  ;; Region base offsets (read from STATE at init)
  (global $STATE_FFI_REQUEST_BASE i32 (i32.const 0x68))  ;; 16 + 0x58
  (global $STATE_ERROR_INFO_BASE i32 (i32.const 0x6C))   ;; 16 + 0x5C
  (global $STATE_SCRATCH_BASE i32 (i32.const 0x70))      ;; 16 + 0x60
  (global $STATE_CALL_STACK_BASE i32 (i32.const 0x74))   ;; 16 + 0x64
  (global $STATE_PENDING_STACK_BASE i32 (i32.const 0x78)) ;; 16 + 0x68
  (global $STATE_TRY_STACK_BASE i32 (i32.const 0x7C))    ;; 16 + 0x6C
  (global $STATE_BUILTINS_BASE i32 (i32.const 0x80))     ;; 16 + 0x70

  ;; BUILTINS segment offsets (relative to builtins_base)
  ;; Typeof string offsets (pre-interned during initialization)
  (global $BUILTIN_TYPEOF_UNDEFINED i32 (i32.const 0x00))
  (global $BUILTIN_TYPEOF_BOOLEAN i32 (i32.const 0x04))
  (global $BUILTIN_TYPEOF_NUMBER i32 (i32.const 0x08))
  (global $BUILTIN_TYPEOF_STRING i32 (i32.const 0x0C))
  (global $BUILTIN_TYPEOF_OBJECT i32 (i32.const 0x10))
  (global $BUILTIN_TYPEOF_FUNCTION i32 (i32.const 0x14))
  ;; Special keywords
  (global $BUILTIN_THIS i32 (i32.const 0x18))
  (global $BUILTIN_LENGTH i32 (i32.const 0x1C))

  ;; Array method names (0x20 - 0x6F)
  (global $BUILTIN_PUSH i32 (i32.const 0x20))
  (global $BUILTIN_POP i32 (i32.const 0x24))
  (global $BUILTIN_SHIFT i32 (i32.const 0x28))
  (global $BUILTIN_UNSHIFT i32 (i32.const 0x2C))
  (global $BUILTIN_SLICE i32 (i32.const 0x30))
  (global $BUILTIN_CONCAT i32 (i32.const 0x34))
  (global $BUILTIN_JOIN i32 (i32.const 0x38))
  (global $BUILTIN_REVERSE i32 (i32.const 0x3C))
  (global $BUILTIN_INDEX_OF i32 (i32.const 0x40))
  (global $BUILTIN_INCLUDES i32 (i32.const 0x44))
  (global $BUILTIN_MAP i32 (i32.const 0x48))
  (global $BUILTIN_FILTER i32 (i32.const 0x4C))
  (global $BUILTIN_REDUCE i32 (i32.const 0x50))
  (global $BUILTIN_FOR_EACH i32 (i32.const 0x54))
  (global $BUILTIN_FIND i32 (i32.const 0x58))
  (global $BUILTIN_FIND_INDEX i32 (i32.const 0x5C))
  (global $BUILTIN_SOME i32 (i32.const 0x60))
  (global $BUILTIN_EVERY i32 (i32.const 0x64))

  ;; String method names (0x80 - 0xDF)
  (global $BUILTIN_CHAR_AT i32 (i32.const 0x80))
  (global $BUILTIN_CHAR_CODE_AT i32 (i32.const 0x84))
  (global $BUILTIN_STRING_INDEX_OF i32 (i32.const 0x88))
  (global $BUILTIN_STRING_INCLUDES i32 (i32.const 0x8C))
  (global $BUILTIN_STRING_SLICE i32 (i32.const 0x90))
  (global $BUILTIN_SUBSTRING i32 (i32.const 0x94))
  (global $BUILTIN_SPLIT i32 (i32.const 0x98))
  (global $BUILTIN_TRIM i32 (i32.const 0x9C))
  (global $BUILTIN_TO_LOWER_CASE i32 (i32.const 0xA0))
  (global $BUILTIN_TO_UPPER_CASE i32 (i32.const 0xA4))
  (global $BUILTIN_STARTS_WITH i32 (i32.const 0xA8))
  (global $BUILTIN_ENDS_WITH i32 (i32.const 0xAC))
  (global $BUILTIN_REPEAT i32 (i32.const 0xB0))
  (global $BUILTIN_PAD_START i32 (i32.const 0xB4))
  (global $BUILTIN_PAD_END i32 (i32.const 0xB8))
  (global $BUILTIN_REPLACE i32 (i32.const 0xBC))

  ;; Literal strings for type conversion (0xC0 - 0xDF)
  (global $BUILTIN_LIT_NULL i32 (i32.const 0xC0))              ;; "null"
  (global $BUILTIN_LIT_TRUE i32 (i32.const 0xC4))              ;; "true"
  (global $BUILTIN_LIT_FALSE i32 (i32.const 0xC8))             ;; "false"
  (global $BUILTIN_LIT_OBJECT_OBJECT i32 (i32.const 0xCC))     ;; "[object Object]"
  (global $BUILTIN_LIT_NAN i32 (i32.const 0xD0))               ;; "NaN"
  (global $BUILTIN_LIT_INFINITY i32 (i32.const 0xD4))          ;; "Infinity"
  (global $BUILTIN_LIT_NEG_INFINITY i32 (i32.const 0xD8))      ;; "-Infinity"
  (global $BUILTIN_LIT_OBJECT_FUNCTION i32 (i32.const 0xDC))   ;; "[object Function]"
  (global $BUILTIN_EMPTY_STRING i32 (i32.const 0xE0))          ;; ""

  ;; Method IDs for TYPE_BOUND_METHOD dispatch
  ;; Array methods (0x01 - 0x12)
  (global $METHOD_PUSH i32 (i32.const 0x01))
  (global $METHOD_POP i32 (i32.const 0x02))
  (global $METHOD_SHIFT i32 (i32.const 0x03))
  (global $METHOD_UNSHIFT i32 (i32.const 0x04))
  (global $METHOD_SLICE i32 (i32.const 0x05))
  (global $METHOD_CONCAT i32 (i32.const 0x06))
  (global $METHOD_JOIN i32 (i32.const 0x07))
  (global $METHOD_REVERSE i32 (i32.const 0x08))
  (global $METHOD_INDEX_OF i32 (i32.const 0x09))
  (global $METHOD_INCLUDES i32 (i32.const 0x0A))
  (global $METHOD_MAP i32 (i32.const 0x0B))
  (global $METHOD_FILTER i32 (i32.const 0x0C))
  (global $METHOD_REDUCE i32 (i32.const 0x0D))
  (global $METHOD_FOR_EACH i32 (i32.const 0x0E))
  (global $METHOD_FIND i32 (i32.const 0x0F))
  (global $METHOD_FIND_INDEX i32 (i32.const 0x10))
  (global $METHOD_SOME i32 (i32.const 0x11))
  (global $METHOD_EVERY i32 (i32.const 0x12))

  ;; String methods (0x20 - 0x2F)
  (global $METHOD_CHAR_AT i32 (i32.const 0x20))
  (global $METHOD_CHAR_CODE_AT i32 (i32.const 0x21))
  (global $METHOD_STRING_INDEX_OF i32 (i32.const 0x22))
  (global $METHOD_STRING_INCLUDES i32 (i32.const 0x23))
  (global $METHOD_STRING_SLICE i32 (i32.const 0x24))
  (global $METHOD_SUBSTRING i32 (i32.const 0x25))
  (global $METHOD_SPLIT i32 (i32.const 0x26))
  (global $METHOD_TRIM i32 (i32.const 0x27))
  (global $METHOD_TO_LOWER_CASE i32 (i32.const 0x28))
  (global $METHOD_TO_UPPER_CASE i32 (i32.const 0x29))
  (global $METHOD_STARTS_WITH i32 (i32.const 0x2A))
  (global $METHOD_ENDS_WITH i32 (i32.const 0x2B))
  (global $METHOD_REPEAT i32 (i32.const 0x2C))
  (global $METHOD_PAD_START i32 (i32.const 0x2D))
  (global $METHOD_PAD_END i32 (i32.const 0x2E))
  (global $METHOD_REPLACE i32 (i32.const 0x2F))

  ;; Math static methods (0x40 - 0x61)
  (global $METHOD_MATH_ABS i32 (i32.const 0x40))
  (global $METHOD_MATH_FLOOR i32 (i32.const 0x41))
  (global $METHOD_MATH_CEIL i32 (i32.const 0x42))
  (global $METHOD_MATH_ROUND i32 (i32.const 0x43))
  (global $METHOD_MATH_TRUNC i32 (i32.const 0x44))
  (global $METHOD_MATH_SIGN i32 (i32.const 0x45))
  (global $METHOD_MATH_MIN i32 (i32.const 0x46))
  (global $METHOD_MATH_MAX i32 (i32.const 0x47))
  (global $METHOD_MATH_POW i32 (i32.const 0x48))
  (global $METHOD_MATH_SQRT i32 (i32.const 0x49))
  (global $METHOD_MATH_CBRT i32 (i32.const 0x4A))
  (global $METHOD_MATH_HYPOT i32 (i32.const 0x4B))
  (global $METHOD_MATH_SIN i32 (i32.const 0x4C))
  (global $METHOD_MATH_COS i32 (i32.const 0x4D))
  (global $METHOD_MATH_TAN i32 (i32.const 0x4E))
  (global $METHOD_MATH_ASIN i32 (i32.const 0x4F))
  (global $METHOD_MATH_ACOS i32 (i32.const 0x50))
  (global $METHOD_MATH_ATAN i32 (i32.const 0x51))
  (global $METHOD_MATH_ATAN2 i32 (i32.const 0x52))
  (global $METHOD_MATH_SINH i32 (i32.const 0x53))
  (global $METHOD_MATH_COSH i32 (i32.const 0x54))
  (global $METHOD_MATH_TANH i32 (i32.const 0x55))
  (global $METHOD_MATH_ASINH i32 (i32.const 0x56))
  (global $METHOD_MATH_ACOSH i32 (i32.const 0x57))
  (global $METHOD_MATH_ATANH i32 (i32.const 0x58))
  (global $METHOD_MATH_LOG i32 (i32.const 0x59))
  (global $METHOD_MATH_LOG10 i32 (i32.const 0x5A))
  (global $METHOD_MATH_LOG2 i32 (i32.const 0x5B))
  (global $METHOD_MATH_LOG1P i32 (i32.const 0x5C))
  (global $METHOD_MATH_EXP i32 (i32.const 0x5D))
  (global $METHOD_MATH_EXPM1 i32 (i32.const 0x5E))
  (global $METHOD_MATH_CLZ32 i32 (i32.const 0x5F))
  (global $METHOD_MATH_IMUL i32 (i32.const 0x60))
  (global $METHOD_MATH_FROUND i32 (i32.const 0x61))

  ;; Constructor method IDs (0xC0 - 0xCF)
  (global $METHOD_ARRAY_CONSTRUCTOR i32 (i32.const 0xC0))
  (global $METHOD_OBJECT_CONSTRUCTOR i32 (i32.const 0xC1))

  ;; Global conversion function IDs (0xB0 - 0xBF)
  (global $METHOD_TO_STRING i32 (i32.const 0xB0))
  (global $METHOD_TO_NUMBER i32 (i32.const 0xB1))
  (global $METHOD_TO_BOOLEAN i32 (i32.const 0xB2))
  (global $METHOD_PARSE_INT i32 (i32.const 0xB3))
  (global $METHOD_PARSE_FLOAT i32 (i32.const 0xB4))
  (global $METHOD_IS_NAN i32 (i32.const 0xB5))
  (global $METHOD_IS_FINITE i32 (i32.const 0xB6))

  ;; Number static method IDs (0xA0 - 0xAF)
  (global $METHOD_NUMBER_IS_NAN i32 (i32.const 0xA0))
  (global $METHOD_NUMBER_IS_FINITE i32 (i32.const 0xA1))
  (global $METHOD_NUMBER_IS_INTEGER i32 (i32.const 0xA2))
  (global $METHOD_NUMBER_PARSE_INT i32 (i32.const 0xA3))
  (global $METHOD_NUMBER_PARSE_FLOAT i32 (i32.const 0xA4))

  ;; String static method IDs
  (global $METHOD_STRING_FROM_CHAR_CODE i32 (i32.const 0x90))
  (global $METHOD_STRING_FROM_CODE_POINT i32 (i32.const 0x91))

  ;; Object static method IDs
  (global $METHOD_OBJECT_KEYS i32 (i32.const 0x70))
  (global $METHOD_OBJECT_VALUES i32 (i32.const 0x71))
  (global $METHOD_OBJECT_ENTRIES i32 (i32.const 0x72))
  (global $METHOD_OBJECT_ASSIGN i32 (i32.const 0x73))

  ;; Array static method IDs
  (global $METHOD_ARRAY_IS_ARRAY i32 (i32.const 0x80))
  (global $METHOD_ARRAY_FROM i32 (i32.const 0x81))

  ;; Closure flags
  (global $CLOSURE_FLAG_ARROW i32 (i32.const 1))

  ;; Status codes
  (global $STATUS_RUNNING i32 (i32.const 0))
  (global $STATUS_DONE i32 (i32.const 1))
  (global $STATUS_PAUSED_FUEL i32 (i32.const 2))
  (global $STATUS_FFI_REQUEST i32 (i32.const 3))
  (global $STATUS_ERROR i32 (i32.const 4))
  (global $STATUS_CALLBACK_DONE i32 (i32.const 5))
  (global $STATUS_THROW i32 (i32.const 6))

  ;; Error codes
  (global $ERR_NONE i32 (i32.const 0))
  (global $ERR_UNDEFINED_VARIABLE i32 (i32.const 1))
  (global $ERR_ASSIGN_UNDEFINED i32 (i32.const 2))
  (global $ERR_NOT_CALLABLE i32 (i32.const 3))
  (global $ERR_NOT_ITERABLE i32 (i32.const 4))
  (global $ERR_PROPERTY_NULL i32 (i32.const 5))
  (global $ERR_INVALID_OPERAND i32 (i32.const 6))
  (global $ERR_STACK_OVERFLOW i32 (i32.const 7))
  (global $ERR_TYPE_ERROR i32 (i32.const 8))
  (global $ERR_ARITY i32 (i32.const 9))
  (global $ERR_USER_THROW i32 (i32.const 12))
  (global $ERR_MSGPACK_READONLY i32 (i32.const 13))
  (global $ERR_MSGPACK_INVALID i32 (i32.const 14))
  (global $ERR_NOT_SUPPORTED i32 (i32.const 15))  ;; NotSupportedError: new String(), etc.
  (global $ERR_REDECLARATION i32 (i32.const 16))  ;; Variable already declared in scope

  ;; Completion types
  (global $COMPLETION_NORMAL i32 (i32.const 0))
  (global $COMPLETION_THROW i32 (i32.const 1))
  (global $COMPLETION_RETURN i32 (i32.const 2))

  ;; Type tags (16-byte value format: [type:i32][flags:i32][data:i64])
  (global $TYPE_NULL i32 (i32.const 0x00))
  (global $TYPE_UNDEFINED i32 (i32.const 0x01))
  (global $TYPE_BOOLEAN i32 (i32.const 0x02))
  (global $TYPE_INTEGER i32 (i32.const 0x03))
  (global $TYPE_FLOAT i32 (i32.const 0x04))
  (global $TYPE_STRING i32 (i32.const 0x05))
  (global $TYPE_ARRAY i32 (i32.const 0x06))
  (global $TYPE_OBJECT i32 (i32.const 0x07))
  (global $TYPE_CLOSURE i32 (i32.const 0x08))
  (global $TYPE_SCOPE i32 (i32.const 0x09))
  (global $TYPE_BOUND_METHOD i32 (i32.const 0x0e))
  (global $TYPE_CONSTRUCTOR i32 (i32.const 0x0f))  ;; callable object (Array, Object, String, Number)
  (global $TYPE_FFI_REF i32 (i32.const 0x0a))      ;; opaque FFI handle
  (global $TYPE_FFI_METHOD i32 (i32.const 0x0c))   ;; FFI method (refId + method name offset)
  (global $TYPE_MSGPACK_REF i32 (i32.const 0x0d))  ;; read-only msgpack reference

  ;; GC object types
  (global $OBJ_ARRAY i32 (i32.const 0))
  (global $OBJ_OBJECT i32 (i32.const 1))
  (global $OBJ_SCOPE i32 (i32.const 2))
  (global $OBJ_CLOSURE i32 (i32.const 3))
  (global $OBJ_ARRAY_DATA i32 (i32.const 5))
  (global $OBJ_CODE_BLOCK i32 (i32.const 6))
  (global $OBJ_OBJECT_DATA i32 (i32.const 7))

  ;; Sizes
  (global $VALUE_SIZE i32 (i32.const 16))
  (global $INSTRUCTION_SIZE i32 (i32.const 16))
  (global $GC_HEADER_SIZE i32 (i32.const 8))
  (global $FRAME_SIZE i32 (i32.const 56))

  ;; Frame field offsets
  (global $FRAME_CODE_BLOCK i32 (i32.const 0x00))
  (global $FRAME_INSTRUCTION_INDEX i32 (i32.const 0x04))
  (global $FRAME_SCOPE_POINTER i32 (i32.const 0x08))
  (global $FRAME_PENDING_BASE i32 (i32.const 0x0C))
  (global $FRAME_PENDING_COUNT i32 (i32.const 0x10))
  (global $FRAME_FLAGS i32 (i32.const 0x14))
  (global $FRAME_SOURCE_START i32 (i32.const 0x18))
  (global $FRAME_SOURCE_END i32 (i32.const 0x1C))
  ;; Iteration fields (for callback methods like map, filter, reduce)
  (global $FRAME_ITER_INDEX i32 (i32.const 0x20))
  (global $FRAME_ITER_LENGTH i32 (i32.const 0x24))
  (global $FRAME_ITER_RESULT i32 (i32.const 0x28))
  (global $FRAME_ITER_METHOD i32 (i32.const 0x2C))
  (global $FRAME_ITER_RECEIVER i32 (i32.const 0x30))
  (global $FRAME_ITER_CALLBACK i32 (i32.const 0x34))

  ;; Frame flags
  (global $FRAME_FLAG_CALLBACK i32 (i32.const 0x01))
  (global $FRAME_FLAG_TAIL i32 (i32.const 0x02))
  (global $FRAME_FLAG_ITERATING i32 (i32.const 0x04))

  ;; Opcodes
  (global $OP_LIT_INT i32 (i32.const 0x01))
  (global $OP_LIT_FLOAT i32 (i32.const 0x02))
  (global $OP_LIT_STRING i32 (i32.const 0x03))
  (global $OP_LIT_NULL i32 (i32.const 0x04))
  (global $OP_LIT_UNDEFINED i32 (i32.const 0x05))
  (global $OP_LIT_TRUE i32 (i32.const 0x06))
  (global $OP_LIT_FALSE i32 (i32.const 0x07))

  (global $OP_GET_VAR i32 (i32.const 0x10))
  (global $OP_SET_VAR i32 (i32.const 0x11))
  (global $OP_LET_VAR i32 (i32.const 0x12))

  (global $OP_ADD i32 (i32.const 0x20))
  (global $OP_SUB i32 (i32.const 0x21))
  (global $OP_MUL i32 (i32.const 0x22))
  (global $OP_DIV i32 (i32.const 0x23))
  (global $OP_MOD i32 (i32.const 0x24))
  (global $OP_NEG i32 (i32.const 0x25))
  (global $OP_POW i32 (i32.const 0x26))
  (global $OP_BAND i32 (i32.const 0x27))
  (global $OP_BOR i32 (i32.const 0x28))
  (global $OP_BXOR i32 (i32.const 0x29))
  (global $OP_BNOT i32 (i32.const 0x2A))
  (global $OP_SHL i32 (i32.const 0x2B))
  (global $OP_SHR i32 (i32.const 0x2C))
  (global $OP_USHR i32 (i32.const 0x2D))

  (global $OP_EQ i32 (i32.const 0x30))
  (global $OP_NEQ i32 (i32.const 0x31))
  (global $OP_LT i32 (i32.const 0x32))
  (global $OP_GT i32 (i32.const 0x33))
  (global $OP_LTE i32 (i32.const 0x34))
  (global $OP_GTE i32 (i32.const 0x35))

  (global $OP_NOT i32 (i32.const 0x40))
  (global $OP_AND i32 (i32.const 0x41))
  (global $OP_OR i32 (i32.const 0x42))
  (global $OP_NULLISH i32 (i32.const 0x43))
  (global $OP_TYPEOF i32 (i32.const 0x91))

  (global $OP_JUMP i32 (i32.const 0x50))
  (global $OP_JUMP_IF_FALSE i32 (i32.const 0x51))
  (global $OP_JUMP_IF_TRUE i32 (i32.const 0x52))

  (global $OP_SCOPE_PUSH i32 (i32.const 0x60))
  (global $OP_SCOPE_POP i32 (i32.const 0x61))

  (global $OP_MAKE_CLOSURE i32 (i32.const 0x70))
  (global $OP_CALL i32 (i32.const 0x71))
  (global $OP_RETURN i32 (i32.const 0x72))
  (global $OP_RETURN_UNDEFINED i32 (i32.const 0x73))
  (global $OP_MAKE_ARROW_CLOSURE i32 (i32.const 0x74))
  (global $OP_CALL_METHOD i32 (i32.const 0x75))
  (global $OP_NEW i32 (i32.const 0x76))           ;; new Constructor(args)

  (global $OP_MAKE_ARRAY i32 (i32.const 0x80))
  (global $OP_MAKE_OBJECT i32 (i32.const 0x81))
  (global $OP_GET_PROP i32 (i32.const 0x82))
  (global $OP_SET_PROP i32 (i32.const 0x83))
  (global $OP_GET_INDEX i32 (i32.const 0x84))
  (global $OP_SET_INDEX i32 (i32.const 0x85))

  (global $OP_THROW i32 (i32.const 0xB0))
  (global $OP_TRY_PUSH i32 (i32.const 0xB1))
  (global $OP_TRY_POP i32 (i32.const 0xB2))
  (global $OP_FINALLY_END i32 (i32.const 0xB3))

  (global $OP_POP i32 (i32.const 0xF0))
  (global $OP_DUP i32 (i32.const 0xF1))
  (global $OP_SWAP i32 (i32.const 0xF2))

  ;; Region base offsets (mutable, read from STATE at init)
  ;; These replace the old hardcoded LAYOUT offsets
  (global $ffi_request_base (mut i32) (i32.const 0))
  (global $error_info_base (mut i32) (i32.const 0))
  (global $call_stack_base (mut i32) (i32.const 0))
  (global $pending_stack_base (mut i32) (i32.const 0))
  (global $try_stack_base (mut i32) (i32.const 0))
  (global $builtins_base (mut i32) (i32.const 0))

  ;; Try entry layout
  (global $TRY_ENTRY_SIZE i32 (i32.const 16))
  (global $TRY_CODE_BLOCK i32 (i32.const 0))
  (global $TRY_CATCH_INDEX i32 (i32.const 4))
  (global $TRY_FINALLY_INDEX i32 (i32.const 8))
  (global $TRY_FRAME_DEPTH i32 (i32.const 12))

  ;; Current scope (mutable, set during execution)
  (global $current_scope (mut i32) (i32.const 0))

  ;; ==========================================================================
  ;; Helper: abs(offset) → absolute address
  ;; ==========================================================================
  (func $abs (param $offset i32) (result i32)
    (i32.add (global.get $base_offset) (local.get $offset))
  )

  ;; ==========================================================================
  ;; Helper: read state field
  ;; ==========================================================================
  (func $read_state (param $field i32) (result i32)
    (i32.load (call $abs (local.get $field)))
  )

  (func $write_state (param $field i32) (param $value i32)
    (i32.store (call $abs (local.get $field)) (local.get $value))
  )

  ;; Helper: read from BUILTINS segment
  ;; ==========================================================================
  (func $read_builtin (param $offset i32) (result i32)
    (i32.load (call $abs (i32.add (global.get $builtins_base) (local.get $offset))))
  )

  ;; ==========================================================================
  ;; f64_to_string: Convert a float to an interned string
  ;; Handles: NaN, Infinity, -Infinity, integers, and decimal floats
  ;; ==========================================================================
  (func $f64_to_string (param $val f64) (result i32)
    (local $int_part i64)
    (local $frac_part f64)
    (local $is_negative i32)
    (local $temp_ptr i32)
    (local $len i32)
    (local $digit i32)
    (local $abs_val f64)
    (local $i i32)
    (local $frac_digits i32)
    (local $temp i64)
    (local $byte_ptr i32)

    ;; Check for NaN
    (if (f64.ne (local.get $val) (local.get $val))
      (then
        (return (call $read_builtin (global.get $BUILTIN_LIT_NAN)))
      )
    )

    ;; Check for positive infinity
    (if (f64.eq (local.get $val) (f64.const inf))
      (then
        (return (call $read_builtin (global.get $BUILTIN_LIT_INFINITY)))
      )
    )

    ;; Check for negative infinity
    (if (f64.eq (local.get $val) (f64.const -inf))
      (then
        (return (call $read_builtin (global.get $BUILTIN_LIT_NEG_INFINITY)))
      )
    )

    ;; Use a scratch area in memory for building the string
    ;; Use the last 64 bytes of the pending stack area as scratch
    (local.set $temp_ptr (call $abs (i32.sub (global.get $pending_stack_base) (i32.const 64))))
    (local.set $byte_ptr (i32.add (local.get $temp_ptr) (i32.const 4))) ;; skip length prefix

    ;; Handle negative
    (local.set $is_negative (i32.const 0))
    (if (f64.lt (local.get $val) (f64.const 0))
      (then
        (local.set $is_negative (i32.const 1))
        (local.set $val (f64.neg (local.get $val)))
      )
    )
    (local.set $abs_val (local.get $val))

    ;; Check if it's an integer (no fractional part)
    (local.set $int_part (i64.trunc_f64_s (local.get $val)))
    (local.set $frac_part (f64.sub (local.get $val) (f64.convert_i64_s (local.get $int_part))))

    ;; If negative, write '-'
    (if (local.get $is_negative)
      (then
        (i32.store8 (local.get $byte_ptr) (i32.const 45)) ;; '-'
        (local.set $byte_ptr (i32.add (local.get $byte_ptr) (i32.const 1)))
        (local.set $len (i32.const 1))
      )
      (else
        (local.set $len (i32.const 0))
      )
    )

    ;; Handle zero specially
    (if (i64.eqz (local.get $int_part))
      (then
        (i32.store8 (local.get $byte_ptr) (i32.const 48)) ;; '0'
        (local.set $byte_ptr (i32.add (local.get $byte_ptr) (i32.const 1)))
        (local.set $len (i32.add (local.get $len) (i32.const 1)))
      )
      (else
        ;; Convert integer part - write digits in reverse, then reverse
        (local.set $i (local.get $len))
        (local.set $temp (local.get $int_part))
        (block $int_done
          (loop $int_loop
            (br_if $int_done (i64.eqz (local.get $temp)))
            (local.set $digit (i32.wrap_i64 (i64.rem_u (local.get $temp) (i64.const 10))))
            (i32.store8 (local.get $byte_ptr) (i32.add (i32.const 48) (local.get $digit)))
            (local.set $byte_ptr (i32.add (local.get $byte_ptr) (i32.const 1)))
            (local.set $len (i32.add (local.get $len) (i32.const 1)))
            (local.set $temp (i64.div_u (local.get $temp) (i64.const 10)))
            (br $int_loop)
          )
        )
        ;; Reverse the integer digits (from position $i to $len-1)
        (call $reverse_bytes
          (i32.add (i32.add (local.get $temp_ptr) (i32.const 4)) (local.get $i))
          (i32.sub (local.get $len) (local.get $i)))
        (local.set $byte_ptr (i32.add (i32.add (local.get $temp_ptr) (i32.const 4)) (local.get $len)))
      )
    )

    ;; Add fractional part if present (only if abs(frac) > 1e-10)
    (if (f64.gt (f64.abs (local.get $frac_part)) (f64.const 1e-10))
      (then
        ;; Add decimal point
        (i32.store8 (local.get $byte_ptr) (i32.const 46)) ;; '.'
        (local.set $byte_ptr (i32.add (local.get $byte_ptr) (i32.const 1)))
        (local.set $len (i32.add (local.get $len) (i32.const 1)))

        ;; Add fractional digits (up to 10, trimming trailing zeros)
        (local.set $frac_digits (i32.const 0))
        (block $frac_done
          (loop $frac_loop
            (br_if $frac_done (i32.ge_u (local.get $frac_digits) (i32.const 10)))
            (local.set $frac_part (f64.mul (local.get $frac_part) (f64.const 10)))
            (local.set $digit (i32.trunc_f64_s (local.get $frac_part)))
            (local.set $frac_part (f64.sub (local.get $frac_part) (f64.convert_i32_s (local.get $digit))))
            (i32.store8 (local.get $byte_ptr) (i32.add (i32.const 48) (local.get $digit)))
            (local.set $byte_ptr (i32.add (local.get $byte_ptr) (i32.const 1)))
            (local.set $len (i32.add (local.get $len) (i32.const 1)))
            (local.set $frac_digits (i32.add (local.get $frac_digits) (i32.const 1)))
            ;; Stop if remaining fraction is negligible
            (br_if $frac_done (f64.lt (f64.abs (local.get $frac_part)) (f64.const 1e-10)))
            (br $frac_loop)
          )
        )

        ;; Trim trailing zeros (but keep at least one digit after decimal)
        (block $trim_done
          (loop $trim_loop
            ;; Stop if we're at the decimal point + 1 digit
            (br_if $trim_done (i32.le_u (local.get $len) (i32.add (local.get $i) (i32.const 2))))
            ;; Check last byte
            (local.set $byte_ptr (i32.sub (local.get $byte_ptr) (i32.const 1)))
            (br_if $trim_done (i32.ne (i32.load8_u (local.get $byte_ptr)) (i32.const 48)))
            (local.set $len (i32.sub (local.get $len) (i32.const 1)))
            (br $trim_loop)
          )
        )
      )
    )

    ;; Store length at temp_ptr
    (i32.store (local.get $temp_ptr) (local.get $len))

    ;; Intern the string
    (call $intern_string (i32.add (local.get $temp_ptr) (i32.const 4)) (local.get $len))
  )

  ;; Helper: reverse bytes in place
  (func $reverse_bytes (param $ptr i32) (param $len i32)
    (local $left i32)
    (local $right i32)
    (local $tmp i32)

    (local.set $left (local.get $ptr))
    (local.set $right (i32.sub (i32.add (local.get $ptr) (local.get $len)) (i32.const 1)))

    (block $done
      (loop $loop
        (br_if $done (i32.ge_u (local.get $left) (local.get $right)))
        (local.set $tmp (i32.load8_u (local.get $left)))
        (i32.store8 (local.get $left) (i32.load8_u (local.get $right)))
        (i32.store8 (local.get $right) (local.get $tmp))
        (local.set $left (i32.add (local.get $left) (i32.const 1)))
        (local.set $right (i32.sub (local.get $right) (i32.const 1)))
        (br $loop)
      )
    )
  )

  ;; ==========================================================================
  ;; UTF-8 helpers
  ;; ==========================================================================

  ;; Count UTF-8 characters in a string
  ;; str_ptr: string table offset (layout: [byte_len:4][bytes:N])
  ;; Returns: number of Unicode characters
  (func $utf8_char_count (param $str_ptr i32) (result i32)
    (local $byte_len i32)
    (local $i i32)
    (local $count i32)
    (local $byte i32)

    (local.set $byte_len (i32.load (call $abs (local.get $str_ptr))))
    (local.set $i (i32.const 0))
    (local.set $count (i32.const 0))

    (block $done
      (loop $loop
        (br_if $done (i32.ge_u (local.get $i) (local.get $byte_len)))

        (local.set $byte (i32.load8_u (call $abs (i32.add
          (i32.add (local.get $str_ptr) (i32.const 4))
          (local.get $i)))))

        ;; Count this byte as a character start if:
        ;; - It's ASCII (0x00-0x7F), or
        ;; - It's a UTF-8 leading byte (0xC0-0xFF)
        ;; Skip continuation bytes (0x80-0xBF)
        (if (i32.or
              (i32.lt_u (local.get $byte) (i32.const 0x80))
              (i32.ge_u (local.get $byte) (i32.const 0xC0)))
          (then
            (local.set $count (i32.add (local.get $count) (i32.const 1)))
          )
        )

        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $loop)
      )
    )

    (local.get $count)
  )

  ;; Get byte offset for character index in UTF-8 string
  ;; str_ptr: string table offset
  ;; char_idx: character index (0-based)
  ;; Returns: byte offset from start of string data, or -1 if out of bounds
  (func $utf8_char_to_byte_offset (param $str_ptr i32) (param $char_idx i32) (result i32)
    (local $byte_len i32)
    (local $i i32)
    (local $char_count i32)
    (local $byte i32)

    (local.set $byte_len (i32.load (call $abs (local.get $str_ptr))))
    (local.set $i (i32.const 0))
    (local.set $char_count (i32.const 0))

    (block $done
      (loop $loop
        (br_if $done (i32.ge_u (local.get $i) (local.get $byte_len)))

        (local.set $byte (i32.load8_u (call $abs (i32.add
          (i32.add (local.get $str_ptr) (i32.const 4))
          (local.get $i)))))

        ;; Check if this is a character start (ASCII or UTF-8 leading byte)
        (if (i32.or
              (i32.lt_u (local.get $byte) (i32.const 0x80))
              (i32.ge_u (local.get $byte) (i32.const 0xC0)))
          (then
            ;; If we've found the target character, return this byte offset
            (if (i32.eq (local.get $char_count) (local.get $char_idx))
              (then (return (local.get $i)))
            )
            ;; Otherwise increment character count
            (local.set $char_count (i32.add (local.get $char_count) (i32.const 1)))
          )
        )

        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $loop)
      )
    )

    ;; Check if char_idx equals total count (pointing to end)
    (if (i32.eq (local.get $char_count) (local.get $char_idx))
      (then (return (local.get $byte_len)))
    )

    ;; Out of bounds
    (i32.const -1)
  )

  ;; Decode UTF-8 code point at byte offset
  ;; str_ptr: string table offset
  ;; byte_offset: offset into string data
  ;; Returns: Unicode code point
  (func $utf8_decode_at (param $str_ptr i32) (param $byte_offset i32) (result i32)
    (local $byte_len i32)
    (local $b0 i32)
    (local $b1 i32)
    (local $b2 i32)
    (local $b3 i32)
    (local $data_ptr i32)

    (local.set $byte_len (i32.load (call $abs (local.get $str_ptr))))
    (local.set $data_ptr (i32.add (local.get $str_ptr) (i32.const 4)))

    ;; Out of bounds check
    (if (i32.ge_u (local.get $byte_offset) (local.get $byte_len))
      (then (return (i32.const -1)))
    )

    (local.set $b0 (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $byte_offset)))))

    ;; ASCII (0x00-0x7F): 1 byte
    (if (i32.lt_u (local.get $b0) (i32.const 0x80))
      (then (return (local.get $b0)))
    )

    ;; 2-byte sequence (110xxxxx 10xxxxxx): 0xC0-0xDF
    (if (i32.and
          (i32.ge_u (local.get $b0) (i32.const 0xC0))
          (i32.lt_u (local.get $b0) (i32.const 0xE0)))
      (then
        (if (i32.ge_u (i32.add (local.get $byte_offset) (i32.const 1)) (local.get $byte_len))
          (then (return (i32.const -1)))
        )
        (local.set $b1 (i32.load8_u (call $abs (i32.add (local.get $data_ptr)
          (i32.add (local.get $byte_offset) (i32.const 1))))))
        (return (i32.or
          (i32.shl (i32.and (local.get $b0) (i32.const 0x1F)) (i32.const 6))
          (i32.and (local.get $b1) (i32.const 0x3F))))
      )
    )

    ;; 3-byte sequence (1110xxxx 10xxxxxx 10xxxxxx): 0xE0-0xEF
    (if (i32.and
          (i32.ge_u (local.get $b0) (i32.const 0xE0))
          (i32.lt_u (local.get $b0) (i32.const 0xF0)))
      (then
        (if (i32.ge_u (i32.add (local.get $byte_offset) (i32.const 2)) (local.get $byte_len))
          (then (return (i32.const -1)))
        )
        (local.set $b1 (i32.load8_u (call $abs (i32.add (local.get $data_ptr)
          (i32.add (local.get $byte_offset) (i32.const 1))))))
        (local.set $b2 (i32.load8_u (call $abs (i32.add (local.get $data_ptr)
          (i32.add (local.get $byte_offset) (i32.const 2))))))
        (return (i32.or
          (i32.or
            (i32.shl (i32.and (local.get $b0) (i32.const 0x0F)) (i32.const 12))
            (i32.shl (i32.and (local.get $b1) (i32.const 0x3F)) (i32.const 6)))
          (i32.and (local.get $b2) (i32.const 0x3F))))
      )
    )

    ;; 4-byte sequence (11110xxx 10xxxxxx 10xxxxxx 10xxxxxx): 0xF0-0xF7
    (if (i32.and
          (i32.ge_u (local.get $b0) (i32.const 0xF0))
          (i32.lt_u (local.get $b0) (i32.const 0xF8)))
      (then
        (if (i32.ge_u (i32.add (local.get $byte_offset) (i32.const 3)) (local.get $byte_len))
          (then (return (i32.const -1)))
        )
        (local.set $b1 (i32.load8_u (call $abs (i32.add (local.get $data_ptr)
          (i32.add (local.get $byte_offset) (i32.const 1))))))
        (local.set $b2 (i32.load8_u (call $abs (i32.add (local.get $data_ptr)
          (i32.add (local.get $byte_offset) (i32.const 2))))))
        (local.set $b3 (i32.load8_u (call $abs (i32.add (local.get $data_ptr)
          (i32.add (local.get $byte_offset) (i32.const 3))))))
        (return (i32.or
          (i32.or
            (i32.or
              (i32.shl (i32.and (local.get $b0) (i32.const 0x07)) (i32.const 18))
              (i32.shl (i32.and (local.get $b1) (i32.const 0x3F)) (i32.const 12)))
            (i32.shl (i32.and (local.get $b2) (i32.const 0x3F)) (i32.const 6)))
          (i32.and (local.get $b3) (i32.const 0x3F))))
      )
    )

    ;; Invalid UTF-8, return the byte as-is
    (local.get $b0)
  )

  ;; Get byte length of UTF-8 character at byte offset
  ;; Returns: 1, 2, 3, or 4
  (func $utf8_char_byte_len (param $str_ptr i32) (param $byte_offset i32) (result i32)
    (local $b0 i32)

    (local.set $b0 (i32.load8_u (call $abs (i32.add
      (i32.add (local.get $str_ptr) (i32.const 4))
      (local.get $byte_offset)))))

    ;; ASCII
    (if (i32.lt_u (local.get $b0) (i32.const 0x80))
      (then (return (i32.const 1)))
    )
    ;; 2-byte
    (if (i32.lt_u (local.get $b0) (i32.const 0xE0))
      (then (return (i32.const 2)))
    )
    ;; 3-byte
    (if (i32.lt_u (local.get $b0) (i32.const 0xF0))
      (then (return (i32.const 3)))
    )
    ;; 4-byte
    (i32.const 4)
  )

  ;; Encode a Unicode code point to UTF-8 at dest_ptr
  ;; Returns: number of bytes written (1-4)
  (func $utf8_encode_at (param $dest_ptr i32) (param $code_point i32) (result i32)
    ;; ASCII (0x00-0x7F): 1 byte
    (if (i32.lt_u (local.get $code_point) (i32.const 0x80))
      (then
        (i32.store8 (call $abs (local.get $dest_ptr)) (local.get $code_point))
        (return (i32.const 1))
      )
    )
    ;; 2-byte (0x80-0x7FF): 110xxxxx 10xxxxxx
    (if (i32.lt_u (local.get $code_point) (i32.const 0x800))
      (then
        (i32.store8 (call $abs (local.get $dest_ptr))
          (i32.or (i32.const 0xC0) (i32.shr_u (local.get $code_point) (i32.const 6))))
        (i32.store8 (call $abs (i32.add (local.get $dest_ptr) (i32.const 1)))
          (i32.or (i32.const 0x80) (i32.and (local.get $code_point) (i32.const 0x3F))))
        (return (i32.const 2))
      )
    )
    ;; 3-byte (0x800-0xFFFF): 1110xxxx 10xxxxxx 10xxxxxx
    (if (i32.lt_u (local.get $code_point) (i32.const 0x10000))
      (then
        (i32.store8 (call $abs (local.get $dest_ptr))
          (i32.or (i32.const 0xE0) (i32.shr_u (local.get $code_point) (i32.const 12))))
        (i32.store8 (call $abs (i32.add (local.get $dest_ptr) (i32.const 1)))
          (i32.or (i32.const 0x80) (i32.and (i32.shr_u (local.get $code_point) (i32.const 6)) (i32.const 0x3F))))
        (i32.store8 (call $abs (i32.add (local.get $dest_ptr) (i32.const 2)))
          (i32.or (i32.const 0x80) (i32.and (local.get $code_point) (i32.const 0x3F))))
        (return (i32.const 3))
      )
    )
    ;; 4-byte (0x10000-0x10FFFF): 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
    (i32.store8 (call $abs (local.get $dest_ptr))
      (i32.or (i32.const 0xF0) (i32.shr_u (local.get $code_point) (i32.const 18))))
    (i32.store8 (call $abs (i32.add (local.get $dest_ptr) (i32.const 1)))
      (i32.or (i32.const 0x80) (i32.and (i32.shr_u (local.get $code_point) (i32.const 12)) (i32.const 0x3F))))
    (i32.store8 (call $abs (i32.add (local.get $dest_ptr) (i32.const 2)))
      (i32.or (i32.const 0x80) (i32.and (i32.shr_u (local.get $code_point) (i32.const 6)) (i32.const 0x3F))))
    (i32.store8 (call $abs (i32.add (local.get $dest_ptr) (i32.const 3)))
      (i32.or (i32.const 0x80) (i32.and (local.get $code_point) (i32.const 0x3F))))
    (i32.const 4)
  )

  ;; Convert code point to uppercase
  ;; Returns: transformed code point
  (func $to_upper_code_point (param $cp i32) (result i32)
    ;; ASCII lowercase a-z (0x61-0x7A) -> A-Z (0x41-0x5A)
    (if (i32.and
          (i32.ge_u (local.get $cp) (i32.const 0x61))
          (i32.le_u (local.get $cp) (i32.const 0x7A)))
      (then (return (i32.sub (local.get $cp) (i32.const 32))))
    )
    ;; Latin-1 lowercase: à-ö (0xE0-0xF6) -> À-Ö (0xC0-0xD6)
    (if (i32.and
          (i32.ge_u (local.get $cp) (i32.const 0xE0))
          (i32.le_u (local.get $cp) (i32.const 0xF6)))
      (then (return (i32.sub (local.get $cp) (i32.const 32))))
    )
    ;; Latin-1 lowercase: ø-þ (0xF8-0xFE) -> Ø-Þ (0xD8-0xDE)
    (if (i32.and
          (i32.ge_u (local.get $cp) (i32.const 0xF8))
          (i32.le_u (local.get $cp) (i32.const 0xFE)))
      (then (return (i32.sub (local.get $cp) (i32.const 32))))
    )
    ;; No transformation
    (local.get $cp)
  )

  ;; Convert code point to lowercase
  ;; Returns: transformed code point
  (func $to_lower_code_point (param $cp i32) (result i32)
    ;; ASCII uppercase A-Z (0x41-0x5A) -> a-z (0x61-0x7A)
    (if (i32.and
          (i32.ge_u (local.get $cp) (i32.const 0x41))
          (i32.le_u (local.get $cp) (i32.const 0x5A)))
      (then (return (i32.add (local.get $cp) (i32.const 32))))
    )
    ;; Latin-1 uppercase: À-Ö (0xC0-0xD6) -> à-ö (0xE0-0xF6)
    (if (i32.and
          (i32.ge_u (local.get $cp) (i32.const 0xC0))
          (i32.le_u (local.get $cp) (i32.const 0xD6)))
      (then (return (i32.add (local.get $cp) (i32.const 32))))
    )
    ;; Latin-1 uppercase: Ø-Þ (0xD8-0xDE) -> ø-þ (0xF8-0xFE)
    (if (i32.and
          (i32.ge_u (local.get $cp) (i32.const 0xD8))
          (i32.le_u (local.get $cp) (i32.const 0xDE)))
      (then (return (i32.add (local.get $cp) (i32.const 32))))
    )
    ;; No transformation
    (local.get $cp)
  )

  ;; ==========================================================================
  ;; parseInt helper: parse integer from string with radix
  ;; str_ptr: string table offset
  ;; radix: 2-36 (0 means auto-detect)
  ;; Returns: parsed integer as f64, or NaN on failure
  ;; ==========================================================================
  (func $parse_int (param $str_ptr i32) (param $radix i32) (result f64)
    (local $len i32) (local $data_ptr i32) (local $i i32) (local $byte i32)
    (local $sign f64) (local $result f64) (local $digit i32) (local $has_digits i32)

    (local.set $len (i32.load (call $abs (local.get $str_ptr))))
    (local.set $data_ptr (i32.add (local.get $str_ptr) (i32.const 4)))
    (local.set $i (i32.const 0))
    (local.set $sign (f64.const 1))
    (local.set $result (f64.const 0))
    (local.set $has_digits (i32.const 0))

    ;; Skip leading whitespace
    (block $skip_ws_done
      (loop $skip_ws
        (br_if $skip_ws_done (i32.ge_u (local.get $i) (local.get $len)))
        (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))
        ;; Space, tab, newline, carriage return
        (br_if $skip_ws_done (i32.and
          (i32.ne (local.get $byte) (i32.const 0x20))
          (i32.and
            (i32.ne (local.get $byte) (i32.const 0x09))
            (i32.and
              (i32.ne (local.get $byte) (i32.const 0x0A))
              (i32.ne (local.get $byte) (i32.const 0x0D))))))
        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $skip_ws)
      )
    )

    ;; Check for sign
    (if (i32.lt_u (local.get $i) (local.get $len))
      (then
        (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))
        (if (i32.eq (local.get $byte) (i32.const 0x2D)) ;; '-'
          (then
            (local.set $sign (f64.const -1))
            (local.set $i (i32.add (local.get $i) (i32.const 1)))
          )
          (else
            (if (i32.eq (local.get $byte) (i32.const 0x2B)) ;; '+'
              (then (local.set $i (i32.add (local.get $i) (i32.const 1))))
            )
          )
        )
      )
    )

    ;; Auto-detect radix if 0
    (if (i32.eqz (local.get $radix))
      (then
        (local.set $radix (i32.const 10))
        (if (i32.lt_u (local.get $i) (local.get $len))
          (then
            (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))
            (if (i32.eq (local.get $byte) (i32.const 0x30)) ;; '0'
              (then
                (if (i32.lt_u (i32.add (local.get $i) (i32.const 1)) (local.get $len))
                  (then
                    (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (i32.add (local.get $i) (i32.const 1))))))
                    (if (i32.or (i32.eq (local.get $byte) (i32.const 0x78)) (i32.eq (local.get $byte) (i32.const 0x58))) ;; 'x' or 'X'
                      (then
                        (local.set $radix (i32.const 16))
                        (local.set $i (i32.add (local.get $i) (i32.const 2)))
                      )
                    )
                  )
                )
              )
            )
          )
        )
      )
      (else
        ;; Skip 0x prefix for explicit radix 16
        (if (i32.eq (local.get $radix) (i32.const 16))
          (then
            (if (i32.lt_u (i32.add (local.get $i) (i32.const 1)) (local.get $len))
              (then
                (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))
                (if (i32.eq (local.get $byte) (i32.const 0x30)) ;; '0'
                  (then
                    (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (i32.add (local.get $i) (i32.const 1))))))
                    (if (i32.or (i32.eq (local.get $byte) (i32.const 0x78)) (i32.eq (local.get $byte) (i32.const 0x58)))
                      (then (local.set $i (i32.add (local.get $i) (i32.const 2))))
                    )
                  )
                )
              )
            )
          )
        )
      )
    )

    ;; Validate radix
    (if (i32.or (i32.lt_s (local.get $radix) (i32.const 2)) (i32.gt_s (local.get $radix) (i32.const 36)))
      (then (return (f64.div (f64.const 0) (f64.const 0)))) ;; NaN
    )

    ;; Parse digits
    (block $parse_done
      (loop $parse
        (br_if $parse_done (i32.ge_u (local.get $i) (local.get $len)))
        (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))

        ;; Convert to digit value
        (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0x30)) (i32.le_u (local.get $byte) (i32.const 0x39)))
          (then (local.set $digit (i32.sub (local.get $byte) (i32.const 0x30)))) ;; 0-9
          (else
            (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0x41)) (i32.le_u (local.get $byte) (i32.const 0x5A)))
              (then (local.set $digit (i32.add (i32.sub (local.get $byte) (i32.const 0x41)) (i32.const 10)))) ;; A-Z
              (else
                (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0x61)) (i32.le_u (local.get $byte) (i32.const 0x7A)))
                  (then (local.set $digit (i32.add (i32.sub (local.get $byte) (i32.const 0x61)) (i32.const 10)))) ;; a-z
                  (else (br $parse_done)) ;; Not a digit, stop parsing
                )
              )
            )
          )
        )

        ;; Check digit is valid for radix
        (br_if $parse_done (i32.ge_s (local.get $digit) (local.get $radix)))

        ;; Accumulate: result = result * radix + digit
        (local.set $result (f64.add
          (f64.mul (local.get $result) (f64.convert_i32_s (local.get $radix)))
          (f64.convert_i32_s (local.get $digit))))
        (local.set $has_digits (i32.const 1))

        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $parse)
      )
    )

    ;; If no digits parsed, return NaN
    (if (i32.eqz (local.get $has_digits))
      (then (return (f64.div (f64.const 0) (f64.const 0))))
    )

    ;; Apply sign and return
    (f64.mul (local.get $sign) (local.get $result))
  )

  ;; ==========================================================================
  ;; parseFloat helper: parse float from string
  ;; str_ptr: string table offset
  ;; Returns: parsed float, or NaN on failure
  ;; ==========================================================================
  (func $parse_float (param $str_ptr i32) (result f64)
    (local $len i32) (local $data_ptr i32) (local $i i32) (local $byte i32)
    (local $sign f64) (local $int_part f64) (local $frac_part f64) (local $frac_div f64)
    (local $exp_sign i32) (local $exp_val i32) (local $has_digits i32) (local $has_dot i32)

    (local.set $len (i32.load (call $abs (local.get $str_ptr))))
    (local.set $data_ptr (i32.add (local.get $str_ptr) (i32.const 4)))
    (local.set $i (i32.const 0))
    (local.set $sign (f64.const 1))
    (local.set $int_part (f64.const 0))
    (local.set $frac_part (f64.const 0))
    (local.set $frac_div (f64.const 1))
    (local.set $exp_sign (i32.const 1))
    (local.set $exp_val (i32.const 0))
    (local.set $has_digits (i32.const 0))
    (local.set $has_dot (i32.const 0))

    ;; Skip leading whitespace
    (block $skip_ws_done
      (loop $skip_ws
        (br_if $skip_ws_done (i32.ge_u (local.get $i) (local.get $len)))
        (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))
        (br_if $skip_ws_done (i32.and
          (i32.ne (local.get $byte) (i32.const 0x20))
          (i32.and
            (i32.ne (local.get $byte) (i32.const 0x09))
            (i32.and
              (i32.ne (local.get $byte) (i32.const 0x0A))
              (i32.ne (local.get $byte) (i32.const 0x0D))))))
        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $skip_ws)
      )
    )

    ;; Check for sign
    (if (i32.lt_u (local.get $i) (local.get $len))
      (then
        (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))
        (if (i32.eq (local.get $byte) (i32.const 0x2D))
          (then
            (local.set $sign (f64.const -1))
            (local.set $i (i32.add (local.get $i) (i32.const 1)))
          )
          (else
            (if (i32.eq (local.get $byte) (i32.const 0x2B))
              (then (local.set $i (i32.add (local.get $i) (i32.const 1))))
            )
          )
        )
      )
    )

    ;; Check for Infinity
    (if (i32.ge_u (i32.sub (local.get $len) (local.get $i)) (i32.const 8))
      (then
        (if (i32.and
              (i32.eq (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))) (i32.const 0x49)) ;; I
              (i32.and
                (i32.eq (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (i32.add (local.get $i) (i32.const 1))))) (i32.const 0x6E)) ;; n
                (i32.and
                  (i32.eq (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (i32.add (local.get $i) (i32.const 2))))) (i32.const 0x66)) ;; f
                  (i32.and
                    (i32.eq (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (i32.add (local.get $i) (i32.const 3))))) (i32.const 0x69)) ;; i
                    (i32.and
                      (i32.eq (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (i32.add (local.get $i) (i32.const 4))))) (i32.const 0x6E)) ;; n
                      (i32.and
                        (i32.eq (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (i32.add (local.get $i) (i32.const 5))))) (i32.const 0x69)) ;; i
                        (i32.and
                          (i32.eq (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (i32.add (local.get $i) (i32.const 6))))) (i32.const 0x74)) ;; t
                          (i32.eq (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (i32.add (local.get $i) (i32.const 7))))) (i32.const 0x79))))))))) ;; y
          (then
            (return (f64.mul (local.get $sign) (f64.div (f64.const 1) (f64.const 0))))
          )
        )
      )
    )

    ;; Parse integer part
    (block $int_done
      (loop $int_loop
        (br_if $int_done (i32.ge_u (local.get $i) (local.get $len)))
        (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))

        ;; Check for decimal point
        (if (i32.eq (local.get $byte) (i32.const 0x2E)) ;; '.'
          (then
            (local.set $has_dot (i32.const 1))
            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $int_done)
          )
        )

        ;; Check for exponent
        (if (i32.or (i32.eq (local.get $byte) (i32.const 0x65)) (i32.eq (local.get $byte) (i32.const 0x45))) ;; 'e' or 'E'
          (then (br $int_done))
        )

        ;; Check for digit 0-9
        (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0x30)) (i32.le_u (local.get $byte) (i32.const 0x39)))
          (then
            (local.set $int_part (f64.add
              (f64.mul (local.get $int_part) (f64.const 10))
              (f64.convert_i32_s (i32.sub (local.get $byte) (i32.const 0x30)))))
            (local.set $has_digits (i32.const 1))
            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $int_loop)
          )
          (else (br $int_done))
        )
      )
    )

    ;; Parse fractional part
    (if (local.get $has_dot)
      (then
        (block $frac_done
          (loop $frac_loop
            (br_if $frac_done (i32.ge_u (local.get $i) (local.get $len)))
            (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))

            ;; Check for exponent
            (if (i32.or (i32.eq (local.get $byte) (i32.const 0x65)) (i32.eq (local.get $byte) (i32.const 0x45)))
              (then (br $frac_done))
            )

            ;; Check for digit
            (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0x30)) (i32.le_u (local.get $byte) (i32.const 0x39)))
              (then
                (local.set $frac_div (f64.mul (local.get $frac_div) (f64.const 10)))
                (local.set $frac_part (f64.add (local.get $frac_part)
                  (f64.div (f64.convert_i32_s (i32.sub (local.get $byte) (i32.const 0x30))) (local.get $frac_div))))
                (local.set $has_digits (i32.const 1))
                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $frac_loop)
              )
              (else (br $frac_done))
            )
          )
        )
      )
    )

    ;; Parse exponent
    (if (i32.lt_u (local.get $i) (local.get $len))
      (then
        (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))
        (if (i32.or (i32.eq (local.get $byte) (i32.const 0x65)) (i32.eq (local.get $byte) (i32.const 0x45)))
          (then
            (local.set $i (i32.add (local.get $i) (i32.const 1)))

            ;; Check exponent sign
            (if (i32.lt_u (local.get $i) (local.get $len))
              (then
                (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))
                (if (i32.eq (local.get $byte) (i32.const 0x2D))
                  (then
                    (local.set $exp_sign (i32.const -1))
                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                  )
                  (else
                    (if (i32.eq (local.get $byte) (i32.const 0x2B))
                      (then (local.set $i (i32.add (local.get $i) (i32.const 1))))
                    )
                  )
                )
              )
            )

            ;; Parse exponent digits
            (block $exp_done
              (loop $exp_loop
                (br_if $exp_done (i32.ge_u (local.get $i) (local.get $len)))
                (local.set $byte (i32.load8_u (call $abs (i32.add (local.get $data_ptr) (local.get $i)))))
                (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0x30)) (i32.le_u (local.get $byte) (i32.const 0x39)))
                  (then
                    (local.set $exp_val (i32.add
                      (i32.mul (local.get $exp_val) (i32.const 10))
                      (i32.sub (local.get $byte) (i32.const 0x30))))
                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                    (br $exp_loop)
                  )
                  (else (br $exp_done))
                )
              )
            )
          )
        )
      )
    )

    ;; If no digits parsed, return NaN
    (if (i32.eqz (local.get $has_digits))
      (then (return (f64.div (f64.const 0) (f64.const 0))))
    )

    ;; Compute final result: sign * (int + frac) * 10^exp
    (f64.mul
      (local.get $sign)
      (f64.mul
        (f64.add (local.get $int_part) (local.get $frac_part))
        (call $f64_pow (f64.const 10) (f64.convert_i32_s (i32.mul (local.get $exp_sign) (local.get $exp_val))))))
  )

  ;; ==========================================================================
  ;; MsgPack parsing functions
  ;; ==========================================================================

  ;; MsgPack type codes (returned by $msgpack_peek_type)
  (global $MSGPACK_NIL i32 (i32.const 0))
  (global $MSGPACK_BOOL i32 (i32.const 1))
  (global $MSGPACK_INT i32 (i32.const 2))
  (global $MSGPACK_FLOAT i32 (i32.const 3))
  (global $MSGPACK_STRING i32 (i32.const 4))
  (global $MSGPACK_ARRAY i32 (i32.const 5))
  (global $MSGPACK_MAP i32 (i32.const 6))
  (global $MSGPACK_ERROR i32 (i32.const -1))

  ;; Read u16 big-endian from absolute address
  (func $read_u16_be (param $addr i32) (result i32)
    (i32.or
      (i32.shl (i32.load8_u (local.get $addr)) (i32.const 8))
      (i32.load8_u (i32.add (local.get $addr) (i32.const 1))))
  )

  ;; Read u32 big-endian from absolute address
  (func $read_u32_be (param $addr i32) (result i32)
    (i32.or
      (i32.or
        (i32.shl (i32.load8_u (local.get $addr)) (i32.const 24))
        (i32.shl (i32.load8_u (i32.add (local.get $addr) (i32.const 1))) (i32.const 16)))
      (i32.or
        (i32.shl (i32.load8_u (i32.add (local.get $addr) (i32.const 2))) (i32.const 8))
        (i32.load8_u (i32.add (local.get $addr) (i32.const 3)))))
  )

  ;; Read i16 big-endian (sign-extended) from absolute address
  (func $read_i16_be (param $addr i32) (result i32)
    (i32.extend16_s
      (i32.or
        (i32.shl (i32.load8_u (local.get $addr)) (i32.const 8))
        (i32.load8_u (i32.add (local.get $addr) (i32.const 1)))))
  )

  ;; Read i32 big-endian from absolute address
  (func $read_i32_be (param $addr i32) (result i32)
    (call $read_u32_be (local.get $addr))
  )

  ;; Read f32 big-endian from absolute address
  (func $read_f32_be (param $addr i32) (result f32)
    (f32.reinterpret_i32 (call $read_u32_be (local.get $addr)))
  )

  ;; Read f64 big-endian from absolute address
  (func $read_f64_be (param $addr i32) (result f64)
    (local $hi i32)
    (local $lo i32)
    (local.set $hi (call $read_u32_be (local.get $addr)))
    (local.set $lo (call $read_u32_be (i32.add (local.get $addr) (i32.const 4))))
    (f64.reinterpret_i64
      (i64.or
        (i64.shl (i64.extend_i32_u (local.get $hi)) (i64.const 32))
        (i64.extend_i32_u (local.get $lo))))
  )

  ;; Peek msgpack type at address
  ;; Returns: type (MSGPACK_* constant), -1 on error
  ;; Also stores in scratch: header_size, data_size, count (for arrays/maps)
  ;; Scratch layout: [type:i32][header_size:i32][data_size:i32][count:i32]
  (func $msgpack_peek_type (param $addr i32) (param $end_addr i32) (result i32)
    (local $byte i32)
    (local $type i32)
    (local $header_size i32)
    (local $data_size i32)
    (local $count i32)
    (local $scratch i32)

    ;; Bounds check - need at least 1 byte
    (if (i32.ge_u (local.get $addr) (local.get $end_addr))
      (then (return (global.get $MSGPACK_ERROR))))

    (local.set $byte (i32.load8_u (local.get $addr)))
    (local.set $scratch (call $abs (call $read_state (global.get $STATE_SCRATCH_BASE))))

    ;; Positive fixint (0x00-0x7f): value = byte
    (if (i32.le_u (local.get $byte) (i32.const 0x7f))
      (then
        (local.set $type (global.get $MSGPACK_INT))
        (local.set $header_size (i32.const 1))
        (local.set $data_size (i32.const 0))
        (local.set $count (local.get $byte))  ;; Use count field to store the value
        (i32.store (local.get $scratch) (local.get $type))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (local.get $header_size))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (local.get $data_size))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (local.get $count))
        (return (local.get $type))
      )
    )

    ;; Negative fixint (0xe0-0xff): value = byte as signed
    (if (i32.ge_u (local.get $byte) (i32.const 0xe0))
      (then
        (local.set $type (global.get $MSGPACK_INT))
        (local.set $header_size (i32.const 1))
        (local.set $data_size (i32.const 0))
        ;; Sign-extend from 8 bits: (byte | 0xffffff00) for negative
        (local.set $count (i32.or (local.get $byte) (i32.const 0xffffff00)))
        (i32.store (local.get $scratch) (local.get $type))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (local.get $header_size))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (local.get $data_size))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (local.get $count))
        (return (local.get $type))
      )
    )

    ;; Fixmap (0x80-0x8f): count = byte & 0x0f
    (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0x80))
                 (i32.le_u (local.get $byte) (i32.const 0x8f)))
      (then
        (local.set $type (global.get $MSGPACK_MAP))
        (local.set $header_size (i32.const 1))
        (local.set $count (i32.and (local.get $byte) (i32.const 0x0f)))
        (i32.store (local.get $scratch) (local.get $type))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (local.get $header_size))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (local.get $count))
        (return (local.get $type))
      )
    )

    ;; Fixarray (0x90-0x9f): count = byte & 0x0f
    (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0x90))
                 (i32.le_u (local.get $byte) (i32.const 0x9f)))
      (then
        (local.set $type (global.get $MSGPACK_ARRAY))
        (local.set $header_size (i32.const 1))
        (local.set $count (i32.and (local.get $byte) (i32.const 0x0f)))
        (i32.store (local.get $scratch) (local.get $type))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (local.get $header_size))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (local.get $count))
        (return (local.get $type))
      )
    )

    ;; Fixstr (0xa0-0xbf): length = byte & 0x1f
    (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0xa0))
                 (i32.le_u (local.get $byte) (i32.const 0xbf)))
      (then
        (local.set $type (global.get $MSGPACK_STRING))
        (local.set $header_size (i32.const 1))
        (local.set $data_size (i32.and (local.get $byte) (i32.const 0x1f)))
        ;; Bounds check
        (if (i32.gt_u (i32.add (local.get $addr) (i32.add (local.get $header_size) (local.get $data_size)))
                      (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (local.get $type))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (local.get $header_size))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (local.get $data_size))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (i32.const 0))
        (return (local.get $type))
      )
    )

    ;; nil (0xc0)
    (if (i32.eq (local.get $byte) (i32.const 0xc0))
      (then
        (i32.store (local.get $scratch) (global.get $MSGPACK_NIL))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 1))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (i32.const 0))
        (return (global.get $MSGPACK_NIL))
      )
    )

    ;; 0xc1 is reserved - error
    (if (i32.eq (local.get $byte) (i32.const 0xc1))
      (then (return (global.get $MSGPACK_ERROR))))

    ;; false (0xc2)
    (if (i32.eq (local.get $byte) (i32.const 0xc2))
      (then
        (i32.store (local.get $scratch) (global.get $MSGPACK_BOOL))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 1))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (i32.const 0))
        (return (global.get $MSGPACK_BOOL))
      )
    )

    ;; true (0xc3)
    (if (i32.eq (local.get $byte) (i32.const 0xc3))
      (then
        (i32.store (local.get $scratch) (global.get $MSGPACK_BOOL))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 1))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (i32.const 1))  ;; 1 = true
        (return (global.get $MSGPACK_BOOL))
      )
    )

    ;; bin8, bin16, bin32 (0xc4-0xc6) - unsupported
    (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0xc4))
                 (i32.le_u (local.get $byte) (i32.const 0xc6)))
      (then (return (global.get $MSGPACK_ERROR))))

    ;; ext8, ext16, ext32 (0xc7-0xc9) - unsupported
    (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0xc7))
                 (i32.le_u (local.get $byte) (i32.const 0xc9)))
      (then (return (global.get $MSGPACK_ERROR))))

    ;; float32 (0xca)
    (if (i32.eq (local.get $byte) (i32.const 0xca))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 5)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_FLOAT))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 1))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 4))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (i32.const 0))
        (return (global.get $MSGPACK_FLOAT))
      )
    )

    ;; float64 (0xcb)
    (if (i32.eq (local.get $byte) (i32.const 0xcb))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 9)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_FLOAT))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 1))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 8))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (i32.const 0))
        (return (global.get $MSGPACK_FLOAT))
      )
    )

    ;; uint8 (0xcc)
    (if (i32.eq (local.get $byte) (i32.const 0xcc))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 2)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_INT))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 2))  ;; header + 1 byte
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12))
          (i32.load8_u (i32.add (local.get $addr) (i32.const 1))))
        (return (global.get $MSGPACK_INT))
      )
    )

    ;; uint16 (0xcd)
    (if (i32.eq (local.get $byte) (i32.const 0xcd))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 3)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_INT))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 3))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12))
          (call $read_u16_be (i32.add (local.get $addr) (i32.const 1))))
        (return (global.get $MSGPACK_INT))
      )
    )

    ;; uint32 (0xce)
    (if (i32.eq (local.get $byte) (i32.const 0xce))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 5)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_INT))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 5))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12))
          (call $read_u32_be (i32.add (local.get $addr) (i32.const 1))))
        (return (global.get $MSGPACK_INT))
      )
    )

    ;; uint64 (0xcf) - treat as float since JS numbers are f64
    (if (i32.eq (local.get $byte) (i32.const 0xcf))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 9)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        ;; uint64 needs special handling - we'll read it as float
        (i32.store (local.get $scratch) (global.get $MSGPACK_INT))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 9))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 8))  ;; Mark as 8-byte int
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (i32.const 0))
        (return (global.get $MSGPACK_INT))
      )
    )

    ;; int8 (0xd0)
    (if (i32.eq (local.get $byte) (i32.const 0xd0))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 2)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_INT))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 2))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12))
          (i32.extend8_s (i32.load8_u (i32.add (local.get $addr) (i32.const 1)))))
        (return (global.get $MSGPACK_INT))
      )
    )

    ;; int16 (0xd1)
    (if (i32.eq (local.get $byte) (i32.const 0xd1))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 3)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_INT))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 3))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12))
          (call $read_i16_be (i32.add (local.get $addr) (i32.const 1))))
        (return (global.get $MSGPACK_INT))
      )
    )

    ;; int32 (0xd2)
    (if (i32.eq (local.get $byte) (i32.const 0xd2))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 5)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_INT))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 5))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12))
          (call $read_i32_be (i32.add (local.get $addr) (i32.const 1))))
        (return (global.get $MSGPACK_INT))
      )
    )

    ;; int64 (0xd3) - treat as float since JS numbers are f64
    (if (i32.eq (local.get $byte) (i32.const 0xd3))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 9)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        ;; int64 needs special handling - we'll read it as float
        (i32.store (local.get $scratch) (global.get $MSGPACK_INT))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 9))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const -8))  ;; Mark as signed 8-byte int
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (i32.const 0))
        (return (global.get $MSGPACK_INT))
      )
    )

    ;; fixext1-fixext16 (0xd4-0xd8) - unsupported
    (if (i32.and (i32.ge_u (local.get $byte) (i32.const 0xd4))
                 (i32.le_u (local.get $byte) (i32.const 0xd8)))
      (then (return (global.get $MSGPACK_ERROR))))

    ;; str8 (0xd9)
    (if (i32.eq (local.get $byte) (i32.const 0xd9))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 2)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (local.set $data_size (i32.load8_u (i32.add (local.get $addr) (i32.const 1))))
        (if (i32.gt_u (i32.add (local.get $addr) (i32.add (i32.const 2) (local.get $data_size)))
                      (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_STRING))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 2))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (local.get $data_size))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (i32.const 0))
        (return (global.get $MSGPACK_STRING))
      )
    )

    ;; str16 (0xda)
    (if (i32.eq (local.get $byte) (i32.const 0xda))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 3)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (local.set $data_size (call $read_u16_be (i32.add (local.get $addr) (i32.const 1))))
        (if (i32.gt_u (i32.add (local.get $addr) (i32.add (i32.const 3) (local.get $data_size)))
                      (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_STRING))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 3))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (local.get $data_size))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (i32.const 0))
        (return (global.get $MSGPACK_STRING))
      )
    )

    ;; str32 (0xdb)
    (if (i32.eq (local.get $byte) (i32.const 0xdb))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 5)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (local.set $data_size (call $read_u32_be (i32.add (local.get $addr) (i32.const 1))))
        (if (i32.gt_u (i32.add (local.get $addr) (i32.add (i32.const 5) (local.get $data_size)))
                      (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_STRING))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 5))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (local.get $data_size))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (i32.const 0))
        (return (global.get $MSGPACK_STRING))
      )
    )

    ;; array16 (0xdc)
    (if (i32.eq (local.get $byte) (i32.const 0xdc))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 3)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (local.set $count (call $read_u16_be (i32.add (local.get $addr) (i32.const 1))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_ARRAY))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 3))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (local.get $count))
        (return (global.get $MSGPACK_ARRAY))
      )
    )

    ;; array32 (0xdd)
    (if (i32.eq (local.get $byte) (i32.const 0xdd))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 5)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (local.set $count (call $read_u32_be (i32.add (local.get $addr) (i32.const 1))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_ARRAY))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 5))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (local.get $count))
        (return (global.get $MSGPACK_ARRAY))
      )
    )

    ;; map16 (0xde)
    (if (i32.eq (local.get $byte) (i32.const 0xde))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 3)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (local.set $count (call $read_u16_be (i32.add (local.get $addr) (i32.const 1))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_MAP))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 3))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (local.get $count))
        (return (global.get $MSGPACK_MAP))
      )
    )

    ;; map32 (0xdf)
    (if (i32.eq (local.get $byte) (i32.const 0xdf))
      (then
        (if (i32.gt_u (i32.add (local.get $addr) (i32.const 5)) (local.get $end_addr))
          (then (return (global.get $MSGPACK_ERROR))))
        (local.set $count (call $read_u32_be (i32.add (local.get $addr) (i32.const 1))))
        (i32.store (local.get $scratch) (global.get $MSGPACK_MAP))
        (i32.store (i32.add (local.get $scratch) (i32.const 4)) (i32.const 5))
        (i32.store (i32.add (local.get $scratch) (i32.const 8)) (i32.const 0))
        (i32.store (i32.add (local.get $scratch) (i32.const 12)) (local.get $count))
        (return (global.get $MSGPACK_MAP))
      )
    )

    ;; Unknown type byte - error
    (global.get $MSGPACK_ERROR)
  )

  ;; Read msgpack scratch field (after calling $msgpack_peek_type)
  (func $msgpack_scratch_header_size (result i32)
    (i32.load (i32.add (call $abs (call $read_state (global.get $STATE_SCRATCH_BASE))) (i32.const 4)))
  )

  (func $msgpack_scratch_data_size (result i32)
    (i32.load (i32.add (call $abs (call $read_state (global.get $STATE_SCRATCH_BASE))) (i32.const 8)))
  )

  (func $msgpack_scratch_count (result i32)
    (i32.load (i32.add (call $abs (call $read_state (global.get $STATE_SCRATCH_BASE))) (i32.const 12)))
  )

  ;; Skip over a complete msgpack value
  ;; Returns: next address after value, or -1 on error
  (func $msgpack_skip_value (param $addr i32) (param $end_addr i32) (result i32)
    (local $type i32)
    (local $header_size i32)
    (local $data_size i32)
    (local $count i32)
    (local $i i32)
    (local $next i32)

    (local.set $type (call $msgpack_peek_type (local.get $addr) (local.get $end_addr)))
    (if (i32.eq (local.get $type) (global.get $MSGPACK_ERROR))
      (then (return (i32.const -1))))

    (local.set $header_size (call $msgpack_scratch_header_size))
    (local.set $data_size (call $msgpack_scratch_data_size))
    (local.set $count (call $msgpack_scratch_count))

    ;; For simple types (nil, bool, int with value in count, fixint)
    (if (i32.or
          (i32.eq (local.get $type) (global.get $MSGPACK_NIL))
          (i32.eq (local.get $type) (global.get $MSGPACK_BOOL)))
      (then
        (return (i32.add (local.get $addr) (local.get $header_size)))
      )
    )

    ;; For int, check if value was stored in count (fixint, uint8-32, int8-32)
    ;; or if it's a 64-bit value (data_size = 8 or -8)
    (if (i32.eq (local.get $type) (global.get $MSGPACK_INT))
      (then
        (if (i32.or (i32.eq (local.get $data_size) (i32.const 8))
                    (i32.eq (local.get $data_size) (i32.const -8)))
          (then
            ;; 64-bit int: header (1 byte type code) + 8 bytes data
            (return (i32.add (local.get $addr) (i32.const 9)))
          )
          (else
            ;; Value stored in count field, just skip header
            (return (i32.add (local.get $addr) (local.get $header_size)))
          )
        )
      )
    )

    ;; For float, skip header + data
    (if (i32.eq (local.get $type) (global.get $MSGPACK_FLOAT))
      (then
        (return (i32.add (local.get $addr) (i32.add (local.get $header_size) (local.get $data_size))))
      )
    )

    ;; For string, skip header + data
    (if (i32.eq (local.get $type) (global.get $MSGPACK_STRING))
      (then
        (return (i32.add (local.get $addr) (i32.add (local.get $header_size) (local.get $data_size))))
      )
    )

    ;; For array, skip header then skip each element
    (if (i32.eq (local.get $type) (global.get $MSGPACK_ARRAY))
      (then
        (local.set $next (i32.add (local.get $addr) (local.get $header_size)))
        (local.set $i (i32.const 0))
        (block $array_done
          (loop $array_loop
            (br_if $array_done (i32.ge_u (local.get $i) (local.get $count)))
            (local.set $next (call $msgpack_skip_value (local.get $next) (local.get $end_addr)))
            (if (i32.eq (local.get $next) (i32.const -1))
              (then (return (i32.const -1))))
            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $array_loop)
          )
        )
        (return (local.get $next))
      )
    )

    ;; For map, skip header then skip 2*count values (key-value pairs)
    (if (i32.eq (local.get $type) (global.get $MSGPACK_MAP))
      (then
        (local.set $next (i32.add (local.get $addr) (local.get $header_size)))
        (local.set $i (i32.const 0))
        (block $map_done
          (loop $map_loop
            (br_if $map_done (i32.ge_u (local.get $i) (local.get $count)))
            ;; Skip key
            (local.set $next (call $msgpack_skip_value (local.get $next) (local.get $end_addr)))
            (if (i32.eq (local.get $next) (i32.const -1))
              (then (return (i32.const -1))))
            ;; Skip value
            (local.set $next (call $msgpack_skip_value (local.get $next) (local.get $end_addr)))
            (if (i32.eq (local.get $next) (i32.const -1))
              (then (return (i32.const -1))))
            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $map_loop)
          )
        )
        (return (local.get $next))
      )
    )

    ;; Unknown type
    (i32.const -1)
  )

  ;; Intern a msgpack string from an absolute address
  ;; The $intern_string function takes an absolute pointer to the bytes
  ;; Returns: string table offset
  (func $msgpack_intern_string (param $addr i32) (param $len i32) (result i32)
    ;; $intern_string expects absolute pointer to bytes (not length-prefixed)
    (call $intern_string (local.get $addr) (local.get $len))
  )

  ;; Read a msgpack primitive and push to pending stack
  ;; Assumes $msgpack_peek_type was already called
  ;; Returns: 0 on success, -1 on error
  (func $msgpack_read_primitive (param $addr i32) (param $end_addr i32) (result i32)
    (local $type i32)
    (local $header_size i32)
    (local $data_size i32)
    (local $count i32)
    (local $str_offset i32)
    (local $float_val f64)
    (local $byte i32)
    (local $hi i32)
    (local $lo i32)

    (local.set $type (call $msgpack_peek_type (local.get $addr) (local.get $end_addr)))
    (if (i32.eq (local.get $type) (global.get $MSGPACK_ERROR))
      (then (return (i32.const -1))))

    (local.set $header_size (call $msgpack_scratch_header_size))
    (local.set $data_size (call $msgpack_scratch_data_size))
    (local.set $count (call $msgpack_scratch_count))

    ;; nil
    (if (i32.eq (local.get $type) (global.get $MSGPACK_NIL))
      (then
        (call $pending_push (global.get $TYPE_NULL) (i32.const 0) (i32.const 0) (i32.const 0))
        (return (i32.const 0))
      )
    )

    ;; bool
    (if (i32.eq (local.get $type) (global.get $MSGPACK_BOOL))
      (then
        (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (local.get $count) (i32.const 0))
        (return (i32.const 0))
      )
    )

    ;; int (for most ints, value is in count; for 64-bit, need special handling)
    (if (i32.eq (local.get $type) (global.get $MSGPACK_INT))
      (then
        (if (i32.eq (local.get $data_size) (i32.const 8))
          (then
            ;; uint64 - read as f64
            (local.set $hi (call $read_u32_be (i32.add (local.get $addr) (i32.const 1))))
            (local.set $lo (call $read_u32_be (i32.add (local.get $addr) (i32.const 5))))
            (local.set $float_val
              (f64.add
                (f64.mul (f64.convert_i32_u (local.get $hi)) (f64.const 4294967296))
                (f64.convert_i32_u (local.get $lo))))
            (call $push_f64 (local.get $float_val))
            (return (i32.const 0))
          )
        )
        (if (i32.eq (local.get $data_size) (i32.const -8))
          (then
            ;; int64 - read as f64 (signed)
            (local.set $hi (call $read_i32_be (i32.add (local.get $addr) (i32.const 1))))
            (local.set $lo (call $read_u32_be (i32.add (local.get $addr) (i32.const 5))))
            (local.set $float_val
              (f64.add
                (f64.mul (f64.convert_i32_s (local.get $hi)) (f64.const 4294967296))
                (f64.convert_i32_u (local.get $lo))))
            (call $push_f64 (local.get $float_val))
            (return (i32.const 0))
          )
        )
        ;; Regular int - value is in count
        (call $push_f64 (f64.convert_i32_s (local.get $count)))
        (return (i32.const 0))
      )
    )

    ;; float
    (if (i32.eq (local.get $type) (global.get $MSGPACK_FLOAT))
      (then
        (if (i32.eq (local.get $data_size) (i32.const 4))
          (then
            ;; float32
            (call $push_f64 (f64.promote_f32
              (call $read_f32_be (i32.add (local.get $addr) (i32.const 1)))))
          )
          (else
            ;; float64
            (call $push_f64 (call $read_f64_be (i32.add (local.get $addr) (i32.const 1))))
          )
        )
        (return (i32.const 0))
      )
    )

    ;; string - intern and push
    (if (i32.eq (local.get $type) (global.get $MSGPACK_STRING))
      (then
        (local.set $str_offset
          (call $msgpack_intern_string
            (i32.add (local.get $addr) (local.get $header_size))
            (local.get $data_size)))
        (call $pending_push (global.get $TYPE_STRING) (i32.const 0) (local.get $str_offset) (i32.const 0))
        (return (i32.const 0))
      )
    )

    ;; array or map - these are containers, return new TYPE_MSGPACK_REF
    ;; This shouldn't be called for containers, but handle gracefully
    (if (i32.or
          (i32.eq (local.get $type) (global.get $MSGPACK_ARRAY))
          (i32.eq (local.get $type) (global.get $MSGPACK_MAP)))
      (then
        ;; Push a new msgpack ref pointing to this container
        ;; The caller should handle containers specially, but this is a fallback
        ;; We need to calculate the new length from addr to end_addr
        (call $pending_push
          (global.get $TYPE_MSGPACK_REF)
          (i32.const 0)
          (local.get $addr)
          (i32.sub (local.get $end_addr) (local.get $addr)))
        (return (i32.const 0))
      )
    )

    ;; Unknown type
    (i32.const -1)
  )

  ;; Search a msgpack map for a key (string table offset)
  ;; Returns: address of value if found, -1 if not found
  (func $msgpack_map_get (param $addr i32) (param $end_addr i32) (param $key_str_offset i32) (result i32)
    (local $type i32)
    (local $header_size i32)
    (local $count i32)
    (local $i i32)
    (local $pos i32)
    (local $key_type i32)
    (local $key_header_size i32)
    (local $key_data_size i32)
    (local $key_addr i32)
    (local $value_addr i32)
    (local $key_str_len i32)
    (local $mp_str_len i32)
    (local $j i32)
    (local $match i32)
    (local $key_bytes_addr i32)
    (local $mp_bytes_addr i32)

    ;; Peek map type
    (local.set $type (call $msgpack_peek_type (local.get $addr) (local.get $end_addr)))
    (if (i32.ne (local.get $type) (global.get $MSGPACK_MAP))
      (then (return (i32.const -1))))

    (local.set $header_size (call $msgpack_scratch_header_size))
    (local.set $count (call $msgpack_scratch_count))

    ;; Position after map header
    (local.set $pos (i32.add (local.get $addr) (local.get $header_size)))

    ;; Get the key string bytes from string table
    ;; String table format: [len:4][bytes...]
    (local.set $key_bytes_addr (i32.add (call $abs (local.get $key_str_offset)) (i32.const 4)))
    (local.set $key_str_len (i32.load (call $abs (local.get $key_str_offset))))

    ;; Iterate through key-value pairs
    (local.set $i (i32.const 0))
    (block $found
      (block $not_found
        (loop $map_loop
          (br_if $not_found (i32.ge_u (local.get $i) (local.get $count)))
          (br_if $not_found (i32.ge_u (local.get $pos) (local.get $end_addr)))

          ;; Peek key type
          (local.set $key_type (call $msgpack_peek_type (local.get $pos) (local.get $end_addr)))
          (if (i32.eq (local.get $key_type) (global.get $MSGPACK_ERROR))
            (then (return (i32.const -1))))

          (local.set $key_header_size (call $msgpack_scratch_header_size))
          (local.set $key_data_size (call $msgpack_scratch_data_size))

          ;; Check if key is a string
          (if (i32.eq (local.get $key_type) (global.get $MSGPACK_STRING))
            (then
              ;; Compare string bytes
              (local.set $mp_str_len (local.get $key_data_size))
              (local.set $mp_bytes_addr (i32.add (local.get $pos) (local.get $key_header_size)))

              ;; Length check first
              (if (i32.eq (local.get $key_str_len) (local.get $mp_str_len))
                (then
                  ;; Compare bytes
                  (local.set $match (i32.const 1))
                  (local.set $j (i32.const 0))
                  (block $cmp_done
                    (loop $cmp_loop
                      (br_if $cmp_done (i32.ge_u (local.get $j) (local.get $key_str_len)))
                      (if (i32.ne
                            (i32.load8_u (i32.add (local.get $key_bytes_addr) (local.get $j)))
                            (i32.load8_u (i32.add (local.get $mp_bytes_addr) (local.get $j))))
                        (then
                          (local.set $match (i32.const 0))
                          (br $cmp_done)
                        )
                      )
                      (local.set $j (i32.add (local.get $j) (i32.const 1)))
                      (br $cmp_loop)
                    )
                  )

                  (if (local.get $match)
                    (then
                      ;; Found! Return address of value
                      (local.set $value_addr
                        (i32.add (local.get $pos)
                          (i32.add (local.get $key_header_size) (local.get $key_data_size))))
                      (return (local.get $value_addr))
                    )
                  )
                )
              )
            )
          )

          ;; Skip this key
          (local.set $pos (call $msgpack_skip_value (local.get $pos) (local.get $end_addr)))
          (if (i32.eq (local.get $pos) (i32.const -1))
            (then (return (i32.const -1))))

          ;; Skip the value
          (local.set $pos (call $msgpack_skip_value (local.get $pos) (local.get $end_addr)))
          (if (i32.eq (local.get $pos) (i32.const -1))
            (then (return (i32.const -1))))

          (local.set $i (i32.add (local.get $i) (i32.const 1)))
          (br $map_loop)
        )
      )
    )

    ;; Not found
    (i32.const -1)
  )

  ;; Get element at index from msgpack array
  ;; Returns: address of element, or -1 if out of bounds/error
  (func $msgpack_array_get (param $addr i32) (param $end_addr i32) (param $index i32) (result i32)
    (local $type i32)
    (local $header_size i32)
    (local $count i32)
    (local $i i32)
    (local $pos i32)

    ;; Peek array type
    (local.set $type (call $msgpack_peek_type (local.get $addr) (local.get $end_addr)))
    (if (i32.ne (local.get $type) (global.get $MSGPACK_ARRAY))
      (then (return (i32.const -1))))

    (local.set $header_size (call $msgpack_scratch_header_size))
    (local.set $count (call $msgpack_scratch_count))

    ;; Bounds check
    (if (i32.ge_u (local.get $index) (local.get $count))
      (then (return (i32.const -1))))

    ;; Position after array header
    (local.set $pos (i32.add (local.get $addr) (local.get $header_size)))

    ;; Skip to the requested index
    (local.set $i (i32.const 0))
    (block $done
      (loop $skip_loop
        (br_if $done (i32.ge_u (local.get $i) (local.get $index)))
        (local.set $pos (call $msgpack_skip_value (local.get $pos) (local.get $end_addr)))
        (if (i32.eq (local.get $pos) (i32.const -1))
          (then (return (i32.const -1))))
        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $skip_loop)
      )
    )

    ;; pos now points to element at index
    (local.get $pos)
  )

  ;; ==========================================================================
  ;; msgpack_to_array
  ;;
  ;; Materializes a msgpack array to a heap array.
  ;; Returns pointer to new TYPE_ARRAY, or 0 on error (error already set).
  ;; ==========================================================================
  (func $msgpack_to_array (param $addr i32) (param $end_addr i32) (result i32)
    (local $type i32)
    (local $header_size i32)
    (local $count i32)
    (local $arr_ptr i32)
    (local $data_ptr i32)
    (local $data_size i32)
    (local $heap_ptr i32)
    (local $elem_addr i32)
    (local $i i32)
    (local $elem_type i32)
    (local $val_ptr i32)
    (local $elem_len i32)
    (local $dest i32)

    ;; Peek type - must be array
    (local.set $type (call $msgpack_peek_type (local.get $addr) (local.get $end_addr)))
    (if (i32.ne (local.get $type) (global.get $MSGPACK_ARRAY))
      (then
        (call $set_error (global.get $ERR_TYPE_ERROR) (local.get $type))
        (return (i32.const 0))
      )
    )

    (local.set $header_size (call $msgpack_scratch_header_size))
    (local.set $count (call $msgpack_scratch_count))

    ;; Allocate array inline (same layout as OP_MAKE_ARRAY)
    ;; Array header: GC header (8) + length (4) + capacity (4) + data_ptr (4) = 20 bytes
    (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
    (local.set $arr_ptr (local.get $heap_ptr))

    ;; GC header for array
    (i32.store (call $abs (local.get $arr_ptr))
      (i32.or (i32.const 20) (i32.shl (global.get $OBJ_ARRAY) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 4))) (i32.const 0))

    ;; Allocate data block: GC header (8) + count * 16 bytes
    (local.set $data_size (i32.add (global.get $GC_HEADER_SIZE)
                                   (i32.mul (local.get $count) (global.get $VALUE_SIZE))))
    (local.set $data_ptr (i32.add (local.get $arr_ptr) (i32.const 20)))

    ;; GC header for data block
    (i32.store (call $abs (local.get $data_ptr))
      (i32.or (local.get $data_size) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $data_ptr) (i32.const 4))) (i32.const 0))

    ;; Array header: length, capacity, data_ptr (includes GC header, matching OP_MAKE_ARRAY)
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 8))) (local.get $count))  ;; length
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 12))) (local.get $count)) ;; capacity
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 16))) (local.get $data_ptr)) ;; data_ptr (includes GC header)

    ;; Update heap pointer
    (call $write_state (global.get $STATE_HEAP_POINTER)
      (i32.add (local.get $data_ptr) (local.get $data_size)))

    ;; Point to first element in msgpack
    (local.set $elem_addr (i32.add (local.get $addr) (local.get $header_size)))

    ;; Iterate and copy elements
    ;; Elements start at data_ptr + GC_HEADER_SIZE
    (local.set $i (i32.const 0))
    (block $done
      (loop $iter
        (br_if $done (i32.ge_u (local.get $i) (local.get $count)))

        ;; Peek element type
        (local.set $elem_type (call $msgpack_peek_type (local.get $elem_addr) (local.get $end_addr)))
        (if (i32.eq (local.get $elem_type) (global.get $MSGPACK_ERROR))
          (then
            (call $set_error (global.get $ERR_MSGPACK_INVALID) (i32.const 0))
            (return (i32.const 0))
          )
        )

        ;; Destination in heap array: data_ptr + GC_HEADER_SIZE + i * 16
        (local.set $dest (i32.add
          (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
          (i32.mul (local.get $i) (global.get $VALUE_SIZE))))

        ;; Check if primitive or container
        (if (i32.or
              (i32.or
                (i32.eq (local.get $elem_type) (global.get $MSGPACK_NIL))
                (i32.eq (local.get $elem_type) (global.get $MSGPACK_BOOL)))
              (i32.or
                (i32.or
                  (i32.eq (local.get $elem_type) (global.get $MSGPACK_INT))
                  (i32.eq (local.get $elem_type) (global.get $MSGPACK_FLOAT)))
                (i32.eq (local.get $elem_type) (global.get $MSGPACK_STRING))))
          (then
            ;; Primitive - read directly using msgpack_read_primitive
            ;; It pushes to pending stack, so we need to pop and copy to dest
            (drop (call $msgpack_read_primitive (local.get $elem_addr) (local.get $end_addr)))
            ;; Pop from pending stack
            (local.set $val_ptr (call $pending_pop))
            ;; Copy 16 bytes to dest
            (i32.store (call $abs (local.get $dest))
              (i32.load (call $abs (local.get $val_ptr))))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4)))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8)))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 8)))))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12)))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))))
          )
          (else
            ;; Container - write TYPE_MSGPACK_REF
            ;; Calculate element length by skipping to next
            (local.set $val_ptr (call $msgpack_skip_value (local.get $elem_addr) (local.get $end_addr)))
            (local.set $elem_len (i32.sub (local.get $val_ptr) (local.get $elem_addr)))

            ;; Write TYPE_MSGPACK_REF: [type][flags=0][addr][length]
            (i32.store (call $abs (local.get $dest)) (global.get $TYPE_MSGPACK_REF))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4))) (i32.const 0))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8))) (local.get $elem_addr))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12))) (local.get $elem_len))
          )
        )

        ;; Advance to next element
        (local.set $elem_addr (call $msgpack_skip_value (local.get $elem_addr) (local.get $end_addr)))
        (if (i32.eq (local.get $elem_addr) (i32.const -1))
          (then
            (call $set_error (global.get $ERR_MSGPACK_INVALID) (i32.const 0))
            (return (i32.const 0))
          )
        )

        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $iter)
      )
    )

    (local.get $arr_ptr)
  )

  ;; ==========================================================================
  ;; msgpack_map_keys
  ;;
  ;; Returns heap array of interned key strings from a msgpack map.
  ;; Returns 0 on error (not a map, or invalid msgpack).
  ;; ==========================================================================
  (func $msgpack_map_keys (param $addr i32) (param $end_addr i32) (result i32)
    (local $type i32)
    (local $header_size i32)
    (local $count i32)
    (local $arr_ptr i32)
    (local $data_ptr i32)
    (local $data_size i32)
    (local $heap_ptr i32)
    (local $pos i32)
    (local $i i32)
    (local $key_type i32)
    (local $key_header_size i32)
    (local $key_data_size i32)
    (local $key_offset i32)
    (local $dest i32)

    ;; Peek type - must be map
    (local.set $type (call $msgpack_peek_type (local.get $addr) (local.get $end_addr)))
    (if (i32.ne (local.get $type) (global.get $MSGPACK_MAP))
      (then
        (return (i32.const 0))
      )
    )

    (local.set $header_size (call $msgpack_scratch_header_size))
    (local.set $count (call $msgpack_scratch_count))

    ;; Allocate array inline (same layout as OP_MAKE_ARRAY)
    (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
    (local.set $arr_ptr (local.get $heap_ptr))

    ;; GC header for array
    (i32.store (call $abs (local.get $arr_ptr))
      (i32.or (i32.const 20) (i32.shl (global.get $OBJ_ARRAY) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 4))) (i32.const 0))

    ;; Allocate data block
    (local.set $data_size (i32.add (global.get $GC_HEADER_SIZE)
                                   (i32.mul (local.get $count) (global.get $VALUE_SIZE))))
    (local.set $data_ptr (i32.add (local.get $arr_ptr) (i32.const 20)))

    ;; GC header for data block
    (i32.store (call $abs (local.get $data_ptr))
      (i32.or (local.get $data_size) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $data_ptr) (i32.const 4))) (i32.const 0))

    ;; Array header
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 8))) (local.get $count))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 12))) (local.get $count))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 16))) (local.get $data_ptr))

    ;; Update heap pointer
    (call $write_state (global.get $STATE_HEAP_POINTER)
      (i32.add (local.get $data_ptr) (local.get $data_size)))

    ;; Iterate map entries and collect keys
    (local.set $pos (i32.add (local.get $addr) (local.get $header_size)))
    (local.set $i (i32.const 0))
    (block $done
      (loop $iter
        (br_if $done (i32.ge_u (local.get $i) (local.get $count)))

        ;; Peek key type - must be string
        (local.set $key_type (call $msgpack_peek_type (local.get $pos) (local.get $end_addr)))
        (if (i32.ne (local.get $key_type) (global.get $MSGPACK_STRING))
          (then
            ;; Non-string key - error
            (call $set_error (global.get $ERR_TYPE_ERROR) (local.get $key_type))
            (return (i32.const 0))
          )
        )

        (local.set $key_header_size (call $msgpack_scratch_header_size))
        (local.set $key_data_size (call $msgpack_scratch_data_size))

        ;; Intern the key string
        (local.set $key_offset (call $msgpack_intern_string
          (i32.add (local.get $pos) (local.get $key_header_size))
          (local.get $key_data_size)))

        ;; Write TYPE_STRING to array slot
        (local.set $dest (i32.add
          (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
          (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
        (i32.store (call $abs (local.get $dest)) (global.get $TYPE_STRING))
        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4))) (i32.const 0))
        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8))) (local.get $key_offset))
        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12))) (i32.const 0))

        ;; Skip key
        (local.set $pos (call $msgpack_skip_value (local.get $pos) (local.get $end_addr)))
        ;; Skip value
        (local.set $pos (call $msgpack_skip_value (local.get $pos) (local.get $end_addr)))

        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $iter)
      )
    )

    (local.get $arr_ptr)
  )

  ;; ==========================================================================
  ;; msgpack_array_keys
  ;;
  ;; Returns heap array of string indices ["0", "1", "2", ...] for a msgpack array.
  ;; Returns 0 if not an array.
  ;; ==========================================================================
  (func $msgpack_array_keys (param $addr i32) (param $end_addr i32) (result i32)
    (local $type i32)
    (local $header_size i32)
    (local $count i32)
    (local $arr_ptr i32)
    (local $data_ptr i32)
    (local $data_size i32)
    (local $heap_ptr i32)
    (local $i i32)
    (local $key_offset i32)
    (local $dest i32)

    ;; Peek type - must be array
    (local.set $type (call $msgpack_peek_type (local.get $addr) (local.get $end_addr)))
    (if (i32.ne (local.get $type) (global.get $MSGPACK_ARRAY))
      (then
        (return (i32.const 0))
      )
    )

    (local.set $header_size (call $msgpack_scratch_header_size))
    (local.set $count (call $msgpack_scratch_count))

    ;; Allocate array inline (same layout as OP_MAKE_ARRAY)
    (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
    (local.set $arr_ptr (local.get $heap_ptr))

    ;; GC header for array
    (i32.store (call $abs (local.get $arr_ptr))
      (i32.or (i32.const 20) (i32.shl (global.get $OBJ_ARRAY) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 4))) (i32.const 0))

    ;; Allocate data block
    (local.set $data_size (i32.add (global.get $GC_HEADER_SIZE)
                                   (i32.mul (local.get $count) (global.get $VALUE_SIZE))))
    (local.set $data_ptr (i32.add (local.get $arr_ptr) (i32.const 20)))

    ;; GC header for data block
    (i32.store (call $abs (local.get $data_ptr))
      (i32.or (local.get $data_size) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $data_ptr) (i32.const 4))) (i32.const 0))

    ;; Array header
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 8))) (local.get $count))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 12))) (local.get $count))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 16))) (local.get $data_ptr))

    ;; Update heap pointer
    (call $write_state (global.get $STATE_HEAP_POINTER)
      (i32.add (local.get $data_ptr) (local.get $data_size)))

    ;; Generate index strings ["0", "1", "2", ...]
    (local.set $i (i32.const 0))
    (block $done
      (loop $iter
        (br_if $done (i32.ge_u (local.get $i) (local.get $count)))

        ;; Convert index to interned string
        (local.set $key_offset (call $f64_to_string (f64.convert_i32_u (local.get $i))))

        ;; Write TYPE_STRING to array slot
        (local.set $dest (i32.add
          (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
          (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
        (i32.store (call $abs (local.get $dest)) (global.get $TYPE_STRING))
        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4))) (i32.const 0))
        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8))) (local.get $key_offset))
        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12))) (i32.const 0))

        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $iter)
      )
    )

    (local.get $arr_ptr)
  )

  ;; ==========================================================================
  ;; msgpack_map_values
  ;;
  ;; Returns heap array of values from a msgpack map.
  ;; Primitives are converted; containers become TYPE_MSGPACK_REF.
  ;; Returns 0 on error.
  ;; ==========================================================================
  (func $msgpack_map_values (param $addr i32) (param $end_addr i32) (result i32)
    (local $type i32)
    (local $header_size i32)
    (local $count i32)
    (local $arr_ptr i32)
    (local $data_ptr i32)
    (local $data_size i32)
    (local $heap_ptr i32)
    (local $pos i32)
    (local $i i32)
    (local $val_addr i32)
    (local $val_type i32)
    (local $val_ptr i32)
    (local $val_len i32)
    (local $dest i32)

    ;; Peek type - must be map
    (local.set $type (call $msgpack_peek_type (local.get $addr) (local.get $end_addr)))
    (if (i32.ne (local.get $type) (global.get $MSGPACK_MAP))
      (then
        (return (i32.const 0))
      )
    )

    (local.set $header_size (call $msgpack_scratch_header_size))
    (local.set $count (call $msgpack_scratch_count))

    ;; Allocate array
    (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
    (local.set $arr_ptr (local.get $heap_ptr))

    (i32.store (call $abs (local.get $arr_ptr))
      (i32.or (i32.const 20) (i32.shl (global.get $OBJ_ARRAY) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 4))) (i32.const 0))

    (local.set $data_size (i32.add (global.get $GC_HEADER_SIZE)
                                   (i32.mul (local.get $count) (global.get $VALUE_SIZE))))
    (local.set $data_ptr (i32.add (local.get $arr_ptr) (i32.const 20)))

    (i32.store (call $abs (local.get $data_ptr))
      (i32.or (local.get $data_size) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $data_ptr) (i32.const 4))) (i32.const 0))

    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 8))) (local.get $count))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 12))) (local.get $count))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 16))) (local.get $data_ptr))

    (call $write_state (global.get $STATE_HEAP_POINTER)
      (i32.add (local.get $data_ptr) (local.get $data_size)))

    ;; Iterate map entries and collect values
    (local.set $pos (i32.add (local.get $addr) (local.get $header_size)))
    (local.set $i (i32.const 0))
    (block $done
      (loop $iter
        (br_if $done (i32.ge_u (local.get $i) (local.get $count)))

        ;; Skip key
        (local.set $pos (call $msgpack_skip_value (local.get $pos) (local.get $end_addr)))

        ;; pos now points to value
        (local.set $val_addr (local.get $pos))
        (local.set $val_type (call $msgpack_peek_type (local.get $val_addr) (local.get $end_addr)))

        ;; Destination in array
        (local.set $dest (i32.add
          (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
          (i32.mul (local.get $i) (global.get $VALUE_SIZE))))

        ;; Check if primitive or container
        (if (i32.or
              (i32.or
                (i32.eq (local.get $val_type) (global.get $MSGPACK_NIL))
                (i32.eq (local.get $val_type) (global.get $MSGPACK_BOOL)))
              (i32.or
                (i32.or
                  (i32.eq (local.get $val_type) (global.get $MSGPACK_INT))
                  (i32.eq (local.get $val_type) (global.get $MSGPACK_FLOAT)))
                (i32.eq (local.get $val_type) (global.get $MSGPACK_STRING))))
          (then
            ;; Primitive - read and copy from pending stack
            (drop (call $msgpack_read_primitive (local.get $val_addr) (local.get $end_addr)))
            (local.set $val_ptr (call $pending_pop))
            (i32.store (call $abs (local.get $dest))
              (i32.load (call $abs (local.get $val_ptr))))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4)))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8)))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 8)))))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12)))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))))
          )
          (else
            ;; Container - write TYPE_MSGPACK_REF
            (local.set $val_ptr (call $msgpack_skip_value (local.get $val_addr) (local.get $end_addr)))
            (local.set $val_len (i32.sub (local.get $val_ptr) (local.get $val_addr)))

            (i32.store (call $abs (local.get $dest)) (global.get $TYPE_MSGPACK_REF))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4))) (i32.const 0))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8))) (local.get $val_addr))
            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12))) (local.get $val_len))
          )
        )

        ;; Skip value
        (local.set $pos (call $msgpack_skip_value (local.get $pos) (local.get $end_addr)))

        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $iter)
      )
    )

    (local.get $arr_ptr)
  )

  ;; ==========================================================================
  ;; msgpack_map_entries
  ;;
  ;; Returns heap array of [key, value] pairs from a msgpack map.
  ;; Returns 0 on error.
  ;; ==========================================================================
  (func $msgpack_map_entries (param $addr i32) (param $end_addr i32) (result i32)
    (local $type i32)
    (local $header_size i32)
    (local $count i32)
    (local $arr_ptr i32)
    (local $data_ptr i32)
    (local $data_size i32)
    (local $heap_ptr i32)
    (local $pos i32)
    (local $i i32)
    (local $key_type i32)
    (local $key_header_size i32)
    (local $key_data_size i32)
    (local $key_offset i32)
    (local $val_addr i32)
    (local $val_type i32)
    (local $val_ptr i32)
    (local $val_len i32)
    (local $pair_ptr i32)
    (local $pair_data i32)
    (local $pair_size i32)
    (local $dest i32)
    (local $elem_ptr i32)

    ;; Peek type - must be map
    (local.set $type (call $msgpack_peek_type (local.get $addr) (local.get $end_addr)))
    (if (i32.ne (local.get $type) (global.get $MSGPACK_MAP))
      (then
        (return (i32.const 0))
      )
    )

    (local.set $header_size (call $msgpack_scratch_header_size))
    (local.set $count (call $msgpack_scratch_count))

    ;; Allocate outer array
    (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
    (local.set $arr_ptr (local.get $heap_ptr))

    (i32.store (call $abs (local.get $arr_ptr))
      (i32.or (i32.const 20) (i32.shl (global.get $OBJ_ARRAY) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 4))) (i32.const 0))

    (local.set $data_size (i32.add (global.get $GC_HEADER_SIZE)
                                   (i32.mul (local.get $count) (global.get $VALUE_SIZE))))
    (local.set $data_ptr (i32.add (local.get $arr_ptr) (i32.const 20)))

    (i32.store (call $abs (local.get $data_ptr))
      (i32.or (local.get $data_size) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $data_ptr) (i32.const 4))) (i32.const 0))

    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 8))) (local.get $count))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 12))) (local.get $count))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 16))) (local.get $data_ptr))

    (local.set $heap_ptr (i32.add (local.get $data_ptr) (local.get $data_size)))
    (call $write_state (global.get $STATE_HEAP_POINTER) (local.get $heap_ptr))

    ;; Iterate map entries
    (local.set $pos (i32.add (local.get $addr) (local.get $header_size)))
    (local.set $i (i32.const 0))
    (block $done
      (loop $iter
        (br_if $done (i32.ge_u (local.get $i) (local.get $count)))

        ;; Peek key type - must be string
        (local.set $key_type (call $msgpack_peek_type (local.get $pos) (local.get $end_addr)))
        (if (i32.ne (local.get $key_type) (global.get $MSGPACK_STRING))
          (then
            (call $set_error (global.get $ERR_TYPE_ERROR) (local.get $key_type))
            (return (i32.const 0))
          )
        )

        (local.set $key_header_size (call $msgpack_scratch_header_size))
        (local.set $key_data_size (call $msgpack_scratch_data_size))

        ;; Intern the key
        (local.set $key_offset (call $msgpack_intern_string
          (i32.add (local.get $pos) (local.get $key_header_size))
          (local.get $key_data_size)))

        ;; Skip key
        (local.set $pos (call $msgpack_skip_value (local.get $pos) (local.get $end_addr)))

        ;; pos now points to value
        (local.set $val_addr (local.get $pos))
        (local.set $val_type (call $msgpack_peek_type (local.get $val_addr) (local.get $end_addr)))

        ;; Allocate [key, value] pair array (2 elements)
        (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
        (local.set $pair_ptr (local.get $heap_ptr))

        (i32.store (call $abs (local.get $pair_ptr))
          (i32.or (i32.const 20) (i32.shl (global.get $OBJ_ARRAY) (i32.const 24))))
        (i32.store (call $abs (i32.add (local.get $pair_ptr) (i32.const 4))) (i32.const 0))

        (local.set $pair_size (i32.add (global.get $GC_HEADER_SIZE)
                                       (i32.mul (i32.const 2) (global.get $VALUE_SIZE))))
        (local.set $pair_data (i32.add (local.get $pair_ptr) (i32.const 20)))

        (i32.store (call $abs (local.get $pair_data))
          (i32.or (local.get $pair_size) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
        (i32.store (call $abs (i32.add (local.get $pair_data) (i32.const 4))) (i32.const 0))

        (i32.store (call $abs (i32.add (local.get $pair_ptr) (i32.const 8))) (i32.const 2))
        (i32.store (call $abs (i32.add (local.get $pair_ptr) (i32.const 12))) (i32.const 2))
        (i32.store (call $abs (i32.add (local.get $pair_ptr) (i32.const 16))) (local.get $pair_data))

        (call $write_state (global.get $STATE_HEAP_POINTER)
          (i32.add (local.get $pair_data) (local.get $pair_size)))

        ;; Write key at index 0
        (local.set $elem_ptr (i32.add (local.get $pair_data) (global.get $GC_HEADER_SIZE)))
        (i32.store (call $abs (local.get $elem_ptr)) (global.get $TYPE_STRING))
        (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))
        (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $key_offset))
        (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (i32.const 0))

        ;; Write value at index 1
        (local.set $elem_ptr (i32.add (local.get $elem_ptr) (global.get $VALUE_SIZE)))
        (if (i32.or
              (i32.or
                (i32.eq (local.get $val_type) (global.get $MSGPACK_NIL))
                (i32.eq (local.get $val_type) (global.get $MSGPACK_BOOL)))
              (i32.or
                (i32.or
                  (i32.eq (local.get $val_type) (global.get $MSGPACK_INT))
                  (i32.eq (local.get $val_type) (global.get $MSGPACK_FLOAT)))
                (i32.eq (local.get $val_type) (global.get $MSGPACK_STRING))))
          (then
            ;; Primitive
            (drop (call $msgpack_read_primitive (local.get $val_addr) (local.get $end_addr)))
            (local.set $val_ptr (call $pending_pop))
            (i32.store (call $abs (local.get $elem_ptr))
              (i32.load (call $abs (local.get $val_ptr))))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 8)))))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))))
          )
          (else
            ;; Container
            (local.set $val_ptr (call $msgpack_skip_value (local.get $val_addr) (local.get $end_addr)))
            (local.set $val_len (i32.sub (local.get $val_ptr) (local.get $val_addr)))

            (i32.store (call $abs (local.get $elem_ptr)) (global.get $TYPE_MSGPACK_REF))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $val_addr))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (local.get $val_len))
          )
        )

        ;; Store pair reference in outer array
        (local.set $dest (i32.add
          (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
          (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
        (i32.store (call $abs (local.get $dest)) (global.get $TYPE_ARRAY))
        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4))) (i32.const 0))
        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8))) (local.get $pair_ptr))
        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12))) (i32.const 0))

        ;; Skip value
        (local.set $pos (call $msgpack_skip_value (local.get $pos) (local.get $end_addr)))

        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $iter)
      )
    )

    (local.get $arr_ptr)
  )

  ;; ==========================================================================
  ;; msgpack_assign_to_object
  ;;
  ;; Copies all key-value pairs from a msgpack map to a target heap object.
  ;; ==========================================================================
  (func $msgpack_assign_to_object (param $obj_ptr i32) (param $addr i32) (param $end_addr i32)
    (local $type i32)
    (local $header_size i32)
    (local $count i32)
    (local $pos i32)
    (local $i i32)
    (local $key_type i32)
    (local $key_header_size i32)
    (local $key_data_size i32)
    (local $key_offset i32)
    (local $val_addr i32)
    (local $val_type i32)
    (local $val_ptr i32)
    (local $val_len i32)
    (local $val_flags i32)
    (local $val_lo i32)
    (local $val_hi i32)

    ;; Peek type - must be map
    (local.set $type (call $msgpack_peek_type (local.get $addr) (local.get $end_addr)))
    (if (i32.ne (local.get $type) (global.get $MSGPACK_MAP))
      (then (return))
    )

    (local.set $header_size (call $msgpack_scratch_header_size))
    (local.set $count (call $msgpack_scratch_count))

    ;; Iterate map entries
    (local.set $pos (i32.add (local.get $addr) (local.get $header_size)))
    (local.set $i (i32.const 0))
    (block $done
      (loop $iter
        (br_if $done (i32.ge_u (local.get $i) (local.get $count)))

        ;; Peek key type - must be string
        (local.set $key_type (call $msgpack_peek_type (local.get $pos) (local.get $end_addr)))
        (if (i32.ne (local.get $key_type) (global.get $MSGPACK_STRING))
          (then (return))
        )

        (local.set $key_header_size (call $msgpack_scratch_header_size))
        (local.set $key_data_size (call $msgpack_scratch_data_size))

        ;; Intern the key
        (local.set $key_offset (call $msgpack_intern_string
          (i32.add (local.get $pos) (local.get $key_header_size))
          (local.get $key_data_size)))

        ;; Skip key
        (local.set $pos (call $msgpack_skip_value (local.get $pos) (local.get $end_addr)))

        ;; pos now points to value
        (local.set $val_addr (local.get $pos))
        (local.set $val_type (call $msgpack_peek_type (local.get $val_addr) (local.get $end_addr)))

        ;; Read value
        (if (i32.or
              (i32.or
                (i32.eq (local.get $val_type) (global.get $MSGPACK_NIL))
                (i32.eq (local.get $val_type) (global.get $MSGPACK_BOOL)))
              (i32.or
                (i32.or
                  (i32.eq (local.get $val_type) (global.get $MSGPACK_INT))
                  (i32.eq (local.get $val_type) (global.get $MSGPACK_FLOAT)))
                (i32.eq (local.get $val_type) (global.get $MSGPACK_STRING))))
          (then
            ;; Primitive - read from pending stack
            (drop (call $msgpack_read_primitive (local.get $val_addr) (local.get $end_addr)))
            (local.set $val_ptr (call $pending_pop))
            (local.set $val_type (i32.load (call $abs (local.get $val_ptr))))
            (local.set $val_flags (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
            (local.set $val_lo (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 8)))))
            (local.set $val_hi (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))))
          )
          (else
            ;; Container - create TYPE_MSGPACK_REF
            (local.set $val_ptr (call $msgpack_skip_value (local.get $val_addr) (local.get $end_addr)))
            (local.set $val_len (i32.sub (local.get $val_ptr) (local.get $val_addr)))
            (local.set $val_type (global.get $TYPE_MSGPACK_REF))
            (local.set $val_flags (i32.const 0))
            (local.set $val_lo (local.get $val_addr))
            (local.set $val_hi (local.get $val_len))
          )
        )

        ;; Set property on target object
        (call $object_set_property
          (local.get $obj_ptr)
          (local.get $key_offset)
          (local.get $val_type)
          (local.get $val_flags)
          (local.get $val_lo)
          (local.get $val_hi))

        ;; Skip value
        (local.set $pos (call $msgpack_skip_value (local.get $pos) (local.get $end_addr)))

        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $iter)
      )
    )
  )

  ;; ==========================================================================
  ;; Helper: set error and status
  ;; ==========================================================================
  (func $set_error (param $code i32) (param $detail i32)
    (i32.store (call $abs (global.get $error_info_base)) (local.get $code))
    (i32.store (call $abs (i32.add (global.get $error_info_base) (i32.const 4))) (local.get $detail))
    ;; Save scope to linear memory for debugging/migration on error
    (call $write_state (global.get $STATE_SCOPE) (global.get $current_scope))
    (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_ERROR))
  )

  ;; ==========================================================================
  ;; Pending stack operations
  ;; ==========================================================================

  ;; Push a 16-byte value to pending stack
  (func $pending_push (param $type i32) (param $flags i32) (param $data_lo i32) (param $data_hi i32)
    (local $ptr i32)
    (local.set $ptr (call $read_state (global.get $STATE_PENDING_POINTER)))

    ;; Write value: [type:4][flags:4][data_lo:4][data_hi:4]
    (i32.store (call $abs (local.get $ptr)) (local.get $type))
    (i32.store (call $abs (i32.add (local.get $ptr) (i32.const 4))) (local.get $flags))
    (i32.store (call $abs (i32.add (local.get $ptr) (i32.const 8))) (local.get $data_lo))
    (i32.store (call $abs (i32.add (local.get $ptr) (i32.const 12))) (local.get $data_hi))

    ;; Advance pointer
    (call $write_state (global.get $STATE_PENDING_POINTER)
      (i32.add (local.get $ptr) (global.get $VALUE_SIZE)))
  )

  ;; Pop and return pointer to the value (doesn't copy, just returns location)
  (func $pending_pop (result i32)
    (local $ptr i32)
    (local.set $ptr
      (i32.sub
        (call $read_state (global.get $STATE_PENDING_POINTER))
        (global.get $VALUE_SIZE)))
    (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $ptr))
    (local.get $ptr)
  )

  ;; Peek top value (return pointer without popping)
  (func $pending_peek (result i32)
    (i32.sub
      (call $read_state (global.get $STATE_PENDING_POINTER))
      (global.get $VALUE_SIZE))
  )

  ;; Read type tag from value at pointer
  (func $value_type (param $ptr i32) (result i32)
    (i32.load (call $abs (local.get $ptr)))
  )

  ;; Read data_lo from value at pointer
  (func $value_data_lo (param $ptr i32) (result i32)
    (i32.load (call $abs (i32.add (local.get $ptr) (i32.const 8))))
  )

  ;; Read data_hi from value at pointer
  (func $value_data_hi (param $ptr i32) (result i32)
    (i32.load (call $abs (i32.add (local.get $ptr) (i32.const 12))))
  )

  ;; Write a 16-byte value to a given address
  (func $write_value (param $ptr i32) (param $type i32) (param $flags i32) (param $data_lo i32) (param $data_hi i32)
    (i32.store (call $abs (local.get $ptr)) (local.get $type))
    (i32.store (call $abs (i32.add (local.get $ptr) (i32.const 4))) (local.get $flags))
    (i32.store (call $abs (i32.add (local.get $ptr) (i32.const 8))) (local.get $data_lo))
    (i32.store (call $abs (i32.add (local.get $ptr) (i32.const 12))) (local.get $data_hi))
  )

  ;; Read f64 from value at pointer (combines data_lo and data_hi)
  (func $read_f64 (param $ptr i32) (result f64)
    (local $lo i32)
    (local $hi i32)
    (local.set $lo (call $value_data_lo (local.get $ptr)))
    (local.set $hi (call $value_data_hi (local.get $ptr)))
    (f64.reinterpret_i64
      (i64.or
        (i64.extend_i32_u (local.get $lo))
        (i64.shl (i64.extend_i32_u (local.get $hi)) (i64.const 32))))
  )

  ;; Push f64 to pending stack as TYPE_FLOAT
  (func $push_f64 (param $val f64)
    (local $bits i64)
    (local.set $bits (i64.reinterpret_f64 (local.get $val)))
    (call $pending_push
      (global.get $TYPE_FLOAT)
      (i32.const 0)
      (i32.wrap_i64 (local.get $bits))
      (i32.wrap_i64 (i64.shr_u (local.get $bits) (i64.const 32))))
  )

  ;; ==========================================================================
  ;; allocate_array
  ;;
  ;; Allocates a new array with given capacity. Returns pointer to array header.
  ;; Array structure: [GC header 8][len 4][cap 4][data_ptr 4] = 20 bytes
  ;; Data structure: [GC header 8][values...]
  ;; ==========================================================================
  (func $allocate_array (param $capacity i32) (result i32)
    (local $heap_ptr i32)
    (local $arr_ptr i32)
    (local $data_ptr i32)
    (local $data_size i32)

    (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))

    ;; Allocate array header (20 bytes = 8 header + 4 len + 4 cap + 4 data_ptr)
    (local.set $arr_ptr (local.get $heap_ptr))
    ;; Write GC header: size=20, type=OBJ_ARRAY
    (i32.store (call $abs (local.get $arr_ptr))
      (i32.or (i32.const 20) (i32.shl (global.get $OBJ_ARRAY) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 4))) (i32.const 0))
    (local.set $heap_ptr (i32.add (local.get $heap_ptr) (i32.const 20)))

    ;; Allocate data block (8 header + capacity * 16)
    (local.set $data_size (i32.add (i32.const 8) (i32.mul (local.get $capacity) (global.get $VALUE_SIZE))))
    (local.set $data_ptr (local.get $heap_ptr))
    ;; Write GC header for data
    (i32.store (call $abs (local.get $data_ptr))
      (i32.or (local.get $data_size) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $data_ptr) (i32.const 4))) (i32.const 0))
    (local.set $heap_ptr (i32.add (local.get $heap_ptr) (local.get $data_size)))

    ;; Fill in array header fields
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 8))) (local.get $capacity)) ;; length = capacity (for map)
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 12))) (local.get $capacity)) ;; capacity
    (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 16))) (local.get $data_ptr)) ;; data_ptr (includes GC header, matching OP_MAKE_ARRAY)

    ;; Update heap pointer
    (call $write_state (global.get $STATE_HEAP_POINTER) (local.get $heap_ptr))

    (local.get $arr_ptr)
  )

  ;; ==========================================================================
  ;; object_set_property - Set a property on an object (used by Object.assign)
  ;; Handles both updating existing properties and adding new ones
  ;; ==========================================================================
  (func $object_set_property
    (param $obj_ptr i32)
    (param $key_offset i32)
    (param $val_type i32)
    (param $val_flags i32)
    (param $val_lo i32)
    (param $val_hi i32)

    (local $obj_count i32)
    (local $obj_cap i32)
    (local $entries_ptr i32)
    (local $i i32)
    (local $entry_ptr i32)
    (local $prop_key i32)
    (local $found i32)
    (local $heap_ptr i32)
    (local $new_entries_ptr i32)
    (local $new_entries_size i32)
    (local $new_cap i32)
    (local $elem_ptr i32)
    (local $dest i32)

    ;; Object layout: [GC:8][count:4][capacity:4][entries_ptr:4]
    (local.set $obj_count (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))))
    (local.set $obj_cap (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 12)))))
    (local.set $entries_ptr (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 16)))))

    ;; Search for existing property
    (local.set $i (i32.const 0))
    (local.set $found (i32.const 0))
    (block $prop_found
      (loop $prop_loop
        (br_if $prop_found (i32.ge_u (local.get $i) (local.get $obj_count)))

        ;; entry_ptr = entries_ptr + 8 + i * 20
        (local.set $entry_ptr (i32.add
          (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
          (i32.mul (local.get $i) (i32.const 20))))

        (local.set $prop_key (i32.load (call $abs (local.get $entry_ptr))))

        (if (i32.eq (local.get $prop_key) (local.get $key_offset))
          (then
            ;; Found! Update the value in place
            (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))) (local.get $val_type))
            (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 8))) (local.get $val_flags))
            (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 12))) (local.get $val_lo))
            (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 16))) (local.get $val_hi))
            (local.set $found (i32.const 1))
            (br $prop_found)
          )
        )

        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $prop_loop)
      )
    )

    ;; If not found, add new property
    (if (i32.eqz (local.get $found))
      (then
        ;; Check if we need to grow entries block
        (if (i32.ge_u (local.get $obj_count) (local.get $obj_cap))
          (then
            ;; Need to reallocate entries block
            ;; new_cap = capacity * 2 (or at least 4)
            (local.set $new_cap (i32.mul (local.get $obj_cap) (i32.const 2)))
            (if (i32.lt_u (local.get $new_cap) (i32.const 4))
              (then (local.set $new_cap (i32.const 4)))
            )

            ;; Allocate new entries block
            (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
            (local.set $new_entries_ptr (local.get $heap_ptr))
            (local.set $new_entries_size (i32.add (global.get $GC_HEADER_SIZE)
                                                   (i32.mul (local.get $new_cap) (i32.const 20))))

            ;; GC header for new entries block
            (i32.store (call $abs (local.get $new_entries_ptr))
              (i32.or (local.get $new_entries_size) (i32.shl (global.get $OBJ_OBJECT_DATA) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $new_entries_ptr) (i32.const 4))) (i32.const 0))

            ;; Update heap pointer
            (call $write_state (global.get $STATE_HEAP_POINTER)
              (i32.add (local.get $new_entries_ptr) (local.get $new_entries_size)))

            ;; Copy existing entries from old block
            (local.set $i (i32.const 0))
            (block $copy_done
              (loop $copy_loop
                (br_if $copy_done (i32.ge_u (local.get $i) (local.get $obj_count)))
                ;; src = old_entries_ptr + 8 + i * 20
                (local.set $elem_ptr (i32.add
                  (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (i32.const 20))))
                ;; dest = new_entries_ptr + 8 + i * 20
                (local.set $dest (i32.add
                  (i32.add (local.get $new_entries_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (i32.const 20))))
                ;; Copy 20 bytes (entry size)
                (i32.store (call $abs (local.get $dest))
                  (i32.load (call $abs (local.get $elem_ptr))))
                (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))
                (i32.store (call $abs (i32.add (local.get $dest) (i32.const 16)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 16)))))
                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $copy_loop)
              )
            )

            ;; Update object header with new entries_ptr and capacity
            (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 12))) (local.get $new_cap))
            (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 16))) (local.get $new_entries_ptr))
            (local.set $entries_ptr (local.get $new_entries_ptr))
          )
        )

        ;; Add new entry at count position
        ;; entry_ptr = entries_ptr + 8 + count * 20
        (local.set $entry_ptr (i32.add
          (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
          (i32.mul (local.get $obj_count) (i32.const 20))))

        ;; Write entry: [key][type][flags][data_lo][data_hi]
        (i32.store (call $abs (local.get $entry_ptr)) (local.get $key_offset))
        (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))) (local.get $val_type))
        (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 8))) (local.get $val_flags))
        (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 12))) (local.get $val_lo))
        (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 16))) (local.get $val_hi))

        ;; Increment count
        (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))
          (i32.add (local.get $obj_count) (i32.const 1)))
      )
    )
  )

  ;; ==========================================================================
  ;; IEEE 754 Power Function helpers
  ;; ==========================================================================

  ;; Get high 32 bits of f64
  (func $f64_hi (param $x f64) (result i32)
    (i32.wrap_i64 (i64.shr_u (i64.reinterpret_f64 (local.get $x)) (i64.const 32)))
  )

  ;; Get low 32 bits of f64
  (func $f64_lo (param $x f64) (result i32)
    (i32.wrap_i64 (i64.reinterpret_f64 (local.get $x)))
  )

  ;; Build f64 from hi/lo words
  (func $f64_from_words (param $hi i32) (param $lo i32) (result f64)
    (f64.reinterpret_i64
      (i64.or
        (i64.extend_i32_u (local.get $lo))
        (i64.shl (i64.extend_i32_u (local.get $hi)) (i64.const 32))))
  )

  ;; Set high word of f64
  (func $f64_set_hi (param $x f64) (param $hi i32) (result f64)
    (call $f64_from_words (local.get $hi) (call $f64_lo (local.get $x)))
  )

  ;; Set low word to zero (truncate to 32-bit precision)
  (func $f64_trunc_lo (param $x f64) (result f64)
    (call $f64_from_words (call $f64_hi (local.get $x)) (i32.const 0))
  )

  ;; scalbn(x, n) - multiply x by 2^n
  (func $scalbn (param $x f64) (param $n i32) (result f64)
    (local $hx i32)
    (local $k i32)

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $k (i32.shr_u (i32.and (local.get $hx) (i32.const 0x7ff00000)) (i32.const 20)))

    ;; x is subnormal
    (if (i32.eqz (local.get $k))
      (then
        (local.set $x (f64.mul (local.get $x) (f64.const 1.80143985094819840000e+16))) ;; 2^54
        (local.set $hx (call $f64_hi (local.get $x)))
        (local.set $k (i32.sub
          (i32.shr_u (i32.and (local.get $hx) (i32.const 0x7ff00000)) (i32.const 20))
          (i32.const 54)))
        (if (i32.lt_s (i32.add (local.get $k) (local.get $n)) (i32.const -54))
          (then (return (f64.const 0.0)))
        )
      )
    )

    (local.set $k (i32.add (local.get $k) (local.get $n)))
    (if (i32.gt_s (local.get $k) (i32.const 0x7fe))
      (then (return (f64.mul (local.get $x) (f64.const 1.0e300))))
    )
    (if (i32.gt_s (local.get $k) (i32.const 0))
      (then
        (return (call $f64_set_hi (local.get $x)
          (i32.or
            (i32.and (local.get $hx) (i32.const 0x800fffff))
            (i32.shl (local.get $k) (i32.const 20)))))
      )
    )

    (if (i32.lt_s (local.get $k) (i32.const -54))
      (then (return (f64.const 0.0)))
    )

    (local.set $k (i32.add (local.get $k) (i32.const 54)))
    (local.set $x (call $f64_set_hi (local.get $x)
      (i32.or
        (i32.and (local.get $hx) (i32.const 0x800fffff))
        (i32.shl (local.get $k) (i32.const 20)))))
    (f64.mul (local.get $x) (f64.const 5.55111512312578270212e-17)) ;; 2^-54
  )

  ;; Main pow function: x ** y
  (func $pow (param $x f64) (param $y f64) (result f64)
    (local $z f64) (local $ax f64) (local $z_h f64) (local $z_l f64)
    (local $p_h f64) (local $p_l f64) (local $y1 f64) (local $t f64)
    (local $t1 f64) (local $t2 f64) (local $r f64) (local $s f64)
    (local $u f64) (local $v f64) (local $w f64)
    (local $ss f64) (local $s2 f64) (local $s_h f64) (local $s_l f64)
    (local $t_h f64) (local $t_l f64)
    (local $hx i32) (local $hy i32) (local $ix i32) (local $iy i32)
    (local $lx i32) (local $ly i32) (local $n i32) (local $j i32)
    (local $k i32) (local $yisint i32) (local $i i32)

    ;; Extract IEEE 754 bit patterns
    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $lx (call $f64_lo (local.get $x)))
    (local.set $hy (call $f64_hi (local.get $y)))
    (local.set $ly (call $f64_lo (local.get $y)))
    (local.set $ix (i32.and (local.get $hx) (i32.const 0x7fffffff)))
    (local.set $iy (i32.and (local.get $hy) (i32.const 0x7fffffff)))

    ;; y == 0: x**0 = 1
    (if (i32.eqz (i32.or (local.get $iy) (local.get $ly)))
      (then (return (f64.const 1.0)))
    )

    ;; x == 1: 1**y = 1
    (if (i32.and
          (i32.eq (local.get $hx) (i32.const 0x3ff00000))
          (i32.eqz (local.get $lx)))
      (then (return (f64.const 1.0)))
    )

    ;; NaN check for x
    (if (i32.or
          (i32.gt_u (local.get $ix) (i32.const 0x7ff00000))
          (i32.and
            (i32.eq (local.get $ix) (i32.const 0x7ff00000))
            (i32.ne (local.get $lx) (i32.const 0))))
      (then (return (f64.add (local.get $x) (local.get $y))))
    )

    ;; NaN check for y
    (if (i32.or
          (i32.gt_u (local.get $iy) (i32.const 0x7ff00000))
          (i32.and
            (i32.eq (local.get $iy) (i32.const 0x7ff00000))
            (i32.ne (local.get $ly) (i32.const 0))))
      (then (return (f64.add (local.get $x) (local.get $y))))
    )

    ;; Determine if y is odd integer when x < 0
    ;; yisint: 0=not int, 1=odd, 2=even
    (local.set $yisint (i32.const 0))
    (if (i32.lt_s (local.get $hx) (i32.const 0))
      (then
        (if (i32.ge_s (local.get $iy) (i32.const 0x43400000))
          (then (local.set $yisint (i32.const 2)))
          (else
            (if (i32.ge_s (local.get $iy) (i32.const 0x3ff00000))
              (then
                (local.set $k (i32.sub (i32.shr_u (local.get $iy) (i32.const 20)) (i32.const 0x3ff)))
                (if (i32.gt_s (local.get $k) (i32.const 20))
                  (then
                    (local.set $j (i32.shr_u (local.get $ly) (i32.sub (i32.const 52) (local.get $k))))
                    (if (i32.eq (i32.shl (local.get $j) (i32.sub (i32.const 52) (local.get $k))) (local.get $ly))
                      (then (local.set $yisint (i32.sub (i32.const 2) (i32.and (local.get $j) (i32.const 1)))))
                    )
                  )
                  (else
                    (if (i32.eqz (local.get $ly))
                      (then
                        (local.set $j (i32.shr_u (local.get $iy) (i32.sub (i32.const 20) (local.get $k))))
                        (if (i32.eq (i32.shl (local.get $j) (i32.sub (i32.const 20) (local.get $k))) (local.get $iy))
                          (then (local.set $yisint (i32.sub (i32.const 2) (i32.and (local.get $j) (i32.const 1)))))
                        )
                      )
                    )
                  )
                )
              )
            )
          )
        )
      )
    )

    ;; Special y values when ly==0
    (if (i32.eqz (local.get $ly))
      (then
        ;; y is +-inf
        (if (i32.eq (local.get $iy) (i32.const 0x7ff00000))
          (then
            ;; (-1)**+-inf = 1 (JS spec)
            (if (i32.eqz (i32.or (i32.sub (local.get $ix) (i32.const 0x3ff00000)) (local.get $lx)))
              (then (return (f64.const 1.0)))
            )
            (if (i32.ge_s (local.get $ix) (i32.const 0x3ff00000))
              (then (return (select (local.get $y) (f64.const 0.0) (i32.ge_s (local.get $hy) (i32.const 0)))))
              (else (return (select (f64.neg (local.get $y)) (f64.const 0.0) (i32.lt_s (local.get $hy) (i32.const 0)))))
            )
          )
        )
        ;; y is +-1
        (if (i32.eq (local.get $iy) (i32.const 0x3ff00000))
          (then
            (return (select (f64.div (f64.const 1.0) (local.get $x)) (local.get $x) (i32.lt_s (local.get $hy) (i32.const 0))))
          )
        )
        ;; y is 2
        (if (i32.eq (local.get $hy) (i32.const 0x40000000))
          (then (return (f64.mul (local.get $x) (local.get $x))))
        )
        ;; y is 0.5
        (if (i32.eq (local.get $hy) (i32.const 0x3fe00000))
          (then
            (if (i32.ge_s (local.get $hx) (i32.const 0))
              (then (return (f64.sqrt (local.get $x))))
            )
          )
        )
      )
    )

    (local.set $ax (f64.abs (local.get $x)))

    ;; Special x values (+-0, +-inf, +-1) when lx==0
    (if (i32.eqz (local.get $lx))
      (then
        (if (i32.or
              (i32.eq (local.get $ix) (i32.const 0x7ff00000))
              (i32.or (i32.eqz (local.get $ix)) (i32.eq (local.get $ix) (i32.const 0x3ff00000))))
          (then
            (local.set $z (local.get $ax))
            (if (i32.lt_s (local.get $hy) (i32.const 0))
              (then (local.set $z (f64.div (f64.const 1.0) (local.get $z))))
            )
            (if (i32.lt_s (local.get $hx) (i32.const 0))
              (then
                (if (i32.eqz (i32.or (i32.sub (local.get $ix) (i32.const 0x3ff00000)) (local.get $yisint)))
                  (then (local.set $z (f64.div (f64.sub (local.get $z) (local.get $z)) (f64.sub (local.get $z) (local.get $z)))))
                  (else
                    (if (i32.eq (local.get $yisint) (i32.const 1))
                      (then (local.set $z (f64.neg (local.get $z))))
                    )
                  )
                )
              )
            )
            (return (local.get $z))
          )
        )
      )
    )

    ;; Sign of result
    (local.set $s (f64.const 1.0))
    (if (i32.lt_s (local.get $hx) (i32.const 0))
      (then
        (if (i32.eqz (local.get $yisint))
          (then (return (f64.div (f64.sub (local.get $x) (local.get $x)) (f64.sub (local.get $x) (local.get $x)))))
        )
        (if (i32.eq (local.get $yisint) (i32.const 1))
          (then (local.set $s (f64.const -1.0)))
        )
      )
    )

    ;; |y| is huge
    (if (i32.gt_s (local.get $iy) (i32.const 0x41e00000))
      (then
        ;; |y| > 2^64: must overflow/underflow
        (if (i32.gt_s (local.get $iy) (i32.const 0x43f00000))
          (then
            (if (i32.le_s (local.get $ix) (i32.const 0x3fefffff))
              (then
                (return (select
                  (f64.mul (f64.const 1.0e300) (f64.const 1.0e300))
                  (f64.mul (f64.const 1.0e-300) (f64.const 1.0e-300))
                  (i32.lt_s (local.get $hy) (i32.const 0))))
              )
            )
            (if (i32.ge_s (local.get $ix) (i32.const 0x3ff00000))
              (then
                (return (select
                  (f64.mul (f64.const 1.0e300) (f64.const 1.0e300))
                  (f64.mul (f64.const 1.0e-300) (f64.const 1.0e-300))
                  (i32.gt_s (local.get $hy) (i32.const 0))))
              )
            )
          )
        )
        ;; Over/underflow if x not close to 1
        (if (i32.lt_s (local.get $ix) (i32.const 0x3fefffff))
          (then
            (return (f64.mul (local.get $s)
              (select
                (f64.mul (f64.const 1.0e300) (f64.const 1.0e300))
                (f64.mul (f64.const 1.0e-300) (f64.const 1.0e-300))
                (i32.lt_s (local.get $hy) (i32.const 0)))))
          )
        )
        (if (i32.gt_s (local.get $ix) (i32.const 0x3ff00000))
          (then
            (return (f64.mul (local.get $s)
              (select
                (f64.mul (f64.const 1.0e300) (f64.const 1.0e300))
                (f64.mul (f64.const 1.0e-300) (f64.const 1.0e-300))
                (i32.gt_s (local.get $hy) (i32.const 0)))))
          )
        )
        ;; |1-x| is tiny: use Taylor series for log
        (local.set $t (f64.sub (local.get $ax) (f64.const 1.0)))
        (local.set $w (f64.mul (f64.mul (local.get $t) (local.get $t))
          (f64.sub (f64.const 0.5)
            (f64.mul (local.get $t)
              (f64.sub (f64.const 0.3333333333333333) (f64.mul (local.get $t) (f64.const 0.25)))))))
        (local.set $u (f64.mul (f64.const 1.44269502162933349609e+00) (local.get $t)))
        (local.set $v (f64.sub
          (f64.mul (local.get $t) (f64.const 1.92596299112661746887e-08))
          (f64.mul (local.get $w) (f64.const 1.44269504088896338700e+00))))
        (local.set $t1 (call $f64_trunc_lo (f64.add (local.get $u) (local.get $v))))
        (local.set $t2 (f64.sub (local.get $v) (f64.sub (local.get $t1) (local.get $u))))
      )
      (else
        ;; General case: compute log2(ax)
        (local.set $n (i32.const 0))
        ;; Handle subnormal x
        (if (i32.lt_s (local.get $ix) (i32.const 0x00100000))
          (then
            (local.set $ax (f64.mul (local.get $ax) (f64.const 9007199254740992.0)))
            (local.set $n (i32.sub (local.get $n) (i32.const 53)))
            (local.set $ix (call $f64_hi (local.get $ax)))
          )
        )
        (local.set $n (i32.add (local.get $n) (i32.sub (i32.shr_u (local.get $ix) (i32.const 20)) (i32.const 0x3ff))))
        (local.set $j (i32.and (local.get $ix) (i32.const 0x000fffff)))
        (local.set $ix (i32.or (local.get $j) (i32.const 0x3ff00000)))
        (if (i32.le_s (local.get $j) (i32.const 0x3988E))
          (then (local.set $k (i32.const 0)))
          (else
            (if (i32.lt_s (local.get $j) (i32.const 0xBB67A))
              (then (local.set $k (i32.const 1)))
              (else
                (local.set $k (i32.const 0))
                (local.set $n (i32.add (local.get $n) (i32.const 1)))
                (local.set $ix (i32.sub (local.get $ix) (i32.const 0x00100000)))
              )
            )
          )
        )
        (local.set $ax (call $f64_set_hi (local.get $ax) (local.get $ix)))

        ;; Compute ss = (x-bp[k])/(x+bp[k])
        (if (i32.eqz (local.get $k))
          (then
            (local.set $u (f64.sub (local.get $ax) (f64.const 1.0)))
            (local.set $v (f64.div (f64.const 1.0) (f64.add (local.get $ax) (f64.const 1.0))))
          )
          (else
            (local.set $u (f64.sub (local.get $ax) (f64.const 1.5)))
            (local.set $v (f64.div (f64.const 1.0) (f64.add (local.get $ax) (f64.const 1.5))))
          )
        )
        (local.set $ss (f64.mul (local.get $u) (local.get $v)))
        (local.set $s_h (call $f64_trunc_lo (local.get $ss)))
        (local.set $t_h (call $f64_from_words
          (i32.add
            (i32.add (i32.or (i32.shr_u (local.get $ix) (i32.const 1)) (i32.const 0x20000000)) (i32.const 0x00080000))
            (i32.shl (local.get $k) (i32.const 18)))
          (i32.const 0)))
        (if (i32.eqz (local.get $k))
          (then (local.set $t_l (f64.sub (local.get $ax) (f64.sub (local.get $t_h) (f64.const 1.0)))))
          (else (local.set $t_l (f64.sub (local.get $ax) (f64.sub (local.get $t_h) (f64.const 1.5)))))
        )
        (local.set $s_l (f64.mul (local.get $v)
          (f64.sub (f64.sub (local.get $u) (f64.mul (local.get $s_h) (local.get $t_h)))
            (f64.mul (local.get $s_h) (local.get $t_l)))))

        ;; Compute log(ax) using polynomial
        (local.set $s2 (f64.mul (local.get $ss) (local.get $ss)))
        (local.set $r (f64.mul (f64.mul (local.get $s2) (local.get $s2))
          (f64.add (f64.const 5.99999999999994648725e-01)
            (f64.mul (local.get $s2)
              (f64.add (f64.const 4.28571428578550184252e-01)
                (f64.mul (local.get $s2)
                  (f64.add (f64.const 3.33333329818377432918e-01)
                    (f64.mul (local.get $s2)
                      (f64.add (f64.const 2.72728123808534006489e-01)
                        (f64.mul (local.get $s2)
                          (f64.add (f64.const 2.30660745775561754067e-01)
                            (f64.mul (local.get $s2) (f64.const 2.06975017800338417784e-01)))))))))))))
        (local.set $r (f64.add (local.get $r) (f64.mul (local.get $s_l) (f64.add (local.get $s_h) (local.get $ss)))))
        (local.set $s2 (f64.mul (local.get $s_h) (local.get $s_h)))
        (local.set $t_h (call $f64_trunc_lo (f64.add (f64.add (f64.const 3.0) (local.get $s2)) (local.get $r))))
        (local.set $t_l (f64.sub (local.get $r) (f64.sub (f64.sub (local.get $t_h) (f64.const 3.0)) (local.get $s2))))
        (local.set $u (f64.mul (local.get $s_h) (local.get $t_h)))
        (local.set $v (f64.add (f64.mul (local.get $s_l) (local.get $t_h)) (f64.mul (local.get $t_l) (local.get $ss))))
        (local.set $p_h (call $f64_trunc_lo (f64.add (local.get $u) (local.get $v))))
        (local.set $p_l (f64.sub (local.get $v) (f64.sub (local.get $p_h) (local.get $u))))
        (local.set $z_h (f64.mul (f64.const 9.61796700954437255859e-01) (local.get $p_h)))
        (if (i32.eqz (local.get $k))
          (then
            (local.set $z_l (f64.add
              (f64.add (f64.mul (f64.const -7.02846165095275826516e-09) (local.get $p_h))
                (f64.mul (local.get $p_l) (f64.const 9.61796693925975554329e-01)))
              (f64.const 0.0)))
          )
          (else
            (local.set $z_l (f64.add
              (f64.add (f64.mul (f64.const -7.02846165095275826516e-09) (local.get $p_h))
                (f64.mul (local.get $p_l) (f64.const 9.61796693925975554329e-01)))
              (f64.const 1.35003920212974897128e-08)))
          )
        )
        (local.set $t (f64.convert_i32_s (local.get $n)))
        (if (i32.eqz (local.get $k))
          (then (local.set $t1 (call $f64_trunc_lo (f64.add (f64.add (f64.add (local.get $z_h) (local.get $z_l)) (f64.const 0.0)) (local.get $t)))))
          (else (local.set $t1 (call $f64_trunc_lo (f64.add (f64.add (f64.add (local.get $z_h) (local.get $z_l)) (f64.const 5.84962487220764160156e-01)) (local.get $t)))))
        )
        (if (i32.eqz (local.get $k))
          (then (local.set $t2 (f64.sub (local.get $z_l) (f64.sub (f64.sub (f64.sub (local.get $t1) (local.get $t)) (f64.const 0.0)) (local.get $z_h)))))
          (else (local.set $t2 (f64.sub (local.get $z_l) (f64.sub (f64.sub (f64.sub (local.get $t1) (local.get $t)) (f64.const 5.84962487220764160156e-01)) (local.get $z_h)))))
        )
      )
    )

    ;; Split y and compute (y1+y2)*(t1+t2)
    (local.set $y1 (call $f64_trunc_lo (local.get $y)))
    (local.set $p_l (f64.add (f64.mul (f64.sub (local.get $y) (local.get $y1)) (local.get $t1)) (f64.mul (local.get $y) (local.get $t2))))
    (local.set $p_h (f64.mul (local.get $y1) (local.get $t1)))
    (local.set $z (f64.add (local.get $p_l) (local.get $p_h)))
    (local.set $j (call $f64_hi (local.get $z)))
    (local.set $i (call $f64_lo (local.get $z)))

    ;; Check overflow/underflow
    (if (i32.ge_s (local.get $j) (i32.const 0x40900000))
      (then
        (if (i32.ne (i32.or (i32.sub (local.get $j) (i32.const 0x40900000)) (local.get $i)) (i32.const 0))
          (then (return (f64.mul (local.get $s) (f64.mul (f64.const 1.0e300) (f64.const 1.0e300)))))
        )
        (if (f64.gt (f64.add (local.get $p_l) (f64.const 8.0085662595372944372e-17)) (f64.sub (local.get $z) (local.get $p_h)))
          (then (return (f64.mul (local.get $s) (f64.mul (f64.const 1.0e300) (f64.const 1.0e300)))))
        )
      )
      (else
        (if (i32.ge_s (i32.and (local.get $j) (i32.const 0x7fffffff)) (i32.const 0x4090cc00))
          (then
            (if (i32.ne (i32.or (i32.sub (local.get $j) (i32.const 0xc090cc00)) (local.get $i)) (i32.const 0))
              (then (return (f64.mul (local.get $s) (f64.mul (f64.const 1.0e-300) (f64.const 1.0e-300)))))
            )
            (if (f64.le (local.get $p_l) (f64.sub (local.get $z) (local.get $p_h)))
              (then (return (f64.mul (local.get $s) (f64.mul (f64.const 1.0e-300) (f64.const 1.0e-300)))))
            )
          )
        )
      )
    )

    ;; Compute 2**(p_h+p_l)
    (local.set $i (i32.and (local.get $j) (i32.const 0x7fffffff)))
    (local.set $k (i32.sub (i32.shr_u (local.get $i) (i32.const 20)) (i32.const 0x3ff)))
    (local.set $n (i32.const 0))
    (if (i32.gt_s (local.get $i) (i32.const 0x3fe00000))
      (then
        (local.set $n (i32.add (local.get $j) (i32.shr_u (i32.const 0x00100000) (i32.add (local.get $k) (i32.const 1)))))
        (local.set $k (i32.sub (i32.shr_u (i32.and (local.get $n) (i32.const 0x7fffffff)) (i32.const 20)) (i32.const 0x3ff)))
        (local.set $t (call $f64_from_words
          (i32.and (local.get $n) (i32.xor (i32.shr_u (i32.const 0x000fffff) (local.get $k)) (i32.const 0xffffffff)))
          (i32.const 0)))
        (local.set $n (i32.shr_u
          (i32.or (i32.and (local.get $n) (i32.const 0x000fffff)) (i32.const 0x00100000))
          (i32.sub (i32.const 20) (local.get $k))))
        (if (i32.lt_s (local.get $j) (i32.const 0))
          (then (local.set $n (i32.sub (i32.const 0) (local.get $n))))
        )
        (local.set $p_h (f64.sub (local.get $p_h) (local.get $t)))
      )
    )
    (local.set $t (call $f64_trunc_lo (f64.add (local.get $p_l) (local.get $p_h))))
    (local.set $u (f64.mul (local.get $t) (f64.const 6.93147182464599609375e-01)))
    (local.set $v (f64.add
      (f64.mul (f64.sub (local.get $p_l) (f64.sub (local.get $t) (local.get $p_h))) (f64.const 6.93147180559945286227e-01))
      (f64.mul (local.get $t) (f64.const -1.90465429995776804525e-09))))
    (local.set $z (f64.add (local.get $u) (local.get $v)))
    (local.set $w (f64.sub (local.get $v) (f64.sub (local.get $z) (local.get $u))))
    (local.set $t (f64.mul (local.get $z) (local.get $z)))
    (local.set $t1 (f64.sub (local.get $z)
      (f64.mul (local.get $t)
        (f64.add (f64.const 1.66666666666666019037e-01)
          (f64.mul (local.get $t)
            (f64.add (f64.const -2.77777777770155933842e-03)
              (f64.mul (local.get $t)
                (f64.add (f64.const 6.61375632143793436117e-05)
                  (f64.mul (local.get $t)
                    (f64.add (f64.const -1.65339022054652515390e-06)
                      (f64.mul (local.get $t) (f64.const 4.13813679705723846039e-08))))))))))))
    (local.set $r (f64.sub
      (f64.div (f64.mul (local.get $z) (local.get $t1)) (f64.sub (local.get $t1) (f64.const 2.0)))
      (f64.add (local.get $w) (f64.mul (local.get $z) (local.get $w)))))
    (local.set $z (f64.sub (f64.const 1.0) (f64.sub (local.get $r) (local.get $z))))
    (local.set $j (call $f64_hi (local.get $z)))
    (local.set $j (i32.add (local.get $j) (i32.shl (local.get $n) (i32.const 20))))
    (if (i32.le_s (i32.shr_s (local.get $j) (i32.const 20)) (i32.const 0))
      (then (local.set $z (call $scalbn (local.get $z) (local.get $n))))
      (else (local.set $z (call $f64_set_hi (local.get $z) (local.get $j))))
    )
    (f64.mul (local.get $s) (local.get $z))
  )

  ;; ==========================================================================
  ;; Wrapper functions for consistent naming (f64_* prefix)
  ;; ==========================================================================

  (func $f64_pow (param $x f64) (param $y f64) (result f64)
    (call $pow (local.get $x) (local.get $y))
  )

  ;; ==========================================================================
  ;; Natural logarithm using polynomial approximation
  ;; ln(x) = ln(m * 2^e) = ln(m) + e*ln(2)
  ;; where m is the mantissa normalized to [1, 2)
  ;; ==========================================================================
  ;; ==========================================================================
  ;; Natural logarithm using fdlibm algorithm (exact match to e_log.c)
  ;; Ported from Sun Microsystems fdlibm e_log.c
  ;; ==========================================================================
  (func $f64_log (param $x f64) (result f64)
    (local $hx i32) (local $lx i32) (local $k i32) (local $i i32) (local $j i32)
    (local $f f64) (local $s f64) (local $z f64) (local $w f64)
    (local $R f64) (local $t1 f64) (local $t2 f64) (local $hfsq f64) (local $dk f64)
    ;; fdlibm coefficients for log (minimax polynomial)
    ;; ln2_hi = 6.93147180369123816490e-01  (0x3fe62e42, 0xfee00000)
    ;; ln2_lo = 1.90821492927058770002e-10  (0x3dea39ef, 0x35793c76)

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $lx (call $f64_lo (local.get $x)))

    (local.set $k (i32.const 0))

    ;; Special cases: x < 2**-1022 (subnormal or zero/negative)
    (if (i32.lt_s (local.get $hx) (i32.const 0x00100000))
      (then
        ;; log(+-0) = -inf
        (if (i32.eq (i32.or (i32.and (local.get $hx) (i32.const 0x7fffffff)) (local.get $lx)) (i32.const 0))
          (then (return (f64.div (f64.const -1) (f64.const 0))))
        )
        ;; log(negative) = NaN
        (if (i32.lt_s (local.get $hx) (i32.const 0))
          (then (return (f64.div (f64.const 0) (f64.const 0))))
        )
        ;; subnormal: scale up by 2^54
        (local.set $k (i32.sub (local.get $k) (i32.const 54)))
        (local.set $x (f64.mul (local.get $x) (f64.const 1.80143985094819840000e+16)))
        (local.set $hx (call $f64_hi (local.get $x)))
      )
    )
    ;; x >= 2^1024 (infinity or NaN)
    (if (i32.ge_s (local.get $hx) (i32.const 0x7ff00000))
      (then (return (f64.add (local.get $x) (local.get $x))))
    )

    ;; k += (hx>>20) - 1023
    (local.set $k (i32.add (local.get $k)
      (i32.sub (i32.shr_u (local.get $hx) (i32.const 20)) (i32.const 1023))))
    (local.set $hx (i32.and (local.get $hx) (i32.const 0x000fffff)))

    ;; i = (hx + 0x95f64) & 0x100000  -- determines if we normalize to x or x/2
    (local.set $i (i32.and (i32.add (local.get $hx) (i32.const 0x95f64)) (i32.const 0x100000)))

    ;; Set high word: hx | (i ^ 0x3ff00000) -- normalize x or x/2
    (local.set $x (call $f64_from_words
      (i32.or (local.get $hx) (i32.xor (local.get $i) (i32.const 0x3ff00000)))
      (call $f64_lo (local.get $x))))
    (local.set $k (i32.add (local.get $k) (i32.shr_u (local.get $i) (i32.const 20))))

    (local.set $f (f64.sub (local.get $x) (f64.const 1.0)))

    ;; Fast path for |f| < 2**-20: if ((0x000fffff & (2+hx)) < 3)
    (if (i32.lt_s (i32.and (i32.const 0x000fffff) (i32.add (i32.const 2) (local.get $hx))) (i32.const 3))
      (then
        (if (f64.eq (local.get $f) (f64.const 0))
          (then
            (if (i32.eq (local.get $k) (i32.const 0))
              (then (return (f64.const 0)))
            )
            (local.set $dk (f64.convert_i32_s (local.get $k)))
            (return (f64.add
              (f64.mul (local.get $dk) (f64.const 6.93147180369123816490e-01))
              (f64.mul (local.get $dk) (f64.const 1.90821492927058770002e-10))))
          )
        )
        ;; R = f*f*(0.5 - 0.33333333333333333*f)
        (local.set $R (f64.mul (f64.mul (local.get $f) (local.get $f))
          (f64.sub (f64.const 0.5)
            (f64.mul (f64.const 0.33333333333333333) (local.get $f)))))
        (if (i32.eq (local.get $k) (i32.const 0))
          (then (return (f64.sub (local.get $f) (local.get $R))))
        )
        (local.set $dk (f64.convert_i32_s (local.get $k)))
        (return (f64.sub
          (f64.mul (local.get $dk) (f64.const 6.93147180369123816490e-01))
          (f64.sub
            (f64.sub (local.get $R) (f64.mul (local.get $dk) (f64.const 1.90821492927058770002e-10)))
            (local.get $f))))
      )
    )

    ;; s = f / (2 + f)
    (local.set $s (f64.div (local.get $f) (f64.add (f64.const 2.0) (local.get $f))))
    (local.set $dk (f64.convert_i32_s (local.get $k)))
    (local.set $z (f64.mul (local.get $s) (local.get $s)))
    (local.set $i (i32.sub (local.get $hx) (i32.const 0x6147a)))
    (local.set $w (f64.mul (local.get $z) (local.get $z)))
    (local.set $j (i32.sub (i32.const 0x6b851) (local.get $hx)))

    ;; t1 = w*(Lg2 + w*(Lg4 + w*Lg6))
    (local.set $t1 (f64.mul (local.get $w)
      (f64.add (f64.const 3.999999999940941908e-01)
        (f64.mul (local.get $w)
          (f64.add (f64.const 2.222219843214978396e-01)
            (f64.mul (local.get $w) (f64.const 1.531383769920937332e-01)))))))

    ;; t2 = z*(Lg1 + w*(Lg3 + w*(Lg5 + w*Lg7)))
    (local.set $t2 (f64.mul (local.get $z)
      (f64.add (f64.const 6.666666666666735130e-01)
        (f64.mul (local.get $w)
          (f64.add (f64.const 2.857142874366239149e-01)
            (f64.mul (local.get $w)
              (f64.add (f64.const 1.818357216161805012e-01)
                (f64.mul (local.get $w) (f64.const 1.479819860511658591e-01)))))))))

    (local.set $i (i32.or (local.get $i) (local.get $j)))
    (local.set $R (f64.add (local.get $t2) (local.get $t1)))

    (if (i32.gt_s (local.get $i) (i32.const 0))
      (then
        ;; Use hfsq path: hfsq = 0.5*f*f
        (local.set $hfsq (f64.mul (f64.const 0.5) (f64.mul (local.get $f) (local.get $f))))
        (if (i32.eq (local.get $k) (i32.const 0))
          (then
            ;; return f - (hfsq - s*(hfsq+R))
            (return (f64.sub (local.get $f)
              (f64.sub (local.get $hfsq)
                (f64.mul (local.get $s) (f64.add (local.get $hfsq) (local.get $R))))))
          )
          (else
            ;; return dk*ln2_hi - ((hfsq - (s*(hfsq+R) + dk*ln2_lo)) - f)
            (return (f64.sub
              (f64.mul (local.get $dk) (f64.const 6.93147180369123816490e-01))
              (f64.sub
                (f64.sub (local.get $hfsq)
                  (f64.add
                    (f64.mul (local.get $s) (f64.add (local.get $hfsq) (local.get $R)))
                    (f64.mul (local.get $dk) (f64.const 1.90821492927058770002e-10))))
                (local.get $f))))
          )
        )
      )
      (else
        ;; Non-hfsq path
        (if (i32.eq (local.get $k) (i32.const 0))
          (then
            ;; return f - s*(f - R)
            (return (f64.sub (local.get $f)
              (f64.mul (local.get $s) (f64.sub (local.get $f) (local.get $R)))))
          )
          (else
            ;; return dk*ln2_hi - ((s*(f-R) - dk*ln2_lo) - f)
            (return (f64.sub
              (f64.mul (local.get $dk) (f64.const 6.93147180369123816490e-01))
              (f64.sub
                (f64.sub
                  (f64.mul (local.get $s) (f64.sub (local.get $f) (local.get $R)))
                  (f64.mul (local.get $dk) (f64.const 1.90821492927058770002e-10)))
                (local.get $f))))
          )
        )
      )
    )
    (unreachable)
  )

  ;; ==========================================================================
  ;; Log base 2 using V8/fdlibm algorithm (exact match to ieee754.cc)
  ;; Uses Dekker splitting for extra precision
  ;; ==========================================================================
  (func $f64_log2 (param $x f64) (result f64)
    (local $hx i32) (local $lx i32) (local $k i32) (local $i i32)
    (local $f f64) (local $s f64) (local $z f64) (local $w f64)
    (local $R f64) (local $hfsq f64) (local $y f64)
    (local $hi f64) (local $lo f64) (local $r f64)
    (local $val_hi f64) (local $val_lo f64)
    (local $t1 f64) (local $t2 f64)
    ;; Constants from V8:
    ;; ivln2hi = 1.44269504072144627571e+00  0x3FF71547, 0x65200000
    ;; ivln2lo = 1.67517131648865118353e-10  0x3DE705FC, 0x2EEFA200
    ;; two54   = 1.80143985094819840000e+16  0x43500000, 0x00000000
    ;; Lg1-Lg7 coefficients for k_log1p

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $lx (call $f64_lo (local.get $x)))
    (local.set $k (i32.const 0))

    ;; x < 2**-1022 (subnormal or negative or zero)
    (if (i32.lt_s (local.get $hx) (i32.const 0x00100000))
      (then
        ;; log(+-0) = -inf
        (if (i32.eqz (i32.or (i32.and (local.get $hx) (i32.const 0x7FFFFFFF)) (local.get $lx)))
          (then (return (f64.div (f64.const -1) (f64.const 0))))
        )
        ;; log(-x) = NaN
        (if (i32.lt_s (local.get $hx) (i32.const 0))
          (then (return (f64.div (f64.const 0) (f64.const 0))))
        )
        ;; subnormal: scale up x
        (local.set $k (i32.sub (local.get $k) (i32.const 54)))
        (local.set $x (f64.mul (local.get $x) (f64.const 1.80143985094819840000e+16)))
        (local.set $hx (call $f64_hi (local.get $x)))
      )
    )

    ;; x = inf or NaN
    (if (i32.ge_s (local.get $hx) (i32.const 0x7FF00000))
      (then (return (f64.add (local.get $x) (local.get $x))))
    )

    ;; log(1) = +0
    (if (i32.and
          (i32.eq (local.get $hx) (i32.const 0x3FF00000))
          (i32.eqz (local.get $lx)))
      (then (return (f64.const 0)))
    )

    ;; k += (hx >> 20) - 1023
    (local.set $k (i32.add (local.get $k) (i32.sub (i32.shr_u (local.get $hx) (i32.const 20)) (i32.const 1023))))
    (local.set $hx (i32.and (local.get $hx) (i32.const 0x000FFFFF)))
    (local.set $i (i32.and (i32.add (local.get $hx) (i32.const 0x95F64)) (i32.const 0x100000)))
    ;; normalize x or x/2
    (local.set $x (call $f64_set_hi (local.get $x) (i32.or (local.get $hx) (i32.xor (local.get $i) (i32.const 0x3FF00000)))))
    (local.set $k (i32.add (local.get $k) (i32.shr_u (local.get $i) (i32.const 20))))
    (local.set $y (f64.convert_i32_s (local.get $k)))
    (local.set $f (f64.sub (local.get $x) (f64.const 1.0)))
    (local.set $hfsq (f64.mul (f64.const 0.5) (f64.mul (local.get $f) (local.get $f))))

    ;; k_log1p(f) - returns s*(hfsq+R)
    (local.set $s (f64.div (local.get $f) (f64.add (f64.const 2.0) (local.get $f))))
    (local.set $z (f64.mul (local.get $s) (local.get $s)))
    (local.set $w (f64.mul (local.get $z) (local.get $z)))
    ;; t1 = w*(Lg2 + w*(Lg4 + w*Lg6))
    (local.set $t1 (f64.mul (local.get $w)
      (f64.add (f64.const 3.999999999940941908e-01)
        (f64.mul (local.get $w)
          (f64.add (f64.const 2.222219843214978396e-01)
            (f64.mul (local.get $w) (f64.const 1.531383769920937332e-01)))))))
    ;; t2 = z*(Lg1 + w*(Lg3 + w*(Lg5 + w*Lg7)))
    (local.set $t2 (f64.mul (local.get $z)
      (f64.add (f64.const 6.666666666666735130e-01)
        (f64.mul (local.get $w)
          (f64.add (f64.const 2.857142874366239149e-01)
            (f64.mul (local.get $w)
              (f64.add (f64.const 1.818357216161805012e-01)
                (f64.mul (local.get $w) (f64.const 1.479819860511658591e-01)))))))))
    (local.set $R (f64.add (local.get $t2) (local.get $t1)))
    (local.set $r (f64.mul (local.get $s) (f64.add (local.get $hfsq) (local.get $R))))

    ;; Dekker splitting: hi = f - hfsq with low word cleared
    (local.set $hi (call $f64_trunc_lo (f64.sub (local.get $f) (local.get $hfsq))))
    (local.set $lo (f64.add (f64.sub (f64.sub (local.get $f) (local.get $hi)) (local.get $hfsq)) (local.get $r)))

    ;; val_hi = hi * ivln2hi
    ;; val_lo = (lo + hi) * ivln2lo + lo * ivln2hi
    (local.set $val_hi (f64.mul (local.get $hi) (f64.const 1.44269504072144627571e+00)))
    (local.set $val_lo (f64.add
      (f64.mul (f64.add (local.get $lo) (local.get $hi)) (f64.const 1.67517131648865118353e-10))
      (f64.mul (local.get $lo) (f64.const 1.44269504072144627571e+00))))

    ;; spadd(val_hi, val_lo, y)
    (local.set $w (f64.add (local.get $y) (local.get $val_hi)))
    (local.set $val_lo (f64.add (local.get $val_lo) (f64.add (f64.sub (local.get $y) (local.get $w)) (local.get $val_hi))))
    (local.set $val_hi (local.get $w))

    (f64.add (local.get $val_lo) (local.get $val_hi))
  )

  ;; ==========================================================================
  ;; log1p(x) = log(1+x) using V8/fdlibm algorithm (exact match to ieee754.cc)
  ;; More precise than log(1+x) for small x
  ;; ==========================================================================
  (func $f64_log1p (param $x f64) (result f64)
    (local $hx i32) (local $ax i32) (local $hu i32) (local $k i32)
    (local $f f64) (local $c f64) (local $s f64) (local $z f64) (local $R f64)
    (local $u f64) (local $hfsq f64)
    ;; Constants from V8:
    ;; ln2_hi = 6.93147180369123816490e-01  0x3FE62E42, 0xFEE00000
    ;; ln2_lo = 1.90821492927058770002e-10  0x3DEA39EF, 0x35793C76
    ;; two54  = 1.80143985094819840000e+16  0x43500000, 0x00000000
    ;; Lp1-Lp7 coefficients (same as Lg1-Lg7)

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $ax (i32.and (local.get $hx) (i32.const 0x7FFFFFFF)))
    (local.set $k (i32.const 1))

    ;; 1+x < sqrt(2)+
    (if (i32.lt_s (local.get $hx) (i32.const 0x3FDA827A))
      (then
        ;; x <= -1.0
        (if (i32.ge_u (local.get $ax) (i32.const 0x3FF00000))
          (then
            (if (f64.eq (local.get $x) (f64.const -1.0))
              (then (return (f64.div (f64.const -1) (f64.const 0)))) ;; -inf
              (else (return (f64.div (f64.const 0) (f64.const 0)))) ;; NaN
            )
          )
        )
        ;; |x| < 2**-29
        (if (i32.lt_u (local.get $ax) (i32.const 0x3E200000))
          (then
            ;; |x| < 2**-54: return x
            (if (i32.lt_u (local.get $ax) (i32.const 0x3C900000))
              (then (return (local.get $x)))
              (else (return (f64.sub (local.get $x) (f64.mul (f64.mul (local.get $x) (local.get $x)) (f64.const 0.5)))))
            )
          )
        )
        ;; sqrt(2)/2- <= 1+x < sqrt(2)+
        (if (i32.or
              (i32.gt_s (local.get $hx) (i32.const 0))
              (i32.le_s (local.get $hx) (i32.const 0xBFD2BEC4)))
          (then
            (local.set $k (i32.const 0))
            (local.set $f (local.get $x))
            (local.set $hu (i32.const 1))
            (local.set $c (f64.const 0))
          )
        )
      )
    )

    ;; x = inf or NaN
    (if (i32.ge_s (local.get $hx) (i32.const 0x7FF00000))
      (then (return (f64.add (local.get $x) (local.get $x))))
    )

    (if (i32.ne (local.get $k) (i32.const 0))
      (then
        (if (i32.lt_s (local.get $hx) (i32.const 0x43400000))
          (then
            (local.set $u (f64.add (f64.const 1.0) (local.get $x)))
            (local.set $hu (call $f64_hi (local.get $u)))
            (local.set $k (i32.sub (i32.shr_u (local.get $hu) (i32.const 20)) (i32.const 1023)))
            ;; c = (k > 0) ? 1.0 - (u - x) : x - (u - 1.0)
            (if (i32.gt_s (local.get $k) (i32.const 0))
              (then (local.set $c (f64.sub (f64.const 1.0) (f64.sub (local.get $u) (local.get $x)))))
              (else (local.set $c (f64.sub (local.get $x) (f64.sub (local.get $u) (f64.const 1.0)))))
            )
            (local.set $c (f64.div (local.get $c) (local.get $u)))
          )
          (else
            (local.set $u (local.get $x))
            (local.set $hu (call $f64_hi (local.get $u)))
            (local.set $k (i32.sub (i32.shr_u (local.get $hu) (i32.const 20)) (i32.const 1023)))
            (local.set $c (f64.const 0))
          )
        )
        (local.set $hu (i32.and (local.get $hu) (i32.const 0x000FFFFF)))
        (if (i32.lt_u (local.get $hu) (i32.const 0x6A09E))
          (then
            ;; u ~< sqrt(2): normalize u
            (local.set $u (call $f64_set_hi (local.get $u) (i32.or (local.get $hu) (i32.const 0x3FF00000))))
          )
          (else
            ;; normalize u/2
            (local.set $k (i32.add (local.get $k) (i32.const 1)))
            (local.set $u (call $f64_set_hi (local.get $u) (i32.or (local.get $hu) (i32.const 0x3FE00000))))
            (local.set $hu (i32.shr_u (i32.sub (i32.const 0x00100000) (local.get $hu)) (i32.const 2)))
          )
        )
        (local.set $f (f64.sub (local.get $u) (f64.const 1.0)))
      )
    )

    (local.set $hfsq (f64.mul (f64.const 0.5) (f64.mul (local.get $f) (local.get $f))))

    ;; |f| < 2**-20
    (if (i32.eqz (local.get $hu))
      (then
        (if (f64.eq (local.get $f) (f64.const 0))
          (then
            (if (i32.eqz (local.get $k))
              (then (return (f64.const 0)))
              (else
                (local.set $c (f64.add (local.get $c) (f64.mul (f64.convert_i32_s (local.get $k)) (f64.const 1.90821492927058770002e-10))))
                (return (f64.add (f64.mul (f64.convert_i32_s (local.get $k)) (f64.const 6.93147180369123816490e-01)) (local.get $c)))
              )
            )
          )
        )
        (local.set $R (f64.mul (local.get $hfsq) (f64.sub (f64.const 1.0) (f64.mul (f64.const 0.66666666666666666) (local.get $f)))))
        (if (i32.eqz (local.get $k))
          (then (return (f64.sub (local.get $f) (local.get $R))))
          (else (return (f64.sub
            (f64.mul (f64.convert_i32_s (local.get $k)) (f64.const 6.93147180369123816490e-01))
            (f64.sub
              (f64.sub (local.get $R)
                (f64.add (f64.mul (f64.convert_i32_s (local.get $k)) (f64.const 1.90821492927058770002e-10)) (local.get $c)))
              (local.get $f)))))
        )
      )
    )

    (local.set $s (f64.div (local.get $f) (f64.add (f64.const 2.0) (local.get $f))))
    (local.set $z (f64.mul (local.get $s) (local.get $s)))
    ;; R = z*(Lp1 + z*(Lp2 + z*(Lp3 + z*(Lp4 + z*(Lp5 + z*(Lp6 + z*Lp7))))))
    (local.set $R (f64.mul (local.get $z)
      (f64.add (f64.const 6.666666666666735130e-01)
        (f64.mul (local.get $z)
          (f64.add (f64.const 3.999999999940941908e-01)
            (f64.mul (local.get $z)
              (f64.add (f64.const 2.857142874366239149e-01)
                (f64.mul (local.get $z)
                  (f64.add (f64.const 2.222219843214978396e-01)
                    (f64.mul (local.get $z)
                      (f64.add (f64.const 1.818357216161805012e-01)
                        (f64.mul (local.get $z)
                          (f64.add (f64.const 1.531383769920937332e-01)
                            (f64.mul (local.get $z) (f64.const 1.479819860511658591e-01)))))))))))))))

    (if (i32.eqz (local.get $k))
      (then (return (f64.sub (local.get $f) (f64.sub (local.get $hfsq) (f64.mul (local.get $s) (f64.add (local.get $hfsq) (local.get $R)))))))
      (else (return (f64.sub
        (f64.mul (f64.convert_i32_s (local.get $k)) (f64.const 6.93147180369123816490e-01))
        (f64.sub
          (f64.sub (local.get $hfsq)
            (f64.add (f64.mul (local.get $s) (f64.add (local.get $hfsq) (local.get $R)))
              (f64.add (f64.mul (f64.convert_i32_s (local.get $k)) (f64.const 1.90821492927058770002e-10)) (local.get $c))))
          (local.get $f)))))
    )
    (unreachable)
  )

  ;; ==========================================================================
  ;; Log base 10 using fdlibm algorithm
  ;; log10(x) = k*log10(2) + log10(1+f)
  ;; ==========================================================================
  (func $f64_log10 (param $x f64) (result f64)
    (local $hx i32) (local $k i32) (local $f f64) (local $s f64) (local $z f64)
    (local $R f64) (local $hfsq f64) (local $dk f64) (local $log_result f64)
    ;; ivln10   = 4.34294481903251816668e-01  (1/ln10)
    ;; log10_2hi = 3.01029995663611771306e-01
    ;; log10_2lo = 3.69423907715893078616e-13

    ;; Special cases
    (if (f64.le (local.get $x) (f64.const 0))
      (then
        (if (f64.eq (local.get $x) (f64.const 0))
          (then (return (f64.div (f64.const -1) (f64.const 0))))
        )
        (return (f64.div (f64.const 0) (f64.const 0)))
      )
    )
    (if (f64.ne (local.get $x) (local.get $x))
      (then (return (local.get $x)))
    )
    ;; Infinity returns Infinity
    (if (f64.eq (local.get $x) (f64.div (f64.const 1) (f64.const 0)))
      (then (return (local.get $x)))
    )

    ;; Extract exponent k
    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $k (i32.sub
      (i32.shr_u (i32.and (local.get $hx) (i32.const 0x7ff00000)) (i32.const 20))
      (i32.const 1023)))

    ;; Get f
    (local.set $f (f64.sub
      (call $f64_from_words
        (i32.or (i32.and (local.get $hx) (i32.const 0x000fffff)) (i32.const 0x3ff00000))
        (call $f64_lo (local.get $x)))
      (f64.const 1)))

    ;; Adjust for f > sqrt(2)-1
    (if (f64.gt (local.get $f) (f64.const 0.41421356237309504))
      (then
        (local.set $f (f64.sub (f64.mul (local.get $f) (f64.const 0.5))
                               (f64.const 0.5)))
        (local.set $k (i32.add (local.get $k) (i32.const 1)))
      )
    )

    (local.set $s (f64.div (local.get $f) (f64.add (f64.const 2) (local.get $f))))
    (local.set $z (f64.mul (local.get $s) (local.get $s)))

    (local.set $R
      (f64.mul (local.get $z)
        (f64.add (f64.const 6.666666666666735130e-01)
          (f64.mul (local.get $z)
            (f64.add (f64.const 3.999999999940941908e-01)
              (f64.mul (local.get $z)
                (f64.add (f64.const 2.857142874366239149e-01)
                  (f64.mul (local.get $z)
                    (f64.add (f64.const 2.222219843214978396e-01)
                      (f64.mul (local.get $z)
                        (f64.add (f64.const 1.818357216161805012e-01)
                          (f64.mul (local.get $z)
                            (f64.add (f64.const 1.531383769920937332e-01)
                              (f64.mul (local.get $z) (f64.const 1.479819860511658591e-01)))))))))))))))

    (local.set $hfsq (f64.mul (f64.const 0.5) (f64.mul (local.get $f) (local.get $f))))
    (local.set $dk (f64.convert_i32_s (local.get $k)))

    ;; log(1+f)
    (local.set $log_result
      (f64.sub
        (f64.add (local.get $f)
          (f64.mul (local.get $s) (f64.add (local.get $hfsq) (local.get $R))))
        (local.get $hfsq)))

    ;; result = k*log10_2 + log(1+f)/ln10
    ;; = k*log10_2hi + (k*log10_2lo + ivln10*log_result)
    (f64.add
      (f64.mul (local.get $dk) (f64.const 3.01029995663611771306e-01))  ;; k * log10_2hi
      (f64.add
        (f64.mul (local.get $dk) (f64.const 3.69423907715893078616e-13))  ;; k * log10_2lo
        (f64.mul (local.get $log_result) (f64.const 4.34294481903251816668e-01))))  ;; ivln10 * log_result
  )

  ;; ==========================================================================
  ;; Exponential function using fdlibm algorithm (exact match to e_exp.c)
  ;; Ported from Sun Microsystems fdlibm e_exp.c
  ;; ==========================================================================
  (func $f64_exp (param $x f64) (result f64)
    (local $hx i32) (local $xsb i32) (local $k i32)
    (local $hi f64) (local $lo f64) (local $t f64) (local $c f64) (local $y f64)
    ;; Constants:
    ;; o_threshold = 7.09782712893383973096e+02  (0x40862E42, 0xFEFA39EF)
    ;; u_threshold = -7.45133219101941108420e+02 (0xc0874910, 0xD52D3051)
    ;; ln2HI = 6.93147180369123816490e-01  (0x3fe62e42, 0xfee00000)
    ;; ln2LO = 1.90821492927058770002e-10  (0x3dea39ef, 0x35793c76)
    ;; invln2 = 1.44269504088896338700e+00 (0x3ff71547, 0x652b82fe)
    ;; twom1000 = 9.33263618503218878990e-302 (0x01700000, 0x00000000)

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $xsb (i32.and (i32.shr_u (local.get $hx) (i32.const 31)) (i32.const 1)))
    (local.set $hx (i32.and (local.get $hx) (i32.const 0x7fffffff)))

    ;; Filter out non-finite argument
    (if (i32.ge_u (local.get $hx) (i32.const 0x40862E42))
      (then
        (if (i32.ge_u (local.get $hx) (i32.const 0x7ff00000))
          (then
            ;; NaN or Inf
            (if (i32.ne (i32.or (i32.and (local.get $hx) (i32.const 0xfffff)) (call $f64_lo (local.get $x))) (i32.const 0))
              (then (return (f64.add (local.get $x) (local.get $x)))) ;; NaN
            )
            (if (i32.eq (local.get $xsb) (i32.const 0))
              (then (return (local.get $x))) ;; +inf
              (else (return (f64.const 0))) ;; -inf -> 0
            )
          )
        )
        ;; Overflow
        (if (f64.gt (local.get $x) (f64.const 7.09782712893383973096e+02))
          (then (return (f64.mul (f64.const 1e300) (f64.const 1e300))))
        )
        ;; Underflow
        (if (f64.lt (local.get $x) (f64.const -7.45133219101941108420e+02))
          (then (return (f64.mul (f64.const 9.33263618503218878990e-302) (f64.const 9.33263618503218878990e-302))))
        )
      )
    )

    ;; Argument reduction
    (if (i32.gt_u (local.get $hx) (i32.const 0x3fd62e42))
      (then
        ;; |x| > 0.5*ln2
        (if (i32.lt_u (local.get $hx) (i32.const 0x3FF0A2B2))
          (then
            ;; |x| < 1.5*ln2: k = 1 - 2*xsb
            ;; Special case: exp(1) returns exact E constant (V8 does this too
            ;; because the fdlibm algorithm gets the last bit wrong)
            (if (f64.eq (local.get $x) (f64.const 1.0))
              (then (return (f64.const 2.718281828459045)))  ;; 0x4005BF0A_8B145769
            )
            (local.set $k (i32.sub (i32.const 1) (i32.shl (local.get $xsb) (i32.const 1))))
            (if (i32.eq (local.get $xsb) (i32.const 0))
              (then
                ;; x > 0: hi = x - ln2HI, lo = ln2LO
                (local.set $hi (f64.sub (local.get $x) (f64.const 6.93147180369123816490e-01)))
                (local.set $lo (f64.const 1.90821492927058770002e-10))
              )
              (else
                ;; x < 0: hi = x + ln2HI, lo = -ln2LO
                (local.set $hi (f64.add (local.get $x) (f64.const 6.93147180369123816490e-01)))
                (local.set $lo (f64.const -1.90821492927058770002e-10))
              )
            )
          )
          (else
            ;; |x| >= 1.5*ln2: k = floor(invln2*x + 0.5) or ceiling
            (if (i32.eq (local.get $xsb) (i32.const 0))
              (then
                (local.set $k (i32.trunc_f64_s
                  (f64.add (f64.mul (local.get $x) (f64.const 1.44269504088896338700e+00)) (f64.const 0.5))))
              )
              (else
                (local.set $k (i32.trunc_f64_s
                  (f64.sub (f64.mul (local.get $x) (f64.const 1.44269504088896338700e+00)) (f64.const 0.5))))
              )
            )
            (local.set $t (f64.convert_i32_s (local.get $k)))
            (local.set $hi (f64.sub (local.get $x) (f64.mul (local.get $t) (f64.const 6.93147180369123816490e-01))))
            (local.set $lo (f64.mul (local.get $t) (f64.const 1.90821492927058770002e-10)))
          )
        )
        (local.set $x (f64.sub (local.get $hi) (local.get $lo)))
      )
      (else
        (if (i32.lt_u (local.get $hx) (i32.const 0x3e300000))
          (then
            ;; |x| < 2**-28: return 1+x (trigger inexact)
            (if (f64.gt (f64.add (f64.const 1e300) (local.get $x)) (f64.const 1))
              (then (return (f64.add (f64.const 1) (local.get $x))))
            )
          )
        )
        (local.set $k (i32.const 0))
      )
    )

    ;; x is now in primary range
    (local.set $t (f64.mul (local.get $x) (local.get $x)))
    (local.set $c (f64.sub (local.get $x)
      (f64.mul (local.get $t)
        (f64.add (f64.const 1.66666666666666019037e-01)
          (f64.mul (local.get $t)
            (f64.add (f64.const -2.77777777770155933842e-03)
              (f64.mul (local.get $t)
                (f64.add (f64.const 6.61375632143793436117e-05)
                  (f64.mul (local.get $t)
                    (f64.add (f64.const -1.65339022054652515390e-06)
                      (f64.mul (local.get $t) (f64.const 4.13813679705723846039e-08))))))))))))

    (if (i32.eq (local.get $k) (i32.const 0))
      (then
        ;; return 1 - ((x*c)/(c-2) - x)
        (return (f64.sub (f64.const 1)
          (f64.sub
            (f64.div (f64.mul (local.get $x) (local.get $c))
                     (f64.sub (local.get $c) (f64.const 2)))
            (local.get $x))))
      )
      (else
        ;; y = 1 - ((lo - (x*c)/(2-c)) - hi)
        (local.set $y (f64.sub (f64.const 1)
          (f64.sub
            (f64.sub (local.get $lo)
              (f64.div (f64.mul (local.get $x) (local.get $c))
                       (f64.sub (f64.const 2) (local.get $c))))
            (local.get $hi))))
        ;; Scale by 2^k
        (return (call $scalbn (local.get $y) (local.get $k)))
      )
    )
    (unreachable)
  )

  ;; ==========================================================================
  ;; ==========================================================================
  ;; Sine and Cosine using fdlibm algorithm (sub-ULP accuracy)
  ;; Ported from Sun Microsystems fdlibm k_sin.c, k_cos.c, s_sin.c, e_rem_pio2.c
  ;; ==========================================================================

  ;; Kernel sine: compute sin(x) for x in [-pi/4, pi/4]
  ;; Uses fdlibm minimax polynomial coefficients (degree 13)
  ;; Error bound: |sin(x)/x - poly| <= 2^-58
  ;; kernel_sin(x, y, iy) - V8/fdlibm kernel sine
  ;; x is the reduced argument, y is the tail, iy indicates if y is valid (iy=1 means use y)
  (func $kernel_sin (param $x f64) (param $y f64) (param $iy i32) (result f64)
    (local $z f64) (local $v f64) (local $r f64) (local $ix i32)
    ;; fdlibm coefficients (minimax, NOT Taylor)
    ;; S1 = -1.66666666666666324348e-01  (0xBFC55555, 0x55555549)
    ;; S2 =  8.33333333332248946124e-03  (0x3F811111, 0x1110F8A6)
    ;; S3 = -1.98412698298579493134e-04  (0xBF2A01A0, 0x19C161D5)
    ;; S4 =  2.75573137070700676789e-06  (0x3EC71DE3, 0x57B1FE7D)
    ;; S5 = -2.50507602534068634195e-08  (0xBE5AE5E6, 0x8A2B9CEB)
    ;; S6 =  1.58969099521155010221e-10  (0x3DE5D93A, 0x5ACFD57C)

    (local.set $ix (i32.and (call $f64_hi (local.get $x)) (i32.const 0x7FFFFFFF)))

    ;; |x| < 2^-27: return x (with inexact)
    (if (i32.lt_u (local.get $ix) (i32.const 0x3E400000))
      (then (return (local.get $x)))
    )

    (local.set $z (f64.mul (local.get $x) (local.get $x)))
    (local.set $v (f64.mul (local.get $z) (local.get $x)))

    ;; r = S2 + z*(S3 + z*(S4 + z*(S5 + z*S6)))
    (local.set $r
      (f64.add (f64.const 8.33333333332248946124e-03)
        (f64.mul (local.get $z)
          (f64.add (f64.const -1.98412698298579493134e-04)
            (f64.mul (local.get $z)
              (f64.add (f64.const 2.75573137070700676789e-06)
                (f64.mul (local.get $z)
                  (f64.add (f64.const -2.50507602534068634195e-08)
                    (f64.mul (local.get $z) (f64.const 1.58969099521155010221e-10))))))))))

    ;; if iy == 0: return x + v*(S1 + z*r)
    ;; if iy != 0: return x - ((z*(half*y - v*r) - y) - v*S1)
    (if (i32.eqz (local.get $iy))
      (then
        (return (f64.add (local.get $x)
          (f64.mul (local.get $v)
            (f64.add (f64.const -1.66666666666666324348e-01)
              (f64.mul (local.get $z) (local.get $r))))))
      )
      (else
        (return (f64.sub (local.get $x)
          (f64.sub
            (f64.sub
              (f64.mul (local.get $z)
                (f64.sub (f64.mul (f64.const 0.5) (local.get $y))
                  (f64.mul (local.get $v) (local.get $r))))
              (local.get $y))
            (f64.mul (local.get $v) (f64.const -1.66666666666666324348e-01)))))
      )
    )
    (unreachable)
  )

  ;; kernel_cos(x, y) - V8/fdlibm kernel cosine
  ;; x is the reduced argument, y is the tail
  (func $kernel_cos (param $x f64) (param $y f64) (result f64)
    (local $z f64) (local $r f64) (local $ix i32) (local $qx f64) (local $iz f64) (local $a f64)
    ;; fdlibm coefficients (minimax, NOT Taylor)
    ;; C1 =  4.16666666666666019037e-02  (0x3FA55555, 0x5555554C)
    ;; C2 = -1.38888888888741095749e-03  (0xBF56C16C, 0x16C15177)
    ;; C3 =  2.48015872894767294178e-05  (0x3EFA01A0, 0x19CB1590)
    ;; C4 = -2.75573143513906633035e-07  (0xBE927E4F, 0x809C52AD)
    ;; C5 =  2.08757232129817482790e-09  (0x3E21EE9E, 0xBDB4B1C4)
    ;; C6 = -1.13596475577881948265e-11  (0xBDA8FAE9, 0xBE8838D4)

    (local.set $ix (i32.and (call $f64_hi (local.get $x)) (i32.const 0x7FFFFFFF)))

    ;; |x| < 2^-27: return 1
    (if (i32.lt_u (local.get $ix) (i32.const 0x3E400000))
      (then (return (f64.const 1)))
    )

    (local.set $z (f64.mul (local.get $x) (local.get $x)))
    ;; r = z*(C1 + z*(C2 + z*(C3 + z*(C4 + z*(C5 + z*C6)))))
    (local.set $r
      (f64.mul (local.get $z)
        (f64.add (f64.const 4.16666666666666019037e-02)
          (f64.mul (local.get $z)
            (f64.add (f64.const -1.38888888888741095749e-03)
              (f64.mul (local.get $z)
                (f64.add (f64.const 2.48015872894767294178e-05)
                  (f64.mul (local.get $z)
                    (f64.add (f64.const -2.75573143513906633035e-07)
                      (f64.mul (local.get $z)
                        (f64.add (f64.const 2.08757232129817482790e-09)
                          (f64.mul (local.get $z) (f64.const -1.13596475577881948265e-11)))))))))))))

    ;; if |x| < 0.3: return 1 - (0.5*z - (z*r - x*y))
    (if (i32.lt_u (local.get $ix) (i32.const 0x3FD33333))
      (then
        (return (f64.sub (f64.const 1)
          (f64.sub (f64.mul (f64.const 0.5) (local.get $z))
            (f64.sub (f64.mul (local.get $z) (local.get $r))
              (f64.mul (local.get $x) (local.get $y))))))
      )
    )

    ;; |x| >= 0.3
    (if (i32.gt_u (local.get $ix) (i32.const 0x3FE90000))
      (then
        ;; x > 0.78125: qx = 0.28125
        (local.set $qx (f64.const 0.28125))
      )
      (else
        ;; qx = x/4 (clear low 20 bits, subtract 2 from exponent)
        (local.set $qx (call $f64_from_words (i32.sub (local.get $ix) (i32.const 0x00200000)) (i32.const 0)))
      )
    )
    (local.set $iz (f64.sub (f64.mul (f64.const 0.5) (local.get $z)) (local.get $qx)))
    (local.set $a (f64.sub (f64.const 1) (local.get $qx)))
    (f64.sub (local.get $a)
      (f64.sub (local.get $iz)
        (f64.sub (f64.mul (local.get $z) (local.get $r))
          (f64.mul (local.get $x) (local.get $y)))))
  )

  ;; Range reduction: reduce x to [-pi/4, pi/4] and return quadrant
  ;; Returns n (quadrant info), stores y[0] in $rem_pio2_y0, y[1] in $rem_pio2_y1
  ;; Follows V8/fdlibm algorithm for high precision
  (func $rem_pio2 (param $x f64) (result i32)
    (local $hx i32) (local $ix i32) (local $n i32)
    (local $z f64) (local $r f64) (local $w f64) (local $t f64) (local $fn f64)
    (local $y0 f64) (local $y1 f64)
    ;; Constants from V8:
    ;; invpio2 = 6.36619772367581382433e-01  0x3FE45F30, 0x6DC9C883
    ;; pio2_1  = 1.57079632673412561417e+00  0x3FF921FB, 0x54400000
    ;; pio2_1t = 6.07710050650619224932e-11  0x3DD0B461, 0x1A626331
    ;; pio2_2  = 6.07710050630396597660e-11  0x3DD0B461, 0x1A600000
    ;; pio2_2t = 2.02226624879595063154e-21  0x3BA3198A, 0x2E037073
    ;; pio2_3  = 2.02226624871116645580e-21  0x3BA3198A, 0x2E000000
    ;; pio2_3t = 8.47842766036889956997e-32  0x397B839A, 0x252049C1

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $ix (i32.and (local.get $hx) (i32.const 0x7FFFFFFF)))

    ;; |x| ~<= pi/4, no reduction needed
    (if (i32.le_s (local.get $ix) (i32.const 0x3FE921FB))
      (then
        (global.set $rem_pio2_y0 (local.get $x))
        (global.set $rem_pio2_y1 (f64.const 0))
        (return (i32.const 0))
      )
    )

    ;; |x| < 3pi/4, special case with n=+-1
    (if (i32.lt_s (local.get $ix) (i32.const 0x4002D97C))
      (then
        (if (i32.gt_s (local.get $hx) (i32.const 0))
          (then
            ;; x > 0
            (local.set $z (f64.sub (local.get $x) (f64.const 1.57079632673412561417e+00)))
            (if (i32.ne (local.get $ix) (i32.const 0x3FF921FB))
              (then
                ;; 33+53 bit pi is good enough
                (local.set $y0 (f64.sub (local.get $z) (f64.const 6.07710050650619224932e-11)))
                (local.set $y1 (f64.sub (f64.sub (local.get $z) (local.get $y0)) (f64.const 6.07710050650619224932e-11)))
              )
              (else
                ;; near pi/2, use 33+33+53 bit pi
                (local.set $z (f64.sub (local.get $z) (f64.const 6.07710050630396597660e-11)))
                (local.set $y0 (f64.sub (local.get $z) (f64.const 2.02226624879595063154e-21)))
                (local.set $y1 (f64.sub (f64.sub (local.get $z) (local.get $y0)) (f64.const 2.02226624879595063154e-21)))
              )
            )
            (global.set $rem_pio2_y0 (local.get $y0))
            (global.set $rem_pio2_y1 (local.get $y1))
            (return (i32.const 1))
          )
          (else
            ;; x < 0
            (local.set $z (f64.add (local.get $x) (f64.const 1.57079632673412561417e+00)))
            (if (i32.ne (local.get $ix) (i32.const 0x3FF921FB))
              (then
                ;; 33+53 bit pi is good enough
                (local.set $y0 (f64.add (local.get $z) (f64.const 6.07710050650619224932e-11)))
                (local.set $y1 (f64.add (f64.sub (local.get $z) (local.get $y0)) (f64.const 6.07710050650619224932e-11)))
              )
              (else
                ;; near pi/2, use 33+33+53 bit pi
                (local.set $z (f64.add (local.get $z) (f64.const 6.07710050630396597660e-11)))
                (local.set $y0 (f64.add (local.get $z) (f64.const 2.02226624879595063154e-21)))
                (local.set $y1 (f64.add (f64.sub (local.get $z) (local.get $y0)) (f64.const 2.02226624879595063154e-21)))
              )
            )
            (global.set $rem_pio2_y0 (local.get $y0))
            (global.set $rem_pio2_y1 (local.get $y1))
            (return (i32.const -1))
          )
        )
        (unreachable)
      )
    )

    ;; |x| ~<= 2^19*(pi/2), medium size - most common case
    (if (i32.le_s (local.get $ix) (i32.const 0x413921FB))
      (then
        (local.set $t (f64.abs (local.get $x)))
        (local.set $n (i32.trunc_f64_s (f64.add (f64.mul (local.get $t) (f64.const 6.36619772367581382433e-01)) (f64.const 0.5))))
        (local.set $fn (f64.convert_i32_s (local.get $n)))
        (local.set $r (f64.sub (local.get $t) (f64.mul (local.get $fn) (f64.const 1.57079632673412561417e+00))))
        (local.set $w (f64.mul (local.get $fn) (f64.const 6.07710050650619224932e-11)))
        (local.set $y0 (f64.sub (local.get $r) (local.get $w)))
        (local.set $y1 (f64.sub (f64.sub (local.get $r) (local.get $y0)) (local.get $w)))
        ;; Check if we need more precision (cancellation check)
        ;; j - i > 16 means significant cancellation occurred
        (if (i32.gt_s
              (i32.sub
                (i32.shr_u (local.get $ix) (i32.const 20))
                (i32.and (i32.shr_u (call $f64_hi (local.get $y0)) (i32.const 20)) (i32.const 0x7FF)))
              (i32.const 16))
          (then
            ;; 2nd iteration needed, good to 118 bits
            (local.set $t (local.get $r))
            (local.set $w (f64.mul (local.get $fn) (f64.const 6.07710050630396597660e-11)))
            (local.set $r (f64.sub (local.get $t) (local.get $w)))
            (local.set $w (f64.sub (f64.mul (local.get $fn) (f64.const 2.02226624879595063154e-21)) (f64.sub (f64.sub (local.get $t) (local.get $r)) (local.get $w))))
            (local.set $y0 (f64.sub (local.get $r) (local.get $w)))
            ;; Check if 3rd iteration needed
            (if (i32.gt_s
                  (i32.sub
                    (i32.shr_u (local.get $ix) (i32.const 20))
                    (i32.and (i32.shr_u (call $f64_hi (local.get $y0)) (i32.const 20)) (i32.const 0x7FF)))
                  (i32.const 49))
              (then
                ;; 3rd iteration, 151 bits accuracy
                (local.set $t (local.get $r))
                (local.set $w (f64.mul (local.get $fn) (f64.const 2.02226624871116645580e-21)))
                (local.set $r (f64.sub (local.get $t) (local.get $w)))
                (local.set $w (f64.sub (f64.mul (local.get $fn) (f64.const 8.47842766036889956997e-32)) (f64.sub (f64.sub (local.get $t) (local.get $r)) (local.get $w))))
                (local.set $y0 (f64.sub (local.get $r) (local.get $w)))
              )
            )
            (local.set $y1 (f64.sub (f64.sub (local.get $r) (local.get $y0)) (local.get $w)))
          )
        )
        ;; Apply sign
        (if (i32.lt_s (local.get $hx) (i32.const 0))
          (then
            (global.set $rem_pio2_y0 (f64.neg (local.get $y0)))
            (global.set $rem_pio2_y1 (f64.neg (local.get $y1)))
            (return (i32.sub (i32.const 0) (local.get $n)))
          )
          (else
            (global.set $rem_pio2_y0 (local.get $y0))
            (global.set $rem_pio2_y1 (local.get $y1))
            (return (local.get $n))
          )
        )
        (unreachable)
      )
    )

    ;; |x| is inf or NaN
    (if (i32.ge_s (local.get $ix) (i32.const 0x7FF00000))
      (then
        (local.set $y0 (f64.sub (local.get $x) (local.get $x)))
        (global.set $rem_pio2_y0 (local.get $y0))
        (global.set $rem_pio2_y1 (local.get $y0))
        (return (i32.const 0))
      )
    )

    ;; For very large |x| > 2^19*(pi/2), use simpler reduction
    ;; (The full V8 uses __kernel_rem_pio2 with extended precision tables)
    ;; For now, use iterative reduction which is less precise but works
    (local.set $t (f64.abs (local.get $x)))
    (local.set $n (i32.trunc_f64_s (f64.add (f64.mul (local.get $t) (f64.const 6.36619772367581382433e-01)) (f64.const 0.5))))
    (local.set $fn (f64.convert_i32_s (local.get $n)))
    (local.set $r (f64.sub (local.get $t) (f64.mul (local.get $fn) (f64.const 1.57079632673412561417e+00))))
    (local.set $w (f64.mul (local.get $fn) (f64.const 6.07710050650619224932e-11)))
    (local.set $r (f64.sub (local.get $r) (local.get $w)))
    (local.set $w (f64.mul (local.get $fn) (f64.const 6.07710050630396597660e-11)))
    (local.set $r (f64.sub (local.get $r) (local.get $w)))
    (local.set $w (f64.mul (local.get $fn) (f64.const 2.02226624879595063154e-21)))
    (local.set $y0 (f64.sub (local.get $r) (local.get $w)))
    (local.set $y1 (f64.sub (f64.sub (local.get $r) (local.get $y0)) (local.get $w)))
    (if (i32.lt_s (local.get $hx) (i32.const 0))
      (then
        (global.set $rem_pio2_y0 (f64.neg (local.get $y0)))
        (global.set $rem_pio2_y1 (f64.neg (local.get $y1)))
        (return (i32.sub (i32.const 0) (local.get $n)))
      )
      (else
        (global.set $rem_pio2_y0 (local.get $y0))
        (global.set $rem_pio2_y1 (local.get $y1))
        (return (local.get $n))
      )
    )
    (unreachable)
  )

  ;; Globals to store rem_pio2 results (y[0] and y[1])
  (global $rem_pio2_y0 (mut f64) (f64.const 0))
  (global $rem_pio2_y1 (mut f64) (f64.const 0))
  ;; Keep old name for compatibility during transition
  (global $rem_pio2_result (mut f64) (f64.const 0))

  ;; ==========================================================================
  ;; expm1(x) = exp(x) - 1 using V8/fdlibm algorithm
  ;; More accurate than exp(x) - 1 for small x
  ;; ==========================================================================
  (func $f64_expm1 (param $x f64) (result f64)
    (local $hx i32) (local $xsb i32) (local $k i32)
    (local $y f64) (local $hi f64) (local $lo f64) (local $c f64)
    (local $t f64) (local $e f64) (local $hxs f64) (local $hfx f64) (local $r1 f64) (local $twopk f64)
    ;; Constants from V8:
    ;; o_threshold = 7.09782712893383973096e+02  0x40862E42, 0xFEFA39EF
    ;; ln2_hi = 6.93147180369123816490e-01      0x3FE62E42, 0xFEE00000
    ;; ln2_lo = 1.90821492927058770002e-10      0x3DEA39EF, 0x35793C76
    ;; invln2 = 1.44269504088896338700e+00      0x3FF71547, 0x652B82FE
    ;; Q1-Q5 coefficients

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $xsb (i32.and (local.get $hx) (i32.const 0x80000000)))
    (local.set $hx (i32.and (local.get $hx) (i32.const 0x7FFFFFFF)))

    ;; Filter out huge and non-finite argument
    (if (i32.ge_u (local.get $hx) (i32.const 0x4043687A))
      (then
        (if (i32.ge_u (local.get $hx) (i32.const 0x40862E42))
          (then
            ;; inf or NaN
            (if (i32.ge_u (local.get $hx) (i32.const 0x7FF00000))
              (then
                (if (i32.ne (i32.or (i32.and (local.get $hx) (i32.const 0xFFFFF)) (call $f64_lo (local.get $x))) (i32.const 0))
                  (then (return (f64.add (local.get $x) (local.get $x)))) ;; NaN
                )
                (if (i32.eqz (local.get $xsb))
                  (then (return (local.get $x))) ;; +inf
                  (else (return (f64.const -1))) ;; -inf -> -1
                )
              )
            )
            ;; overflow
            (if (f64.gt (local.get $x) (f64.const 7.09782712893383973096e+02))
              (then (return (f64.mul (f64.const 1e300) (f64.const 1e300))))
            )
          )
        )
        ;; x < -56*ln2: return -1 with inexact
        (if (i32.ne (local.get $xsb) (i32.const 0))
          (then (return (f64.sub (f64.const 1e-300) (f64.const 1))))
        )
      )
    )

    ;; Argument reduction
    (if (i32.gt_u (local.get $hx) (i32.const 0x3FD62E42))
      (then
        (if (i32.lt_u (local.get $hx) (i32.const 0x3FF0A2B2))
          (then
            ;; |x| < 1.5*ln2
            (if (i32.eqz (local.get $xsb))
              (then
                (local.set $hi (f64.sub (local.get $x) (f64.const 6.93147180369123816490e-01)))
                (local.set $lo (f64.const 1.90821492927058770002e-10))
                (local.set $k (i32.const 1))
              )
              (else
                (local.set $hi (f64.add (local.get $x) (f64.const 6.93147180369123816490e-01)))
                (local.set $lo (f64.const -1.90821492927058770002e-10))
                (local.set $k (i32.const -1))
              )
            )
          )
          (else
            ;; |x| >= 1.5*ln2
            (if (i32.eqz (local.get $xsb))
              (then (local.set $k (i32.trunc_f64_s (f64.add (f64.mul (f64.const 1.44269504088896338700e+00) (local.get $x)) (f64.const 0.5)))))
              (else (local.set $k (i32.trunc_f64_s (f64.sub (f64.mul (f64.const 1.44269504088896338700e+00) (local.get $x)) (f64.const 0.5)))))
            )
            (local.set $t (f64.convert_i32_s (local.get $k)))
            (local.set $hi (f64.sub (local.get $x) (f64.mul (local.get $t) (f64.const 6.93147180369123816490e-01))))
            (local.set $lo (f64.mul (local.get $t) (f64.const 1.90821492927058770002e-10)))
          )
        )
        (local.set $x (f64.sub (local.get $hi) (local.get $lo)))
        (local.set $c (f64.sub (f64.sub (local.get $hi) (local.get $x)) (local.get $lo)))
      )
      (else
        ;; |x| < 2**-54: return x
        (if (i32.lt_u (local.get $hx) (i32.const 0x3C900000))
          (then (return (local.get $x)))
        )
        (local.set $k (i32.const 0))
        (local.set $c (f64.const 0))
      )
    )

    ;; x is now in primary range
    (local.set $hfx (f64.mul (f64.const 0.5) (local.get $x)))
    (local.set $hxs (f64.mul (local.get $x) (local.get $hfx)))
    ;; r1 = 1 + hxs*(Q1 + hxs*(Q2 + hxs*(Q3 + hxs*(Q4 + hxs*Q5))))
    (local.set $r1 (f64.add (f64.const 1)
      (f64.mul (local.get $hxs)
        (f64.add (f64.const -3.33333333333331316428e-02)
          (f64.mul (local.get $hxs)
            (f64.add (f64.const 1.58730158725481460165e-03)
              (f64.mul (local.get $hxs)
                (f64.add (f64.const -7.93650757867487942473e-05)
                  (f64.mul (local.get $hxs)
                    (f64.add (f64.const 4.00821782732936239552e-06)
                      (f64.mul (local.get $hxs) (f64.const -2.01099218183624371326e-07))))))))))))
    (local.set $t (f64.sub (f64.const 3.0) (f64.mul (local.get $r1) (local.get $hfx))))
    (local.set $e (f64.mul (local.get $hxs) (f64.div (f64.sub (local.get $r1) (local.get $t)) (f64.sub (f64.const 6.0) (f64.mul (local.get $x) (local.get $t))))))

    (if (i32.eqz (local.get $k))
      (then (return (f64.sub (local.get $x) (f64.sub (f64.mul (local.get $x) (local.get $e)) (local.get $hxs)))))
    )

    ;; 2^k
    (local.set $twopk (call $f64_from_words
      (i32.add (i32.const 0x3FF00000) (i32.shl (local.get $k) (i32.const 20)))
      (i32.const 0)))
    (local.set $e (f64.sub (f64.mul (local.get $x) (f64.sub (local.get $e) (local.get $c))) (local.get $c)))
    (local.set $e (f64.sub (local.get $e) (local.get $hxs)))

    (if (i32.eq (local.get $k) (i32.const -1))
      (then (return (f64.sub (f64.mul (f64.const 0.5) (f64.sub (local.get $x) (local.get $e))) (f64.const 0.5))))
    )
    (if (i32.eq (local.get $k) (i32.const 1))
      (then
        (if (f64.lt (local.get $x) (f64.const -0.25))
          (then (return (f64.mul (f64.const -2.0) (f64.sub (local.get $e) (f64.add (local.get $x) (f64.const 0.5))))))
          (else (return (f64.add (f64.const 1) (f64.mul (f64.const 2.0) (f64.sub (local.get $x) (local.get $e))))))
        )
      )
    )

    ;; k <= -2 or k > 56: return exp(x) - 1
    (if (i32.or (i32.le_s (local.get $k) (i32.const -2)) (i32.gt_s (local.get $k) (i32.const 56)))
      (then
        (local.set $y (f64.sub (f64.const 1) (f64.sub (local.get $e) (local.get $x))))
        (if (i32.eq (local.get $k) (i32.const 1024))
          (then (local.set $y (f64.mul (f64.mul (local.get $y) (f64.const 2.0)) (f64.const 8.98846567431158e+307))))
          (else (local.set $y (f64.mul (local.get $y) (local.get $twopk))))
        )
        (return (f64.sub (local.get $y) (f64.const 1)))
      )
    )

    (if (i32.lt_s (local.get $k) (i32.const 20))
      (then
        ;; t = 1 - 2^-k
        (local.set $t (call $f64_from_words
          (i32.sub (i32.const 0x3FF00000) (i32.shr_u (i32.const 0x200000) (local.get $k)))
          (i32.const 0)))
        (local.set $y (f64.sub (local.get $t) (f64.sub (local.get $e) (local.get $x))))
        (local.set $y (f64.mul (local.get $y) (local.get $twopk)))
      )
      (else
        ;; t = 2^-k
        (local.set $t (call $f64_from_words
          (i32.shl (i32.sub (i32.const 0x3FF) (local.get $k)) (i32.const 20))
          (i32.const 0)))
        (local.set $y (f64.add (f64.sub (local.get $x) (f64.add (local.get $e) (local.get $t))) (f64.const 1)))
        (local.set $y (f64.mul (local.get $y) (local.get $twopk)))
      )
    )
    (local.get $y)
  )

  ;; ==========================================================================
  ;; tanh(x) using V8/fdlibm algorithm with expm1
  ;; ==========================================================================
  (func $f64_tanh (param $x f64) (result f64)
    (local $jx i32) (local $ix i32) (local $t f64) (local $z f64)

    (local.set $jx (call $f64_hi (local.get $x)))
    (local.set $ix (i32.and (local.get $jx) (i32.const 0x7FFFFFFF)))

    ;; |x| >= INF
    (if (i32.ge_u (local.get $ix) (i32.const 0x7FF00000))
      (then
        (if (i32.ge_s (local.get $jx) (i32.const 0))
          (then (return (f64.add (f64.div (f64.const 1) (local.get $x)) (f64.const 1)))) ;; tanh(+inf) = 1
          (else (return (f64.sub (f64.div (f64.const 1) (local.get $x)) (f64.const 1)))) ;; tanh(-inf) = -1, tanh(NaN) = NaN
        )
      )
    )

    ;; |x| < 22
    (if (i32.lt_u (local.get $ix) (i32.const 0x40360000))
      (then
        ;; |x| < 2**-28: return x with inexact
        (if (i32.lt_u (local.get $ix) (i32.const 0x3E300000))
          (then (return (local.get $x)))
        )
        ;; |x| >= 1
        (if (i32.ge_u (local.get $ix) (i32.const 0x3FF00000))
          (then
            (local.set $t (call $f64_expm1 (f64.mul (f64.const 2) (f64.abs (local.get $x)))))
            (local.set $z (f64.sub (f64.const 1) (f64.div (f64.const 2) (f64.add (local.get $t) (f64.const 2)))))
          )
          (else
            ;; |x| < 1
            (local.set $t (call $f64_expm1 (f64.neg (f64.mul (f64.const 2) (f64.abs (local.get $x))))))
            (local.set $z (f64.div (f64.neg (local.get $t)) (f64.add (local.get $t) (f64.const 2))))
          )
        )
      )
      (else
        ;; |x| >= 22: return +-1
        (local.set $z (f64.sub (f64.const 1) (f64.const 1e-300)))
      )
    )

    (if (i32.ge_s (local.get $jx) (i32.const 0))
      (then (return (local.get $z)))
      (else (return (f64.neg (local.get $z))))
    )
    (unreachable)
  )

  ;; ==========================================================================
  ;; sinh(x) using V8 algorithm with expm1
  ;; ==========================================================================
  (func $f64_sinh (param $x f64) (result f64)
    (local $h f64) (local $ax f64) (local $t f64) (local $w f64)
    ;; Constants from V8:
    ;; TWO_M28 = 3.725290298461914e-9  (2^-28)
    ;; LOG_MAXD = 709.7822265625
    ;; KSINH_OVERFLOW = 710.4758600739439

    (local.set $h (select (f64.const -0.5) (f64.const 0.5) (f64.lt (local.get $x) (f64.const 0))))
    (local.set $ax (f64.abs (local.get $x)))

    ;; |x| < 22
    (if (f64.lt (local.get $ax) (f64.const 22))
      (then
        ;; |x| < 2^-28: return x
        (if (f64.lt (local.get $ax) (f64.const 3.725290298461914e-9))
          (then (return (local.get $x)))
        )
        (local.set $t (call $f64_expm1 (local.get $ax)))
        (if (f64.lt (local.get $ax) (f64.const 1))
          (then
            ;; h * (2*t - t*t/(t+1))
            (return (f64.mul (local.get $h)
              (f64.sub
                (f64.mul (f64.const 2) (local.get $t))
                (f64.div (f64.mul (local.get $t) (local.get $t)) (f64.add (local.get $t) (f64.const 1))))))
          )
          (else
            ;; h * (t + t/(t+1))
            (return (f64.mul (local.get $h)
              (f64.add (local.get $t) (f64.div (local.get $t) (f64.add (local.get $t) (f64.const 1))))))
          )
        )
      )
    )

    ;; |x| in [22, log(maxdouble)]
    (if (f64.lt (local.get $ax) (f64.const 709.7822265625))
      (then (return (f64.mul (local.get $h) (call $f64_exp (local.get $ax)))))
    )

    ;; |x| in [log(maxdouble), overflowthreshold]
    (if (f64.le (local.get $ax) (f64.const 710.4758600739439))
      (then
        (local.set $w (call $f64_exp (f64.mul (f64.const 0.5) (local.get $ax))))
        (local.set $t (f64.mul (local.get $h) (local.get $w)))
        (return (f64.mul (local.get $t) (local.get $w)))
      )
    )

    ;; overflow or NaN
    (f64.mul (local.get $x) (f64.const 1e307))
  )

  ;; ==========================================================================
  ;; asinh(x) using V8 algorithm with log1p
  ;; ==========================================================================
  (func $f64_asinh (param $x f64) (result f64)
    (local $hx i32) (local $ix i32) (local $t f64) (local $w f64)
    ;; Constants from V8:
    ;; ln2 = 6.93147180559945286227e-01

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $ix (i32.and (local.get $hx) (i32.const 0x7FFFFFFF)))

    ;; inf or NaN
    (if (i32.ge_u (local.get $ix) (i32.const 0x7FF00000))
      (then (return (f64.add (local.get $x) (local.get $x))))
    )

    ;; |x| < 2^-28: return x with inexact
    (if (i32.lt_u (local.get $ix) (i32.const 0x3E300000))
      (then (return (local.get $x)))
    )

    ;; |x| > 2^28
    (if (i32.gt_u (local.get $ix) (i32.const 0x41B00000))
      (then
        (local.set $w (f64.add (call $f64_log (f64.abs (local.get $x))) (f64.const 6.93147180559945286227e-01)))
      )
      (else
        ;; 2^28 > |x| > 2.0
        (if (i32.gt_u (local.get $ix) (i32.const 0x40000000))
          (then
            (local.set $t (f64.abs (local.get $x)))
            (local.set $w (call $f64_log (f64.add
              (f64.mul (f64.const 2) (local.get $t))
              (f64.div (f64.const 1) (f64.add (f64.sqrt (f64.add (f64.mul (local.get $x) (local.get $x)) (f64.const 1))) (local.get $t))))))
          )
          (else
            ;; 2.0 > |x| > 2^-28
            (local.set $t (f64.mul (local.get $x) (local.get $x)))
            (local.set $w (call $f64_log1p (f64.add
              (f64.abs (local.get $x))
              (f64.div (local.get $t) (f64.add (f64.const 1) (f64.sqrt (f64.add (f64.const 1) (local.get $t))))))))
          )
        )
      )
    )

    (if (i32.gt_s (local.get $hx) (i32.const 0))
      (then (return (local.get $w)))
      (else (return (f64.neg (local.get $w))))
    )
    (unreachable)
  )

  ;; ==========================================================================
  ;; kernel_tan(x, y, iy) - V8/fdlibm kernel tangent on [-pi/4, pi/4]
  ;; iy=1: return tan(x+y), iy=-1: return -1/tan(x+y)
  ;; ==========================================================================
  (func $kernel_tan (param $x f64) (param $y f64) (param $iy i32) (result f64)
    (local $hx i32) (local $ix i32) (local $low i32)
    (local $z f64) (local $r f64) (local $v f64) (local $w f64) (local $s f64)
    (local $a f64) (local $t f64)
    ;; T0-T12 coefficients from V8
    ;; T0  = 3.33333333333334091986e-01   0x3FD55555, 0x55555563
    ;; T1  = 1.33333333333201242699e-01   0x3FC11111, 0x1110FE7A
    ;; T2  = 5.39682539762260521377e-02   0x3FABA1BA, 0x1BB341FE
    ;; T3  = 2.18694882948595424599e-02   0x3F9664F4, 0x8406D637
    ;; T4  = 8.86323982359930005737e-03   0x3F8226E3, 0xE96E8493
    ;; T5  = 3.59207910759131235356e-03   0x3F6D6D22, 0xC9560328
    ;; T6  = 1.45620945432529025516e-03   0x3F57DBC8, 0xFEE08315
    ;; T7  = 5.88041240820264096874e-04   0x3F4344D8, 0xF2F26501
    ;; T8  = 2.46463134818469906812e-04   0x3F3026F7, 0x1A8D1068
    ;; T9  = 7.81794442939557092300e-05   0x3F147E88, 0xA03792A6
    ;; T10 = 7.14072491382608190305e-05   0x3F12B80F, 0x32F0A7E9
    ;; T11 = -1.85586374855275456654e-05  0xBEF375CB, 0xDB605373
    ;; T12 = 2.59073051863633712884e-05   0x3EFB2A70, 0x74BF7AD4
    ;; pio4    = 7.85398163397448278999e-01  0x3FE921FB, 0x54442D18
    ;; pio4lo  = 3.06161699786838301793e-17  0x3C81A626, 0x33145C07

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $ix (i32.and (local.get $hx) (i32.const 0x7FFFFFFF)))

    ;; |x| < 2^-28
    (if (i32.lt_u (local.get $ix) (i32.const 0x3E300000))
      (then
        (local.set $low (call $f64_lo (local.get $x)))
        (if (i32.eqz (i32.or (i32.or (local.get $ix) (local.get $low)) (i32.add (local.get $iy) (i32.const 1))))
          (then (return (f64.div (f64.const 1) (f64.abs (local.get $x)))))
        )
        (if (i32.eq (local.get $iy) (i32.const 1))
          (then (return (local.get $x)))
          (else
            ;; compute -1/(x+y) carefully
            (local.set $z (f64.add (local.get $x) (local.get $y)))
            (local.set $w (local.get $z))
            (local.set $z (call $f64_trunc_lo (local.get $z)))
            (local.set $v (f64.sub (local.get $y) (f64.sub (local.get $z) (local.get $x))))
            (local.set $a (f64.div (f64.const -1) (local.get $w)))
            (local.set $t (call $f64_trunc_lo (local.get $a)))
            (local.set $s (f64.add (f64.const 1) (f64.mul (local.get $t) (local.get $z))))
            (return (f64.add (local.get $t) (f64.mul (local.get $a) (f64.add (local.get $s) (f64.mul (local.get $t) (local.get $v))))))
          )
        )
      )
    )

    ;; |x| >= 0.6744
    (if (i32.ge_u (local.get $ix) (i32.const 0x3FE59428))
      (then
        (if (i32.lt_s (local.get $hx) (i32.const 0))
          (then
            (local.set $x (f64.neg (local.get $x)))
            (local.set $y (f64.neg (local.get $y)))
          )
        )
        (local.set $z (f64.sub (f64.const 7.85398163397448278999e-01) (local.get $x)))
        (local.set $w (f64.sub (f64.const 3.06161699786838301793e-17) (local.get $y)))
        (local.set $x (f64.add (local.get $z) (local.get $w)))
        (local.set $y (f64.const 0))
      )
    )

    (local.set $z (f64.mul (local.get $x) (local.get $x)))
    (local.set $w (f64.mul (local.get $z) (local.get $z)))

    ;; r = T1 + w*(T3 + w*(T5 + w*(T7 + w*(T9 + w*T11))))
    (local.set $r (f64.add (f64.const 1.33333333333201242699e-01)
      (f64.mul (local.get $w)
        (f64.add (f64.const 2.18694882948595424599e-02)
          (f64.mul (local.get $w)
            (f64.add (f64.const 3.59207910759131235356e-03)
              (f64.mul (local.get $w)
                (f64.add (f64.const 5.88041240820264096874e-04)
                  (f64.mul (local.get $w)
                    (f64.add (f64.const 7.81794442939557092300e-05)
                      (f64.mul (local.get $w) (f64.const -1.85586374855275456654e-05))))))))))))

    ;; v = z*(T2 + w*(T4 + w*(T6 + w*(T8 + w*(T10 + w*T12)))))
    (local.set $v (f64.mul (local.get $z)
      (f64.add (f64.const 5.39682539762260521377e-02)
        (f64.mul (local.get $w)
          (f64.add (f64.const 8.86323982359930005737e-03)
            (f64.mul (local.get $w)
              (f64.add (f64.const 1.45620945432529025516e-03)
                (f64.mul (local.get $w)
                  (f64.add (f64.const 2.46463134818469906812e-04)
                    (f64.mul (local.get $w)
                      (f64.add (f64.const 7.14072491382608190305e-05)
                        (f64.mul (local.get $w) (f64.const 2.59073051863633712884e-05)))))))))))))

    (local.set $s (f64.mul (local.get $z) (local.get $x)))
    (local.set $r (f64.add (local.get $y) (f64.mul (local.get $z) (f64.add (f64.mul (local.get $s) (f64.add (local.get $r) (local.get $v))) (local.get $y)))))
    (local.set $r (f64.add (local.get $r) (f64.mul (f64.const 3.33333333333334091986e-01) (local.get $s))))
    (local.set $w (f64.add (local.get $x) (local.get $r)))

    ;; |x| >= 0.6744: special formula
    (if (i32.ge_u (local.get $ix) (i32.const 0x3FE59428))
      (then
        (local.set $v (f64.convert_i32_s (local.get $iy)))
        (return (f64.mul
          (f64.convert_i32_s (i32.sub (i32.const 1) (i32.and (i32.shr_s (local.get $hx) (i32.const 30)) (i32.const 2))))
          (f64.sub (local.get $v)
            (f64.mul (f64.const 2)
              (f64.sub (local.get $x)
                (f64.sub (f64.div (f64.mul (local.get $w) (local.get $w)) (f64.add (local.get $w) (local.get $v))) (local.get $r)))))))
      )
    )

    (if (i32.eq (local.get $iy) (i32.const 1))
      (then (return (local.get $w)))
    )

    ;; compute -1/(x+r) accurately
    (local.set $z (call $f64_trunc_lo (local.get $w)))
    (local.set $v (f64.sub (local.get $r) (f64.sub (local.get $z) (local.get $x))))
    (local.set $a (f64.div (f64.const -1) (local.get $w)))
    (local.set $t (call $f64_trunc_lo (local.get $a)))
    (local.set $s (f64.add (f64.const 1) (f64.mul (local.get $t) (local.get $z))))
    (f64.add (local.get $t) (f64.mul (local.get $a) (f64.add (local.get $s) (f64.mul (local.get $t) (local.get $v)))))
  )

  ;; ==========================================================================
  ;; tan(x) using V8 algorithm with kernel_tan and rem_pio2
  ;; ==========================================================================
  (func $f64_tan (param $x f64) (result f64)
    (local $ix i32) (local $n i32) (local $y0 f64) (local $y1 f64)

    (local.set $ix (i32.and (call $f64_hi (local.get $x)) (i32.const 0x7FFFFFFF)))

    ;; |x| ~< pi/4
    (if (i32.le_u (local.get $ix) (i32.const 0x3FE921FB))
      (then (return (call $kernel_tan (local.get $x) (f64.const 0) (i32.const 1))))
    )

    ;; tan(Inf or NaN) is NaN
    (if (i32.ge_u (local.get $ix) (i32.const 0x7FF00000))
      (then (return (f64.sub (local.get $x) (local.get $x))))
    )

    ;; argument reduction needed
    (local.set $n (call $rem_pio2 (local.get $x)))
    (local.set $y0 (global.get $rem_pio2_y0))
    (local.set $y1 (global.get $rem_pio2_y1))

    ;; 1 -> n even, -1 -> n odd
    (call $kernel_tan (local.get $y0) (local.get $y1) (i32.sub (i32.const 1) (i32.shl (i32.and (local.get $n) (i32.const 1)) (i32.const 1))))
  )

  ;; ==========================================================================
  ;; cbrt(x) using V8/fdlibm algorithm - cube root
  ;; ==========================================================================
  (func $f64_cbrt (param $x f64) (result f64)
    (local $hx i32) (local $sign i32) (local $high i32) (local $low i32)
    (local $r f64) (local $s f64) (local $t f64) (local $w f64)
    (local $bits i64)
    ;; Constants from V8:
    ;; B1 = 715094163  (1023-1023/3-0.03306235651)*2**20
    ;; B2 = 696219795  (1023-1023/3-54/3-0.03306235651)*2**20
    ;; P0-P4 polynomial coefficients for 1/cbrt(r) approximation

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $low (call $f64_lo (local.get $x)))
    (local.set $sign (i32.and (local.get $hx) (i32.const 0x80000000)))
    (local.set $hx (i32.xor (local.get $hx) (local.get $sign)))

    ;; cbrt(NaN, INF) is itself
    (if (i32.ge_u (local.get $hx) (i32.const 0x7FF00000))
      (then (return (f64.add (local.get $x) (local.get $x))))
    )

    ;; cbrt(0) is itself
    (if (i32.eqz (i32.or (local.get $hx) (local.get $low)))
      (then (return (local.get $x)))
    )

    ;; Rough cbrt to 5 bits
    (if (i32.lt_u (local.get $hx) (i32.const 0x00100000))
      (then
        ;; subnormal: scale up
        (local.set $t (call $f64_from_words (i32.const 0x43500000) (i32.const 0)))
        (local.set $t (f64.mul (local.get $t) (local.get $x)))
        (local.set $high (call $f64_hi (local.get $t)))
        (local.set $t (call $f64_from_words
          (i32.or (local.get $sign) (i32.add (i32.div_u (i32.and (local.get $high) (i32.const 0x7FFFFFFF)) (i32.const 3)) (i32.const 696219795)))
          (i32.const 0)))
      )
      (else
        (local.set $t (call $f64_from_words
          (i32.or (local.get $sign) (i32.add (i32.div_u (local.get $hx) (i32.const 3)) (i32.const 715094163)))
          (i32.const 0)))
      )
    )

    ;; New cbrt to 23 bits using polynomial P(r) where r = t^3/x
    (local.set $r (f64.mul (f64.mul (local.get $t) (local.get $t)) (f64.div (local.get $t) (local.get $x))))
    ;; t = t * ((P0 + r*(P1 + r*P2)) + (r*r*r)*(P3 + r*P4))
    (local.set $t (f64.mul (local.get $t)
      (f64.add
        (f64.add (f64.const 1.87595182427177009643)
          (f64.mul (local.get $r)
            (f64.add (f64.const -1.88497979543377169875)
              (f64.mul (local.get $r) (f64.const 1.621429720105354466140)))))
        (f64.mul (f64.mul (f64.mul (local.get $r) (local.get $r)) (local.get $r))
          (f64.add (f64.const -0.758397934778766047437)
            (f64.mul (local.get $r) (f64.const 0.145996192886612446982)))))))

    ;; Round t away from zero to 23 bits
    (local.set $bits (i64.reinterpret_f64 (local.get $t)))
    (local.set $bits (i64.and
      (i64.add (local.get $bits) (i64.const 0x80000000))
      (i64.const 0xFFFFFFFFC0000000)))
    (local.set $t (f64.reinterpret_i64 (local.get $bits)))

    ;; One step Newton iteration to 53 bits
    (local.set $s (f64.mul (local.get $t) (local.get $t)))
    (local.set $r (f64.div (local.get $x) (local.get $s)))
    (local.set $w (f64.add (local.get $t) (local.get $t)))
    (local.set $r (f64.div (f64.sub (local.get $r) (local.get $t)) (f64.add (local.get $w) (local.get $r))))
    (f64.add (local.get $t) (f64.mul (local.get $t) (local.get $r)))
  )

  ;; ==========================================================================
  ;; Sine: full implementation with range reduction
  ;; ==========================================================================
  (func $f64_sin (param $x f64) (result f64)
    (local $n i32) (local $y0 f64) (local $y1 f64) (local $ix i32)

    (local.set $ix (i32.and (call $f64_hi (local.get $x)) (i32.const 0x7FFFFFFF)))

    ;; |x| ~< pi/4, no reduction needed
    (if (i32.le_s (local.get $ix) (i32.const 0x3FE921FB))
      (then (return (call $kernel_sin (local.get $x) (f64.const 0) (i32.const 0))))
    )

    ;; sin(Inf or NaN) is NaN
    (if (i32.ge_s (local.get $ix) (i32.const 0x7FF00000))
      (then (return (f64.sub (local.get $x) (local.get $x))))
    )

    ;; Range reduction
    (local.set $n (call $rem_pio2 (local.get $x)))
    (local.set $y0 (global.get $rem_pio2_y0))
    (local.set $y1 (global.get $rem_pio2_y1))

    ;; Dispatch based on n&3
    ;; 0: sin(y), 1: cos(y), 2: -sin(y), 3: -cos(y)
    (if (i32.eq (i32.and (local.get $n) (i32.const 3)) (i32.const 0))
      (then (return (call $kernel_sin (local.get $y0) (local.get $y1) (i32.const 1))))
    )
    (if (i32.eq (i32.and (local.get $n) (i32.const 3)) (i32.const 1))
      (then (return (call $kernel_cos (local.get $y0) (local.get $y1))))
    )
    (if (i32.eq (i32.and (local.get $n) (i32.const 3)) (i32.const 2))
      (then (return (f64.neg (call $kernel_sin (local.get $y0) (local.get $y1) (i32.const 1)))))
    )
    ;; n&3 == 3
    (f64.neg (call $kernel_cos (local.get $y0) (local.get $y1)))
  )

  ;; ==========================================================================
  ;; Cosine: full implementation with range reduction
  ;; ==========================================================================
  (func $f64_cos (param $x f64) (result f64)
    (local $n i32) (local $y0 f64) (local $y1 f64) (local $ix i32)

    (local.set $ix (i32.and (call $f64_hi (local.get $x)) (i32.const 0x7FFFFFFF)))

    ;; |x| ~< pi/4, no reduction needed
    (if (i32.le_s (local.get $ix) (i32.const 0x3FE921FB))
      (then (return (call $kernel_cos (local.get $x) (f64.const 0))))
    )

    ;; cos(Inf or NaN) is NaN
    (if (i32.ge_s (local.get $ix) (i32.const 0x7FF00000))
      (then (return (f64.sub (local.get $x) (local.get $x))))
    )

    ;; Range reduction
    (local.set $n (call $rem_pio2 (local.get $x)))
    (local.set $y0 (global.get $rem_pio2_y0))
    (local.set $y1 (global.get $rem_pio2_y1))

    ;; Dispatch based on n&3
    ;; 0: cos(y), 1: -sin(y), 2: -cos(y), 3: sin(y)
    (if (i32.eq (i32.and (local.get $n) (i32.const 3)) (i32.const 0))
      (then (return (call $kernel_cos (local.get $y0) (local.get $y1))))
    )
    (if (i32.eq (i32.and (local.get $n) (i32.const 3)) (i32.const 1))
      (then (return (f64.neg (call $kernel_sin (local.get $y0) (local.get $y1) (i32.const 1)))))
    )
    (if (i32.eq (i32.and (local.get $n) (i32.const 3)) (i32.const 2))
      (then (return (f64.neg (call $kernel_cos (local.get $y0) (local.get $y1)))))
    )
    ;; n&3 == 3
    (call $kernel_sin (local.get $y0) (local.get $y1) (i32.const 1))
  )

  ;; ==========================================================================
  ;; Arcsine using V8/fdlibm algorithm (exact match to ieee754.cc)
  ;; ==========================================================================
  (func $f64_asin (param $x f64) (result f64)
    (local $hx i32) (local $ix i32)
    (local $t f64) (local $w f64) (local $p f64) (local $q f64) (local $c f64) (local $r f64) (local $s f64)
    ;; Constants
    ;; pio2_hi = 1.57079632679489655800e+00  0x3FF921FB, 0x54442D18
    ;; pio2_lo = 6.12323399573676603587e-17  0x3C91A626, 0x33145C07
    ;; pio4_hi = 7.85398163397448278999e-01  0x3FE921FB, 0x54442D18
    ;; pS0-pS5, qS1-qS4 coefficients for R(x^2)

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $ix (i32.and (local.get $hx) (i32.const 0x7FFFFFFF)))

    ;; |x| >= 1
    (if (i32.ge_u (local.get $ix) (i32.const 0x3FF00000))
      (then
        ;; asin(1) = +-pi/2 with inexact
        (if (i32.eq (i32.or (i32.sub (local.get $ix) (i32.const 0x3FF00000)) (call $f64_lo (local.get $x))) (i32.const 0))
          (then
            (return (f64.add
              (f64.mul (local.get $x) (f64.const 1.57079632679489655800e+00))
              (f64.mul (local.get $x) (f64.const 6.12323399573676603587e-17))))
          )
        )
        ;; asin(|x|>1) is NaN
        (return (f64.div (f64.const 0) (f64.const 0)))
      )
    )

    ;; |x| < 0.5
    (if (i32.lt_u (local.get $ix) (i32.const 0x3FE00000))
      (then
        ;; |x| < 2^-27: return x
        (if (i32.lt_u (local.get $ix) (i32.const 0x3E400000))
          (then (return (local.get $x)))
        )
        (local.set $t (f64.mul (local.get $x) (local.get $x)))
        ;; p = t*(pS0 + t*(pS1 + t*(pS2 + t*(pS3 + t*(pS4 + t*pS5)))))
        (local.set $p (f64.mul (local.get $t)
          (f64.add (f64.const 1.66666666666666657415e-01)
            (f64.mul (local.get $t)
              (f64.add (f64.const -3.25565818622400915405e-01)
                (f64.mul (local.get $t)
                  (f64.add (f64.const 2.01212532134862925881e-01)
                    (f64.mul (local.get $t)
                      (f64.add (f64.const -4.00555345006794114027e-02)
                        (f64.mul (local.get $t)
                          (f64.add (f64.const 7.91534994289814532176e-04)
                            (f64.mul (local.get $t) (f64.const 3.47933107596021167570e-05)))))))))))))
        ;; q = 1 + t*(qS1 + t*(qS2 + t*(qS3 + t*qS4)))
        (local.set $q (f64.add (f64.const 1)
          (f64.mul (local.get $t)
            (f64.add (f64.const -2.40339491173441421878e+00)
              (f64.mul (local.get $t)
                (f64.add (f64.const 2.02094576023350569471e+00)
                  (f64.mul (local.get $t)
                    (f64.add (f64.const -6.88283971605453293030e-01)
                      (f64.mul (local.get $t) (f64.const 7.70381505559019352791e-02))))))))))
        (local.set $w (f64.div (local.get $p) (local.get $q)))
        (return (f64.add (local.get $x) (f64.mul (local.get $x) (local.get $w))))
      )
    )

    ;; 1 > |x| >= 0.5
    (local.set $w (f64.sub (f64.const 1) (f64.abs (local.get $x))))
    (local.set $t (f64.mul (local.get $w) (f64.const 0.5)))
    ;; p = t*(pS0 + t*(pS1 + t*(pS2 + t*(pS3 + t*(pS4 + t*pS5)))))
    (local.set $p (f64.mul (local.get $t)
      (f64.add (f64.const 1.66666666666666657415e-01)
        (f64.mul (local.get $t)
          (f64.add (f64.const -3.25565818622400915405e-01)
            (f64.mul (local.get $t)
              (f64.add (f64.const 2.01212532134862925881e-01)
                (f64.mul (local.get $t)
                  (f64.add (f64.const -4.00555345006794114027e-02)
                    (f64.mul (local.get $t)
                      (f64.add (f64.const 7.91534994289814532176e-04)
                        (f64.mul (local.get $t) (f64.const 3.47933107596021167570e-05)))))))))))))
    ;; q = 1 + t*(qS1 + t*(qS2 + t*(qS3 + t*qS4)))
    (local.set $q (f64.add (f64.const 1)
      (f64.mul (local.get $t)
        (f64.add (f64.const -2.40339491173441421878e+00)
          (f64.mul (local.get $t)
            (f64.add (f64.const 2.02094576023350569471e+00)
              (f64.mul (local.get $t)
                (f64.add (f64.const -6.88283971605453293030e-01)
                  (f64.mul (local.get $t) (f64.const 7.70381505559019352791e-02))))))))))
    (local.set $s (f64.sqrt (local.get $t)))

    ;; if |x| > 0.975 (ix >= 0x3FEF3333)
    (if (i32.ge_u (local.get $ix) (i32.const 0x3FEF3333))
      (then
        (local.set $w (f64.div (local.get $p) (local.get $q)))
        (local.set $t (f64.sub
          (f64.const 1.57079632679489655800e+00)
          (f64.sub
            (f64.mul (f64.const 2.0) (f64.add (local.get $s) (f64.mul (local.get $s) (local.get $w))))
            (f64.const 6.12323399573676603587e-17))))
      )
      (else
        ;; w = s with low word cleared
        (local.set $w (call $f64_trunc_lo (local.get $s)))
        (local.set $c (f64.div
          (f64.sub (local.get $t) (f64.mul (local.get $w) (local.get $w)))
          (f64.add (local.get $s) (local.get $w))))
        (local.set $r (f64.div (local.get $p) (local.get $q)))
        (local.set $p (f64.sub
          (f64.mul (f64.const 2.0) (f64.mul (local.get $s) (local.get $r)))
          (f64.sub (f64.const 6.12323399573676603587e-17) (f64.mul (f64.const 2.0) (local.get $c)))))
        (local.set $q (f64.sub (f64.const 7.85398163397448278999e-01) (f64.mul (f64.const 2.0) (local.get $w))))
        (local.set $t (f64.sub (f64.const 7.85398163397448278999e-01) (f64.sub (local.get $p) (local.get $q))))
      )
    )

    (if (i32.gt_s (local.get $hx) (i32.const 0))
      (then (return (local.get $t)))
      (else (return (f64.neg (local.get $t))))
    )
    (unreachable)
  )

  ;; ==========================================================================
  ;; Arccosine using V8/fdlibm algorithm (exact match to ieee754.cc)
  ;; ==========================================================================
  (func $f64_acos (param $x f64) (result f64)
    (local $hx i32) (local $ix i32) (local $lx i32)
    (local $z f64) (local $p f64) (local $q f64) (local $r f64) (local $w f64) (local $s f64) (local $c f64) (local $df f64)
    ;; Constants from V8 ieee754.cc:
    ;; pi        = 3.14159265358979311600e+00  0x400921FB, 0x54442D18
    ;; pio2_hi   = 1.57079632679489655800e+00  0x3FF921FB, 0x54442D18
    ;; pio2_lo   = 6.12323399573676603587e-17  0x3C91A626, 0x33145C07
    ;; pS0-pS5, qS1-qS4 same as asin

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $ix (i32.and (local.get $hx) (i32.const 0x7FFFFFFF)))

    ;; |x| >= 1
    (if (i32.ge_u (local.get $ix) (i32.const 0x3FF00000))
      (then
        (local.set $lx (call $f64_lo (local.get $x)))
        ;; |x| == 1
        (if (i32.eq (i32.or (i32.sub (local.get $ix) (i32.const 0x3FF00000)) (local.get $lx)) (i32.const 0))
          (then
            ;; acos(1) = 0, acos(-1) = pi
            (if (i32.gt_s (local.get $hx) (i32.const 0))
              (then (return (f64.const 0)))
              (else (return (f64.add
                (f64.const 3.14159265358979311600e+00)
                (f64.mul (f64.const 2.0) (f64.const 6.12323399573676603587e-17)))))
            )
          )
        )
        ;; acos(|x|>1) is NaN
        (return (f64.div (f64.const 0) (f64.const 0)))
      )
    )

    ;; |x| < 0.5
    (if (i32.lt_u (local.get $ix) (i32.const 0x3FE00000))
      (then
        ;; |x| < 2^-57: return pio2_hi + pio2_lo
        (if (i32.le_u (local.get $ix) (i32.const 0x3C600000))
          (then (return (f64.add
            (f64.const 1.57079632679489655800e+00)
            (f64.const 6.12323399573676603587e-17))))
        )
        (local.set $z (f64.mul (local.get $x) (local.get $x)))
        ;; p = z*(pS0 + z*(pS1 + z*(pS2 + z*(pS3 + z*(pS4 + z*pS5)))))
        (local.set $p (f64.mul (local.get $z)
          (f64.add (f64.const 1.66666666666666657415e-01)
            (f64.mul (local.get $z)
              (f64.add (f64.const -3.25565818622400915405e-01)
                (f64.mul (local.get $z)
                  (f64.add (f64.const 2.01212532134862925881e-01)
                    (f64.mul (local.get $z)
                      (f64.add (f64.const -4.00555345006794114027e-02)
                        (f64.mul (local.get $z)
                          (f64.add (f64.const 7.91534994289814532176e-04)
                            (f64.mul (local.get $z) (f64.const 3.47933107596021167570e-05)))))))))))))
        ;; q = 1 + z*(qS1 + z*(qS2 + z*(qS3 + z*qS4)))
        (local.set $q (f64.add (f64.const 1)
          (f64.mul (local.get $z)
            (f64.add (f64.const -2.40339491173441421878e+00)
              (f64.mul (local.get $z)
                (f64.add (f64.const 2.02094576023350569471e+00)
                  (f64.mul (local.get $z)
                    (f64.add (f64.const -6.88283971605453293030e-01)
                      (f64.mul (local.get $z) (f64.const 7.70381505559019352791e-02))))))))))
        (local.set $r (f64.div (local.get $p) (local.get $q)))
        ;; return pio2_hi - (x - (pio2_lo - x*r))
        (return (f64.sub
          (f64.const 1.57079632679489655800e+00)
          (f64.sub (local.get $x)
            (f64.sub (f64.const 6.12323399573676603587e-17)
              (f64.mul (local.get $x) (local.get $r))))))
      )
    )

    ;; x < -0.5: return pi - 2*(s + w) where s = sqrt((1+x)/2)
    (if (i32.lt_s (local.get $hx) (i32.const 0))
      (then
        (local.set $z (f64.mul (f64.add (f64.const 1) (local.get $x)) (f64.const 0.5)))
        ;; p = z*(pS0 + z*(pS1 + z*(pS2 + z*(pS3 + z*(pS4 + z*pS5)))))
        (local.set $p (f64.mul (local.get $z)
          (f64.add (f64.const 1.66666666666666657415e-01)
            (f64.mul (local.get $z)
              (f64.add (f64.const -3.25565818622400915405e-01)
                (f64.mul (local.get $z)
                  (f64.add (f64.const 2.01212532134862925881e-01)
                    (f64.mul (local.get $z)
                      (f64.add (f64.const -4.00555345006794114027e-02)
                        (f64.mul (local.get $z)
                          (f64.add (f64.const 7.91534994289814532176e-04)
                            (f64.mul (local.get $z) (f64.const 3.47933107596021167570e-05)))))))))))))
        ;; q = 1 + z*(qS1 + z*(qS2 + z*(qS3 + z*qS4)))
        (local.set $q (f64.add (f64.const 1)
          (f64.mul (local.get $z)
            (f64.add (f64.const -2.40339491173441421878e+00)
              (f64.mul (local.get $z)
                (f64.add (f64.const 2.02094576023350569471e+00)
                  (f64.mul (local.get $z)
                    (f64.add (f64.const -6.88283971605453293030e-01)
                      (f64.mul (local.get $z) (f64.const 7.70381505559019352791e-02))))))))))
        (local.set $s (f64.sqrt (local.get $z)))
        (local.set $r (f64.div (local.get $p) (local.get $q)))
        (local.set $w (f64.sub (f64.mul (local.get $r) (local.get $s)) (f64.const 6.12323399573676603587e-17)))
        ;; return pi - 2.0 * (s + w)
        (return (f64.sub
          (f64.const 3.14159265358979311600e+00)
          (f64.mul (f64.const 2.0) (f64.add (local.get $s) (local.get $w)))))
      )
    )

    ;; x > 0.5: return 2*(df + w) where s = sqrt((1-x)/2), df = s with low word cleared
    (local.set $z (f64.mul (f64.sub (f64.const 1) (local.get $x)) (f64.const 0.5)))
    (local.set $s (f64.sqrt (local.get $z)))
    (local.set $df (call $f64_trunc_lo (local.get $s)))
    (local.set $c (f64.div
      (f64.sub (local.get $z) (f64.mul (local.get $df) (local.get $df)))
      (f64.add (local.get $s) (local.get $df))))
    ;; p = z*(pS0 + z*(pS1 + z*(pS2 + z*(pS3 + z*(pS4 + z*pS5)))))
    (local.set $p (f64.mul (local.get $z)
      (f64.add (f64.const 1.66666666666666657415e-01)
        (f64.mul (local.get $z)
          (f64.add (f64.const -3.25565818622400915405e-01)
            (f64.mul (local.get $z)
              (f64.add (f64.const 2.01212532134862925881e-01)
                (f64.mul (local.get $z)
                  (f64.add (f64.const -4.00555345006794114027e-02)
                    (f64.mul (local.get $z)
                      (f64.add (f64.const 7.91534994289814532176e-04)
                        (f64.mul (local.get $z) (f64.const 3.47933107596021167570e-05)))))))))))))
    ;; q = 1 + z*(qS1 + z*(qS2 + z*(qS3 + z*qS4)))
    (local.set $q (f64.add (f64.const 1)
      (f64.mul (local.get $z)
        (f64.add (f64.const -2.40339491173441421878e+00)
          (f64.mul (local.get $z)
            (f64.add (f64.const 2.02094576023350569471e+00)
              (f64.mul (local.get $z)
                (f64.add (f64.const -6.88283971605453293030e-01)
                  (f64.mul (local.get $z) (f64.const 7.70381505559019352791e-02))))))))))
    (local.set $r (f64.div (local.get $p) (local.get $q)))
    (local.set $w (f64.add (f64.mul (local.get $r) (local.get $s)) (local.get $c)))
    ;; return 2.0 * (df + w)
    (f64.mul (f64.const 2.0) (f64.add (local.get $df) (local.get $w)))
  )

  ;; ==========================================================================
  ;; Arctangent using V8/fdlibm algorithm (exact match to ieee754.cc)
  ;; ==========================================================================
  (func $f64_atan (param $x f64) (result f64)
    (local $hx i32) (local $ix i32) (local $id i32)
    (local $z f64) (local $w f64) (local $s1 f64) (local $s2 f64)
    ;; atanhi/atanlo constants for high precision
    ;; atanhi[0] = 4.63647609000806093515e-01  atan(0.5)hi 0x3FDDAC67, 0x0561BB4F
    ;; atanhi[1] = 7.85398163397448278999e-01  atan(1.0)hi 0x3FE921FB, 0x54442D18
    ;; atanhi[2] = 9.82793723247329054082e-01  atan(1.5)hi 0x3FEF730B, 0xD281F69B
    ;; atanhi[3] = 1.57079632679489655800e+00  atan(inf)hi 0x3FF921FB, 0x54442D18
    ;; atanlo[0] = 2.26987774529616870924e-17  0x3C7A2B7F, 0x222F65E2
    ;; atanlo[1] = 3.06161699786838301793e-17  0x3C81A626, 0x33145C07
    ;; atanlo[2] = 1.39033110312309984516e-17  0x3C700788, 0x7AF0CBBD
    ;; atanlo[3] = 6.12323399573676603587e-17  0x3C91A626, 0x33145C07

    (local.set $hx (call $f64_hi (local.get $x)))
    (local.set $ix (i32.and (local.get $hx) (i32.const 0x7FFFFFFF)))

    ;; if |x| >= 2^66, return +-pi/2
    (if (i32.ge_u (local.get $ix) (i32.const 0x44100000))
      (then
        ;; Check for NaN
        (if (i32.gt_u (local.get $ix) (i32.const 0x7FF00000))
          (then (return (f64.add (local.get $x) (local.get $x))))
        )
        (if (i32.eq (local.get $ix) (i32.const 0x7FF00000))
          (then
            (if (i32.ne (call $f64_lo (local.get $x)) (i32.const 0))
              (then (return (f64.add (local.get $x) (local.get $x)))) ;; NaN
            )
          )
        )
        ;; Return +-pi/2
        (if (i32.gt_s (local.get $hx) (i32.const 0))
          (then (return (f64.add
            (f64.const 1.57079632679489655800e+00)
            (f64.const 6.12323399573676603587e-17))))
          (else (return (f64.neg (f64.add
            (f64.const 1.57079632679489655800e+00)
            (f64.const 6.12323399573676603587e-17)))))
        )
      )
    )

    ;; |x| < 0.4375 (0x3FDC0000)
    (if (i32.lt_u (local.get $ix) (i32.const 0x3FDC0000))
      (then
        ;; |x| < 2^-27: return x
        (if (i32.lt_u (local.get $ix) (i32.const 0x3E400000))
          (then (return (local.get $x)))
        )
        (local.set $id (i32.const -1))
      )
      (else
        (local.set $x (f64.abs (local.get $x)))
        (if (i32.lt_u (local.get $ix) (i32.const 0x3FF30000))
          (then
            ;; |x| < 1.1875
            (if (i32.lt_u (local.get $ix) (i32.const 0x3FE60000))
              (then
                ;; 7/16 <= |x| < 11/16: x = (2x-1)/(2+x)
                (local.set $id (i32.const 0))
                (local.set $x (f64.div
                  (f64.sub (f64.mul (f64.const 2.0) (local.get $x)) (f64.const 1.0))
                  (f64.add (f64.const 2.0) (local.get $x))))
              )
              (else
                ;; 11/16 <= |x| < 19/16: x = (x-1)/(x+1)
                (local.set $id (i32.const 1))
                (local.set $x (f64.div
                  (f64.sub (local.get $x) (f64.const 1.0))
                  (f64.add (local.get $x) (f64.const 1.0))))
              )
            )
          )
          (else
            (if (i32.lt_u (local.get $ix) (i32.const 0x40038000))
              (then
                ;; |x| < 2.4375: x = (x-1.5)/(1+1.5*x)
                (local.set $id (i32.const 2))
                (local.set $x (f64.div
                  (f64.sub (local.get $x) (f64.const 1.5))
                  (f64.add (f64.const 1.0) (f64.mul (f64.const 1.5) (local.get $x)))))
              )
              (else
                ;; |x| >= 2.4375: x = -1/x
                (local.set $id (i32.const 3))
                (local.set $x (f64.div (f64.const -1.0) (local.get $x)))
              )
            )
          )
        )
      )
    )

    ;; Polynomial evaluation
    (local.set $z (f64.mul (local.get $x) (local.get $x)))
    (local.set $w (f64.mul (local.get $z) (local.get $z)))

    ;; s1 = z*(aT[0] + w*(aT[2] + w*(aT[4] + w*(aT[6] + w*(aT[8] + w*aT[10])))))
    (local.set $s1 (f64.mul (local.get $z)
      (f64.add (f64.const 3.33333333333329318027e-01)
        (f64.mul (local.get $w)
          (f64.add (f64.const 1.42857142725034663711e-01)
            (f64.mul (local.get $w)
              (f64.add (f64.const 9.09088713343650656196e-02)
                (f64.mul (local.get $w)
                  (f64.add (f64.const 6.66107313738753120669e-02)
                    (f64.mul (local.get $w)
                      (f64.add (f64.const 4.97687799461593236017e-02)
                        (f64.mul (local.get $w) (f64.const 1.62858201153657823623e-02)))))))))))))

    ;; s2 = w*(aT[1] + w*(aT[3] + w*(aT[5] + w*(aT[7] + w*aT[9]))))
    (local.set $s2 (f64.mul (local.get $w)
      (f64.add (f64.const -1.99999999998764832476e-01)
        (f64.mul (local.get $w)
          (f64.add (f64.const -1.11111104054623557880e-01)
            (f64.mul (local.get $w)
              (f64.add (f64.const -7.69187620504482999495e-02)
                (f64.mul (local.get $w)
                  (f64.add (f64.const -5.83357013379057348645e-02)
                    (f64.mul (local.get $w) (f64.const -3.65315727442169155270e-02)))))))))))

    ;; For id < 0: return x - x*(s1+s2)
    (if (i32.lt_s (local.get $id) (i32.const 0))
      (then
        (return (f64.sub (local.get $x)
          (f64.mul (local.get $x) (f64.add (local.get $s1) (local.get $s2)))))
      )
    )

    ;; z = atanhi[id] - ((x*(s1+s2) - atanlo[id]) - x)
    ;; For each id, use the appropriate constants
    (if (i32.eq (local.get $id) (i32.const 0))
      (then
        (local.set $z (f64.sub
          (f64.const 4.63647609000806093515e-01)
          (f64.sub
            (f64.sub (f64.mul (local.get $x) (f64.add (local.get $s1) (local.get $s2)))
                     (f64.const 2.26987774529616870924e-17))
            (local.get $x))))
      )
    )
    (if (i32.eq (local.get $id) (i32.const 1))
      (then
        (local.set $z (f64.sub
          (f64.const 7.85398163397448278999e-01)
          (f64.sub
            (f64.sub (f64.mul (local.get $x) (f64.add (local.get $s1) (local.get $s2)))
                     (f64.const 3.06161699786838301793e-17))
            (local.get $x))))
      )
    )
    (if (i32.eq (local.get $id) (i32.const 2))
      (then
        (local.set $z (f64.sub
          (f64.const 9.82793723247329054082e-01)
          (f64.sub
            (f64.sub (f64.mul (local.get $x) (f64.add (local.get $s1) (local.get $s2)))
                     (f64.const 1.39033110312309984516e-17))
            (local.get $x))))
      )
    )
    (if (i32.eq (local.get $id) (i32.const 3))
      (then
        (local.set $z (f64.sub
          (f64.const 1.57079632679489655800e+00)
          (f64.sub
            (f64.sub (f64.mul (local.get $x) (f64.add (local.get $s1) (local.get $s2)))
                     (f64.const 6.12323399573676603587e-17))
            (local.get $x))))
      )
    )

    (if (i32.lt_s (local.get $hx) (i32.const 0))
      (then (return (f64.neg (local.get $z))))
      (else (return (local.get $z)))
    )
    (unreachable)
  )

  ;; ==========================================================================
  ;; Two-argument arctangent (atan2)
  ;; atan2(y, x) = angle from positive x-axis to point (x, y)
  ;; ==========================================================================
  (func $f64_atan2 (param $y f64) (param $x f64) (result f64)
    ;; Handle special cases
    (if (f64.ne (local.get $y) (local.get $y))
      (then (return (local.get $y))) ;; NaN
    )
    (if (f64.ne (local.get $x) (local.get $x))
      (then (return (local.get $x))) ;; NaN
    )

    ;; x > 0: atan(y/x)
    (if (f64.gt (local.get $x) (f64.const 0))
      (then
        (return (call $f64_atan (f64.div (local.get $y) (local.get $x))))
      )
    )

    ;; x < 0, y >= 0: atan(y/x) + PI
    (if (f64.lt (local.get $x) (f64.const 0))
      (then
        (if (f64.ge (local.get $y) (f64.const 0))
          (then
            (return (f64.add
              (call $f64_atan (f64.div (local.get $y) (local.get $x)))
              (f64.const 3.141592653589793)))
          )
          (else
            ;; x < 0, y < 0: atan(y/x) - PI
            (return (f64.sub
              (call $f64_atan (f64.div (local.get $y) (local.get $x)))
              (f64.const 3.141592653589793)))
          )
        )
      )
    )

    ;; x == 0
    (if (f64.gt (local.get $y) (f64.const 0))
      (then (return (f64.const 1.5707963267948966))) ;; PI/2
    )
    (if (f64.lt (local.get $y) (f64.const 0))
      (then (return (f64.const -1.5707963267948966))) ;; -PI/2
    )

    ;; x == 0, y == 0: return 0
    (f64.const 0)
  )

  ;; ==========================================================================
  ;; Scope operations
  ;; ==========================================================================

  ;; Scope layout (after GC header):
  ;;   [parent:4][count:4][capacity:4][entries_ptr:4]
  ;; Entry layout: [key_offset:4][type:4][flags:4][data_lo:4][data_hi:4] = 20 bytes

  ;; Allocate a new scope with given parent
  (func $scope_create (param $parent i32) (result i32)
    (local $ptr i32)
    (local $entries_ptr i32)
    (local $heap_ptr i32)

    ;; Allocate scope object: GC header (8) + parent (4) + count (4) + capacity (4) + entries_ptr (4) = 24 bytes
    (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
    (local.set $ptr (local.get $heap_ptr))

    ;; GC header: [size | (type << 24)]
    (i32.store (call $abs (local.get $ptr))
      (i32.or (i32.const 24) (i32.shl (global.get $OBJ_SCOPE) (i32.const 24))))
    ;; GC header word 2 (flags, unused for now)
    (i32.store (call $abs (i32.add (local.get $ptr) (i32.const 4))) (i32.const 0))

    ;; Allocate entries array: GC header (8) + 8 entries × 20 bytes = 168 bytes
    (local.set $entries_ptr (i32.add (local.get $heap_ptr) (i32.const 24)))
    (i32.store (call $abs (local.get $entries_ptr))
      (i32.or (i32.const 168) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
    (i32.store (call $abs (i32.add (local.get $entries_ptr) (i32.const 4))) (i32.const 0))

    ;; Scope data (after GC header)
    (i32.store (call $abs (i32.add (local.get $ptr) (i32.const 8))) (local.get $parent))  ;; parent
    (i32.store (call $abs (i32.add (local.get $ptr) (i32.const 12))) (i32.const 0))       ;; count = 0
    (i32.store (call $abs (i32.add (local.get $ptr) (i32.const 16))) (i32.const 8))       ;; capacity = 8
    (i32.store (call $abs (i32.add (local.get $ptr) (i32.const 20))) (local.get $entries_ptr)) ;; entries_ptr

    ;; Update heap pointer
    (call $write_state (global.get $STATE_HEAP_POINTER)
      (i32.add (local.get $entries_ptr) (i32.const 168)))

    ;; Return data pointer (after GC header)
    (i32.add (local.get $ptr) (global.get $GC_HEADER_SIZE))
  )

  ;; Look up variable in scope chain, return pointer to value or 0 if not found
  (func $scope_lookup (param $scope i32) (param $name i32) (result i32)
    (local $count i32)
    (local $entries i32)
    (local $i i32)
    (local $entry_ptr i32)
    (local $entry_key i32)

    (block $not_found
      (loop $scope_loop
        ;; If scope is 0 (null), not found
        (br_if $not_found (i32.eqz (local.get $scope)))

        ;; Read scope fields (scope points to data, after GC header)
        (local.set $count (i32.load (call $abs (i32.add (local.get $scope) (i32.const 4)))))
        (local.set $entries (i32.load (call $abs (i32.add (local.get $scope) (i32.const 12)))))

        ;; Search entries (entries_ptr points to GC object, data starts at +8)
        (local.set $i (i32.const 0))
        (block $found_in_scope
          (loop $entry_loop
            (br_if $found_in_scope (i32.ge_u (local.get $i) (local.get $count)))

            ;; entry_ptr = entries + 8 (GC header) + i * 20
            (local.set $entry_ptr
              (i32.add
                (i32.add (local.get $entries) (global.get $GC_HEADER_SIZE))
                (i32.mul (local.get $i) (i32.const 20))))

            (local.set $entry_key (i32.load (call $abs (local.get $entry_ptr))))

            (if (i32.eq (local.get $entry_key) (local.get $name))
              (then
                ;; Return pointer to value (entry + 4, which is type/flags/data)
                (return (i32.add (local.get $entry_ptr) (i32.const 4)))
              )
            )

            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $entry_loop)
          )
        )

        ;; Not found in this scope, try parent
        (local.set $scope (i32.load (call $abs (local.get $scope))))
        (br $scope_loop)
      )
    )

    (i32.const 0)
  )

  ;; Define variable in current scope (for LET_VAR)
  ;; Grows entries array if needed (doubles capacity)
  ;; Sets ERR_REDECLARATION if name already exists in current scope
  (func $scope_define (param $scope i32) (param $name i32)
                      (param $type i32) (param $flags i32) (param $data_lo i32) (param $data_hi i32)
    (local $count i32)
    (local $capacity i32)
    (local $entries i32)
    (local $entry_ptr i32)
    (local $new_capacity i32)
    (local $new_entries i32)
    (local $new_size i32)
    (local $heap_ptr i32)
    (local $i i32)
    (local $src i32)
    (local $dst i32)

    (local.set $count (i32.load (call $abs (i32.add (local.get $scope) (i32.const 4)))))
    (local.set $capacity (i32.load (call $abs (i32.add (local.get $scope) (i32.const 8)))))
    (local.set $entries (i32.load (call $abs (i32.add (local.get $scope) (i32.const 12)))))

    ;; Check for redeclaration in current scope only (not parent scopes)
    (local.set $i (i32.const 0))
    (block $check_done
      (loop $check_loop
        (br_if $check_done (i32.ge_u (local.get $i) (local.get $count)))

        ;; entry_ptr = entries + 8 (GC header) + i * 20
        (local.set $entry_ptr
          (i32.add
            (i32.add (local.get $entries) (global.get $GC_HEADER_SIZE))
            (i32.mul (local.get $i) (i32.const 20))))

        ;; Check if entry key matches name
        (if (i32.eq (i32.load (call $abs (local.get $entry_ptr))) (local.get $name))
          (then
            ;; Redeclaration error
            (call $set_error (global.get $ERR_REDECLARATION) (local.get $name))
            (return)
          )
        )

        (local.set $i (i32.add (local.get $i) (i32.const 1)))
        (br $check_loop)
      )
    )

    ;; Check if we need to grow
    (if (i32.ge_u (local.get $count) (local.get $capacity))
      (then
        ;; Double the capacity
        (local.set $new_capacity (i32.mul (local.get $capacity) (i32.const 2)))

        ;; Allocate new entries array: GC header (8) + new_capacity * 20 bytes
        (local.set $new_size (i32.add (global.get $GC_HEADER_SIZE)
                                      (i32.mul (local.get $new_capacity) (i32.const 20))))
        (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
        (local.set $new_entries (local.get $heap_ptr))

        ;; Write GC header for new array
        (i32.store (call $abs (local.get $new_entries))
          (i32.or (local.get $new_size) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
        (i32.store (call $abs (i32.add (local.get $new_entries) (i32.const 4))) (i32.const 0))

        ;; Update heap pointer
        (call $write_state (global.get $STATE_HEAP_POINTER)
          (i32.add (local.get $heap_ptr) (local.get $new_size)))

        ;; Copy existing entries (count * 20 bytes)
        (local.set $i (i32.const 0))
        (block $copy_done
          (loop $copy_loop
            (br_if $copy_done (i32.ge_u (local.get $i) (local.get $count)))

            (local.set $src (i32.add
              (i32.add (local.get $entries) (global.get $GC_HEADER_SIZE))
              (i32.mul (local.get $i) (i32.const 20))))
            (local.set $dst (i32.add
              (i32.add (local.get $new_entries) (global.get $GC_HEADER_SIZE))
              (i32.mul (local.get $i) (i32.const 20))))

            ;; Copy 20 bytes (5 i32s)
            (i32.store (call $abs (local.get $dst))
              (i32.load (call $abs (local.get $src))))
            (i32.store (call $abs (i32.add (local.get $dst) (i32.const 4)))
              (i32.load (call $abs (i32.add (local.get $src) (i32.const 4)))))
            (i32.store (call $abs (i32.add (local.get $dst) (i32.const 8)))
              (i32.load (call $abs (i32.add (local.get $src) (i32.const 8)))))
            (i32.store (call $abs (i32.add (local.get $dst) (i32.const 12)))
              (i32.load (call $abs (i32.add (local.get $src) (i32.const 12)))))
            (i32.store (call $abs (i32.add (local.get $dst) (i32.const 16)))
              (i32.load (call $abs (i32.add (local.get $src) (i32.const 16)))))

            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $copy_loop)
          )
        )

        ;; Update scope to point to new entries and new capacity
        (local.set $entries (local.get $new_entries))
        (i32.store (call $abs (i32.add (local.get $scope) (i32.const 8))) (local.get $new_capacity))
        (i32.store (call $abs (i32.add (local.get $scope) (i32.const 12))) (local.get $new_entries))
      )
    )

    ;; entry_ptr = entries + 8 (GC header) + count * 20
    (local.set $entry_ptr
      (i32.add
        (i32.add (local.get $entries) (global.get $GC_HEADER_SIZE))
        (i32.mul (local.get $count) (i32.const 20))))

    ;; Write entry: [key][type][flags][data_lo][data_hi]
    (i32.store (call $abs (local.get $entry_ptr)) (local.get $name))
    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))) (local.get $type))
    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 8))) (local.get $flags))
    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 12))) (local.get $data_lo))
    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 16))) (local.get $data_hi))

    ;; Increment count
    (i32.store (call $abs (i32.add (local.get $scope) (i32.const 4)))
      (i32.add (local.get $count) (i32.const 1)))
  )

  ;; Set variable (find in chain and update, or error)
  (func $scope_set (param $scope i32) (param $name i32)
                   (param $type i32) (param $flags i32) (param $data_lo i32) (param $data_hi i32)
                   (result i32)  ;; returns 1 on success, 0 on not found
    (local $value_ptr i32)

    (local.set $value_ptr (call $scope_lookup (local.get $scope) (local.get $name)))
    (if (i32.eqz (local.get $value_ptr))
      (then (return (i32.const 0)))
    )

    ;; Update value in place
    (i32.store (call $abs (local.get $value_ptr)) (local.get $type))
    (i32.store (call $abs (i32.add (local.get $value_ptr) (i32.const 4))) (local.get $flags))
    (i32.store (call $abs (i32.add (local.get $value_ptr) (i32.const 8))) (local.get $data_lo))
    (i32.store (call $abs (i32.add (local.get $value_ptr) (i32.const 12))) (local.get $data_hi))

    (i32.const 1)
  )

  ;; ==========================================================================
  ;; Try Stack operations
  ;; ==========================================================================

  ;; Get current call frame depth (number of frames)
  (func $get_frame_depth (result i32)
    (i32.div_u
      (i32.sub
        (call $read_state (global.get $STATE_STACK_POINTER))
        (call $read_state (global.get $STATE_STACK_BASE)))
      (global.get $FRAME_SIZE)))

  ;; Push try entry: code_block, catch_index, finally_index, frame_depth
  (func $try_push (param $code_block i32) (param $catch_index i32) (param $finally_index i32)
    (local $try_ptr i32)
    (local $frame_depth i32)

    (local.set $try_ptr (call $read_state (global.get $STATE_TRY_POINTER)))
    (local.set $frame_depth (call $get_frame_depth))

    ;; Write try entry
    (i32.store (call $abs (local.get $try_ptr)) (local.get $code_block))
    (i32.store (call $abs (i32.add (local.get $try_ptr) (i32.const 4))) (local.get $catch_index))
    (i32.store (call $abs (i32.add (local.get $try_ptr) (i32.const 8))) (local.get $finally_index))
    (i32.store (call $abs (i32.add (local.get $try_ptr) (i32.const 12))) (local.get $frame_depth))

    ;; Advance try pointer
    (call $write_state (global.get $STATE_TRY_POINTER)
      (i32.add (local.get $try_ptr) (global.get $TRY_ENTRY_SIZE)))
  )

  ;; Pop try entry (just decrements pointer)
  (func $try_pop
    (call $write_state (global.get $STATE_TRY_POINTER)
      (i32.sub
        (call $read_state (global.get $STATE_TRY_POINTER))
        (global.get $TRY_ENTRY_SIZE)))
  )

  ;; Get try stack depth
  (func $try_depth (result i32)
    (i32.div_u
      (i32.sub
        (call $read_state (global.get $STATE_TRY_POINTER))
        (call $read_state (global.get $STATE_TRY_BASE)))
      (global.get $TRY_ENTRY_SIZE)))

  ;; Read try entry field at given depth (0 = top)
  (func $try_read (param $depth i32) (param $field i32) (result i32)
    (local $entry_ptr i32)
    ;; entry_ptr = try_pointer - (depth + 1) * TRY_ENTRY_SIZE
    (local.set $entry_ptr
      (i32.sub
        (call $read_state (global.get $STATE_TRY_POINTER))
        (i32.mul
          (i32.add (local.get $depth) (i32.const 1))
          (global.get $TRY_ENTRY_SIZE))))
    (i32.load (call $abs (i32.add (local.get $entry_ptr) (local.get $field)))))

  ;; ==========================================================================
  ;; Completion value operations
  ;; The completion value is stored in a reserved 16-byte slot just below
  ;; the pending stack base (PENDING_BASE - 16).
  ;; ==========================================================================

  ;; Copy value from pending stack top to completion value slot
  (func $save_completion_value
    (local $src i32)
    (local $dst i32)

    ;; src = pending_ptr - VALUE_SIZE (current top value)
    (local.set $src (call $pending_peek))
    ;; dst = PENDING_BASE - VALUE_SIZE (reserved completion slot)
    (local.set $dst
      (i32.sub
        (call $read_state (global.get $STATE_PENDING_BASE))
        (global.get $VALUE_SIZE)))

    ;; Copy 16 bytes
    (i32.store (call $abs (local.get $dst))
      (i32.load (call $abs (local.get $src))))
    (i32.store (call $abs (i32.add (local.get $dst) (i32.const 4)))
      (i32.load (call $abs (i32.add (local.get $src) (i32.const 4)))))
    (i32.store (call $abs (i32.add (local.get $dst) (i32.const 8)))
      (i32.load (call $abs (i32.add (local.get $src) (i32.const 8)))))
    (i32.store (call $abs (i32.add (local.get $dst) (i32.const 12)))
      (i32.load (call $abs (i32.add (local.get $src) (i32.const 12)))))

    ;; Update COMPLETION_VALUE pointer
    (call $write_state (global.get $STATE_COMPLETION_VALUE) (local.get $dst))
  )

  ;; Push completion value back to pending stack
  (func $restore_completion_value
    (local $src i32)

    (local.set $src (call $read_state (global.get $STATE_COMPLETION_VALUE)))

    (call $pending_push
      (i32.load (call $abs (local.get $src)))
      (i32.load (call $abs (i32.add (local.get $src) (i32.const 4))))
      (i32.load (call $abs (i32.add (local.get $src) (i32.const 8))))
      (i32.load (call $abs (i32.add (local.get $src) (i32.const 12)))))
  )

  ;; Unwind call frames to a target depth
  ;; Pops frames until frame_depth matches target, restoring scope
  (func $unwind_frames_to (param $target_depth i32)
    (local $stack_ptr i32)
    (local $stack_base i32)
    (local $frame_ptr i32)

    (block $done
      (loop $loop
        ;; current_depth = (stack_ptr - stack_base) / FRAME_SIZE
        (local.set $stack_ptr (call $read_state (global.get $STATE_STACK_POINTER)))
        (local.set $stack_base (call $read_state (global.get $STATE_STACK_BASE)))

        ;; If at target depth, done
        (br_if $done
          (i32.le_u
            (i32.div_u
              (i32.sub (local.get $stack_ptr) (local.get $stack_base))
              (global.get $FRAME_SIZE))
            (local.get $target_depth)))

        ;; Pop frame
        (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
        (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))

        ;; Restore scope from popped frame
        (local.set $frame_ptr (call $abs (local.get $stack_ptr)))
        (global.set $current_scope
          (i32.load (i32.add (local.get $frame_ptr) (i32.const 8))))

        ;; Restore pending pointer from frame
        (call $write_state (global.get $STATE_PENDING_POINTER)
          (i32.load (i32.add (local.get $frame_ptr) (i32.const 12))))

        (br $loop)
      )
    )
  )

  ;; ==========================================================================
  ;; Instruction reading (Reverse-Growing Code Block)
  ;;
  ;; Code block grows backward from STRING_START. Layout:
  ;;   [instr N]...[instr 1][instr 0][count:4][flags:4][GC:8] STRING_START
  ;;   ↑                             ↑                       ↑
  ;;   lowest addr              CODE_START            STRING_START
  ;;
  ;; CODE_START = STRING_START - 16 (fixed header position)
  ;; instr_address = CODE_START - (index + 1) * INSTRUCTION_SIZE
  ;;
  ;; Instruction format: [opcode:1][flags:1][src_start:2][src_end:2][reserved:2][operand1:4][operand2:4]
  ;; ==========================================================================

  ;; Get the fixed code block start (header position)
  (func $get_code_start (result i32)
    (i32.sub
      (call $read_state (global.get $STATE_STRING_START))
      (i32.const 16)))

  ;; Read instruction at index from the reverse-growing code block
  ;; instr_address = CODE_START - (index + 1) * 16
  (func $read_opcode (param $code_block i32) (param $index i32) (result i32)
    (local $instr_ptr i32)
    (local.set $instr_ptr
      (i32.sub
        (call $get_code_start)
        (i32.mul
          (i32.add (local.get $index) (i32.const 1))
          (global.get $INSTRUCTION_SIZE))))
    (i32.load8_u (call $abs (local.get $instr_ptr)))
  )

  (func $read_operand1 (param $code_block i32) (param $index i32) (result i32)
    (local $instr_ptr i32)
    (local.set $instr_ptr
      (i32.sub
        (call $get_code_start)
        (i32.mul
          (i32.add (local.get $index) (i32.const 1))
          (global.get $INSTRUCTION_SIZE))))
    (i32.load (call $abs (i32.add (local.get $instr_ptr) (i32.const 8))))
  )

  (func $read_operand2 (param $code_block i32) (param $index i32) (result i32)
    (local $instr_ptr i32)
    (local.set $instr_ptr
      (i32.sub
        (call $get_code_start)
        (i32.mul
          (i32.add (local.get $index) (i32.const 1))
          (global.get $INSTRUCTION_SIZE))))
    (i32.load (call $abs (i32.add (local.get $instr_ptr) (i32.const 12))))
  )

  ;; Read instruction count from the fixed header
  (func $read_instr_count (param $code_block i32) (result i32)
    (i32.load (call $abs (call $get_code_start)))
  )

  ;; ==========================================================================
  ;; is_truthy: check if value is truthy
  ;; ==========================================================================
  (func $is_truthy (param $ptr i32) (result i32)
    (local $type i32)
    (local $data_lo i32)

    (local.set $type (call $value_type (local.get $ptr)))
    (local.set $data_lo (call $value_data_lo (local.get $ptr)))

    ;; null and undefined are falsy
    (if (i32.eq (local.get $type) (global.get $TYPE_NULL))
      (then (return (i32.const 0))))
    (if (i32.eq (local.get $type) (global.get $TYPE_UNDEFINED))
      (then (return (i32.const 0))))

    ;; boolean: check data_lo
    (if (i32.eq (local.get $type) (global.get $TYPE_BOOLEAN))
      (then (return (local.get $data_lo))))

    ;; integer: 0 is falsy
    (if (i32.eq (local.get $type) (global.get $TYPE_INTEGER))
      (then (return (i32.ne (local.get $data_lo) (i32.const 0)))))

    ;; float: 0.0 is falsy (check if both words are 0)
    (if (i32.eq (local.get $type) (global.get $TYPE_FLOAT))
      (then
        (return (i32.or
          (i32.ne (local.get $data_lo) (i32.const 0))
          (i32.ne (call $value_data_hi (local.get $ptr)) (i32.const 0))))))

    ;; string: empty string is falsy (check length at string offset)
    (if (i32.eq (local.get $type) (global.get $TYPE_STRING))
      (then
        (return (i32.ne
          (i32.load (call $abs (local.get $data_lo)))
          (i32.const 0)))))

    ;; Everything else is truthy
    (i32.const 1)
  )

  ;; ==========================================================================
  ;; is_callback_method
  ;;
  ;; Returns 1 if method_id is a callback method (map, filter, reduce, etc.)
  ;; ==========================================================================
  (func $is_callback_method (param $method_id i32) (result i32)
    (i32.or
      (i32.or
        (i32.or
          (i32.eq (local.get $method_id) (global.get $METHOD_MAP))
          (i32.eq (local.get $method_id) (global.get $METHOD_FILTER)))
        (i32.or
          (i32.eq (local.get $method_id) (global.get $METHOD_REDUCE))
          (i32.eq (local.get $method_id) (global.get $METHOD_FOR_EACH))))
      (i32.or
        (i32.or
          (i32.eq (local.get $method_id) (global.get $METHOD_FIND))
          (i32.eq (local.get $method_id) (global.get $METHOD_FIND_INDEX)))
        (i32.or
          (i32.eq (local.get $method_id) (global.get $METHOD_SOME))
          (i32.eq (local.get $method_id) (global.get $METHOD_EVERY)))))
  )

  ;; ==========================================================================
  ;; is_truthy_inline
  ;;
  ;; Like is_truthy but takes type, data_lo, data_hi directly instead of ptr
  ;; ==========================================================================
  (func $is_truthy_inline (param $type i32) (param $data_lo i32) (param $data_hi i32) (result i32)
    ;; null and undefined are falsy
    (if (i32.eq (local.get $type) (global.get $TYPE_NULL))
      (then (return (i32.const 0))))
    (if (i32.eq (local.get $type) (global.get $TYPE_UNDEFINED))
      (then (return (i32.const 0))))

    ;; boolean: check data_lo
    (if (i32.eq (local.get $type) (global.get $TYPE_BOOLEAN))
      (then (return (local.get $data_lo))))

    ;; integer: 0 is falsy
    (if (i32.eq (local.get $type) (global.get $TYPE_INTEGER))
      (then (return (i32.ne (local.get $data_lo) (i32.const 0)))))

    ;; float: 0.0 is falsy
    (if (i32.eq (local.get $type) (global.get $TYPE_FLOAT))
      (then
        (return (i32.or
          (i32.ne (local.get $data_lo) (i32.const 0))
          (i32.ne (local.get $data_hi) (i32.const 0))))))

    ;; string: empty string is falsy
    (if (i32.eq (local.get $type) (global.get $TYPE_STRING))
      (then
        (return (i32.ne
          (i32.load (call $abs (local.get $data_lo)))
          (i32.const 0)))))

    ;; Everything else is truthy
    (i32.const 1)
  )

  ;; ==========================================================================
  ;; dispatch_builtin_method
  ;;
  ;; Dispatch a built-in method call (TYPE_BOUND_METHOD).
  ;; Pops arguments, executes method, pushes result, cleans up stack.
  ;;
  ;; Parameters:
  ;;   receiver_type: TYPE_ARRAY or TYPE_STRING
  ;;   receiver_ptr: pointer to the receiver (array ptr or string table offset)
  ;;   method_id: which method to call (METHOD_* constant)
  ;;   argc: number of arguments
  ;;   stack_base: position on pending stack where receiver was (for cleanup)
  ;; ==========================================================================
  (func $dispatch_builtin_method
    (param $receiver_type i32)
    (param $receiver_ptr i32)
    (param $method_id i32)
    (param $argc i32)
    (param $stack_base i32)

    (local $pending_ptr i32)
    (local $arg1_ptr i32)
    (local $arg1_type i32)
    (local $arg1_lo i32)
    (local $arg1_hi i32)
    (local $arg2_ptr i32)
    (local $arg2_type i32)
    (local $arg2_lo i32)
    (local $i32_result i32)
    (local $f64_result f64)
    (local $f64_arg2 f64)
    (local $str_len i32)
    (local $char_code i32)
    (local $byte_ptr i32)
    (local $i i32)
    (local $found i32)
    (local $arr_len i32)
    (local $arr_cap i32)
    (local $data_ptr i32)
    (local $elem_ptr i32)
    (local $new_arr i32)
    (local $new_data i32)
    (local $heap_ptr i32)
    (local $new_str_ptr i32)
    (local $str_ptr i32)
    (local $search_len i32)
    (local $search_ptr i32)
    (local $start_idx i32)
    (local $end_idx i32)
    (local $slice_len i32)
    (local $count i32)
    (local $match_pos i32)
    (local $j i32)
    (local $target_len i32)
    (local $pad_len i32)
    (local $fill_ptr i32)
    (local $fill_len i32)
    (local $repl_ptr i32)
    (local $repl_len i32)
    (local $char_count i32)
    (local $start_byte i32)
    (local $end_byte i32)
    (local $arg_ptr i32)
    (local $arg_type i32)
    (local $code_point i32)
    (local $dest i32)
    (local $obj_ptr i32)
    (local $entries_ptr i32)
    (local $entry_ptr i32)
    (local $key_offset i32)
    (local $arr_ptr i32)
    (local $src_ptr i32)
    (local $src_count i32)
    (local $src_entries i32)
    (local $val_type i32)
    (local $val_flags i32)
    (local $val_lo i32)
    (local $val_hi i32)

    ;; Get argument pointers if present
    ;; JS args are pushed left-to-right, so first JS arg is deepest on stack
    ;; For argc=1: arg1 is at pending_ptr - VALUE_SIZE
    ;; For argc=2: arg1 is at pending_ptr - 2*VALUE_SIZE, arg2 is at pending_ptr - VALUE_SIZE
    (local.set $pending_ptr (call $read_state (global.get $STATE_PENDING_POINTER)))

    ;; arg1 (first JS argument)
    (if (i32.ge_u (local.get $argc) (i32.const 1))
      (then
        ;; arg1 is at pending_ptr - argc * VALUE_SIZE (deepest)
        (local.set $arg1_ptr (i32.sub (local.get $pending_ptr)
          (i32.mul (local.get $argc) (global.get $VALUE_SIZE))))
        (local.set $arg1_type (call $value_type (local.get $arg1_ptr)))
        (local.set $arg1_lo (call $value_data_lo (local.get $arg1_ptr)))
        (local.set $arg1_hi (call $value_data_hi (local.get $arg1_ptr)))
      )
    )

    ;; arg2 (second JS argument)
    (if (i32.ge_u (local.get $argc) (i32.const 2))
      (then
        ;; arg2 is at arg1_ptr + VALUE_SIZE (one above arg1)
        (local.set $arg2_ptr (i32.add (local.get $arg1_ptr) (global.get $VALUE_SIZE)))
        (local.set $arg2_type (call $value_type (local.get $arg2_ptr)))
        (local.set $arg2_lo (call $value_data_lo (local.get $arg2_ptr)))
      )
    )

    ;; Dispatch based on receiver type and method ID

    ;; If receiver is a msgpack array, materialize it first
    ;; receiver_type for msgpack has TYPE_MSGPACK_REF in low byte, length in upper bytes
    (if (i32.eq (i32.and (local.get $receiver_type) (i32.const 0xff)) (global.get $TYPE_MSGPACK_REF))
      (then
        ;; Extract msgpack addr (receiver_ptr) and length (upper bits of receiver_type)
        (local.set $arr_ptr (call $msgpack_to_array
          (local.get $receiver_ptr)
          (i32.add (local.get $receiver_ptr) (i32.shr_u (local.get $receiver_type) (i32.const 8)))))
        (if (i32.eqz (local.get $arr_ptr))
          (then
            ;; Error already set by msgpack_to_array
            (return)
          )
        )
        ;; Now treat as regular array
        (local.set $receiver_type (global.get $TYPE_ARRAY))
        (local.set $receiver_ptr (local.get $arr_ptr))
      )
    )

    (if (i32.eq (local.get $receiver_type) (global.get $TYPE_ARRAY))
      (then
        ;; ============================================
        ;; ARRAY METHODS
        ;; ============================================
        ;; Array layout: [GC:8][length:4][capacity:4][data_ptr:4]
        (local.set $arr_len (i32.load (call $abs (i32.add (local.get $receiver_ptr) (i32.const 8)))))
        (local.set $arr_cap (i32.load (call $abs (i32.add (local.get $receiver_ptr) (i32.const 12)))))
        (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $receiver_ptr) (i32.const 16)))))

        ;; Clean up stack first: reset pending pointer to where we'll write the result
        ;; All method handlers will push their result starting from stack_base
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))

        ;; push(value) - adds element at end, returns new length
        (if (i32.eq (local.get $method_id) (global.get $METHOD_PUSH))
          (then
            ;; Array layout: [GC:8][length:4][capacity:4][data_ptr:4]
            ;; Data block: [GC:8][elements...] where element i is at data_ptr + 8 + i * 16

            ;; Check if we need to grow
            (if (i32.ge_u (local.get $arr_len) (local.get $arr_cap))
              (then
                ;; Need to reallocate data block with double capacity
                (local.set $new_arr (i32.shl (local.get $arr_cap) (i32.const 1)))  ;; new capacity = cap * 2
                (if (i32.eqz (local.get $new_arr))
                  (then (local.set $new_arr (i32.const 4))))  ;; minimum capacity of 4

                ;; Allocate new data block
                (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
                (local.set $new_data (local.get $heap_ptr))
                (local.set $i32_result (i32.add (global.get $GC_HEADER_SIZE)
                  (i32.mul (local.get $new_arr) (global.get $VALUE_SIZE))))

                ;; GC header for new data block
                (i32.store (call $abs (local.get $new_data))
                  (i32.or (local.get $i32_result) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
                (i32.store (call $abs (i32.add (local.get $new_data) (i32.const 4))) (i32.const 0))

                ;; Update heap pointer
                (call $write_state (global.get $STATE_HEAP_POINTER)
                  (i32.add (local.get $new_data) (local.get $i32_result)))

                ;; Copy existing elements
                (local.set $i (i32.const 0))
                (block $copy_done
                  (loop $copy_loop
                    (br_if $copy_done (i32.ge_u (local.get $i) (local.get $arr_len)))

                    ;; src = old data_ptr + 8 + i * 16
                    (local.set $elem_ptr (i32.add
                      (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                    ;; dest = new_data + 8 + i * 16
                    (local.set $str_ptr (i32.add
                      (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (global.get $VALUE_SIZE))))

                    ;; Copy 16 bytes
                    (i32.store (call $abs (local.get $str_ptr))
                      (i32.load (call $abs (local.get $elem_ptr))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 4)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 8)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 12)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))

                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                    (br $copy_loop)
                  )
                )

                ;; Update array header with new capacity and data_ptr
                (i32.store (call $abs (i32.add (local.get $receiver_ptr) (i32.const 12))) (local.get $new_arr))
                (i32.store (call $abs (i32.add (local.get $receiver_ptr) (i32.const 16))) (local.get $new_data))
                (local.set $data_ptr (local.get $new_data))
              )
            )

            ;; Now add the new element at position arr_len
            ;; dest = data_ptr + 8 + arr_len * 16
            (local.set $elem_ptr (i32.add
              (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
              (i32.mul (local.get $arr_len) (global.get $VALUE_SIZE))))

            ;; Copy arg1 (the value to push) to element slot
            (i32.store (call $abs (local.get $elem_ptr)) (local.get $arg1_type))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))  ;; flags
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $arg1_lo))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (local.get $arg1_hi))

            ;; Increment length
            (local.set $arr_len (i32.add (local.get $arr_len) (i32.const 1)))
            (i32.store (call $abs (i32.add (local.get $receiver_ptr) (i32.const 8))) (local.get $arr_len))

            ;; Return new length
            (call $push_f64 (f64.convert_i32_u (local.get $arr_len)))
            (return)
          )
        )

        ;; pop() - removes and returns last element
        (if (i32.eq (local.get $method_id) (global.get $METHOD_POP))
          (then
            ;; If array is empty, return undefined
            (if (i32.eqz (local.get $arr_len))
              (then
                (call $pending_push (global.get $TYPE_UNDEFINED) (i32.const 0) (i32.const 0) (i32.const 0))
                (return)
              )
            )

            ;; Get last element
            ;; elem_ptr = data_ptr + 8 + (arr_len - 1) * 16
            (local.set $elem_ptr (i32.add
              (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
              (i32.mul (i32.sub (local.get $arr_len) (i32.const 1)) (global.get $VALUE_SIZE))))

            ;; Push the element value
            (call $pending_push
              (i32.load (call $abs (local.get $elem_ptr)))
              (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))))
              (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))))
              (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))

            ;; Decrement length
            (i32.store (call $abs (i32.add (local.get $receiver_ptr) (i32.const 8)))
              (i32.sub (local.get $arr_len) (i32.const 1)))
            (return)
          )
        )

        ;; shift() - removes and returns first element
        (if (i32.eq (local.get $method_id) (global.get $METHOD_SHIFT))
          (then
            ;; If array is empty, return undefined
            (if (i32.eqz (local.get $arr_len))
              (then
                (call $pending_push (global.get $TYPE_UNDEFINED) (i32.const 0) (i32.const 0) (i32.const 0))
                (return)
              )
            )

            ;; Get first element (to return)
            ;; elem_ptr = data_ptr + 8
            (local.set $elem_ptr (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE)))

            ;; Save the first element to return later
            (local.set $arg1_type (i32.load (call $abs (local.get $elem_ptr))))
            (local.set $i32_result (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
            (local.set $arg1_lo (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
            (local.set $arg1_hi (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))

            ;; Shift all elements down by one
            (local.set $i (i32.const 0))
            (block $shift_done
              (loop $shift_loop
                (br_if $shift_done (i32.ge_u (local.get $i) (i32.sub (local.get $arr_len) (i32.const 1))))

                ;; src = data_ptr + 8 + (i+1) * 16
                (local.set $str_ptr (i32.add
                  (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (i32.add (local.get $i) (i32.const 1)) (global.get $VALUE_SIZE))))
                ;; dest = data_ptr + 8 + i * 16
                (local.set $new_data (i32.add
                  (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (global.get $VALUE_SIZE))))

                ;; Copy 16 bytes
                (i32.store (call $abs (local.get $new_data))
                  (i32.load (call $abs (local.get $str_ptr))))
                (i32.store (call $abs (i32.add (local.get $new_data) (i32.const 4)))
                  (i32.load (call $abs (i32.add (local.get $str_ptr) (i32.const 4)))))
                (i32.store (call $abs (i32.add (local.get $new_data) (i32.const 8)))
                  (i32.load (call $abs (i32.add (local.get $str_ptr) (i32.const 8)))))
                (i32.store (call $abs (i32.add (local.get $new_data) (i32.const 12)))
                  (i32.load (call $abs (i32.add (local.get $str_ptr) (i32.const 12)))))

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $shift_loop)
              )
            )

            ;; Decrement length
            (i32.store (call $abs (i32.add (local.get $receiver_ptr) (i32.const 8)))
              (i32.sub (local.get $arr_len) (i32.const 1)))

            ;; Return the saved first element
            (call $pending_push (local.get $arg1_type) (local.get $i32_result)
              (local.get $arg1_lo) (local.get $arg1_hi))
            (return)
          )
        )

        ;; unshift(value) - adds element at start, returns new length
        (if (i32.eq (local.get $method_id) (global.get $METHOD_UNSHIFT))
          (then
            ;; Check if we need to grow (same logic as push)
            (if (i32.ge_u (local.get $arr_len) (local.get $arr_cap))
              (then
                ;; Need to reallocate data block with double capacity
                (local.set $new_arr (i32.shl (local.get $arr_cap) (i32.const 1)))
                (if (i32.eqz (local.get $new_arr))
                  (then (local.set $new_arr (i32.const 4))))

                (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
                (local.set $new_data (local.get $heap_ptr))
                (local.set $i32_result (i32.add (global.get $GC_HEADER_SIZE)
                  (i32.mul (local.get $new_arr) (global.get $VALUE_SIZE))))

                (i32.store (call $abs (local.get $new_data))
                  (i32.or (local.get $i32_result) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
                (i32.store (call $abs (i32.add (local.get $new_data) (i32.const 4))) (i32.const 0))

                (call $write_state (global.get $STATE_HEAP_POINTER)
                  (i32.add (local.get $new_data) (local.get $i32_result)))

                ;; Copy existing elements, but shifted by 1
                (local.set $i (i32.const 0))
                (block $copy_done2
                  (loop $copy_loop2
                    (br_if $copy_done2 (i32.ge_u (local.get $i) (local.get $arr_len)))

                    ;; src = old data_ptr + 8 + i * 16
                    (local.set $elem_ptr (i32.add
                      (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                    ;; dest = new_data + 8 + (i+1) * 16  (shifted by 1)
                    (local.set $str_ptr (i32.add
                      (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
                      (i32.mul (i32.add (local.get $i) (i32.const 1)) (global.get $VALUE_SIZE))))

                    (i32.store (call $abs (local.get $str_ptr))
                      (i32.load (call $abs (local.get $elem_ptr))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 4)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 8)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 12)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))

                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                    (br $copy_loop2)
                  )
                )

                (i32.store (call $abs (i32.add (local.get $receiver_ptr) (i32.const 12))) (local.get $new_arr))
                (i32.store (call $abs (i32.add (local.get $receiver_ptr) (i32.const 16))) (local.get $new_data))
                (local.set $data_ptr (local.get $new_data))
              )
              (else
                ;; No realloc needed - shift existing elements up by 1
                ;; Go backwards to avoid overwriting
                (local.set $i (local.get $arr_len))
                (block $shift_up_done
                  (loop $shift_up_loop
                    (br_if $shift_up_done (i32.eqz (local.get $i)))
                    (local.set $i (i32.sub (local.get $i) (i32.const 1)))

                    ;; src = data_ptr + 8 + i * 16
                    (local.set $elem_ptr (i32.add
                      (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                    ;; dest = data_ptr + 8 + (i+1) * 16
                    (local.set $str_ptr (i32.add
                      (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (i32.add (local.get $i) (i32.const 1)) (global.get $VALUE_SIZE))))

                    (i32.store (call $abs (local.get $str_ptr))
                      (i32.load (call $abs (local.get $elem_ptr))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 4)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 8)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 12)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))

                    (br $shift_up_loop)
                  )
                )
              )
            )

            ;; Insert the new element at position 0
            (local.set $elem_ptr (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE)))
            (i32.store (call $abs (local.get $elem_ptr)) (local.get $arg1_type))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $arg1_lo))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (local.get $arg1_hi))

            ;; Increment length
            (local.set $arr_len (i32.add (local.get $arr_len) (i32.const 1)))
            (i32.store (call $abs (i32.add (local.get $receiver_ptr) (i32.const 8))) (local.get $arr_len))

            ;; Return new length
            (call $push_f64 (f64.convert_i32_u (local.get $arr_len)))
            (return)
          )
        )

        ;; indexOf(value) - returns index or -1
        (if (i32.eq (local.get $method_id) (global.get $METHOD_INDEX_OF))
          (then
            ;; Search for arg1 in array using strict equality
            (local.set $i (i32.const 0))
            (local.set $found (i32.const 0))
            (block $index_found
              (loop $index_loop
                (br_if $index_found (i32.ge_u (local.get $i) (local.get $arr_len)))

                ;; Get element at index i
                (local.set $elem_ptr (i32.add
                  (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (global.get $VALUE_SIZE))))

                ;; Compare type and value
                (local.set $arg2_type (i32.load (call $abs (local.get $elem_ptr))))
                (local.set $arg2_lo (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))

                ;; Strict equality: types must match
                (if (i32.eq (local.get $arg1_type) (local.get $arg2_type))
                  (then
                    ;; For primitives, compare data_lo
                    (if (i32.or
                          (i32.or
                            (i32.eq (local.get $arg1_type) (global.get $TYPE_NULL))
                            (i32.eq (local.get $arg1_type) (global.get $TYPE_UNDEFINED)))
                          (i32.or
                            (i32.eq (local.get $arg1_type) (global.get $TYPE_BOOLEAN))
                            (i32.eq (local.get $arg1_type) (global.get $TYPE_INTEGER))))
                      (then
                        (if (i32.eq (local.get $arg1_lo) (local.get $arg2_lo))
                          (then
                            (local.set $found (i32.const 1))
                            (br $index_found)
                          )
                        )
                      )
                    )
                    ;; For floats, need to compare both lo and hi
                    (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
                      (then
                        (if (i32.and
                              (i32.eq (local.get $arg1_lo) (local.get $arg2_lo))
                              (i32.eq (local.get $arg1_hi)
                                (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))))))
                          (then
                            (local.set $found (i32.const 1))
                            (br $index_found)
                          )
                        )
                      )
                    )
                    ;; For strings, compare string offsets (interned)
                    (if (i32.eq (local.get $arg1_type) (global.get $TYPE_STRING))
                      (then
                        (if (i32.eq (local.get $arg1_lo) (local.get $arg2_lo))
                          (then
                            (local.set $found (i32.const 1))
                            (br $index_found)
                          )
                        )
                      )
                    )
                    ;; For objects/arrays, compare by reference (pointer)
                    (if (i32.or
                          (i32.eq (local.get $arg1_type) (global.get $TYPE_ARRAY))
                          (i32.eq (local.get $arg1_type) (global.get $TYPE_OBJECT)))
                      (then
                        (if (i32.eq (local.get $arg1_lo) (local.get $arg2_lo))
                          (then
                            (local.set $found (i32.const 1))
                            (br $index_found)
                          )
                        )
                      )
                    )
                  )
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $index_loop)
              )
            )

            (if (local.get $found)
              (then (call $push_f64 (f64.convert_i32_s (local.get $i))))
              (else (call $push_f64 (f64.const -1)))
            )
            (return)
          )
        )

        ;; includes(value) - returns boolean
        (if (i32.eq (local.get $method_id) (global.get $METHOD_INCLUDES))
          (then
            ;; Same logic as indexOf but returns boolean
            (local.set $i (i32.const 0))
            (local.set $found (i32.const 0))
            (block $incl_found
              (loop $incl_loop
                (br_if $incl_found (i32.ge_u (local.get $i) (local.get $arr_len)))

                (local.set $elem_ptr (i32.add
                  (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (global.get $VALUE_SIZE))))

                (local.set $arg2_type (i32.load (call $abs (local.get $elem_ptr))))
                (local.set $arg2_lo (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))

                (if (i32.eq (local.get $arg1_type) (local.get $arg2_type))
                  (then
                    ;; For primitives (null, undefined, boolean, integer)
                    (if (i32.or
                          (i32.or
                            (i32.eq (local.get $arg1_type) (global.get $TYPE_NULL))
                            (i32.eq (local.get $arg1_type) (global.get $TYPE_UNDEFINED)))
                          (i32.or
                            (i32.eq (local.get $arg1_type) (global.get $TYPE_BOOLEAN))
                            (i32.eq (local.get $arg1_type) (global.get $TYPE_INTEGER))))
                      (then
                        (if (i32.eq (local.get $arg1_lo) (local.get $arg2_lo))
                          (then (local.set $found (i32.const 1)) (br $incl_found))
                        )
                      )
                    )
                    ;; For floats
                    (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
                      (then
                        (if (i32.and
                              (i32.eq (local.get $arg1_lo) (local.get $arg2_lo))
                              (i32.eq (local.get $arg1_hi)
                                (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))))))
                          (then (local.set $found (i32.const 1)) (br $incl_found))
                        )
                      )
                    )
                    ;; For strings
                    (if (i32.eq (local.get $arg1_type) (global.get $TYPE_STRING))
                      (then
                        (if (i32.eq (local.get $arg1_lo) (local.get $arg2_lo))
                          (then (local.set $found (i32.const 1)) (br $incl_found))
                        )
                      )
                    )
                    ;; For objects/arrays
                    (if (i32.or
                          (i32.eq (local.get $arg1_type) (global.get $TYPE_ARRAY))
                          (i32.eq (local.get $arg1_type) (global.get $TYPE_OBJECT)))
                      (then
                        (if (i32.eq (local.get $arg1_lo) (local.get $arg2_lo))
                          (then (local.set $found (i32.const 1)) (br $incl_found))
                        )
                      )
                    )
                  )
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $incl_loop)
              )
            )

            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (local.get $found) (i32.const 0))
            (return)
          )
        )

        ;; slice(start, end) - returns new array
        (if (i32.eq (local.get $method_id) (global.get $METHOD_SLICE))
          (then
            ;; Handle start index (arg1)
            ;; If no args, start = 0
            (if (i32.eqz (local.get $argc))
              (then (local.set $i (i32.const 0)))
              (else
                ;; Convert arg1 to int
                (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
                  (then
                    (local.set $i (i32.trunc_f64_s (call $read_f64 (local.get $arg1_ptr))))
                  )
                  (else (local.set $i (i32.const 0)))
                )
                ;; Handle negative index
                (if (i32.lt_s (local.get $i) (i32.const 0))
                  (then
                    (local.set $i (i32.add (local.get $arr_len) (local.get $i)))
                    (if (i32.lt_s (local.get $i) (i32.const 0))
                      (then (local.set $i (i32.const 0)))
                    )
                  )
                )
                ;; Clamp to arr_len
                (if (i32.gt_s (local.get $i) (local.get $arr_len))
                  (then (local.set $i (local.get $arr_len)))
                )
              )
            )

            ;; Handle end index (arg2 or arr_len)
            (if (i32.lt_u (local.get $argc) (i32.const 2))
              (then (local.set $found (local.get $arr_len)))  ;; reuse $found as end index
              (else
                (if (i32.eq (local.get $arg2_type) (global.get $TYPE_FLOAT))
                  (then
                    (local.set $found (i32.trunc_f64_s (call $read_f64 (local.get $arg2_ptr))))
                  )
                  (else (local.set $found (local.get $arr_len)))
                )
                ;; Handle negative index
                (if (i32.lt_s (local.get $found) (i32.const 0))
                  (then
                    (local.set $found (i32.add (local.get $arr_len) (local.get $found)))
                    (if (i32.lt_s (local.get $found) (i32.const 0))
                      (then (local.set $found (i32.const 0)))
                    )
                  )
                )
                ;; Clamp to arr_len
                (if (i32.gt_s (local.get $found) (local.get $arr_len))
                  (then (local.set $found (local.get $arr_len)))
                )
              )
            )

            ;; Calculate slice length
            (local.set $search_len (i32.sub (local.get $found) (local.get $i)))
            (if (i32.lt_s (local.get $search_len) (i32.const 0))
              (then (local.set $search_len (i32.const 0)))
            )

            ;; Allocate new array
            (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
            (local.set $new_arr (local.get $heap_ptr))

            ;; Array header
            (i32.store (call $abs (local.get $new_arr))
              (i32.or (i32.const 20) (i32.shl (global.get $OBJ_ARRAY) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 4))) (i32.const 0))

            ;; Data block
            (local.set $new_data (i32.add (local.get $new_arr) (i32.const 20)))
            (local.set $i32_result (i32.add (global.get $GC_HEADER_SIZE)
              (i32.mul (local.get $search_len) (global.get $VALUE_SIZE))))

            (i32.store (call $abs (local.get $new_data))
              (i32.or (local.get $i32_result) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $new_data) (i32.const 4))) (i32.const 0))

            ;; Array header fields
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 8))) (local.get $search_len))  ;; length
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 12))) (local.get $search_len)) ;; capacity
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 16))) (local.get $new_data))   ;; data_ptr

            ;; Update heap pointer
            (call $write_state (global.get $STATE_HEAP_POINTER)
              (i32.add (local.get $new_data) (local.get $i32_result)))

            ;; Copy elements from [i, found)
            (local.set $char_code (i32.const 0))  ;; reuse as copy index
            (block $slice_copy_done
              (loop $slice_copy_loop
                (br_if $slice_copy_done (i32.ge_u (local.get $char_code) (local.get $search_len)))

                ;; src = data_ptr + 8 + (i + char_code) * 16
                (local.set $elem_ptr (i32.add
                  (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (i32.add (local.get $i) (local.get $char_code)) (global.get $VALUE_SIZE))))
                ;; dest = new_data + 8 + char_code * 16
                (local.set $str_ptr (i32.add
                  (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $char_code) (global.get $VALUE_SIZE))))

                ;; Copy 16 bytes
                (i32.store (call $abs (local.get $str_ptr))
                  (i32.load (call $abs (local.get $elem_ptr))))
                (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 4)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 8)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 12)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))

                (local.set $char_code (i32.add (local.get $char_code) (i32.const 1)))
                (br $slice_copy_loop)
              )
            )

            ;; Push new array reference
            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
            (return)
          )
        )

        ;; concat(arr) - returns new array
        (if (i32.eq (local.get $method_id) (global.get $METHOD_CONCAT))
          (then
            ;; Get the other array's length and data
            (if (i32.ne (local.get $arg1_type) (global.get $TYPE_ARRAY))
              (then
                ;; If arg is not an array, just return a copy of this array
                ;; (simplified - real JS would wrap in array)
                (local.set $search_len (i32.const 0))
              )
              (else
                ;; arg1_lo is the other array pointer
                (local.set $search_len (i32.load (call $abs (i32.add (local.get $arg1_lo) (i32.const 8)))))
                (local.set $search_ptr (i32.load (call $abs (i32.add (local.get $arg1_lo) (i32.const 16)))))
              )
            )

            ;; Total length = arr_len + search_len
            (local.set $i32_result (i32.add (local.get $arr_len) (local.get $search_len)))

            ;; Allocate new array
            (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
            (local.set $new_arr (local.get $heap_ptr))

            ;; Array header
            (i32.store (call $abs (local.get $new_arr))
              (i32.or (i32.const 20) (i32.shl (global.get $OBJ_ARRAY) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 4))) (i32.const 0))

            ;; Data block
            (local.set $new_data (i32.add (local.get $new_arr) (i32.const 20)))
            (local.set $char_code (i32.add (global.get $GC_HEADER_SIZE)
              (i32.mul (local.get $i32_result) (global.get $VALUE_SIZE))))

            (i32.store (call $abs (local.get $new_data))
              (i32.or (local.get $char_code) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $new_data) (i32.const 4))) (i32.const 0))

            ;; Array header fields
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 8))) (local.get $i32_result))
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 12))) (local.get $i32_result))
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 16))) (local.get $new_data))

            ;; Update heap pointer
            (call $write_state (global.get $STATE_HEAP_POINTER)
              (i32.add (local.get $new_data) (local.get $char_code)))

            ;; Copy elements from first array
            (local.set $i (i32.const 0))
            (block $concat1_done
              (loop $concat1_loop
                (br_if $concat1_done (i32.ge_u (local.get $i) (local.get $arr_len)))

                (local.set $elem_ptr (i32.add
                  (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                (local.set $str_ptr (i32.add
                  (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (global.get $VALUE_SIZE))))

                (i32.store (call $abs (local.get $str_ptr))
                  (i32.load (call $abs (local.get $elem_ptr))))
                (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 4)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 8)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 12)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $concat1_loop)
              )
            )

            ;; Copy elements from second array (if it was an array)
            (if (i32.gt_u (local.get $search_len) (i32.const 0))
              (then
                (local.set $i (i32.const 0))
                (block $concat2_done
                  (loop $concat2_loop
                    (br_if $concat2_done (i32.ge_u (local.get $i) (local.get $search_len)))

                    (local.set $elem_ptr (i32.add
                      (i32.add (local.get $search_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                    (local.set $str_ptr (i32.add
                      (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
                      (i32.mul (i32.add (local.get $arr_len) (local.get $i)) (global.get $VALUE_SIZE))))

                    (i32.store (call $abs (local.get $str_ptr))
                      (i32.load (call $abs (local.get $elem_ptr))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 4)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 8)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                    (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 12)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))

                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                    (br $concat2_loop)
                  )
                )
              )
            )

            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
            (return)
          )
        )

        ;; join(sep) - returns string
        ;; Joins array elements with separator. Handles strings directly,
        ;; integers as decimal, other types as empty string.
        (if (i32.eq (local.get $method_id) (global.get $METHOD_JOIN))
          (then
            ;; Get separator (default to ",")
            (if (i32.and
                  (i32.ge_u (local.get $argc) (i32.const 1))
                  (i32.eq (local.get $arg1_type) (global.get $TYPE_STRING)))
              (then
                (local.set $search_ptr (local.get $arg1_lo))
                (local.set $search_len (i32.load (call $abs (local.get $search_ptr))))
              )
              (else
                ;; Default separator is ","
                ;; Allocate a comma string
                (local.set $search_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                (i32.store (call $abs (local.get $search_ptr)) (i32.const 1))
                (i32.store8 (call $abs (i32.add (local.get $search_ptr) (i32.const 4))) (i32.const 44)) ;; ','
                (call $write_state (global.get $STATE_STRING_POINTER)
                  (i32.add (local.get $search_ptr) (i32.const 8)))
                (local.set $search_len (i32.const 1))
              )
            )

            ;; Empty array returns empty string
            (if (i32.eqz (local.get $arr_len))
              (then
                (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                (i32.store (call $abs (local.get $new_str_ptr)) (i32.const 0))
                (call $write_state (global.get $STATE_STRING_POINTER)
                  (i32.add (local.get $new_str_ptr) (i32.const 4)))
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $new_str_ptr) (i32.const 0))
                (return)
              )
            )

            ;; First pass: calculate total length needed
            (local.set $slice_len (i32.const 0))
            (local.set $i (i32.const 0))
            (block $len_done
              (loop $len_loop
                (br_if $len_done (i32.ge_u (local.get $i) (local.get $arr_len)))

                ;; Add separator length (except before first element)
                (if (i32.gt_u (local.get $i) (i32.const 0))
                  (then
                    (local.set $slice_len (i32.add (local.get $slice_len) (local.get $search_len)))
                  )
                )

                ;; Get element
                (local.set $elem_ptr (i32.add
                  (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                (local.set $char_code (call $value_type (local.get $elem_ptr)))

                ;; Add element length based on type
                (if (i32.eq (local.get $char_code) (global.get $TYPE_STRING))
                  (then
                    ;; String: add its byte length
                    (local.set $str_ptr (call $value_data_lo (local.get $elem_ptr)))
                    (local.set $slice_len (i32.add (local.get $slice_len)
                      (i32.load (call $abs (local.get $str_ptr)))))
                  )
                )
                ;; Other types contribute 0 length for now (simplified)

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $len_loop)
              )
            )

            ;; Allocate result string
            (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
            (i32.store (call $abs (local.get $new_str_ptr)) (local.get $slice_len))

            ;; Second pass: copy data
            (local.set $count (i32.const 0))  ;; write position
            (local.set $i (i32.const 0))
            (block $copy_done
              (loop $copy_loop
                (br_if $copy_done (i32.ge_u (local.get $i) (local.get $arr_len)))

                ;; Add separator (except before first element)
                (if (i32.gt_u (local.get $i) (i32.const 0))
                  (then
                    (local.set $j (i32.const 0))
                    (block $sep_done
                      (loop $sep_loop
                        (br_if $sep_done (i32.ge_u (local.get $j) (local.get $search_len)))
                        (i32.store8 (call $abs (i32.add
                          (i32.add (local.get $new_str_ptr) (i32.const 4))
                          (local.get $count)))
                          (i32.load8_u (call $abs (i32.add
                            (i32.add (local.get $search_ptr) (i32.const 4))
                            (local.get $j)))))
                        (local.set $count (i32.add (local.get $count) (i32.const 1)))
                        (local.set $j (i32.add (local.get $j) (i32.const 1)))
                        (br $sep_loop)
                      )
                    )
                  )
                )

                ;; Get element
                (local.set $elem_ptr (i32.add
                  (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                (local.set $char_code (call $value_type (local.get $elem_ptr)))

                ;; Copy element based on type
                (if (i32.eq (local.get $char_code) (global.get $TYPE_STRING))
                  (then
                    (local.set $str_ptr (call $value_data_lo (local.get $elem_ptr)))
                    (local.set $target_len (i32.load (call $abs (local.get $str_ptr))))
                    (local.set $j (i32.const 0))
                    (block $elem_done
                      (loop $elem_loop
                        (br_if $elem_done (i32.ge_u (local.get $j) (local.get $target_len)))
                        (i32.store8 (call $abs (i32.add
                          (i32.add (local.get $new_str_ptr) (i32.const 4))
                          (local.get $count)))
                          (i32.load8_u (call $abs (i32.add
                            (i32.add (local.get $str_ptr) (i32.const 4))
                            (local.get $j)))))
                        (local.set $count (i32.add (local.get $count) (i32.const 1)))
                        (local.set $j (i32.add (local.get $j) (i32.const 1)))
                        (br $elem_loop)
                      )
                    )
                  )
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $copy_loop)
              )
            )

            ;; Update string pointer (align)
            (call $write_state (global.get $STATE_STRING_POINTER)
              (i32.and
                (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (i32.add (local.get $slice_len) (i32.const 3)))
                (i32.const 0xFFFFFFFC)))

            (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
              (local.get $new_str_ptr) (i32.const 0))
            (return)
          )
        )

        ;; reverse() - mutates array in place and returns it
        (if (i32.eq (local.get $method_id) (global.get $METHOD_REVERSE))
          (then
            ;; Swap elements from both ends towards middle
            (local.set $i (i32.const 0))
            (local.set $found (i32.sub (local.get $arr_len) (i32.const 1)))  ;; end index

            (block $reverse_done
              (loop $reverse_loop
                (br_if $reverse_done (i32.ge_u (local.get $i) (local.get $found)))

                ;; Get pointers to elements at i and found
                (local.set $elem_ptr (i32.add
                  (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                (local.set $str_ptr (i32.add
                  (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $found) (global.get $VALUE_SIZE))))

                ;; Save element at i (using locals)
                (local.set $arg1_type (i32.load (call $abs (local.get $elem_ptr))))
                (local.set $i32_result (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                (local.set $arg1_lo (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                (local.set $arg1_hi (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))

                ;; Copy element at found to i
                (i32.store (call $abs (local.get $elem_ptr))
                  (i32.load (call $abs (local.get $str_ptr))))
                (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))
                  (i32.load (call $abs (i32.add (local.get $str_ptr) (i32.const 4)))))
                (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))
                  (i32.load (call $abs (i32.add (local.get $str_ptr) (i32.const 8)))))
                (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))
                  (i32.load (call $abs (i32.add (local.get $str_ptr) (i32.const 12)))))

                ;; Copy saved element to found
                (i32.store (call $abs (local.get $str_ptr)) (local.get $arg1_type))
                (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 4))) (local.get $i32_result))
                (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 8))) (local.get $arg1_lo))
                (i32.store (call $abs (i32.add (local.get $str_ptr) (i32.const 12))) (local.get $arg1_hi))

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (local.set $found (i32.sub (local.get $found) (i32.const 1)))
                (br $reverse_loop)
              )
            )

            ;; Return the same array (now reversed)
            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $receiver_ptr) (i32.const 0))
            (return)
          )
        )

        ;; Callback methods (map, filter, reduce, forEach, find, findIndex, some, every)
        ;; These are handled directly in OP_CALL_METHOD, not here
        ;; This should never be reached for callback methods
        (if (call $is_callback_method (local.get $method_id))
          (then
            ;; Should not reach here - callback methods handled in OP_CALL_METHOD
            (call $set_error (global.get $ERR_NOT_CALLABLE) (local.get $method_id))
            (return)
          )
        )
      )
    )

    (if (i32.eq (local.get $receiver_type) (global.get $TYPE_STRING))
      (then
        ;; ============================================
        ;; STRING METHODS
        ;; ============================================
        ;; String table entry: [length:4][bytes:N]
        ;; receiver_ptr is the string table offset
        (local.set $str_len (i32.load (call $abs (local.get $receiver_ptr))))

        ;; Clean up stack first: reset pending pointer to where we'll write the result
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))

        ;; charAt(index) - returns single character string (UTF-8 aware)
        (if (i32.eq (local.get $method_id) (global.get $METHOD_CHAR_AT))
          (then
            ;; Get character index - convert from f64 if needed
            (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (then
                (local.set $i (i32.trunc_f64_s (call $read_f64 (local.get $arg1_ptr))))
              )
              (else
                (if (i32.eq (local.get $arg1_type) (global.get $TYPE_INTEGER))
                  (then (local.set $i (local.get $arg1_lo)))
                  (else (local.set $i (i32.const 0)))
                )
              )
            )

            ;; Get byte offset for character index
            (local.set $start_idx (call $utf8_char_to_byte_offset (local.get $receiver_ptr) (local.get $i)))

            ;; Check bounds - return empty string if out of range
            (if (i32.or
                  (i32.lt_s (local.get $i) (i32.const 0))
                  (i32.lt_s (local.get $start_idx) (i32.const 0)))
              (then
                ;; Empty string: allocate 4 bytes (length=0)
                (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                (i32.store (call $abs (local.get $new_str_ptr)) (i32.const 0))
                (call $write_state (global.get $STATE_STRING_POINTER)
                  (i32.add (local.get $new_str_ptr) (i32.const 4)))
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $new_str_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Also check if byte offset equals or exceeds byte length
            (if (i32.ge_s (local.get $start_idx) (local.get $str_len))
              (then
                (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                (i32.store (call $abs (local.get $new_str_ptr)) (i32.const 0))
                (call $write_state (global.get $STATE_STRING_POINTER)
                  (i32.add (local.get $new_str_ptr) (i32.const 4)))
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $new_str_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Get byte length of this UTF-8 character
            (local.set $slice_len (call $utf8_char_byte_len (local.get $receiver_ptr) (local.get $start_idx)))

            ;; Allocate new string with appropriate byte length
            (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
            (i32.store (call $abs (local.get $new_str_ptr)) (local.get $slice_len))

            ;; Copy the UTF-8 bytes
            (local.set $j (i32.const 0))
            (block $copy_done
              (loop $copy_loop
                (br_if $copy_done (i32.ge_u (local.get $j) (local.get $slice_len)))
                (i32.store8 (call $abs (i32.add
                  (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (local.get $j)))
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (i32.add (local.get $start_idx) (local.get $j))))))
                (local.set $j (i32.add (local.get $j) (i32.const 1)))
                (br $copy_loop)
              )
            )

            ;; Update string pointer (align to 4 bytes)
            (call $write_state (global.get $STATE_STRING_POINTER)
              (i32.and
                (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (i32.add (local.get $slice_len) (i32.const 3)))
                (i32.const 0xFFFFFFFC)))

            (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
              (local.get $new_str_ptr) (i32.const 0))
            (return)
          )
        )

        ;; charCodeAt(index) - returns Unicode code point
        (if (i32.eq (local.get $method_id) (global.get $METHOD_CHAR_CODE_AT))
          (then
            ;; Get character index - convert from f64 if needed
            (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (then
                (local.set $i (i32.trunc_f64_s (call $read_f64 (local.get $arg1_ptr))))
              )
              (else
                (if (i32.eq (local.get $arg1_type) (global.get $TYPE_INTEGER))
                  (then (local.set $i (local.get $arg1_lo)))
                  (else (local.set $i (i32.const 0)))
                )
              )
            )

            ;; Get byte offset for character index
            (local.set $start_idx (call $utf8_char_to_byte_offset (local.get $receiver_ptr) (local.get $i)))

            ;; Check bounds - return NaN if out of range (JS behavior)
            (if (i32.or
                  (i32.lt_s (local.get $i) (i32.const 0))
                  (i32.lt_s (local.get $start_idx) (i32.const 0)))
              (then
                ;; Return NaN (0x7FF8000000000000)
                (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0)
                  (i32.const 0x00000000)  ;; lo bits of NaN
                  (i32.const 0x7FF80000)) ;; hi bits of NaN
                (return)
              )
            )

            ;; Also check if byte offset equals byte length (past end)
            (if (i32.ge_s (local.get $start_idx) (local.get $str_len))
              (then
                (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0)
                  (i32.const 0x00000000)
                  (i32.const 0x7FF80000))
                (return)
              )
            )

            ;; Decode UTF-8 code point at byte offset
            (local.set $char_code (call $utf8_decode_at (local.get $receiver_ptr) (local.get $start_idx)))

            ;; Return the code as a number
            (call $push_f64 (f64.convert_i32_u (local.get $char_code)))
            (return)
          )
        )

        ;; indexOf(search) - returns index or -1
        (if (i32.eq (local.get $method_id) (global.get $METHOD_STRING_INDEX_OF))
          (then
            ;; Get search string info
            (if (i32.ne (local.get $arg1_type) (global.get $TYPE_STRING))
              (then
                ;; Non-string search returns -1
                (call $push_f64 (f64.const -1))
                (return)
              )
            )

            ;; search_ptr = arg1_lo (string table offset)
            (local.set $search_ptr (local.get $arg1_lo))
            (local.set $search_len (i32.load (call $abs (local.get $search_ptr))))

            ;; Empty search string found at index 0
            (if (i32.eqz (local.get $search_len))
              (then
                (call $push_f64 (f64.const 0))
                (return)
              )
            )

            ;; Search string longer than haystack - not found
            (if (i32.gt_u (local.get $search_len) (local.get $str_len))
              (then
                (call $push_f64 (f64.const -1))
                (return)
              )
            )

            ;; Search through the string
            ;; Can stop at str_len - search_len + 1
            (local.set $i (i32.const 0))
            (local.set $found (i32.const 0))
            (block $str_indexOf_done
              (loop $str_indexOf_loop
                (br_if $str_indexOf_done
                  (i32.gt_u (local.get $i) (i32.sub (local.get $str_len) (local.get $search_len))))

                ;; Compare bytes at position i
                (local.set $found (i32.const 1))
                (local.set $char_code (i32.const 0))
                (block $cmp_done
                  (loop $cmp_loop
                    (br_if $cmp_done (i32.ge_u (local.get $char_code) (local.get $search_len)))

                    ;; Get byte from haystack at receiver_ptr + 4 + i + char_code
                    (local.set $byte_ptr
                      (i32.load8_u (call $abs (i32.add
                        (i32.add (local.get $receiver_ptr) (i32.const 4))
                        (i32.add (local.get $i) (local.get $char_code))))))

                    ;; Get byte from needle at search_ptr + 4 + char_code
                    (local.set $elem_ptr
                      (i32.load8_u (call $abs (i32.add
                        (i32.add (local.get $search_ptr) (i32.const 4))
                        (local.get $char_code)))))

                    ;; Compare
                    (if (i32.ne (local.get $byte_ptr) (local.get $elem_ptr))
                      (then
                        (local.set $found (i32.const 0))
                        (br $cmp_done)
                      )
                    )

                    (local.set $char_code (i32.add (local.get $char_code) (i32.const 1)))
                    (br $cmp_loop)
                  )
                )

                (if (local.get $found)
                  (then (br $str_indexOf_done))
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $str_indexOf_loop)
              )
            )

            (if (local.get $found)
              (then (call $push_f64 (f64.convert_i32_s (local.get $i))))
              (else (call $push_f64 (f64.const -1)))
            )
            (return)
          )
        )

        ;; includes(search) - returns boolean (uses indexOf logic)
        (if (i32.eq (local.get $method_id) (global.get $METHOD_STRING_INCLUDES))
          (then
            ;; Get search string info
            (if (i32.ne (local.get $arg1_type) (global.get $TYPE_STRING))
              (then
                ;; Non-string search returns false
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
                (return)
              )
            )

            (local.set $search_ptr (local.get $arg1_lo))
            (local.set $search_len (i32.load (call $abs (local.get $search_ptr))))

            ;; Empty search string always found
            (if (i32.eqz (local.get $search_len))
              (then
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))
                (return)
              )
            )

            ;; Search string longer than haystack
            (if (i32.gt_u (local.get $search_len) (local.get $str_len))
              (then
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
                (return)
              )
            )

            ;; Search through the string
            (local.set $i (i32.const 0))
            (local.set $found (i32.const 0))
            (block $str_incl_done
              (loop $str_incl_loop
                (br_if $str_incl_done
                  (i32.gt_u (local.get $i) (i32.sub (local.get $str_len) (local.get $search_len))))

                (local.set $found (i32.const 1))
                (local.set $char_code (i32.const 0))
                (block $cmp_done2
                  (loop $cmp_loop2
                    (br_if $cmp_done2 (i32.ge_u (local.get $char_code) (local.get $search_len)))

                    (local.set $byte_ptr
                      (i32.load8_u (call $abs (i32.add
                        (i32.add (local.get $receiver_ptr) (i32.const 4))
                        (i32.add (local.get $i) (local.get $char_code))))))

                    (local.set $elem_ptr
                      (i32.load8_u (call $abs (i32.add
                        (i32.add (local.get $search_ptr) (i32.const 4))
                        (local.get $char_code)))))

                    (if (i32.ne (local.get $byte_ptr) (local.get $elem_ptr))
                      (then
                        (local.set $found (i32.const 0))
                        (br $cmp_done2)
                      )
                    )

                    (local.set $char_code (i32.add (local.get $char_code) (i32.const 1)))
                    (br $cmp_loop2)
                  )
                )

                (if (local.get $found)
                  (then (br $str_incl_done))
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $str_incl_loop)
              )
            )

            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (local.get $found) (i32.const 0))
            (return)
          )
        )

        ;; slice(start, end) - returns substring
        (if (i32.eq (local.get $method_id) (global.get $METHOD_STRING_SLICE))
          (then
            ;; Get character count for proper bounds checking
            (local.set $char_count (call $utf8_char_count (local.get $receiver_ptr)))

            ;; Get start index (character index)
            (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (then
                (local.set $start_idx (i32.trunc_f64_s (call $read_f64 (local.get $arg1_ptr))))
              )
              (else
                (if (i32.eq (local.get $arg1_type) (global.get $TYPE_INTEGER))
                  (then (local.set $start_idx (local.get $arg1_lo)))
                  (else (local.set $start_idx (i32.const 0)))
                )
              )
            )

            ;; Handle negative start (relative to char count)
            (if (i32.lt_s (local.get $start_idx) (i32.const 0))
              (then
                (local.set $start_idx (i32.add (local.get $char_count) (local.get $start_idx)))
                (if (i32.lt_s (local.get $start_idx) (i32.const 0))
                  (then (local.set $start_idx (i32.const 0)))
                )
              )
            )

            ;; Get end index (default to char_count if not provided)
            (if (i32.ge_u (local.get $argc) (i32.const 2))
              (then
                (if (i32.eq (local.get $arg2_type) (global.get $TYPE_FLOAT))
                  (then
                    (local.set $end_idx (i32.trunc_f64_s (call $read_f64 (local.get $arg2_ptr))))
                  )
                  (else
                    (if (i32.eq (local.get $arg2_type) (global.get $TYPE_INTEGER))
                      (then (local.set $end_idx (call $value_data_lo (local.get $arg2_ptr))))
                      (else (local.set $end_idx (local.get $char_count)))
                    )
                  )
                )
              )
              (else
                (local.set $end_idx (local.get $char_count))
              )
            )

            ;; Handle negative end
            (if (i32.lt_s (local.get $end_idx) (i32.const 0))
              (then
                (local.set $end_idx (i32.add (local.get $char_count) (local.get $end_idx)))
                (if (i32.lt_s (local.get $end_idx) (i32.const 0))
                  (then (local.set $end_idx (i32.const 0)))
                )
              )
            )

            ;; Clamp end to char_count
            (if (i32.gt_s (local.get $end_idx) (local.get $char_count))
              (then (local.set $end_idx (local.get $char_count)))
            )

            ;; Clamp start to char_count
            (if (i32.gt_s (local.get $start_idx) (local.get $char_count))
              (then (local.set $start_idx (local.get $char_count)))
            )

            ;; If start >= end, return empty string
            (if (i32.ge_s (local.get $start_idx) (local.get $end_idx))
              (then
                (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                (i32.store (call $abs (local.get $new_str_ptr)) (i32.const 0))
                (call $write_state (global.get $STATE_STRING_POINTER)
                  (i32.add (local.get $new_str_ptr) (i32.const 4)))
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $new_str_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Convert character indices to byte offsets
            (local.set $start_byte (call $utf8_char_to_byte_offset (local.get $receiver_ptr) (local.get $start_idx)))
            (local.set $end_byte (call $utf8_char_to_byte_offset (local.get $receiver_ptr) (local.get $end_idx)))

            ;; Calculate byte length of slice
            (local.set $slice_len (i32.sub (local.get $end_byte) (local.get $start_byte)))

            ;; Allocate new string
            (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
            (i32.store (call $abs (local.get $new_str_ptr)) (local.get $slice_len))

            ;; Copy bytes
            (local.set $i (i32.const 0))
            (block $copy_done
              (loop $copy_loop
                (br_if $copy_done (i32.ge_u (local.get $i) (local.get $slice_len)))

                ;; Get byte from source at start_byte + i
                (local.set $char_code
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (i32.add (local.get $start_byte) (local.get $i))))))

                ;; Store byte to destination
                (i32.store8 (call $abs (i32.add
                  (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (local.get $i)))
                  (local.get $char_code))

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $copy_loop)
              )
            )

            ;; Update string pointer (align to 4 bytes)
            (call $write_state (global.get $STATE_STRING_POINTER)
              (i32.and
                (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4)) (i32.add (local.get $slice_len) (i32.const 3)))
                (i32.const 0xFFFFFFFC)))

            (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
              (local.get $new_str_ptr) (i32.const 0))
            (return)
          )
        )

        ;; substring(start, end) - returns substring
        ;; Unlike slice: negative becomes 0, if start > end they swap
        (if (i32.eq (local.get $method_id) (global.get $METHOD_SUBSTRING))
          (then
            ;; Get character count for proper bounds checking
            (local.set $char_count (call $utf8_char_count (local.get $receiver_ptr)))

            ;; Get start index (character index)
            (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (then
                (local.set $start_idx (i32.trunc_f64_s (call $read_f64 (local.get $arg1_ptr))))
              )
              (else
                (if (i32.eq (local.get $arg1_type) (global.get $TYPE_INTEGER))
                  (then (local.set $start_idx (local.get $arg1_lo)))
                  (else (local.set $start_idx (i32.const 0)))
                )
              )
            )

            ;; Negative becomes 0
            (if (i32.lt_s (local.get $start_idx) (i32.const 0))
              (then (local.set $start_idx (i32.const 0)))
            )

            ;; Get end index (default to char_count if not provided)
            (if (i32.ge_u (local.get $argc) (i32.const 2))
              (then
                (if (i32.eq (local.get $arg2_type) (global.get $TYPE_FLOAT))
                  (then
                    (local.set $end_idx (i32.trunc_f64_s (call $read_f64 (local.get $arg2_ptr))))
                  )
                  (else
                    (if (i32.eq (local.get $arg2_type) (global.get $TYPE_INTEGER))
                      (then (local.set $end_idx (call $value_data_lo (local.get $arg2_ptr))))
                      (else (local.set $end_idx (local.get $char_count)))
                    )
                  )
                )
              )
              (else
                (local.set $end_idx (local.get $char_count))
              )
            )

            ;; Negative becomes 0
            (if (i32.lt_s (local.get $end_idx) (i32.const 0))
              (then (local.set $end_idx (i32.const 0)))
            )

            ;; Clamp to char_count
            (if (i32.gt_s (local.get $start_idx) (local.get $char_count))
              (then (local.set $start_idx (local.get $char_count)))
            )
            (if (i32.gt_s (local.get $end_idx) (local.get $char_count))
              (then (local.set $end_idx (local.get $char_count)))
            )

            ;; If start > end, swap them (substring behavior)
            (if (i32.gt_s (local.get $start_idx) (local.get $end_idx))
              (then
                (local.set $i (local.get $start_idx))
                (local.set $start_idx (local.get $end_idx))
                (local.set $end_idx (local.get $i))
              )
            )

            ;; If start == end, return empty string
            (if (i32.eq (local.get $start_idx) (local.get $end_idx))
              (then
                (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                (i32.store (call $abs (local.get $new_str_ptr)) (i32.const 0))
                (call $write_state (global.get $STATE_STRING_POINTER)
                  (i32.add (local.get $new_str_ptr) (i32.const 4)))
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $new_str_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Convert character indices to byte offsets
            (local.set $start_byte (call $utf8_char_to_byte_offset (local.get $receiver_ptr) (local.get $start_idx)))
            (local.set $end_byte (call $utf8_char_to_byte_offset (local.get $receiver_ptr) (local.get $end_idx)))

            ;; Calculate byte length of slice
            (local.set $slice_len (i32.sub (local.get $end_byte) (local.get $start_byte)))

            ;; Allocate new string
            (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
            (i32.store (call $abs (local.get $new_str_ptr)) (local.get $slice_len))

            ;; Copy bytes
            (local.set $i (i32.const 0))
            (block $ss_copy_done
              (loop $ss_copy_loop
                (br_if $ss_copy_done (i32.ge_u (local.get $i) (local.get $slice_len)))

                ;; Get byte from source at start_byte + i
                (local.set $char_code
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (i32.add (local.get $start_byte) (local.get $i))))))

                ;; Store byte to destination
                (i32.store8 (call $abs (i32.add
                  (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (local.get $i)))
                  (local.get $char_code))

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $ss_copy_loop)
              )
            )

            ;; Update string pointer (align to 4 bytes)
            (call $write_state (global.get $STATE_STRING_POINTER)
              (i32.and
                (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4)) (i32.add (local.get $slice_len) (i32.const 3)))
                (i32.const 0xFFFFFFFC)))

            (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
              (local.get $new_str_ptr) (i32.const 0))
            (return)
          )
        )

        ;; split(sep) - returns array of strings
        (if (i32.eq (local.get $method_id) (global.get $METHOD_SPLIT))
          (then
            ;; Get separator string
            (if (i32.ne (local.get $arg1_type) (global.get $TYPE_STRING))
              (then
                ;; If no separator or not a string, return array with whole string
                (local.set $count (i32.const 1))
              )
              (else
                (local.set $search_ptr (local.get $arg1_lo))
                (local.set $search_len (i32.load (call $abs (local.get $search_ptr))))

                ;; Empty separator - split into individual characters
                (if (i32.eqz (local.get $search_len))
                  (then (local.set $count (local.get $str_len)))
                  (else
                    ;; Count occurrences to determine array size
                    (local.set $count (i32.const 1))
                    (local.set $i (i32.const 0))
                    (block $count_done
                      (loop $count_loop
                        (br_if $count_done (i32.gt_u (local.get $i)
                          (i32.sub (local.get $str_len) (local.get $search_len))))

                        ;; Check for match at position i
                        (local.set $found (i32.const 1))
                        (local.set $j (i32.const 0))
                        (block $match_done
                          (loop $match_loop
                            (br_if $match_done (i32.ge_u (local.get $j) (local.get $search_len)))
                            (if (i32.ne
                                  (i32.load8_u (call $abs (i32.add
                                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                                    (i32.add (local.get $i) (local.get $j)))))
                                  (i32.load8_u (call $abs (i32.add
                                    (i32.add (local.get $search_ptr) (i32.const 4))
                                    (local.get $j)))))
                              (then
                                (local.set $found (i32.const 0))
                                (br $match_done)
                              )
                            )
                            (local.set $j (i32.add (local.get $j) (i32.const 1)))
                            (br $match_loop)
                          )
                        )

                        (if (local.get $found)
                          (then
                            (local.set $count (i32.add (local.get $count) (i32.const 1)))
                            (local.set $i (i32.add (local.get $i) (local.get $search_len)))
                          )
                          (else
                            (local.set $i (i32.add (local.get $i) (i32.const 1)))
                          )
                        )
                        (br $count_loop)
                      )
                    )
                  )
                )
              )
            )

            ;; Allocate array
            (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
            (local.set $new_arr (local.get $heap_ptr))

            ;; Array header
            (i32.store (call $abs (local.get $new_arr))
              (i32.or (i32.const 20) (i32.shl (global.get $OBJ_ARRAY) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 4))) (i32.const 0))

            ;; Data block
            (local.set $new_data (i32.add (local.get $new_arr) (i32.const 20)))
            (local.set $i32_result (i32.add (global.get $GC_HEADER_SIZE)
              (i32.mul (local.get $count) (global.get $VALUE_SIZE))))

            (i32.store (call $abs (local.get $new_data))
              (i32.or (local.get $i32_result) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $new_data) (i32.const 4))) (i32.const 0))

            ;; Array header fields
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 8))) (local.get $count))
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 12))) (local.get $count))
            (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 16))) (local.get $new_data))

            ;; Update heap pointer
            (call $write_state (global.get $STATE_HEAP_POINTER)
              (i32.add (local.get $new_data) (local.get $i32_result)))

            ;; Now fill the array with string pieces
            (if (i32.ne (local.get $arg1_type) (global.get $TYPE_STRING))
              (then
                ;; No separator - store the whole string
                (local.set $elem_ptr (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE)))
                (i32.store (call $abs (local.get $elem_ptr)) (global.get $TYPE_STRING))
                (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))
                (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $receiver_ptr))
                (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (i32.const 0))
              )
              (else
                ;; Split with separator
                (if (i32.eqz (local.get $search_len))
                  (then
                    ;; Empty separator - split into characters
                    (local.set $i (i32.const 0))
                    (block $char_split_done
                      (loop $char_split_loop
                        (br_if $char_split_done (i32.ge_u (local.get $i) (local.get $str_len)))

                        ;; Create single-char string
                        (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                        (i32.store (call $abs (local.get $new_str_ptr)) (i32.const 1))
                        (i32.store8 (call $abs (i32.add (local.get $new_str_ptr) (i32.const 4)))
                          (i32.load8_u (call $abs (i32.add
                            (i32.add (local.get $receiver_ptr) (i32.const 4))
                            (local.get $i)))))
                        (call $write_state (global.get $STATE_STRING_POINTER)
                          (i32.add (local.get $new_str_ptr) (i32.const 8)))

                        ;; Store in array
                        (local.set $elem_ptr (i32.add
                          (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
                          (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                        (i32.store (call $abs (local.get $elem_ptr)) (global.get $TYPE_STRING))
                        (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))
                        (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $new_str_ptr))
                        (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (i32.const 0))

                        (local.set $i (i32.add (local.get $i) (i32.const 1)))
                        (br $char_split_loop)
                      )
                    )
                  )
                  (else
                    ;; Split with actual separator
                    (local.set $start_idx (i32.const 0))
                    (local.set $count (i32.const 0))  ;; reuse as array index
                    (local.set $i (i32.const 0))

                    (block $split_done
                      (loop $split_loop
                        ;; Check if we can find separator starting at i
                        (if (i32.le_u (local.get $i)
                              (i32.sub (local.get $str_len) (local.get $search_len)))
                          (then
                            (local.set $found (i32.const 1))
                            (local.set $j (i32.const 0))
                            (block $sep_match_done
                              (loop $sep_match_loop
                                (br_if $sep_match_done (i32.ge_u (local.get $j) (local.get $search_len)))
                                (if (i32.ne
                                      (i32.load8_u (call $abs (i32.add
                                        (i32.add (local.get $receiver_ptr) (i32.const 4))
                                        (i32.add (local.get $i) (local.get $j)))))
                                      (i32.load8_u (call $abs (i32.add
                                        (i32.add (local.get $search_ptr) (i32.const 4))
                                        (local.get $j)))))
                                  (then
                                    (local.set $found (i32.const 0))
                                    (br $sep_match_done)
                                  )
                                )
                                (local.set $j (i32.add (local.get $j) (i32.const 1)))
                                (br $sep_match_loop)
                              )
                            )

                            (if (local.get $found)
                              (then
                                ;; Found separator at i, create substring from start_idx to i
                                (local.set $slice_len (i32.sub (local.get $i) (local.get $start_idx)))

                                ;; Allocate new string
                                (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                                (i32.store (call $abs (local.get $new_str_ptr)) (local.get $slice_len))

                                ;; Copy bytes
                                (local.set $j (i32.const 0))
                                (block $piece_copy_done
                                  (loop $piece_copy_loop
                                    (br_if $piece_copy_done (i32.ge_u (local.get $j) (local.get $slice_len)))
                                    (i32.store8 (call $abs (i32.add
                                      (i32.add (local.get $new_str_ptr) (i32.const 4))
                                      (local.get $j)))
                                      (i32.load8_u (call $abs (i32.add
                                        (i32.add (local.get $receiver_ptr) (i32.const 4))
                                        (i32.add (local.get $start_idx) (local.get $j))))))
                                    (local.set $j (i32.add (local.get $j) (i32.const 1)))
                                    (br $piece_copy_loop)
                                  )
                                )

                                ;; Update string pointer (align)
                                (call $write_state (global.get $STATE_STRING_POINTER)
                                  (i32.and
                                    (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4))
                                      (i32.add (local.get $slice_len) (i32.const 3)))
                                    (i32.const 0xFFFFFFFC)))

                                ;; Store in array
                                (local.set $elem_ptr (i32.add
                                  (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
                                  (i32.mul (local.get $count) (global.get $VALUE_SIZE))))
                                (i32.store (call $abs (local.get $elem_ptr)) (global.get $TYPE_STRING))
                                (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))
                                (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $new_str_ptr))
                                (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (i32.const 0))

                                (local.set $count (i32.add (local.get $count) (i32.const 1)))
                                (local.set $i (i32.add (local.get $i) (local.get $search_len)))
                                (local.set $start_idx (local.get $i))
                                (br $split_loop)
                              )
                              (else
                                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                                (br $split_loop)
                              )
                            )
                          )
                          (else
                            ;; Past end, add final piece
                            (br $split_done)
                          )
                        )
                      )
                    )

                    ;; Add final piece from start_idx to end
                    (local.set $slice_len (i32.sub (local.get $str_len) (local.get $start_idx)))
                    (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                    (i32.store (call $abs (local.get $new_str_ptr)) (local.get $slice_len))

                    (local.set $j (i32.const 0))
                    (block $final_copy_done
                      (loop $final_copy_loop
                        (br_if $final_copy_done (i32.ge_u (local.get $j) (local.get $slice_len)))
                        (i32.store8 (call $abs (i32.add
                          (i32.add (local.get $new_str_ptr) (i32.const 4))
                          (local.get $j)))
                          (i32.load8_u (call $abs (i32.add
                            (i32.add (local.get $receiver_ptr) (i32.const 4))
                            (i32.add (local.get $start_idx) (local.get $j))))))
                        (local.set $j (i32.add (local.get $j) (i32.const 1)))
                        (br $final_copy_loop)
                      )
                    )

                    (call $write_state (global.get $STATE_STRING_POINTER)
                      (i32.and
                        (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4))
                          (i32.add (local.get $slice_len) (i32.const 3)))
                        (i32.const 0xFFFFFFFC)))

                    (local.set $elem_ptr (i32.add
                      (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $count) (global.get $VALUE_SIZE))))
                    (i32.store (call $abs (local.get $elem_ptr)) (global.get $TYPE_STRING))
                    (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))
                    (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $new_str_ptr))
                    (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (i32.const 0))
                  )
                )
              )
            )

            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
            (return)
          )
        )

        ;; trim() - returns trimmed string
        (if (i32.eq (local.get $method_id) (global.get $METHOD_TRIM))
          (then
            ;; Find start index (skip leading whitespace)
            (local.set $start_idx (i32.const 0))
            (block $trim_start_done
              (loop $trim_start_loop
                (br_if $trim_start_done (i32.ge_u (local.get $start_idx) (local.get $str_len)))
                (local.set $char_code
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (local.get $start_idx)))))
                ;; Check for space (0x20), tab (0x09), newline (0x0A), carriage return (0x0D)
                (br_if $trim_start_done
                  (i32.and
                    (i32.and
                      (i32.ne (local.get $char_code) (i32.const 0x20))
                      (i32.ne (local.get $char_code) (i32.const 0x09)))
                    (i32.and
                      (i32.ne (local.get $char_code) (i32.const 0x0A))
                      (i32.ne (local.get $char_code) (i32.const 0x0D)))))
                (local.set $start_idx (i32.add (local.get $start_idx) (i32.const 1)))
                (br $trim_start_loop)
              )
            )

            ;; Find end index (skip trailing whitespace)
            (local.set $end_idx (local.get $str_len))
            (block $trim_end_done
              (loop $trim_end_loop
                (br_if $trim_end_done (i32.le_u (local.get $end_idx) (local.get $start_idx)))
                (local.set $char_code
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (i32.sub (local.get $end_idx) (i32.const 1))))))
                ;; Check for whitespace
                (br_if $trim_end_done
                  (i32.and
                    (i32.and
                      (i32.ne (local.get $char_code) (i32.const 0x20))
                      (i32.ne (local.get $char_code) (i32.const 0x09)))
                    (i32.and
                      (i32.ne (local.get $char_code) (i32.const 0x0A))
                      (i32.ne (local.get $char_code) (i32.const 0x0D)))))
                (local.set $end_idx (i32.sub (local.get $end_idx) (i32.const 1)))
                (br $trim_end_loop)
              )
            )

            ;; Calculate slice length
            (local.set $slice_len (i32.sub (local.get $end_idx) (local.get $start_idx)))

            ;; If empty, return empty string
            (if (i32.le_s (local.get $slice_len) (i32.const 0))
              (then
                (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                (i32.store (call $abs (local.get $new_str_ptr)) (i32.const 0))
                (call $write_state (global.get $STATE_STRING_POINTER)
                  (i32.add (local.get $new_str_ptr) (i32.const 4)))
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $new_str_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Allocate new string
            (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
            (i32.store (call $abs (local.get $new_str_ptr)) (local.get $slice_len))

            ;; Copy bytes
            (local.set $i (i32.const 0))
            (block $trim_copy_done
              (loop $trim_copy_loop
                (br_if $trim_copy_done (i32.ge_u (local.get $i) (local.get $slice_len)))
                (local.set $char_code
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (i32.add (local.get $start_idx) (local.get $i))))))
                (i32.store8 (call $abs (i32.add
                  (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (local.get $i)))
                  (local.get $char_code))
                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $trim_copy_loop)
              )
            )

            ;; Update string pointer (align to 4 bytes)
            (call $write_state (global.get $STATE_STRING_POINTER)
              (i32.and
                (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4)) (i32.add (local.get $slice_len) (i32.const 3)))
                (i32.const 0xFFFFFFFC)))

            (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
              (local.get $new_str_ptr) (i32.const 0))
            (return)
          )
        )

        ;; toLowerCase() - returns lowercase string
        ;; toLowerCase() - returns lowercase string (UTF-8 aware)
        (if (i32.eq (local.get $method_id) (global.get $METHOD_TO_LOWER_CASE))
          (then
            ;; Empty string case
            (if (i32.eqz (local.get $str_len))
              (then
                (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                (i32.store (call $abs (local.get $new_str_ptr)) (i32.const 0))
                (call $write_state (global.get $STATE_STRING_POINTER)
                  (i32.add (local.get $new_str_ptr) (i32.const 4)))
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $new_str_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Allocate new string
            (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))

            ;; Iterate through UTF-8 code points
            (local.set $i (i32.const 0))  ;; source byte index
            (local.set $j (i32.const 0))  ;; dest byte index
            (block $lower_done
              (loop $lower_loop
                (br_if $lower_done (i32.ge_u (local.get $i) (local.get $str_len)))

                ;; Decode code point at byte offset i
                (local.set $char_code (call $utf8_decode_at (local.get $receiver_ptr) (local.get $i)))

                ;; Get byte length of source character
                (local.set $slice_len (call $utf8_char_byte_len (local.get $receiver_ptr) (local.get $i)))

                ;; Transform to lowercase
                (local.set $char_code (call $to_lower_code_point (local.get $char_code)))

                ;; Encode to destination
                (local.set $count (call $utf8_encode_at
                  (i32.add (local.get $new_str_ptr) (i32.add (i32.const 4) (local.get $j)))
                  (local.get $char_code)))

                ;; Advance source and dest
                (local.set $i (i32.add (local.get $i) (local.get $slice_len)))
                (local.set $j (i32.add (local.get $j) (local.get $count)))
                (br $lower_loop)
              )
            )

            ;; Write final byte length
            (i32.store (call $abs (local.get $new_str_ptr)) (local.get $j))

            ;; Update string pointer (align to 4 bytes)
            (call $write_state (global.get $STATE_STRING_POINTER)
              (i32.and
                (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4)) (i32.add (local.get $j) (i32.const 3)))
                (i32.const 0xFFFFFFFC)))

            (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
              (local.get $new_str_ptr) (i32.const 0))
            (return)
          )
        )

        ;; toUpperCase() - returns uppercase string (UTF-8 aware)
        (if (i32.eq (local.get $method_id) (global.get $METHOD_TO_UPPER_CASE))
          (then
            ;; Empty string case
            (if (i32.eqz (local.get $str_len))
              (then
                (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                (i32.store (call $abs (local.get $new_str_ptr)) (i32.const 0))
                (call $write_state (global.get $STATE_STRING_POINTER)
                  (i32.add (local.get $new_str_ptr) (i32.const 4)))
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $new_str_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Allocate new string (same byte length, case changes don't affect size for ASCII/Latin-1)
            (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))

            ;; Iterate through UTF-8 code points
            (local.set $i (i32.const 0))  ;; source byte index
            (local.set $j (i32.const 0))  ;; dest byte index
            (block $upper_done
              (loop $upper_loop
                (br_if $upper_done (i32.ge_u (local.get $i) (local.get $str_len)))

                ;; Decode code point at byte offset i
                (local.set $char_code (call $utf8_decode_at (local.get $receiver_ptr) (local.get $i)))

                ;; Get byte length of source character
                (local.set $slice_len (call $utf8_char_byte_len (local.get $receiver_ptr) (local.get $i)))

                ;; Transform to uppercase
                (local.set $char_code (call $to_upper_code_point (local.get $char_code)))

                ;; Encode to destination
                (local.set $count (call $utf8_encode_at
                  (i32.add (local.get $new_str_ptr) (i32.add (i32.const 4) (local.get $j)))
                  (local.get $char_code)))

                ;; Advance source and dest
                (local.set $i (i32.add (local.get $i) (local.get $slice_len)))
                (local.set $j (i32.add (local.get $j) (local.get $count)))
                (br $upper_loop)
              )
            )

            ;; Write final byte length
            (i32.store (call $abs (local.get $new_str_ptr)) (local.get $j))

            ;; Update string pointer (align to 4 bytes)
            (call $write_state (global.get $STATE_STRING_POINTER)
              (i32.and
                (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4)) (i32.add (local.get $j) (i32.const 3)))
                (i32.const 0xFFFFFFFC)))

            (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
              (local.get $new_str_ptr) (i32.const 0))
            (return)
          )
        )

        ;; startsWith(search) - returns boolean
        (if (i32.eq (local.get $method_id) (global.get $METHOD_STARTS_WITH))
          (then
            ;; Get search string info
            (if (i32.ne (local.get $arg1_type) (global.get $TYPE_STRING))
              (then
                ;; Non-string search returns false
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
                (return)
              )
            )

            (local.set $search_ptr (local.get $arg1_lo))
            (local.set $search_len (i32.load (call $abs (local.get $search_ptr))))

            ;; Empty search string always matches
            (if (i32.eqz (local.get $search_len))
              (then
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))
                (return)
              )
            )

            ;; Search string longer than haystack - can't match
            (if (i32.gt_u (local.get $search_len) (local.get $str_len))
              (then
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
                (return)
              )
            )

            ;; Compare bytes from start
            (local.set $found (i32.const 1))
            (local.set $i (i32.const 0))
            (block $sw_done
              (loop $sw_loop
                (br_if $sw_done (i32.ge_u (local.get $i) (local.get $search_len)))

                ;; Get byte from receiver string
                (local.set $byte_ptr
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (local.get $i)))))

                ;; Get byte from search string
                (local.set $elem_ptr
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $search_ptr) (i32.const 4))
                    (local.get $i)))))

                ;; Compare
                (if (i32.ne (local.get $byte_ptr) (local.get $elem_ptr))
                  (then
                    (local.set $found (i32.const 0))
                    (br $sw_done)
                  )
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $sw_loop)
              )
            )

            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (local.get $found) (i32.const 0))
            (return)
          )
        )

        ;; endsWith(search) - returns boolean
        (if (i32.eq (local.get $method_id) (global.get $METHOD_ENDS_WITH))
          (then
            ;; Get search string info
            (if (i32.ne (local.get $arg1_type) (global.get $TYPE_STRING))
              (then
                ;; Non-string search returns false
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
                (return)
              )
            )

            (local.set $search_ptr (local.get $arg1_lo))
            (local.set $search_len (i32.load (call $abs (local.get $search_ptr))))

            ;; Empty search string always matches
            (if (i32.eqz (local.get $search_len))
              (then
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))
                (return)
              )
            )

            ;; Search string longer than haystack - can't match
            (if (i32.gt_u (local.get $search_len) (local.get $str_len))
              (then
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
                (return)
              )
            )

            ;; Compare bytes from end
            ;; Start position in receiver = str_len - search_len
            (local.set $start_idx (i32.sub (local.get $str_len) (local.get $search_len)))
            (local.set $found (i32.const 1))
            (local.set $i (i32.const 0))
            (block $ew_done
              (loop $ew_loop
                (br_if $ew_done (i32.ge_u (local.get $i) (local.get $search_len)))

                ;; Get byte from receiver string at (start_idx + i)
                (local.set $byte_ptr
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (i32.add (local.get $start_idx) (local.get $i))))))

                ;; Get byte from search string at i
                (local.set $elem_ptr
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $search_ptr) (i32.const 4))
                    (local.get $i)))))

                ;; Compare
                (if (i32.ne (local.get $byte_ptr) (local.get $elem_ptr))
                  (then
                    (local.set $found (i32.const 0))
                    (br $ew_done)
                  )
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $ew_loop)
              )
            )

            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (local.get $found) (i32.const 0))
            (return)
          )
        )

        ;; repeat(count) - returns repeated string
        (if (i32.eq (local.get $method_id) (global.get $METHOD_REPEAT))
          (then
            ;; Get count
            (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (then
                (local.set $count (i32.trunc_f64_s (call $read_f64 (local.get $arg1_ptr))))
              )
              (else
                (if (i32.eq (local.get $arg1_type) (global.get $TYPE_INTEGER))
                  (then (local.set $count (local.get $arg1_lo)))
                  (else (local.set $count (i32.const 0)))
                )
              )
            )

            ;; If count <= 0 or string is empty, return empty string
            (if (i32.or (i32.le_s (local.get $count) (i32.const 0))
                        (i32.eqz (local.get $str_len)))
              (then
                (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
                (i32.store (call $abs (local.get $new_str_ptr)) (i32.const 0))
                (call $write_state (global.get $STATE_STRING_POINTER)
                  (i32.add (local.get $new_str_ptr) (i32.const 4)))
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $new_str_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Calculate new length
            (local.set $slice_len (i32.mul (local.get $str_len) (local.get $count)))

            ;; Allocate new string
            (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
            (i32.store (call $abs (local.get $new_str_ptr)) (local.get $slice_len))

            ;; Copy the string count times
            (local.set $i (i32.const 0))
            (block $repeat_done
              (loop $repeat_loop
                (br_if $repeat_done (i32.ge_u (local.get $i) (local.get $count)))

                ;; Copy str_len bytes
                (local.set $j (i32.const 0))
                (block $copy_done
                  (loop $copy_loop
                    (br_if $copy_done (i32.ge_u (local.get $j) (local.get $str_len)))
                    (i32.store8 (call $abs (i32.add
                      (i32.add (local.get $new_str_ptr) (i32.const 4))
                      (i32.add (i32.mul (local.get $i) (local.get $str_len)) (local.get $j))))
                      (i32.load8_u (call $abs (i32.add
                        (i32.add (local.get $receiver_ptr) (i32.const 4))
                        (local.get $j)))))
                    (local.set $j (i32.add (local.get $j) (i32.const 1)))
                    (br $copy_loop)
                  )
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $repeat_loop)
              )
            )

            ;; Update string pointer (align)
            (call $write_state (global.get $STATE_STRING_POINTER)
              (i32.and
                (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (i32.add (local.get $slice_len) (i32.const 3)))
                (i32.const 0xFFFFFFFC)))

            (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
              (local.get $new_str_ptr) (i32.const 0))
            (return)
          )
        )

        ;; padStart(len, fill) - returns padded string
        (if (i32.eq (local.get $method_id) (global.get $METHOD_PAD_START))
          (then
            ;; Get target length
            (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (then
                (local.set $target_len (i32.trunc_f64_s (call $read_f64 (local.get $arg1_ptr))))
              )
              (else
                (if (i32.eq (local.get $arg1_type) (global.get $TYPE_INTEGER))
                  (then (local.set $target_len (local.get $arg1_lo)))
                  (else (local.set $target_len (i32.const 0)))
                )
              )
            )

            ;; If target_len <= str_len, return original string
            (if (i32.le_s (local.get $target_len) (local.get $str_len))
              (then
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $receiver_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Get fill string (default to space)
            (local.set $fill_len (i32.const 1))
            (if (i32.ge_u (local.get $argc) (i32.const 2))
              (then
                (if (i32.eq (local.get $arg2_type) (global.get $TYPE_STRING))
                  (then
                    (local.set $fill_ptr (call $value_data_lo (local.get $arg2_ptr)))
                    (local.set $fill_len (i32.load (call $abs (local.get $fill_ptr))))
                  )
                )
              )
            )

            ;; If fill is empty, just return original
            (if (i32.eqz (local.get $fill_len))
              (then
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $receiver_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Calculate pad length
            (local.set $pad_len (i32.sub (local.get $target_len) (local.get $str_len)))

            ;; Allocate new string
            (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
            (i32.store (call $abs (local.get $new_str_ptr)) (local.get $target_len))

            ;; Fill with pad characters first
            (local.set $i (i32.const 0))
            (block $pad_done
              (loop $pad_loop
                (br_if $pad_done (i32.ge_u (local.get $i) (local.get $pad_len)))

                ;; Get fill character (cycle through fill string)
                (if (i32.and (i32.ge_u (local.get $argc) (i32.const 2))
                             (i32.eq (local.get $arg2_type) (global.get $TYPE_STRING)))
                  (then
                    (local.set $char_code
                      (i32.load8_u (call $abs (i32.add
                        (i32.add (local.get $fill_ptr) (i32.const 4))
                        (i32.rem_u (local.get $i) (local.get $fill_len))))))
                  )
                  (else
                    (local.set $char_code (i32.const 0x20))  ;; space
                  )
                )

                (i32.store8 (call $abs (i32.add
                  (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (local.get $i)))
                  (local.get $char_code))

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $pad_loop)
              )
            )

            ;; Copy original string after padding
            (local.set $i (i32.const 0))
            (block $orig_done
              (loop $orig_loop
                (br_if $orig_done (i32.ge_u (local.get $i) (local.get $str_len)))
                (i32.store8 (call $abs (i32.add
                  (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (i32.add (local.get $pad_len) (local.get $i))))
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (local.get $i)))))
                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $orig_loop)
              )
            )

            ;; Update string pointer (align)
            (call $write_state (global.get $STATE_STRING_POINTER)
              (i32.and
                (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (i32.add (local.get $target_len) (i32.const 3)))
                (i32.const 0xFFFFFFFC)))

            (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
              (local.get $new_str_ptr) (i32.const 0))
            (return)
          )
        )

        ;; padEnd(len, fill) - returns padded string
        (if (i32.eq (local.get $method_id) (global.get $METHOD_PAD_END))
          (then
            ;; Get target length
            (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (then
                (local.set $target_len (i32.trunc_f64_s (call $read_f64 (local.get $arg1_ptr))))
              )
              (else
                (if (i32.eq (local.get $arg1_type) (global.get $TYPE_INTEGER))
                  (then (local.set $target_len (local.get $arg1_lo)))
                  (else (local.set $target_len (i32.const 0)))
                )
              )
            )

            ;; If target_len <= str_len, return original string
            (if (i32.le_s (local.get $target_len) (local.get $str_len))
              (then
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $receiver_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Get fill string (default to space)
            (local.set $fill_len (i32.const 1))
            (if (i32.ge_u (local.get $argc) (i32.const 2))
              (then
                (if (i32.eq (local.get $arg2_type) (global.get $TYPE_STRING))
                  (then
                    (local.set $fill_ptr (call $value_data_lo (local.get $arg2_ptr)))
                    (local.set $fill_len (i32.load (call $abs (local.get $fill_ptr))))
                  )
                )
              )
            )

            ;; If fill is empty, just return original
            (if (i32.eqz (local.get $fill_len))
              (then
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $receiver_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Calculate pad length
            (local.set $pad_len (i32.sub (local.get $target_len) (local.get $str_len)))

            ;; Allocate new string
            (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
            (i32.store (call $abs (local.get $new_str_ptr)) (local.get $target_len))

            ;; Copy original string first
            (local.set $i (i32.const 0))
            (block $orig_done2
              (loop $orig_loop2
                (br_if $orig_done2 (i32.ge_u (local.get $i) (local.get $str_len)))
                (i32.store8 (call $abs (i32.add
                  (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (local.get $i)))
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (local.get $i)))))
                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $orig_loop2)
              )
            )

            ;; Fill with pad characters after
            (local.set $i (i32.const 0))
            (block $pad_done2
              (loop $pad_loop2
                (br_if $pad_done2 (i32.ge_u (local.get $i) (local.get $pad_len)))

                ;; Get fill character (cycle through fill string)
                (if (i32.and (i32.ge_u (local.get $argc) (i32.const 2))
                             (i32.eq (local.get $arg2_type) (global.get $TYPE_STRING)))
                  (then
                    (local.set $char_code
                      (i32.load8_u (call $abs (i32.add
                        (i32.add (local.get $fill_ptr) (i32.const 4))
                        (i32.rem_u (local.get $i) (local.get $fill_len))))))
                  )
                  (else
                    (local.set $char_code (i32.const 0x20))  ;; space
                  )
                )

                (i32.store8 (call $abs (i32.add
                  (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (i32.add (local.get $str_len) (local.get $i))))
                  (local.get $char_code))

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $pad_loop2)
              )
            )

            ;; Update string pointer (align)
            (call $write_state (global.get $STATE_STRING_POINTER)
              (i32.and
                (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (i32.add (local.get $target_len) (i32.const 3)))
                (i32.const 0xFFFFFFFC)))

            (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
              (local.get $new_str_ptr) (i32.const 0))
            (return)
          )
        )

        ;; replace(search, replacement) - returns new string (first occurrence only)
        (if (i32.eq (local.get $method_id) (global.get $METHOD_REPLACE))
          (then
            ;; Get search string
            (if (i32.ne (local.get $arg1_type) (global.get $TYPE_STRING))
              (then
                ;; No search string - return original
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $receiver_ptr) (i32.const 0))
                (return)
              )
            )
            (local.set $search_ptr (local.get $arg1_lo))
            (local.set $search_len (i32.load (call $abs (local.get $search_ptr))))

            ;; Get replacement string
            (if (i32.and (i32.ge_u (local.get $argc) (i32.const 2))
                         (i32.eq (local.get $arg2_type) (global.get $TYPE_STRING)))
              (then
                (local.set $repl_ptr (call $value_data_lo (local.get $arg2_ptr)))
                (local.set $repl_len (i32.load (call $abs (local.get $repl_ptr))))
              )
              (else
                ;; No replacement - use empty string
                (local.set $repl_len (i32.const 0))
              )
            )

            ;; Empty search - return original (JS behavior for empty string search is complex, simplify)
            (if (i32.eqz (local.get $search_len))
              (then
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $receiver_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Search for first occurrence
            (local.set $match_pos (i32.const -1))
            (local.set $i (i32.const 0))
            (block $search_done
              (loop $search_loop
                (br_if $search_done (i32.gt_u (local.get $i)
                  (i32.sub (local.get $str_len) (local.get $search_len))))

                (local.set $found (i32.const 1))
                (local.set $j (i32.const 0))
                (block $match_check_done
                  (loop $match_check_loop
                    (br_if $match_check_done (i32.ge_u (local.get $j) (local.get $search_len)))
                    (if (i32.ne
                          (i32.load8_u (call $abs (i32.add
                            (i32.add (local.get $receiver_ptr) (i32.const 4))
                            (i32.add (local.get $i) (local.get $j)))))
                          (i32.load8_u (call $abs (i32.add
                            (i32.add (local.get $search_ptr) (i32.const 4))
                            (local.get $j)))))
                      (then
                        (local.set $found (i32.const 0))
                        (br $match_check_done)
                      )
                    )
                    (local.set $j (i32.add (local.get $j) (i32.const 1)))
                    (br $match_check_loop)
                  )
                )

                (if (local.get $found)
                  (then
                    (local.set $match_pos (local.get $i))
                    (br $search_done)
                  )
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $search_loop)
              )
            )

            ;; If not found, return original
            (if (i32.lt_s (local.get $match_pos) (i32.const 0))
              (then
                (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
                  (local.get $receiver_ptr) (i32.const 0))
                (return)
              )
            )

            ;; Calculate new length: str_len - search_len + repl_len
            (local.set $target_len (i32.add
              (i32.sub (local.get $str_len) (local.get $search_len))
              (local.get $repl_len)))

            ;; Allocate new string
            (local.set $new_str_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
            (i32.store (call $abs (local.get $new_str_ptr)) (local.get $target_len))

            ;; Copy part before match
            (local.set $i (i32.const 0))
            (block $before_done
              (loop $before_loop
                (br_if $before_done (i32.ge_u (local.get $i) (local.get $match_pos)))
                (i32.store8 (call $abs (i32.add
                  (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (local.get $i)))
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (local.get $i)))))
                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $before_loop)
              )
            )

            ;; Copy replacement
            (local.set $j (i32.const 0))
            (block $repl_done
              (loop $repl_loop
                (br_if $repl_done (i32.ge_u (local.get $j) (local.get $repl_len)))
                (i32.store8 (call $abs (i32.add
                  (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (i32.add (local.get $match_pos) (local.get $j))))
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $repl_ptr) (i32.const 4))
                    (local.get $j)))))
                (local.set $j (i32.add (local.get $j) (i32.const 1)))
                (br $repl_loop)
              )
            )

            ;; Copy part after match
            (local.set $i (i32.add (local.get $match_pos) (local.get $search_len)))
            (block $after_done
              (loop $after_loop
                (br_if $after_done (i32.ge_u (local.get $i) (local.get $str_len)))
                (i32.store8 (call $abs (i32.add
                  (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (i32.add (local.get $match_pos)
                    (i32.add (local.get $repl_len)
                      (i32.sub (local.get $i) (i32.add (local.get $match_pos) (local.get $search_len)))))))
                  (i32.load8_u (call $abs (i32.add
                    (i32.add (local.get $receiver_ptr) (i32.const 4))
                    (local.get $i)))))
                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $after_loop)
              )
            )

            ;; Update string pointer (align)
            (call $write_state (global.get $STATE_STRING_POINTER)
              (i32.and
                (i32.add (i32.add (local.get $new_str_ptr) (i32.const 4))
                  (i32.add (local.get $target_len) (i32.const 3)))
                (i32.const 0xFFFFFFFC)))

            (call $pending_push (global.get $TYPE_STRING) (i32.const 0)
              (local.get $new_str_ptr) (i32.const 0))
            (return)
          )
        )
      )
    )

    ;; ============================================
    ;; MATH STATIC METHODS (receiver_type = TYPE_OBJECT for Math object)
    ;; Math methods have receiver=Math object but we dispatch on method_id
    ;; ============================================

    ;; Math.abs(x)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_ABS))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (f64.abs (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.floor(x)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_FLOOR))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (f64.floor (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.ceil(x)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_CEIL))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (f64.ceil (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.round(x)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_ROUND))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            ;; JS Math.round: round half away from zero
            (call $push_f64 (f64.floor (f64.add (local.get $f64_result) (f64.const 0.5))))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.trunc(x)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_TRUNC))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (f64.trunc (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.sqrt(x)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_SQRT))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (f64.sqrt (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.pow(x, y) - using V8/fdlibm algorithm
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_POW))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.and
              (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (i32.eq (local.get $arg2_type) (global.get $TYPE_FLOAT)))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (local.set $f64_arg2 (f64.load (call $abs (i32.add (local.get $arg2_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_pow (local.get $f64_result) (local.get $f64_arg2)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.min(a, b)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_MIN))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.and
              (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (i32.eq (local.get $arg2_type) (global.get $TYPE_FLOAT)))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (f64.min
              (local.get $f64_result)
              (f64.load (call $abs (i32.add (local.get $arg2_ptr) (i32.const 8))))))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.max(a, b)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_MAX))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.and
              (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (i32.eq (local.get $arg2_type) (global.get $TYPE_FLOAT)))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (f64.max
              (local.get $f64_result)
              (f64.load (call $abs (i32.add (local.get $arg2_ptr) (i32.const 8))))))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.sign(x)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_SIGN))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (if (f64.gt (local.get $f64_result) (f64.const 0))
              (then (call $push_f64 (f64.const 1)))
              (else
                (if (f64.lt (local.get $f64_result) (f64.const 0))
                  (then (call $push_f64 (f64.const -1)))
                  (else (call $push_f64 (f64.const 0)))
                )
              )
            )
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.cbrt(x) - cube root using V8 algorithm
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_CBRT))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_cbrt (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.log(x) - natural logarithm
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_LOG))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_log (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.exp(x) - e^x
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_EXP))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_exp (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.log10(x) - base-10 logarithm using fdlibm
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_LOG10))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_log10 (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.log2(x) - base-2 logarithm using fdlibm
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_LOG2))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_log2 (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.log1p(x) - log(1 + x), using V8/fdlibm algorithm for precision
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_LOG1P))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_log1p (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.expm1(x) - e^x - 1, more precise for small x
    ;; Math.expm1(x) - e^x - 1, using V8/fdlibm algorithm
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_EXPM1))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_expm1 (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.clz32(x) - count leading zeros in 32-bit representation
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_CLZ32))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            ;; Convert to i32 (truncate), then count leading zeros
            (call $push_f64 (f64.convert_i32_u (i32.clz (i32.trunc_f64_s (local.get $f64_result)))))
          )
          (else
            (call $push_f64 (f64.const 32)) ;; Non-number returns 32
          )
        )
        (return)
      )
    )

    ;; Math.imul(a, b) - 32-bit integer multiply
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_IMUL))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.and
              (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (i32.eq (local.get $arg2_type) (global.get $TYPE_FLOAT)))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            ;; Convert both to i32, multiply, convert back to f64
            (call $push_f64 (f64.convert_i32_s (i32.mul
              (i32.trunc_f64_s (local.get $f64_result))
              (i32.trunc_f64_s (f64.load (call $abs (i32.add (local.get $arg2_ptr) (i32.const 8))))))))
          )
          (else
            (call $push_f64 (f64.const 0)) ;; Non-numbers return 0
          )
        )
        (return)
      )
    )

    ;; Math.fround(x) - round to f32 and back
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_FROUND))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            ;; Convert to f32 and back to f64
            (call $push_f64 (f64.promote_f32 (f32.demote_f64 (local.get $f64_result))))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.hypot(a, b) - sqrt(a^2 + b^2)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_HYPOT))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.and
              (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (i32.eq (local.get $arg2_type) (global.get $TYPE_FLOAT)))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (local.set $f64_arg2 (f64.load (call $abs (i32.add (local.get $arg2_ptr) (i32.const 8)))))
            (call $push_f64 (f64.sqrt (f64.add
              (f64.mul (local.get $f64_result) (local.get $f64_result))
              (f64.mul (local.get $f64_arg2) (local.get $f64_arg2)))))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.sin(x) - sine using Taylor series
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_SIN))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_sin (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.cos(x) - cosine using Taylor series
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_COS))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_cos (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.tan(x) - tangent using V8 kernel_tan algorithm
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_TAN))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_tan (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.asin(x) - arcsine
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_ASIN))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_asin (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.acos(x) - arccosine using V8 algorithm
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_ACOS))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_acos (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.atan(x) - arctangent
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_ATAN))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_atan (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.atan2(y, x) - two-argument arctangent
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_ATAN2))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.and
              (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
              (i32.eq (local.get $arg2_type) (global.get $TYPE_FLOAT)))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8))))) ;; y
            (local.set $f64_arg2 (f64.load (call $abs (i32.add (local.get $arg2_ptr) (i32.const 8))))) ;; x
            (call $push_f64 (call $f64_atan2 (local.get $f64_result) (local.get $f64_arg2)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.sinh(x) - hyperbolic sine = (e^x - e^-x) / 2
    ;; Math.sinh(x) - using V8 algorithm with expm1
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_SINH))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_sinh (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.cosh(x) - hyperbolic cosine = (e^x + e^-x) / 2
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_COSH))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (f64.mul
              (f64.const 0.5)
              (f64.add
                (call $f64_exp (local.get $f64_result))
                (call $f64_exp (f64.neg (local.get $f64_result))))))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.tanh(x) - hyperbolic tangent using V8/fdlibm algorithm
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_TANH))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_tanh (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.asinh(x) - inverse hyperbolic sine = ln(x + sqrt(x^2 + 1))
    ;; Math.asinh(x) - using V8 algorithm with log1p
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_ASINH))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (call $f64_asinh (local.get $f64_result)))
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.acosh(x) - inverse hyperbolic cosine = ln(x + sqrt(x^2 - 1)), x >= 1
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_ACOSH))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            ;; acosh(x) = ln(x + sqrt(x^2 - 1)), returns NaN for x < 1
            (if (f64.lt (local.get $f64_result) (f64.const 1))
              (then
                (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
              )
              (else
                (call $push_f64 (call $f64_log (f64.add
                  (local.get $f64_result)
                  (f64.sqrt (f64.sub
                    (f64.mul (local.get $f64_result) (local.get $f64_result))
                    (f64.const 1))))))
              )
            )
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; Math.atanh(x) - inverse hyperbolic tangent using V8 algorithm
    (if (i32.eq (local.get $method_id) (global.get $METHOD_MATH_ATANH))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            ;; |x| >= 1: NaN or inf
            (if (f64.ge (f64.abs (local.get $f64_result)) (f64.const 1))
              (then
                (if (f64.eq (f64.abs (local.get $f64_result)) (f64.const 1))
                  (then
                    ;; atanh(1) = +inf, atanh(-1) = -inf
                    (if (f64.gt (local.get $f64_result) (f64.const 0))
                      (then (call $push_f64 (f64.div (f64.const 1) (f64.const 0))))
                      (else (call $push_f64 (f64.div (f64.const -1) (f64.const 0))))
                    )
                  )
                  (else
                    (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
                  )
                )
              )
              (else
                ;; V8 formula: 0.5 * log1p((x + x) / (1 - x)) for |x| >= 0.5
                ;; or: 0.5 * log1p(t + t * x / (1 - x)) where t = 2x for |x| < 0.5
                (local.set $f64_arg2 (f64.abs (local.get $f64_result)))
                (if (f64.lt (local.get $f64_arg2) (f64.const 0.5))
                  (then
                    ;; t = 2x, result = 0.5 * log1p(t + t*x/(1-x))
                    (local.set $f64_arg2 (f64.mul (f64.const 2) (local.get $f64_arg2)))
                    (local.set $f64_arg2 (f64.mul (f64.const 0.5)
                      (call $f64_log1p (f64.add (local.get $f64_arg2)
                        (f64.div
                          (f64.mul (local.get $f64_arg2) (f64.abs (local.get $f64_result)))
                          (f64.sub (f64.const 1) (f64.abs (local.get $f64_result))))))))
                  )
                  (else
                    ;; result = 0.5 * log1p((x + x) / (1 - x))
                    (local.set $f64_arg2 (f64.mul (f64.const 0.5)
                      (call $f64_log1p (f64.div
                        (f64.mul (f64.const 2) (local.get $f64_arg2))
                        (f64.sub (f64.const 1) (local.get $f64_arg2))))))
                  )
                )
                ;; Apply sign
                (if (f64.lt (local.get $f64_result) (f64.const 0))
                  (then (call $push_f64 (f64.neg (local.get $f64_arg2))))
                  (else (call $push_f64 (local.get $f64_arg2)))
                )
              )
            )
          )
          (else
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000)) ;; NaN
          )
        )
        (return)
      )
    )

    ;; ============================================
    ;; GLOBAL FUNCTIONS (receiver_type = 0 for global functions)
    ;; ============================================

    ;; isNaN(x) - global version with coercion
    ;; Returns true if argument is NaN (after coercion to number)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_IS_NAN))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            ;; NaN is the only value where x != x
            (if (f64.ne (local.get $f64_result) (local.get $f64_result))
              (then (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0)))
              (else (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0)))
            )
          )
          (else
            ;; Non-float: coerce would give NaN for most types, but we'll simplify
            ;; undefined -> NaN (true), null -> 0 (false), bool -> 0/1 (false), string -> depends
            (if (i32.eq (local.get $arg1_type) (global.get $TYPE_UNDEFINED))
              (then (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0)))
              (else (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0)))
            )
          )
        )
        (return)
      )
    )

    ;; isFinite(x) - global version with coercion
    ;; Returns true if argument is a finite number (not NaN, not Infinity)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_IS_FINITE))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            ;; Check: not NaN and not Infinity
            ;; x == x (not NaN) and abs(x) < inf
            (if (i32.and
                  (f64.eq (local.get $f64_result) (local.get $f64_result))
                  (f64.lt (f64.abs (local.get $f64_result)) (f64.const inf)))
              (then (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0)))
              (else (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0)))
            )
          )
          (else
            ;; Non-float: false (coercion would give NaN for undefined, objects, etc.)
            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
          )
        )
        (return)
      )
    )

    ;; parseInt(string, radix?) - parse string to integer
    (if (i32.eq (local.get $method_id) (global.get $METHOD_PARSE_INT))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        ;; First arg must be string
        (if (i32.ne (local.get $arg1_type) (global.get $TYPE_STRING))
          (then
            ;; Non-string: return NaN
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000))
            (return)
          )
        )
        ;; Get radix from second argument (default 0 = auto-detect)
        (local.set $i (i32.const 0))
        (if (i32.gt_u (local.get $argc) (i32.const 1))
          (then
            (local.set $arg2_ptr (i32.add (local.get $arg1_ptr) (global.get $VALUE_SIZE)))
            (local.set $arg2_type (call $value_type (local.get $arg2_ptr)))
            (if (i32.eq (local.get $arg2_type) (global.get $TYPE_FLOAT))
              (then
                (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg2_ptr) (i32.const 8)))))
                (local.set $i (i32.trunc_f64_s (local.get $f64_result)))
              )
            )
          )
        )
        ;; Parse the integer
        (call $push_f64 (call $parse_int (local.get $arg1_lo) (local.get $i)))
        (return)
      )
    )

    ;; parseFloat(string) - parse string to float
    (if (i32.eq (local.get $method_id) (global.get $METHOD_PARSE_FLOAT))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        ;; First arg must be string
        (if (i32.ne (local.get $arg1_type) (global.get $TYPE_STRING))
          (then
            ;; Non-string: return NaN
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000))
            (return)
          )
        )
        ;; Parse the float
        (call $push_f64 (call $parse_float (local.get $arg1_lo)))
        (return)
      )
    )

    ;; ============================================
    ;; TYPE CONVERSION FUNCTIONS (String(), Number(), Boolean())
    ;; Called without 'new', receiver_type = TYPE_CONSTRUCTOR
    ;; ============================================

    ;; String(x) - convert to string
    (if (i32.eq (local.get $method_id) (global.get $METHOD_TO_STRING))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        ;; Handle different types
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_STRING))
          (then
            ;; Already a string, return as-is
            (call $pending_push (global.get $TYPE_STRING) (i32.const 0) (local.get $arg1_lo) (i32.const 0))
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_UNDEFINED))
          (then
            ;; "undefined" - reuse the typeof string
            (local.set $str_ptr (call $read_builtin (global.get $BUILTIN_TYPEOF_UNDEFINED)))
            (call $pending_push (global.get $TYPE_STRING) (i32.const 0) (local.get $str_ptr) (i32.const 0))
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_NULL))
          (then
            (local.set $str_ptr (call $read_builtin (global.get $BUILTIN_LIT_NULL)))
            (call $pending_push (global.get $TYPE_STRING) (i32.const 0) (local.get $str_ptr) (i32.const 0))
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_BOOLEAN))
          (then
            (if (local.get $arg1_lo)
              (then (local.set $str_ptr (call $read_builtin (global.get $BUILTIN_LIT_TRUE))))
              (else (local.set $str_ptr (call $read_builtin (global.get $BUILTIN_LIT_FALSE))))
            )
            (call $pending_push (global.get $TYPE_STRING) (i32.const 0) (local.get $str_ptr) (i32.const 0))
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            ;; Convert float to string
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (local.set $str_ptr (call $f64_to_string (local.get $f64_result)))
            (call $pending_push (global.get $TYPE_STRING) (i32.const 0) (local.get $str_ptr) (i32.const 0))
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_CLOSURE))
          (then
            (local.set $str_ptr (call $read_builtin (global.get $BUILTIN_LIT_OBJECT_FUNCTION)))
            (call $pending_push (global.get $TYPE_STRING) (i32.const 0) (local.get $str_ptr) (i32.const 0))
            (return)
          )
        )
        ;; Default for objects/arrays: "[object Object]"
        (local.set $str_ptr (call $read_builtin (global.get $BUILTIN_LIT_OBJECT_OBJECT)))
        (call $pending_push (global.get $TYPE_STRING) (i32.const 0) (local.get $str_ptr) (i32.const 0))
        (return)
      )
    )

    ;; Number(x) - convert to number
    (if (i32.eq (local.get $method_id) (global.get $METHOD_TO_NUMBER))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            ;; Already a number
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (call $push_f64 (local.get $f64_result))
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_UNDEFINED))
          (then
            ;; undefined -> NaN
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000))
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_NULL))
          (then
            ;; null -> 0
            (call $push_f64 (f64.const 0))
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_BOOLEAN))
          (then
            ;; true -> 1, false -> 0
            (if (local.get $arg1_lo)
              (then (call $push_f64 (f64.const 1)))
              (else (call $push_f64 (f64.const 0)))
            )
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_STRING))
          (then
            ;; Parse string as number
            (call $push_f64 (call $parse_float (local.get $arg1_lo)))
            (return)
          )
        )
        ;; Array/Object -> NaN
        (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (i32.const 0) (i32.const 0x7FF80000))
        (return)
      )
    )

    ;; Boolean(x) - convert to boolean
    (if (i32.eq (local.get $method_id) (global.get $METHOD_TO_BOOLEAN))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        ;; Falsy values: undefined, null, false, 0, -0, NaN, ""
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_UNDEFINED))
          (then
            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_NULL))
          (then
            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_BOOLEAN))
          (then
            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (local.get $arg1_lo) (i32.const 0))
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            ;; false if 0, -0, or NaN
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (if (i32.or
                  (f64.eq (local.get $f64_result) (f64.const 0))
                  (f64.ne (local.get $f64_result) (local.get $f64_result))) ;; NaN check
              (then (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0)))
              (else (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0)))
            )
            (return)
          )
        )
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_STRING))
          (then
            ;; false if empty string
            (local.set $str_len (i32.load (call $abs (local.get $arg1_lo))))
            (if (i32.eqz (local.get $str_len))
              (then (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0)))
              (else (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0)))
            )
            (return)
          )
        )
        ;; Arrays and objects are always truthy
        (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))
        (return)
      )
    )

    ;; ============================================
    ;; NUMBER STATIC METHODS (receiver_type = TYPE_CONSTRUCTOR for Number)
    ;; ============================================

    ;; Number.isNaN(x) - strict version, no coercion
    ;; Returns true only if argument is exactly NaN (not coerced)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_NUMBER_IS_NAN))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        ;; Only returns true for actual NaN float values
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (if (f64.ne (local.get $f64_result) (local.get $f64_result))
              (then (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0)))
              (else (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0)))
            )
          )
          (else
            ;; Non-float: always false (no coercion)
            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
          )
        )
        (return)
      )
    )

    ;; Number.isFinite(x) - strict version, no coercion
    ;; Returns true only for finite float values (not coerced)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_NUMBER_IS_FINITE))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            (if (i32.and
                  (f64.eq (local.get $f64_result) (local.get $f64_result))
                  (f64.lt (f64.abs (local.get $f64_result)) (f64.const inf)))
              (then (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0)))
              (else (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0)))
            )
          )
          (else
            ;; Non-float: always false (no coercion)
            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
          )
        )
        (return)
      )
    )

    ;; Number.isInteger(x) - strict version, no coercion
    ;; Returns true only for integer float values (no fractional part)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_NUMBER_IS_INTEGER))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_FLOAT))
          (then
            (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg1_ptr) (i32.const 8)))))
            ;; Check: is finite and trunc(x) == x
            (if (i32.and
                  (i32.and
                    (f64.eq (local.get $f64_result) (local.get $f64_result))
                    (f64.lt (f64.abs (local.get $f64_result)) (f64.const inf)))
                  (f64.eq (f64.trunc (local.get $f64_result)) (local.get $f64_result)))
              (then (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0)))
              (else (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0)))
            )
          )
          (else
            ;; Non-float: always false (no coercion)
            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
          )
        )
        (return)
      )
    )

    ;; ============================================
    ;; STRING STATIC METHODS (receiver_type = TYPE_CONSTRUCTOR for String)
    ;; ============================================

    ;; String.fromCharCode(code1, code2, ...) - create string from char codes
    ;; Variadic: takes argc arguments, each is a code point (truncated to 16-bit)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_STRING_FROM_CHAR_CODE))
      (then
        ;; Allocate string buffer - worst case 3 bytes per char for UTF-8
        (local.set $data_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
        (local.set $dest (i32.add (local.get $data_ptr) (i32.const 4))) ;; skip length field

        ;; Process each argument
        (local.set $i (i32.const 0))
        (local.set $arg_ptr (local.get $arg1_ptr))
        (block $args_done
          (loop $args_loop
            (br_if $args_done (i32.ge_u (local.get $i) (local.get $argc)))

            ;; Get code point from argument
            (local.set $arg_type (call $value_type (local.get $arg_ptr)))
            (if (i32.eq (local.get $arg_type) (global.get $TYPE_FLOAT))
              (then
                (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg_ptr) (i32.const 8)))))
                (local.set $code_point (i32.and (i32.trunc_f64_u (local.get $f64_result)) (i32.const 0xFFFF)))
              )
              (else (local.set $code_point (i32.const 0)))
            )

            ;; Encode UTF-8
            (local.set $dest (i32.add (local.get $dest)
              (call $utf8_encode_at (local.get $dest) (local.get $code_point))))

            (local.set $arg_ptr (i32.add (local.get $arg_ptr) (global.get $VALUE_SIZE)))
            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $args_loop)
          )
        )

        ;; Calculate length and write it
        (local.set $str_len (i32.sub (local.get $dest) (i32.add (local.get $data_ptr) (i32.const 4))))
        (i32.store (call $abs (local.get $data_ptr)) (local.get $str_len))

        ;; Update string pointer
        (call $write_state (global.get $STATE_STRING_POINTER)
          (i32.add (local.get $dest) (i32.and (i32.add (local.get $str_len) (i32.const 7)) (i32.const -4))))

        ;; Push result string
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (call $pending_push (global.get $TYPE_STRING) (i32.const 0) (local.get $data_ptr) (i32.const 0))
        (return)
      )
    )

    ;; String.fromCodePoint(cp1, cp2, ...) - create string from code points
    ;; Same as fromCharCode but allows full Unicode (up to 0x10FFFF)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_STRING_FROM_CODE_POINT))
      (then
        ;; Allocate string buffer - worst case 4 bytes per code point for UTF-8
        (local.set $data_ptr (call $read_state (global.get $STATE_STRING_POINTER)))
        (local.set $dest (i32.add (local.get $data_ptr) (i32.const 4))) ;; skip length field

        ;; Process each argument
        (local.set $i (i32.const 0))
        (local.set $arg_ptr (local.get $arg1_ptr))
        (block $args_done
          (loop $args_loop
            (br_if $args_done (i32.ge_u (local.get $i) (local.get $argc)))

            ;; Get code point from argument
            (local.set $arg_type (call $value_type (local.get $arg_ptr)))
            (if (i32.eq (local.get $arg_type) (global.get $TYPE_FLOAT))
              (then
                (local.set $f64_result (f64.load (call $abs (i32.add (local.get $arg_ptr) (i32.const 8)))))
                (local.set $code_point (i32.trunc_f64_u (local.get $f64_result)))
                ;; Validate: must be <= 0x10FFFF
                (if (i32.gt_u (local.get $code_point) (i32.const 0x10FFFF))
                  (then (local.set $code_point (i32.const 0xFFFD))) ;; replacement character
                )
              )
              (else (local.set $code_point (i32.const 0)))
            )

            ;; Encode UTF-8
            (local.set $dest (i32.add (local.get $dest)
              (call $utf8_encode_at (local.get $dest) (local.get $code_point))))

            (local.set $arg_ptr (i32.add (local.get $arg_ptr) (global.get $VALUE_SIZE)))
            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $args_loop)
          )
        )

        ;; Calculate length and write it
        (local.set $str_len (i32.sub (local.get $dest) (i32.add (local.get $data_ptr) (i32.const 4))))
        (i32.store (call $abs (local.get $data_ptr)) (local.get $str_len))

        ;; Update string pointer
        (call $write_state (global.get $STATE_STRING_POINTER)
          (i32.add (local.get $dest) (i32.and (i32.add (local.get $str_len) (i32.const 7)) (i32.const -4))))

        ;; Push result string
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        (call $pending_push (global.get $TYPE_STRING) (i32.const 0) (local.get $data_ptr) (i32.const 0))
        (return)
      )
    )

    ;; ============================================
    ;; OBJECT STATIC METHODS
    ;; ============================================

    ;; Object.keys(obj) - returns array of own property names
    (if (i32.eq (local.get $method_id) (global.get $METHOD_OBJECT_KEYS))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))

        ;; Handle TYPE_MSGPACK_REF
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_MSGPACK_REF))
          (then
            (local.set $new_arr (call $msgpack_map_keys
              (local.get $arg1_lo)
              (i32.add (local.get $arg1_lo) (local.get $arg1_hi))))
            (if (i32.eqz (local.get $new_arr))
              (then
                ;; Not a map - try array
                (local.set $new_arr (call $msgpack_array_keys
                  (local.get $arg1_lo)
                  (i32.add (local.get $arg1_lo) (local.get $arg1_hi))))
                (if (i32.eqz (local.get $new_arr))
                  (then
                    ;; Neither map nor array - return empty array
                    (local.set $new_arr (call $allocate_array (i32.const 0)))
                  )
                )
              )
            )
            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
            (return)
          )
        )

        ;; If arg is not an object, return empty array
        (if (i32.ne (local.get $arg1_type) (global.get $TYPE_OBJECT))
          (then
            ;; Create empty array
            (local.set $new_arr (call $allocate_array (i32.const 0)))
            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
            (return)
          )
        )
        ;; Get object properties
        (local.set $obj_ptr (local.get $arg1_lo))
        (local.set $count (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 8))))) ;; count
        (local.set $entries_ptr (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 16))))) ;; entries_ptr

        ;; Allocate result array
        (local.set $new_arr (call $allocate_array (local.get $count)))
        (local.set $new_data (i32.load (call $abs (i32.add (local.get $new_arr) (i32.const 16)))))

        ;; Copy keys
        (local.set $i (i32.const 0))
        (block $keys_done
          (loop $keys_loop
            (br_if $keys_done (i32.ge_u (local.get $i) (local.get $count)))
            ;; Entry format: [key_offset:4][value:16]
            (local.set $entry_ptr (i32.add
              (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
              (i32.mul (local.get $i) (i32.const 20))))
            (local.set $key_offset (i32.load (call $abs (local.get $entry_ptr))))

            ;; Write key as string to array
            (local.set $elem_ptr (i32.add
              (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
              (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
            (i32.store (call $abs (local.get $elem_ptr)) (global.get $TYPE_STRING))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $key_offset))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (i32.const 0))

            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $keys_loop)
          )
        )

        ;; Set array length
        (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 8))) (local.get $count))

        (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
        (return)
      )
    )

    ;; Object.values(obj) - returns array of own property values
    (if (i32.eq (local.get $method_id) (global.get $METHOD_OBJECT_VALUES))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))

        ;; Handle TYPE_MSGPACK_REF
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_MSGPACK_REF))
          (then
            (local.set $new_arr (call $msgpack_map_values
              (local.get $arg1_lo)
              (i32.add (local.get $arg1_lo) (local.get $arg1_hi))))
            (if (i32.eqz (local.get $new_arr))
              (then
                ;; Not a map - try array (msgpack_to_array materializes it)
                (local.set $new_arr (call $msgpack_to_array
                  (local.get $arg1_lo)
                  (i32.add (local.get $arg1_lo) (local.get $arg1_hi))))
                (if (i32.eqz (local.get $new_arr))
                  (then
                    ;; Neither map nor array - return empty array
                    (local.set $new_arr (call $allocate_array (i32.const 0)))
                  )
                )
              )
            )
            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
            (return)
          )
        )

        (if (i32.ne (local.get $arg1_type) (global.get $TYPE_OBJECT))
          (then
            (local.set $new_arr (call $allocate_array (i32.const 0)))
            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
            (return)
          )
        )
        (local.set $obj_ptr (local.get $arg1_lo))
        (local.set $count (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))))
        (local.set $entries_ptr (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 16)))))

        (local.set $new_arr (call $allocate_array (local.get $count)))
        (local.set $new_data (i32.load (call $abs (i32.add (local.get $new_arr) (i32.const 16)))))

        (local.set $i (i32.const 0))
        (block $values_done
          (loop $values_loop
            (br_if $values_done (i32.ge_u (local.get $i) (local.get $count)))
            ;; Entry format: [key_offset:4][value:16]
            (local.set $entry_ptr (i32.add
              (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
              (i32.mul (local.get $i) (i32.const 20))))

            ;; Copy value (16 bytes at entry_ptr+4)
            (local.set $elem_ptr (i32.add
              (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
              (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
            (i32.store (call $abs (local.get $elem_ptr))
              (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 4)))))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))
              (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 8)))))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))
              (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 12)))))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))
              (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 16)))))

            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $values_loop)
          )
        )

        (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 8))) (local.get $count))
        (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
        (return)
      )
    )

    ;; Object.entries(obj) - returns array of [key, value] pairs
    (if (i32.eq (local.get $method_id) (global.get $METHOD_OBJECT_ENTRIES))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))

        ;; Handle TYPE_MSGPACK_REF
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_MSGPACK_REF))
          (then
            (local.set $new_arr (call $msgpack_map_entries
              (local.get $arg1_lo)
              (i32.add (local.get $arg1_lo) (local.get $arg1_hi))))
            (if (i32.eqz (local.get $new_arr))
              (then
                (local.set $new_arr (call $allocate_array (i32.const 0)))
              )
            )
            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
            (return)
          )
        )

        (if (i32.ne (local.get $arg1_type) (global.get $TYPE_OBJECT))
          (then
            (local.set $new_arr (call $allocate_array (i32.const 0)))
            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
            (return)
          )
        )
        (local.set $obj_ptr (local.get $arg1_lo))
        (local.set $count (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))))
        (local.set $entries_ptr (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 16)))))

        ;; Allocate outer array
        (local.set $new_arr (call $allocate_array (local.get $count)))
        (local.set $new_data (i32.load (call $abs (i32.add (local.get $new_arr) (i32.const 16)))))

        (local.set $i (i32.const 0))
        (block $entries_done
          (loop $entries_loop
            (br_if $entries_done (i32.ge_u (local.get $i) (local.get $count)))
            ;; Entry format: [key_offset:4][value:16]
            (local.set $entry_ptr (i32.add
              (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
              (i32.mul (local.get $i) (i32.const 20))))
            (local.set $key_offset (i32.load (call $abs (local.get $entry_ptr))))

            ;; Allocate [key, value] pair array
            (local.set $arr_ptr (call $allocate_array (i32.const 2)))
            (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 16)))))

            ;; Write key at index 0
            (local.set $elem_ptr (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE)))
            (i32.store (call $abs (local.get $elem_ptr)) (global.get $TYPE_STRING))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $key_offset))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (i32.const 0))

            ;; Write value at index 1 (copy from entry_ptr+4)
            (local.set $elem_ptr (i32.add (local.get $elem_ptr) (global.get $VALUE_SIZE)))
            (i32.store (call $abs (local.get $elem_ptr))
              (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 4)))))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))
              (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 8)))))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))
              (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 12)))))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))
              (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 16)))))

            ;; Set pair length to 2
            (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 8))) (i32.const 2))

            ;; Write pair to outer array
            (local.set $elem_ptr (i32.add
              (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
              (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
            (i32.store (call $abs (local.get $elem_ptr)) (global.get $TYPE_ARRAY))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $arr_ptr))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (i32.const 0))

            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $entries_loop)
          )
        )

        (i32.store (call $abs (i32.add (local.get $new_arr) (i32.const 8))) (local.get $count))
        (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
        (return)
      )
    )

    ;; Object.assign(target, ...sources) - copy properties from sources to target
    (if (i32.eq (local.get $method_id) (global.get $METHOD_OBJECT_ASSIGN))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))

        ;; Target must be an object
        (if (i32.ne (local.get $arg1_type) (global.get $TYPE_OBJECT))
          (then
            ;; Return target as-is if not an object (JS returns the value)
            (call $pending_push (local.get $arg1_type) (i32.const 0) (local.get $arg1_lo) (i32.const 0))
            (return)
          )
        )

        (local.set $obj_ptr (local.get $arg1_lo))  ;; target object

        ;; Process each source argument (starting from arg2)
        (local.set $i (i32.const 1))  ;; skip target (arg1)
        (local.set $arg_ptr (i32.add (local.get $arg1_ptr) (global.get $VALUE_SIZE)))
        (block $sources_done
          (loop $sources_loop
            (br_if $sources_done (i32.ge_u (local.get $i) (local.get $argc)))

            ;; Get source type
            (local.set $arg_type (call $value_type (local.get $arg_ptr)))

            ;; Handle TYPE_MSGPACK_REF source
            (if (i32.eq (local.get $arg_type) (global.get $TYPE_MSGPACK_REF))
              (then
                (call $msgpack_assign_to_object
                  (local.get $obj_ptr)
                  (call $value_data_lo (local.get $arg_ptr))
                  (i32.add
                    (call $value_data_lo (local.get $arg_ptr))
                    (call $value_data_hi (local.get $arg_ptr))))
              )
              (else
                ;; Handle TYPE_OBJECT source (skip if not an object)
                (if (i32.eq (local.get $arg_type) (global.get $TYPE_OBJECT))
                  (then
                    (local.set $src_ptr (call $value_data_lo (local.get $arg_ptr)))
                    (local.set $src_count (i32.load (call $abs (i32.add (local.get $src_ptr) (i32.const 8)))))
                    (local.set $src_entries (i32.load (call $abs (i32.add (local.get $src_ptr) (i32.const 16)))))

                    ;; Iterate through source entries
                    (local.set $j (i32.const 0))
                    (block $entries_done
                      (loop $entries_loop
                        (br_if $entries_done (i32.ge_u (local.get $j) (local.get $src_count)))

                        ;; Get source entry
                        (local.set $entry_ptr (i32.add
                          (i32.add (local.get $src_entries) (global.get $GC_HEADER_SIZE))
                          (i32.mul (local.get $j) (i32.const 20))))

                        ;; Read key and value from source entry
                        (local.set $key_offset (i32.load (call $abs (local.get $entry_ptr))))
                        (local.set $val_type (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 4)))))
                        (local.set $val_flags (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 8)))))
                        (local.set $val_lo (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 12)))))
                        (local.set $val_hi (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 16)))))

                        ;; Set property on target (call helper)
                        (call $object_set_property
                          (local.get $obj_ptr)
                          (local.get $key_offset)
                          (local.get $val_type)
                          (local.get $val_flags)
                          (local.get $val_lo)
                          (local.get $val_hi))

                        (local.set $j (i32.add (local.get $j) (i32.const 1)))
                        (br $entries_loop)
                      )
                    )
                  )
                )
              )
            )

            (local.set $arg_ptr (i32.add (local.get $arg_ptr) (global.get $VALUE_SIZE)))
            (local.set $i (i32.add (local.get $i) (i32.const 1)))
            (br $sources_loop)
          )
        )

        ;; Return target object
        (call $pending_push (global.get $TYPE_OBJECT) (i32.const 0) (local.get $obj_ptr) (i32.const 0))
        (return)
      )
    )

    ;; Array.isArray(x) - returns true if x is an array (including msgpack arrays)
    (if (i32.eq (local.get $method_id) (global.get $METHOD_ARRAY_IS_ARRAY))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
        ;; Check for TYPE_ARRAY or TYPE_MSGPACK_REF that's an array
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_ARRAY))
          (then (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0)))
          (else
            (if (i32.eq (local.get $arg1_type) (global.get $TYPE_MSGPACK_REF))
              (then
                ;; Check if msgpack is array type
                (if (i32.eq
                      (call $msgpack_peek_type (local.get $arg1_lo) (i32.add (local.get $arg1_lo) (local.get $arg1_hi)))
                      (global.get $MSGPACK_ARRAY))
                  (then (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0)))
                  (else (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0)))
                )
              )
              (else (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0)))
            )
          )
        )
        (return)
      )
    )

    ;; Array.from(arrayLike) - creates array from array-like or iterable
    (if (i32.eq (local.get $method_id) (global.get $METHOD_ARRAY_FROM))
      (then
        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))

        ;; If already a heap array, clone it
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_ARRAY))
          (then
            ;; Get array info
            (local.set $count (i32.load (call $abs (i32.add (local.get $arg1_lo) (i32.const 8)))))
            (local.set $new_arr (call $allocate_array (local.get $count)))
            (local.set $new_data (i32.load (call $abs (i32.add (local.get $new_arr) (i32.const 16)))))
            (local.set $entries_ptr (i32.load (call $abs (i32.add (local.get $arg1_lo) (i32.const 16)))))

            ;; Copy elements
            ;; Note: entries_ptr and new_data point to GC header, need to add GC_HEADER_SIZE
            (local.set $i (i32.const 0))
            (block $copy_done
              (loop $copy_loop
                (br_if $copy_done (i32.ge_u (local.get $i) (local.get $count)))
                (local.set $elem_ptr (i32.add
                  (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                (local.set $entry_ptr (i32.add
                  (i32.add (local.get $new_data) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                ;; Copy 16 bytes
                (i32.store (call $abs (local.get $entry_ptr))
                  (i32.load (call $abs (local.get $elem_ptr))))
                (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 4)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 8)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 12)))
                  (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))
                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $copy_loop)
              )
            )
            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
            (return)
          )
        )

        ;; If msgpack ref, materialize if it's an array
        (if (i32.eq (local.get $arg1_type) (global.get $TYPE_MSGPACK_REF))
          (then
            ;; Check if it's a msgpack array before calling msgpack_to_array
            ;; (to avoid setting error state for non-arrays)
            (if (i32.eq
                  (call $msgpack_peek_type (local.get $arg1_lo) (i32.add (local.get $arg1_lo) (local.get $arg1_hi)))
                  (global.get $MSGPACK_ARRAY))
              (then
                (local.set $new_arr (call $msgpack_to_array
                  (local.get $arg1_lo)
                  (i32.add (local.get $arg1_lo) (local.get $arg1_hi))))
              )
              (else
                ;; Not an array - return empty array
                (local.set $new_arr (call $allocate_array (i32.const 0)))
              )
            )
            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
            (return)
          )
        )

        ;; For other types, return empty array (simplified - full impl would handle iterables)
        (local.set $new_arr (call $allocate_array (i32.const 0)))
        (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $new_arr) (i32.const 0))
        (return)
      )
    )

    ;; Unknown method - reset stack and push undefined
    (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $stack_base))
    (call $pending_push (global.get $TYPE_UNDEFINED) (i32.const 0) (i32.const 0) (i32.const 0))
  )

  ;; ==========================================================================
  ;; run(fuel) → status
  ;;
  ;; Execute instructions until fuel runs out, completion, or error.
  ;; ==========================================================================
  (func $run (export "run") (param $fuel i32) (result i32)
    (local $code_block i32)
    (local $pc i32)
    (local $instr_count i32)
    (local $opcode i32)
    (local $operand1 i32)
    (local $operand2 i32)
    (local $val_ptr i32)
    (local $val_ptr2 i32)
    (local $type1 i32)
    (local $type2 i32)
    (local $data1_lo i32)
    (local $data1_hi i32)
    (local $data2_lo i32)
    (local $data2_hi i32)
    (local $result_i32 i32)
    (local $result_f64 f64)
    (local $found i32)
    (local $new_scope i32)
    ;; For MAKE_ARRAY / MAKE_OBJECT / GET_INDEX / SET_INDEX / GET_PROP
    (local $count i32)
    (local $heap_ptr i32)
    (local $arr_ptr i32)
    (local $obj_ptr i32)
    (local $data_ptr i32)
    (local $data_size i32)
    (local $obj_size i32)
    (local $i i32)
    (local $dest i32)
    (local $entry_ptr i32)
    (local $key_offset i32)
    (local $index i32)
    (local $arr_len i32)
    (local $arr_cap i32)
    (local $new_cap i32)
    (local $new_data_ptr i32)
    (local $new_data_size i32)
    (local $elem_ptr i32)
    (local $obj_count i32)
    (local $obj_cap i32)
    (local $entries_ptr i32)
    (local $new_entries_ptr i32)
    (local $new_entries_size i32)
    (local $prop_key i32)
    ;; For SET_INDEX value storage
    (local $set_val_type i32)
    (local $set_val_flags i32)
    (local $set_val_lo i32)
    (local $set_val_hi i32)
    ;; For CALL / RETURN / CALL_METHOD
    (local $closure_ptr i32)
    (local $start_instr i32)
    (local $captured_scope i32)
    (local $stack_ptr i32)
    (local $frame_ptr i32)
    (local $pending_ptr i32)
    (local $result_type i32)
    (local $result_flags i32)
    (local $result_lo i32)
    (local $result_hi i32)
    (local $return_pc i32)
    (local $caller_scope i32)
    (local $closure_flags i32)
    ;; For SWAP
    (local $swap_flags i32)
    ;; For bound method dispatch
    (local $method_id i32)
    (local $receiver_type i32)
    (local $receiver_ptr i32)
    ;; For msgpack parsing
    (local $mp_type i32)
    (local $mp_addr i32)
    (local $mp_end i32)
    (local $mp_length i32)
    (local $mp_header_size i32)
    (local $mp_data_size i32)
    (local $mp_count i32)

    ;; Load execution state
    (local.set $code_block (call $read_state (global.get $STATE_CODE_BLOCK)))
    (local.set $pc (call $read_state (global.get $STATE_INSTRUCTION_INDEX)))
    ;; Load current scope
    (global.set $current_scope (call $read_state (global.get $STATE_SCOPE)))

    ;; If already done or error, return current status (no-op)
    (if (i32.eq (call $read_state (global.get $STATE_STATUS)) (global.get $STATUS_DONE))
      (then
        (return (global.get $STATUS_DONE))
      )
    )
    (if (i32.eq (call $read_state (global.get $STATE_STATUS)) (global.get $STATUS_ERROR))
      (then
        (return (global.get $STATUS_ERROR))
      )
    )

    ;; If no code block set, done
    (if (i32.eqz (local.get $code_block))
      (then
        (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_DONE))
        (return (global.get $STATUS_DONE))
      )
    )

    (local.set $instr_count (call $read_instr_count (local.get $code_block)))

    ;; Set status to running (unless we're handling a throw)
    (if (i32.ne (call $read_state (global.get $STATE_STATUS)) (global.get $STATUS_THROW))
      (then
        (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_RUNNING))
      )
    )

    ;; Main execution loop
    (block $exit
      (loop $loop
        ;; Check for STATUS_THROW (FFI threw an exception)
        ;; Exception value is on pending stack, process it like OP_THROW
        (if (i32.eq (call $read_state (global.get $STATE_STATUS)) (global.get $STATUS_THROW))
          (then
            ;; Set status back to running
            (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_RUNNING))

            ;; Pop exception value (already on pending stack from orchestrator)
            (local.set $val_ptr (call $pending_pop))

            ;; Push it back so save_completion_value can read it
            (call $pending_push
              (call $value_type (local.get $val_ptr))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4))))
              (call $value_data_lo (local.get $val_ptr))
              (call $value_data_hi (local.get $val_ptr)))

            ;; Save exception to completion value slot
            (call $save_completion_value)

            ;; Pop it again (we saved it)
            (drop (call $pending_pop))

            ;; Set completion type to throw
            (call $write_state (global.get $STATE_COMPLETION_TYPE) (global.get $COMPLETION_THROW))

            ;; Search try stack for handler
            (local.set $i (i32.const 0))
            (local.set $found (i32.const 0))

            (block $ffi_handler_found
              (block $ffi_no_handler
                (loop $ffi_search_loop
                  (br_if $ffi_no_handler (i32.ge_u (local.get $i) (call $try_depth)))

                  (local.set $operand1 (call $try_read (local.get $i) (global.get $TRY_CATCH_INDEX)))
                  (local.set $operand2 (call $try_read (local.get $i) (global.get $TRY_FINALLY_INDEX)))
                  (local.set $index (call $try_read (local.get $i) (global.get $TRY_FRAME_DEPTH)))

                  (call $try_pop)

                  (if (i32.ne (local.get $operand1) (i32.const 0))
                    (then
                      (call $unwind_frames_to (local.get $index))
                      (call $restore_completion_value)
                      (call $write_state (global.get $STATE_COMPLETION_TYPE) (global.get $COMPLETION_NORMAL))
                      (local.set $pc (local.get $operand1))
                      (local.set $found (i32.const 1))
                      (br $ffi_handler_found)
                    )
                  )

                  (if (i32.ne (local.get $operand2) (i32.const 0))
                    (then
                      (call $unwind_frames_to (local.get $index))
                      (local.set $pc (local.get $operand2))
                      (local.set $found (i32.const 1))
                      (br $ffi_handler_found)
                    )
                  )

                  (br $ffi_search_loop)
                )
              )

              ;; No handler found - uncaught exception
              (call $set_error (global.get $ERR_USER_THROW) (i32.const 0))
              (br $exit)
            )

            ;; Handler found, continue execution
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; Check fuel
        (if (i32.le_s (local.get $fuel) (i32.const 0))
          (then
            ;; Save state and pause
            (call $write_state (global.get $STATE_CODE_BLOCK) (local.get $code_block))
            (call $write_state (global.get $STATE_INSTRUCTION_INDEX) (local.get $pc))
            (call $write_state (global.get $STATE_SCOPE) (global.get $current_scope))
            (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_PAUSED_FUEL))
            (return (global.get $STATUS_PAUSED_FUEL))
          )
        )

        ;; Check if we've reached the end
        (if (i32.ge_u (local.get $pc) (local.get $instr_count))
          (then
            (call $write_state (global.get $STATE_SCOPE) (global.get $current_scope))
            (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_DONE))
            (br $exit)
          )
        )

        ;; Fetch instruction
        (local.set $opcode (call $read_opcode (local.get $code_block) (local.get $pc)))
        (local.set $operand1 (call $read_operand1 (local.get $code_block) (local.get $pc)))
        (local.set $operand2 (call $read_operand2 (local.get $code_block) (local.get $pc)))

        ;; Decode and execute
        ;; =====================================================================
        ;; LITERALS
        ;; =====================================================================
        ;; Note: LIT_INT is no longer emitted by parser (all numbers are f64)
        ;; Keep handler for backwards compatibility with any existing bytecode

        (if (i32.eq (local.get $opcode) (global.get $OP_LIT_FLOAT))
          (then
            (call $pending_push (global.get $TYPE_FLOAT) (i32.const 0) (local.get $operand1) (local.get $operand2))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_LIT_STRING))
          (then
            (call $pending_push (global.get $TYPE_STRING) (i32.const 0) (local.get $operand1) (i32.const 0))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_LIT_NULL))
          (then
            (call $pending_push (global.get $TYPE_NULL) (i32.const 0) (i32.const 0) (i32.const 0))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_LIT_UNDEFINED))
          (then
            (call $pending_push (global.get $TYPE_UNDEFINED) (i32.const 0) (i32.const 0) (i32.const 0))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_LIT_TRUE))
          (then
            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_LIT_FALSE))
          (then
            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; =====================================================================
        ;; VARIABLES
        ;; =====================================================================
        (if (i32.eq (local.get $opcode) (global.get $OP_GET_VAR))
          (then
            ;; Look up variable
            (local.set $val_ptr (call $scope_lookup (global.get $current_scope) (local.get $operand1)))
            (if (i32.eqz (local.get $val_ptr))
              (then
                (call $set_error (global.get $ERR_UNDEFINED_VARIABLE) (local.get $operand1))
                (br $exit)
              )
            )
            ;; Check if TYPE_MSGPACK_REF
            (local.set $type1 (i32.load (call $abs (local.get $val_ptr))))
            (if (i32.eq (local.get $type1) (global.get $TYPE_MSGPACK_REF))
              (then
                ;; data_lo = msgpack address (absolute), data_hi = length
                (local.set $data1_lo (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 8)))))
                (local.set $data1_hi (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))))
                ;; Peek the type to determine if scalar or container
                (local.set $mp_type (call $msgpack_peek_type
                  (local.get $data1_lo)
                  (i32.add (local.get $data1_lo) (local.get $data1_hi))))
                (if (i32.eq (local.get $mp_type) (global.get $MSGPACK_ERROR))
                  (then
                    (call $set_error (global.get $ERR_MSGPACK_INVALID) (i32.const 0))
                    (br $exit)
                  )
                )
                ;; For scalars (nil, bool, int, float, string), read the primitive
                (if (i32.or
                      (i32.or
                        (i32.eq (local.get $mp_type) (global.get $MSGPACK_NIL))
                        (i32.eq (local.get $mp_type) (global.get $MSGPACK_BOOL)))
                      (i32.or
                        (i32.or
                          (i32.eq (local.get $mp_type) (global.get $MSGPACK_INT))
                          (i32.eq (local.get $mp_type) (global.get $MSGPACK_FLOAT)))
                        (i32.eq (local.get $mp_type) (global.get $MSGPACK_STRING))))
                  (then
                    ;; Read primitive directly to pending stack
                    (if (i32.eq (call $msgpack_read_primitive
                                  (local.get $data1_lo)
                                  (i32.add (local.get $data1_lo) (local.get $data1_hi)))
                                (i32.const -1))
                      (then
                        (call $set_error (global.get $ERR_MSGPACK_INVALID) (i32.const 1))
                        (br $exit)
                      )
                    )
                    (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )
                ;; For containers (array, map), keep as TYPE_MSGPACK_REF
                (call $pending_push
                  (global.get $TYPE_MSGPACK_REF)
                  (i32.const 0)
                  (local.get $data1_lo)
                  (local.get $data1_hi))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )
            ;; Push value to pending stack (non-msgpack case)
            (call $pending_push
              (i32.load (call $abs (local.get $val_ptr)))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4))))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 8))))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_SET_VAR))
          (then
            ;; Pop value
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $data1_lo (call $value_data_lo (local.get $val_ptr)))
            (local.set $data1_hi (call $value_data_hi (local.get $val_ptr)))

            ;; Set in scope chain
            (local.set $found (call $scope_set
              (global.get $current_scope)
              (local.get $operand1)
              (local.get $type1)
              (i32.const 0)
              (local.get $data1_lo)
              (local.get $data1_hi)))

            (if (i32.eqz (local.get $found))
              (then
                (call $set_error (global.get $ERR_ASSIGN_UNDEFINED) (local.get $operand1))
                (br $exit)
              )
            )

            ;; Push value back (assignment is an expression)
            (call $pending_push (local.get $type1) (i32.const 0) (local.get $data1_lo) (local.get $data1_hi))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_LET_VAR))
          (then
            ;; Pop value
            (local.set $val_ptr (call $pending_pop))

            ;; Define in current scope
            (call $scope_define
              (global.get $current_scope)
              (local.get $operand1)
              (call $value_type (local.get $val_ptr))
              (i32.const 0)
              (call $value_data_lo (local.get $val_ptr))
              (call $value_data_hi (local.get $val_ptr)))

            ;; Check if redeclaration error occurred
            (if (i32.eq (call $read_state (global.get $STATE_STATUS)) (global.get $STATUS_ERROR))
              (then (br $exit)))

            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; =====================================================================
        ;; ARITHMETIC (f64 - JS semantics: all numbers are floats)
        ;; =====================================================================
        (if (i32.eq (local.get $opcode) (global.get $OP_ADD))
          (then
            (local.set $val_ptr2 (call $pending_pop))  ;; right
            (local.set $val_ptr (call $pending_pop))   ;; left
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            ;; Float + Float
            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
                  (i32.eq (local.get $type2) (global.get $TYPE_FLOAT)))
              (then
                (call $push_f64
                  (f64.add
                    (call $read_f64 (local.get $val_ptr))
                    (call $read_f64 (local.get $val_ptr2))))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; String + String concatenation
            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_STRING))
                  (i32.eq (local.get $type2) (global.get $TYPE_STRING)))
              (then
                (call $pending_push
                  (global.get $TYPE_STRING)
                  (i32.const 0)
                  (call $string_concat
                    (call $value_data_lo (local.get $val_ptr))
                    (call $value_data_lo (local.get $val_ptr2)))
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; String + other (coerce to string) - left is string
            (if (i32.eq (local.get $type1) (global.get $TYPE_STRING))
              (then
                (call $pending_push
                  (global.get $TYPE_STRING)
                  (i32.const 0)
                  (call $string_concat
                    (call $value_data_lo (local.get $val_ptr))
                    (call $value_to_string (local.get $val_ptr2)))
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; other + String (coerce to string) - right is string
            (if (i32.eq (local.get $type2) (global.get $TYPE_STRING))
              (then
                (call $pending_push
                  (global.get $TYPE_STRING)
                  (i32.const 0)
                  (call $string_concat
                    (call $value_to_string (local.get $val_ptr))
                    (call $value_data_lo (local.get $val_ptr2)))
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            (call $set_error (global.get $ERR_INVALID_OPERAND) (i32.const 0))
            (br $exit)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_SUB))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
                  (i32.eq (local.get $type2) (global.get $TYPE_FLOAT)))
              (then
                (call $push_f64
                  (f64.sub
                    (call $read_f64 (local.get $val_ptr))
                    (call $read_f64 (local.get $val_ptr2))))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )
            (call $set_error (global.get $ERR_INVALID_OPERAND) (i32.const 0))
            (br $exit)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_MUL))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
                  (i32.eq (local.get $type2) (global.get $TYPE_FLOAT)))
              (then
                (call $push_f64
                  (f64.mul
                    (call $read_f64 (local.get $val_ptr))
                    (call $read_f64 (local.get $val_ptr2))))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )
            (call $set_error (global.get $ERR_INVALID_OPERAND) (i32.const 0))
            (br $exit)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_DIV))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
                  (i32.eq (local.get $type2) (global.get $TYPE_FLOAT)))
              (then
                (call $push_f64
                  (f64.div
                    (call $read_f64 (local.get $val_ptr))
                    (call $read_f64 (local.get $val_ptr2))))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )
            (call $set_error (global.get $ERR_INVALID_OPERAND) (i32.const 0))
            (br $exit)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_MOD))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
                  (i32.eq (local.get $type2) (global.get $TYPE_FLOAT)))
              (then
                ;; a % b = a - trunc(a/b) * b
                (call $push_f64
                  (f64.sub
                    (call $read_f64 (local.get $val_ptr))
                    (f64.mul
                      (f64.trunc
                        (f64.div
                          (call $read_f64 (local.get $val_ptr))
                          (call $read_f64 (local.get $val_ptr2))))
                      (call $read_f64 (local.get $val_ptr2)))))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )
            (call $set_error (global.get $ERR_INVALID_OPERAND) (i32.const 0))
            (br $exit)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_NEG))
          (then
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))

            (if (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
              (then
                (call $push_f64
                  (f64.neg (call $read_f64 (local.get $val_ptr))))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )
            (call $set_error (global.get $ERR_INVALID_OPERAND) (i32.const 0))
            (br $exit)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_POW))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
                  (i32.eq (local.get $type2) (global.get $TYPE_FLOAT)))
              (then
                (call $push_f64
                  (call $pow
                    (call $read_f64 (local.get $val_ptr))
                    (call $read_f64 (local.get $val_ptr2))))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )
            (call $set_error (global.get $ERR_INVALID_OPERAND) (i32.const 0))
            (br $exit)
          )
        )

        ;; =====================================================================
        ;; BITWISE OPERATORS
        ;; =====================================================================

        ;; BAND: bitwise AND
        (if (i32.eq (local.get $opcode) (global.get $OP_BAND))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (call $push_f64 (f64.convert_i32_s
              (i32.and
                (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr)))
                (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr2))))))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; BOR: bitwise OR
        (if (i32.eq (local.get $opcode) (global.get $OP_BOR))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (call $push_f64 (f64.convert_i32_s
              (i32.or
                (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr)))
                (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr2))))))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; BXOR: bitwise XOR
        (if (i32.eq (local.get $opcode) (global.get $OP_BXOR))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (call $push_f64 (f64.convert_i32_s
              (i32.xor
                (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr)))
                (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr2))))))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; BNOT: bitwise NOT (unary)
        (if (i32.eq (local.get $opcode) (global.get $OP_BNOT))
          (then
            (local.set $val_ptr (call $pending_pop))
            (call $push_f64 (f64.convert_i32_s
              (i32.xor
                (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr)))
                (i32.const -1))))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; SHL: left shift
        (if (i32.eq (local.get $opcode) (global.get $OP_SHL))
          (then
            (local.set $val_ptr2 (call $pending_pop))  ;; shift amount
            (local.set $val_ptr (call $pending_pop))   ;; value
            (call $push_f64 (f64.convert_i32_s
              (i32.shl
                (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr)))
                (i32.and
                  (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr2)))
                  (i32.const 31)))))  ;; mask to 5 bits per JS spec
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; SHR: signed right shift
        (if (i32.eq (local.get $opcode) (global.get $OP_SHR))
          (then
            (local.set $val_ptr2 (call $pending_pop))  ;; shift amount
            (local.set $val_ptr (call $pending_pop))   ;; value
            (call $push_f64 (f64.convert_i32_s
              (i32.shr_s
                (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr)))
                (i32.and
                  (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr2)))
                  (i32.const 31)))))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; USHR: unsigned right shift (result is always positive)
        (if (i32.eq (local.get $opcode) (global.get $OP_USHR))
          (then
            (local.set $val_ptr2 (call $pending_pop))  ;; shift amount
            (local.set $val_ptr (call $pending_pop))   ;; value
            (call $push_f64 (f64.convert_i32_u
              (i32.shr_u
                (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr)))
                (i32.and
                  (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr2)))
                  (i32.const 31)))))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; =====================================================================
        ;; COMPARISON
        ;; =====================================================================
        (if (i32.eq (local.get $opcode) (global.get $OP_EQ))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            ;; Different types are never equal (strict equality)
            (if (i32.ne (local.get $type1) (local.get $type2))
              (then
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Float comparison: use f64.eq (handles NaN !== NaN correctly)
            (if (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
              (then
                (call $pending_push
                  (global.get $TYPE_BOOLEAN)
                  (i32.const 0)
                  (f64.eq
                    (call $read_f64 (local.get $val_ptr))
                    (call $read_f64 (local.get $val_ptr2)))
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Other types: compare data_lo (booleans, strings by ref, etc.)
            (local.set $data1_lo (call $value_data_lo (local.get $val_ptr)))
            (local.set $data2_lo (call $value_data_lo (local.get $val_ptr2)))
            (if (i32.eq (local.get $data1_lo) (local.get $data2_lo))
              (then
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))
              )
              (else
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
              )
            )
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_LT))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            ;; Float < Float
            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
                  (i32.eq (local.get $type2) (global.get $TYPE_FLOAT)))
              (then
                (call $pending_push
                  (global.get $TYPE_BOOLEAN)
                  (i32.const 0)
                  (f64.lt
                    (call $read_f64 (local.get $val_ptr))
                    (call $read_f64 (local.get $val_ptr2)))
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )
            (call $set_error (global.get $ERR_INVALID_OPERAND) (i32.const 0))
            (br $exit)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_GT))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            ;; Float > Float
            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
                  (i32.eq (local.get $type2) (global.get $TYPE_FLOAT)))
              (then
                (call $pending_push
                  (global.get $TYPE_BOOLEAN)
                  (i32.const 0)
                  (f64.gt
                    (call $read_f64 (local.get $val_ptr))
                    (call $read_f64 (local.get $val_ptr2)))
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )
            (call $set_error (global.get $ERR_INVALID_OPERAND) (i32.const 0))
            (br $exit)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_LTE))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            ;; Float <= Float
            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
                  (i32.eq (local.get $type2) (global.get $TYPE_FLOAT)))
              (then
                (call $pending_push
                  (global.get $TYPE_BOOLEAN)
                  (i32.const 0)
                  (f64.le
                    (call $read_f64 (local.get $val_ptr))
                    (call $read_f64 (local.get $val_ptr2)))
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )
            (call $set_error (global.get $ERR_INVALID_OPERAND) (i32.const 0))
            (br $exit)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_GTE))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            ;; Float >= Float
            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
                  (i32.eq (local.get $type2) (global.get $TYPE_FLOAT)))
              (then
                (call $pending_push
                  (global.get $TYPE_BOOLEAN)
                  (i32.const 0)
                  (f64.ge
                    (call $read_f64 (local.get $val_ptr))
                    (call $read_f64 (local.get $val_ptr2)))
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )
            (call $set_error (global.get $ERR_INVALID_OPERAND) (i32.const 0))
            (br $exit)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_NEQ))
          (then
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            ;; Different types are never equal (so NEQ is true)
            (if (i32.ne (local.get $type1) (local.get $type2))
              (then
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Float comparison: use f64.ne
            (if (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
              (then
                (call $pending_push
                  (global.get $TYPE_BOOLEAN)
                  (i32.const 0)
                  (f64.ne
                    (call $read_f64 (local.get $val_ptr))
                    (call $read_f64 (local.get $val_ptr2)))
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Other types: compare data_lo
            (local.set $data1_lo (call $value_data_lo (local.get $val_ptr)))
            (local.set $data2_lo (call $value_data_lo (local.get $val_ptr2)))
            (if (i32.ne (local.get $data1_lo) (local.get $data2_lo))
              (then
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))
              )
              (else
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
              )
            )
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; =====================================================================
        ;; LOGIC
        ;; =====================================================================
        (if (i32.eq (local.get $opcode) (global.get $OP_NOT))
          (then
            (local.set $val_ptr (call $pending_pop))
            (if (call $is_truthy (local.get $val_ptr))
              (then
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
              )
              (else
                (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))
              )
            )
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; TYPEOF: pop value, push type string
        ;; Type mappings (JS semantics):
        ;;   TYPE_NULL (0) -> "object" (JS quirk)
        ;;   TYPE_UNDEFINED (1) -> "undefined"
        ;;   TYPE_BOOLEAN (2) -> "boolean"
        ;;   TYPE_FLOAT (4) -> "number"
        ;;   TYPE_STRING (5) -> "string"
        ;;   TYPE_ARRAY (6) -> "object"
        ;;   TYPE_OBJECT (7) -> "object"
        ;;   TYPE_CLOSURE (8) -> "function"
        (if (i32.eq (local.get $opcode) (global.get $OP_TYPEOF))
          (then
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))

            ;; Select the appropriate string offset based on type
            ;; Now reads from BUILTINS segment instead of STATE
            (if (i32.eq (local.get $type1) (global.get $TYPE_UNDEFINED))
              (then
                (local.set $key_offset (call $read_builtin (global.get $BUILTIN_TYPEOF_UNDEFINED)))
              )
              (else (if (i32.eq (local.get $type1) (global.get $TYPE_BOOLEAN))
                (then
                  (local.set $key_offset (call $read_builtin (global.get $BUILTIN_TYPEOF_BOOLEAN)))
                )
                (else (if (i32.eq (local.get $type1) (global.get $TYPE_FLOAT))
                  (then
                    (local.set $key_offset (call $read_builtin (global.get $BUILTIN_TYPEOF_NUMBER)))
                  )
                  (else (if (i32.eq (local.get $type1) (global.get $TYPE_STRING))
                    (then
                      (local.set $key_offset (call $read_builtin (global.get $BUILTIN_TYPEOF_STRING)))
                    )
                    (else (if (i32.or
                              (i32.eq (local.get $type1) (global.get $TYPE_CLOSURE))
                              (i32.eq (local.get $type1) (global.get $TYPE_BOUND_METHOD)))
                      (then
                        (local.set $key_offset (call $read_builtin (global.get $BUILTIN_TYPEOF_FUNCTION)))
                      )
                      (else
                        ;; TYPE_NULL, TYPE_ARRAY, TYPE_OBJECT all return "object"
                        (local.set $key_offset (call $read_builtin (global.get $BUILTIN_TYPEOF_OBJECT)))
                      )
                    ))
                  ))
                ))
              ))
            )

            ;; Push TYPE_STRING with the string offset
            (call $pending_push
              (global.get $TYPE_STRING)
              (i32.const 0)
              (local.get $key_offset)
              (i32.const 0))

            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; AND: short-circuit - if left is falsy, keep it and jump; else pop and continue
        (if (i32.eq (local.get $opcode) (global.get $OP_AND))
          (then
            (local.set $val_ptr (call $pending_peek))
            (if (call $is_truthy (local.get $val_ptr))
              (then
                ;; Truthy: pop left (discard), continue to evaluate right
                (drop (call $pending_pop))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
              )
              (else
                ;; Falsy: keep left on stack, jump past right
                (local.set $pc (local.get $operand1))
              )
            )
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; OR: short-circuit - if left is truthy, keep it and jump; else pop and continue
        (if (i32.eq (local.get $opcode) (global.get $OP_OR))
          (then
            (local.set $val_ptr (call $pending_peek))
            (if (call $is_truthy (local.get $val_ptr))
              (then
                ;; Truthy: keep left on stack, jump past right
                (local.set $pc (local.get $operand1))
              )
              (else
                ;; Falsy: pop left (discard), continue to evaluate right
                (drop (call $pending_pop))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
              )
            )
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; NULLISH: short-circuit - if left is nullish, pop and evaluate right; else keep and jump
        (if (i32.eq (local.get $opcode) (global.get $OP_NULLISH))
          (then
            (local.set $val_ptr (call $pending_peek))
            (local.set $type1 (i32.load (local.get $val_ptr)))
            ;; Check if type is NULL (0) or UNDEFINED (1)
            (if (i32.or
                  (i32.eq (local.get $type1) (global.get $TYPE_NULL))
                  (i32.eq (local.get $type1) (global.get $TYPE_UNDEFINED)))
              (then
                ;; Nullish: pop left (discard), continue to evaluate right
                (drop (call $pending_pop))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
              )
              (else
                ;; Not nullish: keep left on stack, jump past right
                (local.set $pc (local.get $operand1))
              )
            )
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; =====================================================================
        ;; CONTROL FLOW
        ;; =====================================================================
        (if (i32.eq (local.get $opcode) (global.get $OP_JUMP))
          (then
            (local.set $pc (local.get $operand1))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_JUMP_IF_FALSE))
          (then
            (local.set $val_ptr (call $pending_pop))
            (if (call $is_truthy (local.get $val_ptr))
              (then
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
              )
              (else
                (local.set $pc (local.get $operand1))
              )
            )
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_JUMP_IF_TRUE))
          (then
            (local.set $val_ptr (call $pending_pop))
            (if (call $is_truthy (local.get $val_ptr))
              (then
                (local.set $pc (local.get $operand1))
              )
              (else
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
              )
            )
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; =====================================================================
        ;; OBJECTS/ARRAYS
        ;; =====================================================================

        ;; MAKE_ARRAY: pop N values, create array, push array reference
        ;; operand1 = count
        (if (i32.eq (local.get $opcode) (global.get $OP_MAKE_ARRAY))
          (then
            (local.set $count (local.get $operand1))

            ;; Allocate array header: GC header (8) + length (4) + capacity (4) + data_ptr (4) = 20 bytes
            (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
            (local.set $arr_ptr (local.get $heap_ptr))

            ;; GC header for array
            (i32.store (call $abs (local.get $arr_ptr))
              (i32.or (i32.const 20) (i32.shl (global.get $OBJ_ARRAY) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 4))) (i32.const 0))

            ;; Allocate data block: GC header (8) + count * 16 bytes
            (local.set $data_size (i32.add (global.get $GC_HEADER_SIZE)
                                           (i32.mul (local.get $count) (global.get $VALUE_SIZE))))
            (local.set $data_ptr (i32.add (local.get $arr_ptr) (i32.const 20)))

            ;; GC header for data block
            (i32.store (call $abs (local.get $data_ptr))
              (i32.or (local.get $data_size) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $data_ptr) (i32.const 4))) (i32.const 0))

            ;; Array header: length, capacity, data_ptr
            (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 8))) (local.get $count))  ;; length
            (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 12))) (local.get $count)) ;; capacity
            (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 16))) (local.get $data_ptr)) ;; data_ptr (includes GC header)

            ;; Update heap pointer
            (call $write_state (global.get $STATE_HEAP_POINTER)
              (i32.add (local.get $data_ptr) (local.get $data_size)))

            ;; Pop values in reverse order and copy to data block
            ;; Values are on stack: [first, second, ..., last] with last on top
            ;; We need to pop last first and store at index (count-1)
            (local.set $i (local.get $count))
            (block $pop_done
              (loop $pop_loop
                (br_if $pop_done (i32.eqz (local.get $i)))
                (local.set $i (i32.sub (local.get $i) (i32.const 1)))

                (local.set $val_ptr (call $pending_pop))
                ;; dest = data_ptr + GC_HEADER_SIZE + i * 16
                (local.set $dest (i32.add
                  (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (global.get $VALUE_SIZE))))

                ;; Copy 16 bytes
                (i32.store (call $abs (local.get $dest))
                  (i32.load (call $abs (local.get $val_ptr))))
                (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4)))
                  (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
                (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8)))
                  (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 8)))))
                (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12)))
                  (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))))

                (br $pop_loop)
              )
            )

            ;; Push array reference (type=ARRAY, data_lo=arr_ptr)
            (call $pending_push
              (global.get $TYPE_ARRAY)
              (i32.const 0)
              (local.get $arr_ptr)
              (i32.const 0))

            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; MAKE_OBJECT: pop N key-value pairs, create object, push object reference
        ;; operand1 = count (number of key-value pairs)
        ;; Stack has: key1, val1, key2, val2, ... with last pair on top
        ;; New layout: Object header (20) + separate entries block
        ;;   Header: [GC:8][count:4][capacity:4][entries_ptr:4]
        ;;   Entries: [GC:8][entries: capacity * 20]
        (if (i32.eq (local.get $opcode) (global.get $OP_MAKE_OBJECT))
          (then
            (local.set $count (local.get $operand1))
            ;; Initial capacity: max(count, 4) to allow for growth
            (local.set $arr_cap (local.get $count))
            (if (i32.lt_u (local.get $arr_cap) (i32.const 4))
              (then (local.set $arr_cap (i32.const 4)))
            )

            (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))

            ;; Allocate object header: 20 bytes
            (local.set $obj_ptr (local.get $heap_ptr))
            (i32.store (call $abs (local.get $obj_ptr))
              (i32.or (i32.const 20) (i32.shl (global.get $OBJ_OBJECT) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 4))) (i32.const 0))

            ;; Allocate entries block: GC header (8) + capacity * 20
            (local.set $entries_ptr (i32.add (local.get $obj_ptr) (i32.const 20)))
            (local.set $data_size (i32.add (global.get $GC_HEADER_SIZE)
                                           (i32.mul (local.get $arr_cap) (i32.const 20))))
            (i32.store (call $abs (local.get $entries_ptr))
              (i32.or (local.get $data_size) (i32.shl (global.get $OBJ_OBJECT_DATA) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $entries_ptr) (i32.const 4))) (i32.const 0))

            ;; Object header: count, capacity, entries_ptr
            (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 8))) (local.get $count))
            (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 12))) (local.get $arr_cap))
            (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 16))) (local.get $entries_ptr))

            ;; Update heap pointer
            (call $write_state (global.get $STATE_HEAP_POINTER)
              (i32.add (local.get $entries_ptr) (local.get $data_size)))

            ;; Pop key-value pairs in reverse order
            ;; Stack: [key1, val1, key2, val2] with val2 on top
            ;; Entry layout: [key_ptr:4][type:4][flags:4][data_lo:4][data_hi:4]
            (local.set $i (local.get $count))
            (block $obj_pop_done
              (loop $obj_pop_loop
                (br_if $obj_pop_done (i32.eqz (local.get $i)))
                (local.set $i (i32.sub (local.get $i) (i32.const 1)))

                ;; entry_ptr = entries_ptr + 8 (GC header) + i * 20
                (local.set $entry_ptr (i32.add
                  (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (i32.const 20))))

                ;; Pop value first (it's on top)
                (local.set $val_ptr (call $pending_pop))

                ;; Pop key (should be a string - get its data_lo which is the string offset)
                (local.set $val_ptr2 (call $pending_pop))
                (local.set $key_offset (call $value_data_lo (local.get $val_ptr2)))

                ;; Write entry: [key][type][flags][data_lo][data_hi]
                (i32.store (call $abs (local.get $entry_ptr)) (local.get $key_offset))
                (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 4)))
                  (call $value_type (local.get $val_ptr)))
                (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 8)))
                  (i32.const 0))  ;; flags
                (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 12)))
                  (call $value_data_lo (local.get $val_ptr)))
                (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 16)))
                  (call $value_data_hi (local.get $val_ptr)))

                (br $obj_pop_loop)
              )
            )

            ;; Push object reference (type=OBJECT, data_lo=obj_ptr)
            (call $pending_push
              (global.get $TYPE_OBJECT)
              (i32.const 0)
              (local.get $obj_ptr)
              (i32.const 0))

            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; GET_INDEX: pop index, pop array/object, push element
        (if (i32.eq (local.get $opcode) (global.get $OP_GET_INDEX))
          (then
            ;; Pop index
            (local.set $val_ptr (call $pending_pop))
            (local.set $type2 (call $value_type (local.get $val_ptr)))

            ;; Pop array/object
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr2)))
            (local.set $arr_ptr (call $value_data_lo (local.get $val_ptr2)))

            ;; Check if array
            (if (i32.eq (local.get $type1) (global.get $TYPE_ARRAY))
              (then
                ;; Array indexing - convert index to i32
                (local.set $index (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr))))

                ;; Array: get length and bounds check
                (local.set $arr_len (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 8)))))

                (if (i32.ge_u (local.get $index) (local.get $arr_len))
                  (then
                    ;; Out of bounds - push undefined
                    (call $pending_push
                      (global.get $TYPE_UNDEFINED)
                      (i32.const 0)
                      (i32.const 0)
                      (i32.const 0))
                  )
                  (else
                    ;; Get data pointer and element
                    ;; data_ptr points to start of data block (including GC header), add 8 to skip it
                    (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 16)))))
                    (local.set $elem_ptr (i32.add
                      (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $index) (global.get $VALUE_SIZE))))

                    ;; Push element value
                    (call $pending_push
                      (i32.load (call $abs (local.get $elem_ptr)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))
                  )
                )

                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Check if object with string key
            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_OBJECT))
                  (i32.eq (local.get $type2) (global.get $TYPE_STRING)))
              (then
                ;; Object property lookup by string key
                (local.set $obj_ptr (local.get $arr_ptr))
                (local.set $key_offset (call $value_data_lo (local.get $val_ptr)))

                ;; Object layout: [GC:8][count:4][capacity:4][entries_ptr:4]
                (local.set $obj_count (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))))
                (local.set $entries_ptr (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 16)))))

                ;; Linear search for property
                (local.set $i (i32.const 0))
                (local.set $found (i32.const 0))
                (block $idx_prop_found
                  (loop $idx_prop_loop
                    (br_if $idx_prop_found (i32.ge_u (local.get $i) (local.get $obj_count)))

                    ;; entry_ptr = entries_ptr + 8 (GC header) + i * 20
                    (local.set $entry_ptr (i32.add
                      (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (i32.const 20))))

                    (local.set $prop_key (i32.load (call $abs (local.get $entry_ptr))))

                    (if (i32.eq (local.get $prop_key) (local.get $key_offset))
                      (then
                        ;; Found! Push the value
                        (call $pending_push
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))))   ;; type
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 8))))   ;; flags
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 12))))  ;; data_lo
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 16))))) ;; data_hi
                        (local.set $found (i32.const 1))
                        (br $idx_prop_found)
                      )
                    )

                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                    (br $idx_prop_loop)
                  )
                )

                ;; If not found, push undefined
                (if (i32.eqz (local.get $found))
                  (then
                    (call $pending_push
                      (global.get $TYPE_UNDEFINED)
                      (i32.const 0)
                      (i32.const 0)
                      (i32.const 0))
                  )
                )

                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Check if TYPE_MSGPACK_REF (read-only msgpack array)
            (if (i32.eq (local.get $type1) (global.get $TYPE_MSGPACK_REF))
              (then
                ;; arr_ptr = msgpack address (absolute)
                (local.set $data1_hi (call $value_data_hi (local.get $val_ptr2)))
                (local.set $mp_addr (local.get $arr_ptr))
                (local.set $mp_end (i32.add (local.get $mp_addr) (local.get $data1_hi)))

                ;; Convert index to i32
                (local.set $index (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr))))

                ;; Use msgpack_array_get to find the element
                (local.set $found (call $msgpack_array_get (local.get $mp_addr) (local.get $mp_end) (local.get $index)))

                (if (i32.eq (local.get $found) (i32.const -1))
                  (then
                    ;; Out of bounds or not an array - push undefined
                    (call $pending_push
                      (global.get $TYPE_UNDEFINED)
                      (i32.const 0)
                      (i32.const 0)
                      (i32.const 0))
                    (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )

                ;; found = address of element
                ;; Peek the element type
                (local.set $mp_type (call $msgpack_peek_type (local.get $found) (local.get $mp_end)))
                (if (i32.eq (local.get $mp_type) (global.get $MSGPACK_ERROR))
                  (then
                    (call $set_error (global.get $ERR_MSGPACK_INVALID) (i32.const 20))
                    (br $exit)
                  )
                )

                ;; For primitives, read directly
                (if (i32.or
                      (i32.or
                        (i32.eq (local.get $mp_type) (global.get $MSGPACK_NIL))
                        (i32.eq (local.get $mp_type) (global.get $MSGPACK_BOOL)))
                      (i32.or
                        (i32.or
                          (i32.eq (local.get $mp_type) (global.get $MSGPACK_INT))
                          (i32.eq (local.get $mp_type) (global.get $MSGPACK_FLOAT)))
                        (i32.eq (local.get $mp_type) (global.get $MSGPACK_STRING))))
                  (then
                    (if (i32.eq (call $msgpack_read_primitive (local.get $found) (local.get $mp_end))
                                (i32.const -1))
                      (then
                        (call $set_error (global.get $ERR_MSGPACK_INVALID) (i32.const 21))
                        (br $exit)
                      )
                    )
                    (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )

                ;; For containers (array/map), return new TYPE_MSGPACK_REF
                (call $pending_push
                  (global.get $TYPE_MSGPACK_REF)
                  (i32.const 0)
                  (local.get $found)
                  (i32.sub (local.get $mp_end) (local.get $found)))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Not an array or object[string] or msgpack - error
            (call $set_error (global.get $ERR_INVALID_OPERAND) (local.get $type1))
            (br $exit)
          )
        )

        ;; SET_INDEX: pop value, pop index, pop container, set container[index] = value, push value
        ;; Handles both arrays (numeric index) and objects (string index)
        (if (i32.eq (local.get $opcode) (global.get $OP_SET_INDEX))
          (then
            ;; Pop value to store
            (local.set $val_ptr (call $pending_pop))
            (local.set $set_val_type (i32.load (call $abs (local.get $val_ptr))))
            (local.set $set_val_flags (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
            (local.set $set_val_lo (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 8)))))
            (local.set $set_val_hi (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))))

            ;; Pop index
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $type2 (call $value_type (local.get $val_ptr2)))

            ;; Pop container
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))

            ;; Check if msgpack ref (read-only, cannot assign)
            (if (i32.eq (local.get $type1) (global.get $TYPE_MSGPACK_REF))
              (then
                (call $set_error (global.get $ERR_MSGPACK_READONLY) (local.get $type1))
                (br $exit)
              )
            )

            ;; Branch based on container type and index type
            (if (i32.and
                  (i32.eq (local.get $type1) (global.get $TYPE_OBJECT))
                  (i32.eq (local.get $type2) (global.get $TYPE_STRING)))
              (then
                ;; Object with string key - same logic as SET_PROP
                (local.set $obj_ptr (call $value_data_lo (local.get $val_ptr)))
                (local.set $key_offset (call $value_data_lo (local.get $val_ptr2)))

                ;; Object layout: [GC:8][count:4][capacity:4][entries_ptr:4]
                (local.set $obj_count (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))))
                (local.set $obj_cap (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 12)))))
                (local.set $entries_ptr (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 16)))))

                ;; Search for existing property
                (local.set $i (i32.const 0))
                (local.set $found (i32.const 0))
                (block $idx_prop_set_found
                  (loop $idx_prop_set_loop
                    (br_if $idx_prop_set_found (i32.ge_u (local.get $i) (local.get $obj_count)))

                    ;; entry_ptr = entries_ptr + 8 + i * 20
                    (local.set $entry_ptr (i32.add
                      (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (i32.const 20))))

                    (local.set $prop_key (i32.load (call $abs (local.get $entry_ptr))))

                    (if (i32.eq (local.get $prop_key) (local.get $key_offset))
                      (then
                        ;; Found! Update the value in place
                        (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))) (local.get $set_val_type))
                        (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 8))) (local.get $set_val_flags))
                        (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 12))) (local.get $set_val_lo))
                        (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 16))) (local.get $set_val_hi))
                        (local.set $found (i32.const 1))
                        (br $idx_prop_set_found)
                      )
                    )

                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                    (br $idx_prop_set_loop)
                  )
                )

                ;; If not found, add new property
                (if (i32.eqz (local.get $found))
                  (then
                    ;; Check if we need to grow entries block
                    (if (i32.ge_u (local.get $obj_count) (local.get $obj_cap))
                      (then
                        ;; Need to reallocate entries block
                        ;; new_cap = capacity * 2 (or at least 4)
                        (local.set $new_cap (i32.mul (local.get $obj_cap) (i32.const 2)))
                        (if (i32.lt_u (local.get $new_cap) (i32.const 4))
                          (then (local.set $new_cap (i32.const 4)))
                        )

                        ;; Allocate new entries block
                        (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
                        (local.set $new_entries_ptr (local.get $heap_ptr))
                        (local.set $new_entries_size (i32.add (global.get $GC_HEADER_SIZE)
                                                             (i32.mul (local.get $new_cap) (i32.const 20))))

                        ;; GC header for new entries block
                        (i32.store (call $abs (local.get $new_entries_ptr))
                          (i32.or (local.get $new_entries_size) (i32.shl (global.get $OBJ_OBJECT_DATA) (i32.const 24))))
                        (i32.store (call $abs (i32.add (local.get $new_entries_ptr) (i32.const 4))) (i32.const 0))

                        ;; Update heap pointer
                        (call $write_state (global.get $STATE_HEAP_POINTER)
                          (i32.add (local.get $new_entries_ptr) (local.get $new_entries_size)))

                        ;; Copy existing entries from old block
                        (local.set $i (i32.const 0))
                        (block $idx_copy_entries_done
                          (loop $idx_copy_entries_loop
                            (br_if $idx_copy_entries_done (i32.ge_u (local.get $i) (local.get $obj_count)))
                            ;; src = old_entries_ptr + 8 + i * 20
                            (local.set $elem_ptr (i32.add
                              (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                              (i32.mul (local.get $i) (i32.const 20))))
                            ;; dest = new_entries_ptr + 8 + i * 20
                            (local.set $dest (i32.add
                              (i32.add (local.get $new_entries_ptr) (global.get $GC_HEADER_SIZE))
                              (i32.mul (local.get $i) (i32.const 20))))
                            ;; Copy 20 bytes (entry size)
                            (i32.store (call $abs (local.get $dest))
                              (i32.load (call $abs (local.get $elem_ptr))))
                            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4)))
                              (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8)))
                              (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12)))
                              (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))
                            (i32.store (call $abs (i32.add (local.get $dest) (i32.const 16)))
                              (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 16)))))
                            (local.set $i (i32.add (local.get $i) (i32.const 1)))
                            (br $idx_copy_entries_loop)
                          )
                        )

                        ;; Update object header with new entries_ptr and capacity
                        (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 12))) (local.get $new_cap))
                        (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 16))) (local.get $new_entries_ptr))
                        (local.set $entries_ptr (local.get $new_entries_ptr))
                      )
                    )

                    ;; Add new entry at count position
                    ;; entry_ptr = entries_ptr + 8 + count * 20
                    (local.set $entry_ptr (i32.add
                      (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $obj_count) (i32.const 20))))

                    ;; Write entry: [key][type][flags][data_lo][data_hi]
                    (i32.store (call $abs (local.get $entry_ptr)) (local.get $key_offset))
                    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))) (local.get $set_val_type))
                    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 8))) (local.get $set_val_flags))
                    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 12))) (local.get $set_val_lo))
                    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 16))) (local.get $set_val_hi))

                    ;; Increment count
                    (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))
                      (i32.add (local.get $obj_count) (i32.const 1)))
                  )
                )

                ;; Push value back (assignment is an expression)
                (call $pending_push
                  (local.get $set_val_type)
                  (local.get $set_val_flags)
                  (local.get $set_val_lo)
                  (local.get $set_val_hi))

                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Array with numeric index - original logic
            (local.set $index (i32.trunc_f64_s (call $read_f64 (local.get $val_ptr2))))
            (local.set $arr_ptr (call $value_data_lo (local.get $val_ptr)))

            ;; Check if array
            (if (i32.ne (local.get $type1) (global.get $TYPE_ARRAY))
              (then
                (call $set_error (global.get $ERR_INVALID_OPERAND) (local.get $type1))
                (br $exit)
              )
            )

            ;; Check for negative index
            (if (i32.lt_s (local.get $index) (i32.const 0))
              (then
                (call $set_error (global.get $ERR_INVALID_OPERAND) (local.get $index))
                (br $exit)
              )
            )

            ;; Get array length, capacity, data_ptr
            (local.set $arr_len (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 8)))))
            (local.set $arr_cap (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 12)))))
            (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 16)))))

            ;; Check if we need to grow the array
            (if (i32.ge_u (local.get $index) (local.get $arr_cap))
              (then
                ;; Need to reallocate data block
                ;; new_cap = max(index + 1, capacity * 2)
                (local.set $new_cap (i32.add (local.get $index) (i32.const 1)))
                (if (i32.gt_u (i32.mul (local.get $arr_cap) (i32.const 2)) (local.get $new_cap))
                  (then
                    (local.set $new_cap (i32.mul (local.get $arr_cap) (i32.const 2)))
                  )
                )

                ;; Allocate new data block
                (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
                (local.set $new_data_ptr (local.get $heap_ptr))
                (local.set $new_data_size (i32.add (global.get $GC_HEADER_SIZE)
                                                   (i32.mul (local.get $new_cap) (global.get $VALUE_SIZE))))

                ;; GC header for new data block
                (i32.store (call $abs (local.get $new_data_ptr))
                  (i32.or (local.get $new_data_size) (i32.shl (global.get $OBJ_ARRAY_DATA) (i32.const 24))))
                (i32.store (call $abs (i32.add (local.get $new_data_ptr) (i32.const 4))) (i32.const 0))

                ;; Update heap pointer
                (call $write_state (global.get $STATE_HEAP_POINTER)
                  (i32.add (local.get $new_data_ptr) (local.get $new_data_size)))

                ;; Copy existing elements from old data block
                (local.set $i (i32.const 0))
                (block $copy_done
                  (loop $copy_loop
                    (br_if $copy_done (i32.ge_u (local.get $i) (local.get $arr_len)))
                    ;; src = old_data_ptr + 8 + i * 16
                    (local.set $elem_ptr (i32.add
                      (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                    ;; dest = new_data_ptr + 8 + i * 16
                    (local.set $dest (i32.add
                      (i32.add (local.get $new_data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                    ;; Copy 16 bytes
                    (i32.store (call $abs (local.get $dest))
                      (i32.load (call $abs (local.get $elem_ptr))))
                    (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                    (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                    (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12)))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))
                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                    (br $copy_loop)
                  )
                )

                ;; Fill gap with undefined if index > arr_len
                (local.set $i (local.get $arr_len))
                (block $fill_done
                  (loop $fill_loop
                    (br_if $fill_done (i32.ge_u (local.get $i) (local.get $index)))
                    ;; dest = new_data_ptr + 8 + i * 16
                    (local.set $dest (i32.add
                      (i32.add (local.get $new_data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                    ;; Write undefined
                    (i32.store (call $abs (local.get $dest)) (global.get $TYPE_UNDEFINED))
                    (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4))) (i32.const 0))
                    (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8))) (i32.const 0))
                    (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12))) (i32.const 0))
                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                    (br $fill_loop)
                  )
                )

                ;; Update array header with new data_ptr and capacity
                (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 12))) (local.get $new_cap))
                (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 16))) (local.get $new_data_ptr))
                (local.set $data_ptr (local.get $new_data_ptr))
              )
              (else
                ;; Within capacity, but may need to fill gap if index > length
                (if (i32.gt_u (local.get $index) (local.get $arr_len))
                  (then
                    (local.set $i (local.get $arr_len))
                    (block $fill_done2
                      (loop $fill_loop2
                        (br_if $fill_done2 (i32.ge_u (local.get $i) (local.get $index)))
                        (local.set $dest (i32.add
                          (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                          (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                        (i32.store (call $abs (local.get $dest)) (global.get $TYPE_UNDEFINED))
                        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4))) (i32.const 0))
                        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8))) (i32.const 0))
                        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12))) (i32.const 0))
                        (local.set $i (i32.add (local.get $i) (i32.const 1)))
                        (br $fill_loop2)
                      )
                    )
                  )
                )
              )
            )

            ;; Write value at index
            (local.set $elem_ptr (i32.add
              (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
              (i32.mul (local.get $index) (global.get $VALUE_SIZE))))
            (i32.store (call $abs (local.get $elem_ptr)) (local.get $set_val_type))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (local.get $set_val_flags))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (local.get $set_val_lo))
            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (local.get $set_val_hi))

            ;; Update length if index >= arr_len
            (if (i32.ge_u (local.get $index) (local.get $arr_len))
              (then
                (i32.store (call $abs (i32.add (local.get $arr_ptr) (i32.const 8)))
                  (i32.add (local.get $index) (i32.const 1)))
              )
            )

            ;; Push value back (assignment is an expression)
            (call $pending_push
              (local.get $set_val_type)
              (local.get $set_val_flags)
              (local.get $set_val_lo)
              (local.get $set_val_hi))

            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; GET_PROP: operand1 = property name (string offset), pop object, push value
        (if (i32.eq (local.get $opcode) (global.get $OP_GET_PROP))
          (then
            (local.set $key_offset (local.get $operand1))

            ;; Pop object
            (local.set $val_ptr (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr)))
            (local.set $obj_ptr (call $value_data_lo (local.get $val_ptr)))

            ;; Check for .length and methods on arrays
            (if (i32.eq (local.get $type1) (global.get $TYPE_ARRAY))
              (then
                ;; Check if property is "length"
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_LENGTH)))
                  (then
                    ;; Array layout: [GC:8][length:4][capacity:4][data_ptr:4]
                    ;; obj_ptr = arr_ptr (start of allocation including GC header)
                    ;; Length is at obj_ptr + 8 (after 8-byte GC header)
                    (local.set $arr_len (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))))

                    ;; Push length as float (JS semantics - all numbers are f64)
                    (call $push_f64 (f64.convert_i32_u (local.get $arr_len)))

                    (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )

                ;; Check for array methods - return TYPE_BOUND_METHOD
                ;; Format: type=BOUND_METHOD, flags=receiver_type, data_lo=receiver_ptr, data_hi=method_id
                (local.set $method_id (i32.const 0))

                ;; push
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_PUSH)))
                  (then (local.set $method_id (global.get $METHOD_PUSH))))
                ;; pop
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_POP)))
                  (then (local.set $method_id (global.get $METHOD_POP))))
                ;; shift
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_SHIFT)))
                  (then (local.set $method_id (global.get $METHOD_SHIFT))))
                ;; unshift
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_UNSHIFT)))
                  (then (local.set $method_id (global.get $METHOD_UNSHIFT))))
                ;; slice
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_SLICE)))
                  (then (local.set $method_id (global.get $METHOD_SLICE))))
                ;; concat
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_CONCAT)))
                  (then (local.set $method_id (global.get $METHOD_CONCAT))))
                ;; join
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_JOIN)))
                  (then (local.set $method_id (global.get $METHOD_JOIN))))
                ;; reverse
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_REVERSE)))
                  (then (local.set $method_id (global.get $METHOD_REVERSE))))
                ;; indexOf
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_INDEX_OF)))
                  (then (local.set $method_id (global.get $METHOD_INDEX_OF))))
                ;; includes
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_INCLUDES)))
                  (then (local.set $method_id (global.get $METHOD_INCLUDES))))
                ;; map (callback method)
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_MAP)))
                  (then (local.set $method_id (global.get $METHOD_MAP))))
                ;; filter (callback method)
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_FILTER)))
                  (then (local.set $method_id (global.get $METHOD_FILTER))))
                ;; reduce (callback method)
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_REDUCE)))
                  (then (local.set $method_id (global.get $METHOD_REDUCE))))
                ;; forEach (callback method)
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_FOR_EACH)))
                  (then (local.set $method_id (global.get $METHOD_FOR_EACH))))
                ;; find (callback method)
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_FIND)))
                  (then (local.set $method_id (global.get $METHOD_FIND))))
                ;; findIndex (callback method)
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_FIND_INDEX)))
                  (then (local.set $method_id (global.get $METHOD_FIND_INDEX))))
                ;; some (callback method)
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_SOME)))
                  (then (local.set $method_id (global.get $METHOD_SOME))))
                ;; every (callback method)
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_EVERY)))
                  (then (local.set $method_id (global.get $METHOD_EVERY))))

                ;; If we found a method, push TYPE_BOUND_METHOD
                (if (i32.ne (local.get $method_id) (i32.const 0))
                  (then
                    (call $pending_push
                      (global.get $TYPE_BOUND_METHOD)
                      (global.get $TYPE_ARRAY)           ;; flags = receiver type
                      (local.get $obj_ptr)               ;; data_lo = receiver pointer
                      (local.get $method_id))            ;; data_hi = method ID
                    (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )

                ;; Not a known property - return undefined
                (call $pending_push
                  (global.get $TYPE_UNDEFINED)
                  (i32.const 0)
                  (i32.const 0)
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Check for .length and methods on strings
            (if (i32.eq (local.get $type1) (global.get $TYPE_STRING))
              (then
                ;; Check if property is "length"
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_LENGTH)))
                  (then
                    ;; String table entry: [byte_len:4][bytes:N]
                    ;; obj_ptr is the string table offset
                    ;; Count UTF-8 characters (not bytes) to match JS behavior
                    (local.set $arr_len (call $utf8_char_count (local.get $obj_ptr)))

                    ;; Push length as float
                    (call $push_f64 (f64.convert_i32_u (local.get $arr_len)))

                    (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )

                ;; Check for string methods - return TYPE_BOUND_METHOD
                (local.set $method_id (i32.const 0))

                ;; charAt
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_CHAR_AT)))
                  (then (local.set $method_id (global.get $METHOD_CHAR_AT))))
                ;; charCodeAt
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_CHAR_CODE_AT)))
                  (then (local.set $method_id (global.get $METHOD_CHAR_CODE_AT))))
                ;; indexOf (string version)
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_STRING_INDEX_OF)))
                  (then (local.set $method_id (global.get $METHOD_STRING_INDEX_OF))))
                ;; includes (string version)
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_STRING_INCLUDES)))
                  (then (local.set $method_id (global.get $METHOD_STRING_INCLUDES))))
                ;; slice (string version)
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_STRING_SLICE)))
                  (then (local.set $method_id (global.get $METHOD_STRING_SLICE))))
                ;; substring
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_SUBSTRING)))
                  (then (local.set $method_id (global.get $METHOD_SUBSTRING))))
                ;; split
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_SPLIT)))
                  (then (local.set $method_id (global.get $METHOD_SPLIT))))
                ;; trim
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_TRIM)))
                  (then (local.set $method_id (global.get $METHOD_TRIM))))
                ;; toLowerCase
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_TO_LOWER_CASE)))
                  (then (local.set $method_id (global.get $METHOD_TO_LOWER_CASE))))
                ;; toUpperCase
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_TO_UPPER_CASE)))
                  (then (local.set $method_id (global.get $METHOD_TO_UPPER_CASE))))
                ;; startsWith
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_STARTS_WITH)))
                  (then (local.set $method_id (global.get $METHOD_STARTS_WITH))))
                ;; endsWith
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_ENDS_WITH)))
                  (then (local.set $method_id (global.get $METHOD_ENDS_WITH))))
                ;; repeat
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_REPEAT)))
                  (then (local.set $method_id (global.get $METHOD_REPEAT))))
                ;; padStart
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_PAD_START)))
                  (then (local.set $method_id (global.get $METHOD_PAD_START))))
                ;; padEnd
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_PAD_END)))
                  (then (local.set $method_id (global.get $METHOD_PAD_END))))
                ;; replace
                (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_REPLACE)))
                  (then (local.set $method_id (global.get $METHOD_REPLACE))))

                ;; If we found a method, push TYPE_BOUND_METHOD
                (if (i32.ne (local.get $method_id) (i32.const 0))
                  (then
                    (call $pending_push
                      (global.get $TYPE_BOUND_METHOD)
                      (global.get $TYPE_STRING)          ;; flags = receiver type
                      (local.get $obj_ptr)               ;; data_lo = receiver pointer (string table offset)
                      (local.get $method_id))            ;; data_hi = method ID
                    (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )

                ;; Not a known property - return undefined
                (call $pending_push
                  (global.get $TYPE_UNDEFINED)
                  (i32.const 0)
                  (i32.const 0)
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Check if TYPE_CONSTRUCTOR (Array, Object, String, Number)
            ;; These have an underlying object with static methods
            ;; Layout: [type=CONSTRUCTOR][padding][object_ptr][method_id]
            (if (i32.eq (local.get $type1) (global.get $TYPE_CONSTRUCTOR))
              (then
                ;; obj_ptr already has data_lo which is the underlying object pointer
                ;; Do the same object property lookup as TYPE_OBJECT
                (local.set $obj_count (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))))
                (local.set $entries_ptr (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 16)))))

                ;; Linear search for property
                (local.set $i (i32.const 0))
                (local.set $found (i32.const 0))
                (block $ctor_prop_found
                  (loop $ctor_prop_loop
                    (br_if $ctor_prop_found (i32.ge_u (local.get $i) (local.get $obj_count)))

                    ;; entry_ptr = entries_ptr + 8 (GC header) + i * 20
                    (local.set $entry_ptr (i32.add
                      (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (i32.const 20))))

                    (local.set $prop_key (i32.load (call $abs (local.get $entry_ptr))))

                    (if (i32.eq (local.get $prop_key) (local.get $key_offset))
                      (then
                        ;; Found! Push the value
                        (call $pending_push
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))))   ;; type
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 8))))   ;; flags
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 12))))  ;; data_lo
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 16))))) ;; data_hi
                        (local.set $found (i32.const 1))
                        (br $ctor_prop_found)
                      )
                    )

                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                    (br $ctor_prop_loop)
                  )
                )

                ;; If not found, push undefined
                (if (i32.eqz (local.get $found))
                  (then
                    (call $pending_push
                      (global.get $TYPE_UNDEFINED)
                      (i32.const 0)
                      (i32.const 0)
                      (i32.const 0))
                  )
                )

                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Check if object
            (if (i32.eq (local.get $type1) (global.get $TYPE_OBJECT))
              (then
                ;; New object layout: [GC:8][count:4][capacity:4][entries_ptr:4]
                ;; Entry: [key:4][type:4][flags:4][data_lo:4][data_hi:4]
                (local.set $obj_count (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))))
                (local.set $entries_ptr (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 16)))))

                ;; Linear search for property
                (local.set $i (i32.const 0))
                (local.set $found (i32.const 0))
                (block $prop_found
                  (loop $prop_loop
                    (br_if $prop_found (i32.ge_u (local.get $i) (local.get $obj_count)))

                    ;; entry_ptr = entries_ptr + 8 (GC header) + i * 20
                    (local.set $entry_ptr (i32.add
                      (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $i) (i32.const 20))))

                    (local.set $prop_key (i32.load (call $abs (local.get $entry_ptr))))

                    (if (i32.eq (local.get $prop_key) (local.get $key_offset))
                      (then
                        ;; Found! Push the value
                        (call $pending_push
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))))   ;; type
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 8))))   ;; flags
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 12))))  ;; data_lo
                          (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 16))))) ;; data_hi
                        (local.set $found (i32.const 1))
                        (br $prop_found)
                      )
                    )

                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                    (br $prop_loop)
                  )
                )

                ;; If not found, push undefined
                (if (i32.eqz (local.get $found))
                  (then
                    (call $pending_push
                      (global.get $TYPE_UNDEFINED)
                      (i32.const 0)
                      (i32.const 0)
                      (i32.const 0))
                  )
                )

                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Check if TYPE_FFI_REF (opaque FFI handle)
            ;; When accessing a property on an FFI ref, return TYPE_FFI_METHOD
            ;; Format: [type=FFI_METHOD][flags=0][refId][methodNameOffset]
            (if (i32.eq (local.get $type1) (global.get $TYPE_FFI_REF))
              (then
                ;; obj_ptr contains the refId (stored in data_lo)
                ;; key_offset is the method name (string table offset)
                ;; Push TYPE_FFI_METHOD with refId and method name
                (call $pending_push
                  (global.get $TYPE_FFI_METHOD)
                  (i32.const 0)                    ;; flags
                  (local.get $obj_ptr)             ;; data_lo = refId
                  (local.get $key_offset))         ;; data_hi = method name offset
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Check if TYPE_MSGPACK_REF (read-only msgpack data)
            (if (i32.eq (local.get $type1) (global.get $TYPE_MSGPACK_REF))
              (then
                ;; obj_ptr = msgpack address (absolute), data_hi = length
                (local.set $mp_length (call $value_data_hi (local.get $val_ptr)))
                (local.set $mp_addr (local.get $obj_ptr))
                (local.set $mp_end (i32.add (local.get $mp_addr) (local.get $mp_length)))

                ;; Peek the msgpack type
                (local.set $mp_type (call $msgpack_peek_type (local.get $mp_addr) (local.get $mp_end)))
                (if (i32.eq (local.get $mp_type) (global.get $MSGPACK_ERROR))
                  (then
                    (call $set_error (global.get $ERR_MSGPACK_INVALID) (i32.const 10))
                    (br $exit)
                  )
                )

                ;; If it's an array and property is "length", return array length
                (if (i32.eq (local.get $mp_type) (global.get $MSGPACK_ARRAY))
                  (then
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_LENGTH)))
                      (then
                        ;; Count is in scratch from peek_type
                        (local.set $mp_count (call $msgpack_scratch_count))
                        (call $push_f64 (f64.convert_i32_u (local.get $mp_count)))
                        (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                        (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                        (br $loop)
                      )
                    )

                    ;; Check for array methods - return TYPE_BOUND_METHOD
                    (local.set $method_id (i32.const 0))

                    ;; map
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_MAP)))
                      (then (local.set $method_id (global.get $METHOD_MAP))))
                    ;; filter
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_FILTER)))
                      (then (local.set $method_id (global.get $METHOD_FILTER))))
                    ;; forEach
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_FOR_EACH)))
                      (then (local.set $method_id (global.get $METHOD_FOR_EACH))))
                    ;; find
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_FIND)))
                      (then (local.set $method_id (global.get $METHOD_FIND))))
                    ;; findIndex
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_FIND_INDEX)))
                      (then (local.set $method_id (global.get $METHOD_FIND_INDEX))))
                    ;; some
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_SOME)))
                      (then (local.set $method_id (global.get $METHOD_SOME))))
                    ;; every
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_EVERY)))
                      (then (local.set $method_id (global.get $METHOD_EVERY))))
                    ;; reduce
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_REDUCE)))
                      (then (local.set $method_id (global.get $METHOD_REDUCE))))
                    ;; indexOf
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_INDEX_OF)))
                      (then (local.set $method_id (global.get $METHOD_INDEX_OF))))
                    ;; includes
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_INCLUDES)))
                      (then (local.set $method_id (global.get $METHOD_INCLUDES))))
                    ;; slice
                    (if (i32.eq (local.get $key_offset) (call $read_builtin (global.get $BUILTIN_SLICE)))
                      (then (local.set $method_id (global.get $METHOD_SLICE))))

                    ;; If we found a method, push TYPE_BOUND_METHOD
                    (if (i32.ne (local.get $method_id) (i32.const 0))
                      (then
                        ;; Encode: flags = TYPE_MSGPACK_REF | (mp_length << 8)
                        ;; data_lo = mp_addr, data_hi = method_id
                        (call $pending_push
                          (global.get $TYPE_BOUND_METHOD)
                          (i32.or
                            (global.get $TYPE_MSGPACK_REF)
                            (i32.shl (local.get $mp_length) (i32.const 8)))
                          (local.get $mp_addr)
                          (local.get $method_id))
                        (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                        (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                        (br $loop)
                      )
                    )

                    ;; Array doesn't support other properties - return undefined
                    (call $pending_push
                      (global.get $TYPE_UNDEFINED)
                      (i32.const 0)
                      (i32.const 0)
                      (i32.const 0))
                    (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )

                ;; If it's a map, search for the key
                (if (i32.eq (local.get $mp_type) (global.get $MSGPACK_MAP))
                  (then
                    ;; Call msgpack_map_get to find the key
                    (local.set $found
                      (call $msgpack_map_get
                        (local.get $mp_addr)
                        (local.get $mp_end)
                        (local.get $key_offset)))

                    (if (i32.eq (local.get $found) (i32.const -1))
                      (then
                        ;; Key not found - return undefined
                        (call $pending_push
                          (global.get $TYPE_UNDEFINED)
                          (i32.const 0)
                          (i32.const 0)
                          (i32.const 0))
                        (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                        (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                        (br $loop)
                      )
                    )

                    ;; found = address of value in msgpack
                    ;; Peek the value type
                    (local.set $mp_type (call $msgpack_peek_type (local.get $found) (local.get $mp_end)))
                    (if (i32.eq (local.get $mp_type) (global.get $MSGPACK_ERROR))
                      (then
                        (call $set_error (global.get $ERR_MSGPACK_INVALID) (i32.const 11))
                        (br $exit)
                      )
                    )

                    ;; For primitives, read directly
                    (if (i32.or
                          (i32.or
                            (i32.eq (local.get $mp_type) (global.get $MSGPACK_NIL))
                            (i32.eq (local.get $mp_type) (global.get $MSGPACK_BOOL)))
                          (i32.or
                            (i32.or
                              (i32.eq (local.get $mp_type) (global.get $MSGPACK_INT))
                              (i32.eq (local.get $mp_type) (global.get $MSGPACK_FLOAT)))
                            (i32.eq (local.get $mp_type) (global.get $MSGPACK_STRING))))
                      (then
                        (if (i32.eq (call $msgpack_read_primitive (local.get $found) (local.get $mp_end))
                                    (i32.const -1))
                          (then
                            (call $set_error (global.get $ERR_MSGPACK_INVALID) (i32.const 12))
                            (br $exit)
                          )
                        )
                        (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                        (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                        (br $loop)
                      )
                    )

                    ;; For containers (array/map), return new TYPE_MSGPACK_REF
                    ;; Calculate length from found to mp_end (conservative - actual value may be shorter)
                    (call $pending_push
                      (global.get $TYPE_MSGPACK_REF)
                      (i32.const 0)
                      (local.get $found)
                      (i32.sub (local.get $mp_end) (local.get $found)))
                    (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )

                ;; Not a map or array - can't access property, return undefined
                (call $pending_push
                  (global.get $TYPE_UNDEFINED)
                  (i32.const 0)
                  (i32.const 0)
                  (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Not an object/array/string/ffi_ref/msgpack_ref - push undefined (match JS behavior)
            (call $pending_push
              (global.get $TYPE_UNDEFINED)
              (i32.const 0)
              (i32.const 0)
              (i32.const 0))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; SET_PROP: operand1 = property name (string offset), pop value, pop object
        ;; Set obj.prop = value, push value back
        (if (i32.eq (local.get $opcode) (global.get $OP_SET_PROP))
          (then
            (local.set $key_offset (local.get $operand1))

            ;; Pop value to store
            (local.set $val_ptr (call $pending_pop))
            (local.set $set_val_type (i32.load (call $abs (local.get $val_ptr))))
            (local.set $set_val_flags (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
            (local.set $set_val_lo (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 8)))))
            (local.set $set_val_hi (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))))

            ;; Pop object
            (local.set $val_ptr2 (call $pending_pop))
            (local.set $type1 (call $value_type (local.get $val_ptr2)))
            (local.set $obj_ptr (call $value_data_lo (local.get $val_ptr2)))

            ;; Check if msgpack ref (read-only, cannot assign)
            (if (i32.eq (local.get $type1) (global.get $TYPE_MSGPACK_REF))
              (then
                (call $set_error (global.get $ERR_MSGPACK_READONLY) (local.get $type1))
                (br $exit)
              )
            )

            ;; Check if object
            (if (i32.ne (local.get $type1) (global.get $TYPE_OBJECT))
              (then
                (call $set_error (global.get $ERR_PROPERTY_NULL) (local.get $type1))
                (br $exit)
              )
            )

            ;; Object layout: [GC:8][count:4][capacity:4][entries_ptr:4]
            (local.set $obj_count (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))))
            (local.set $obj_cap (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 12)))))
            (local.set $entries_ptr (i32.load (call $abs (i32.add (local.get $obj_ptr) (i32.const 16)))))

            ;; Search for existing property
            (local.set $i (i32.const 0))
            (local.set $found (i32.const 0))
            (block $prop_set_found
              (loop $prop_set_loop
                (br_if $prop_set_found (i32.ge_u (local.get $i) (local.get $obj_count)))

                ;; entry_ptr = entries_ptr + 8 + i * 20
                (local.set $entry_ptr (i32.add
                  (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $i) (i32.const 20))))

                (local.set $prop_key (i32.load (call $abs (local.get $entry_ptr))))

                (if (i32.eq (local.get $prop_key) (local.get $key_offset))
                  (then
                    ;; Found! Update the value in place
                    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))) (local.get $set_val_type))
                    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 8))) (local.get $set_val_flags))
                    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 12))) (local.get $set_val_lo))
                    (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 16))) (local.get $set_val_hi))
                    (local.set $found (i32.const 1))
                    (br $prop_set_found)
                  )
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $prop_set_loop)
              )
            )

            ;; If not found, add new property
            (if (i32.eqz (local.get $found))
              (then
                ;; Check if we need to grow entries block
                (if (i32.ge_u (local.get $obj_count) (local.get $obj_cap))
                  (then
                    ;; Need to reallocate entries block
                    ;; new_cap = capacity * 2 (or at least 4)
                    (local.set $new_cap (i32.mul (local.get $obj_cap) (i32.const 2)))
                    (if (i32.lt_u (local.get $new_cap) (i32.const 4))
                      (then (local.set $new_cap (i32.const 4)))
                    )

                    ;; Allocate new entries block
                    (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
                    (local.set $new_entries_ptr (local.get $heap_ptr))
                    (local.set $new_entries_size (i32.add (global.get $GC_HEADER_SIZE)
                                                         (i32.mul (local.get $new_cap) (i32.const 20))))

                    ;; GC header for new entries block
                    (i32.store (call $abs (local.get $new_entries_ptr))
                      (i32.or (local.get $new_entries_size) (i32.shl (global.get $OBJ_OBJECT_DATA) (i32.const 24))))
                    (i32.store (call $abs (i32.add (local.get $new_entries_ptr) (i32.const 4))) (i32.const 0))

                    ;; Update heap pointer
                    (call $write_state (global.get $STATE_HEAP_POINTER)
                      (i32.add (local.get $new_entries_ptr) (local.get $new_entries_size)))

                    ;; Copy existing entries from old block
                    (local.set $i (i32.const 0))
                    (block $copy_entries_done
                      (loop $copy_entries_loop
                        (br_if $copy_entries_done (i32.ge_u (local.get $i) (local.get $obj_count)))
                        ;; src = old_entries_ptr + 8 + i * 20
                        (local.set $elem_ptr (i32.add
                          (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                          (i32.mul (local.get $i) (i32.const 20))))
                        ;; dest = new_entries_ptr + 8 + i * 20
                        (local.set $dest (i32.add
                          (i32.add (local.get $new_entries_ptr) (global.get $GC_HEADER_SIZE))
                          (i32.mul (local.get $i) (i32.const 20))))
                        ;; Copy 20 bytes (entry size)
                        (i32.store (call $abs (local.get $dest))
                          (i32.load (call $abs (local.get $elem_ptr))))
                        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4)))
                          (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4)))))
                        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8)))
                          (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 8)))))
                        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12)))
                          (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 12)))))
                        (i32.store (call $abs (i32.add (local.get $dest) (i32.const 16)))
                          (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 16)))))
                        (local.set $i (i32.add (local.get $i) (i32.const 1)))
                        (br $copy_entries_loop)
                      )
                    )

                    ;; Update object header with new entries_ptr and capacity
                    (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 12))) (local.get $new_cap))
                    (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 16))) (local.get $new_entries_ptr))
                    (local.set $entries_ptr (local.get $new_entries_ptr))
                  )
                )

                ;; Add new entry at count position
                ;; entry_ptr = entries_ptr + 8 + count * 20
                (local.set $entry_ptr (i32.add
                  (i32.add (local.get $entries_ptr) (global.get $GC_HEADER_SIZE))
                  (i32.mul (local.get $obj_count) (i32.const 20))))

                ;; Write entry: [key][type][flags][data_lo][data_hi]
                (i32.store (call $abs (local.get $entry_ptr)) (local.get $key_offset))
                (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))) (local.get $set_val_type))
                (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 8))) (local.get $set_val_flags))
                (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 12))) (local.get $set_val_lo))
                (i32.store (call $abs (i32.add (local.get $entry_ptr) (i32.const 16))) (local.get $set_val_hi))

                ;; Increment count
                (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 8)))
                  (i32.add (local.get $obj_count) (i32.const 1)))
              )
            )

            ;; Push value back (assignment is an expression)
            (call $pending_push
              (local.get $set_val_type)
              (local.get $set_val_flags)
              (local.get $set_val_lo)
              (local.get $set_val_hi))

            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; =====================================================================
        ;; SCOPE
        ;; =====================================================================
        (if (i32.eq (local.get $opcode) (global.get $OP_SCOPE_PUSH))
          (then
            (local.set $new_scope (call $scope_create (global.get $current_scope)))
            (global.set $current_scope (local.get $new_scope))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        (if (i32.eq (local.get $opcode) (global.get $OP_SCOPE_POP))
          (then
            ;; Restore parent scope
            (global.set $current_scope
              (i32.load (call $abs (global.get $current_scope))))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; =====================================================================
        ;; TRY/CATCH/FINALLY
        ;; =====================================================================

        ;; TRY_PUSH: operand1 = catch_index (0 if none), operand2 = finally_index (0 if none)
        ;; Push try entry to track exception handlers
        (if (i32.eq (local.get $opcode) (global.get $OP_TRY_PUSH))
          (then
            (call $try_push (local.get $code_block) (local.get $operand1) (local.get $operand2))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; TRY_POP: Normal exit from try block
        ;; Pop try entry, set completion=normal, jump to finally if present
        (if (i32.eq (local.get $opcode) (global.get $OP_TRY_POP))
          (then
            ;; Read finally_index before popping
            (local.set $operand2 (call $try_read (i32.const 0) (global.get $TRY_FINALLY_INDEX)))

            ;; Pop try entry
            (call $try_pop)

            ;; Set completion type to normal
            (call $write_state (global.get $STATE_COMPLETION_TYPE) (global.get $COMPLETION_NORMAL))

            ;; If finally_index != 0, jump to finally
            (if (i32.ne (local.get $operand2) (i32.const 0))
              (then
                (local.set $pc (local.get $operand2))
              )
              (else
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
              )
            )
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; THROW: Pop exception value, search for handler
        ;; If handler found: unwind frames, jump to catch/finally
        ;; If not found: set STATUS_ERROR
        (if (i32.eq (local.get $opcode) (global.get $OP_THROW))
          (then
            ;; Pop exception value
            (local.set $val_ptr (call $pending_pop))

            ;; Push it back so save_completion_value can read it
            (call $pending_push
              (call $value_type (local.get $val_ptr))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4))))
              (call $value_data_lo (local.get $val_ptr))
              (call $value_data_hi (local.get $val_ptr)))

            ;; Save exception to completion value slot
            (call $save_completion_value)

            ;; Pop it again (we saved it)
            (drop (call $pending_pop))

            ;; Set completion type to throw
            (call $write_state (global.get $STATE_COMPLETION_TYPE) (global.get $COMPLETION_THROW))

            ;; Search try stack for handler
            ;; We need locals for the search
            (local.set $i (i32.const 0))  ;; depth in try stack
            (local.set $found (i32.const 0))

            (block $handler_found
              (block $no_handler
                (loop $search_loop
                  ;; If we've searched all entries, no handler
                  (br_if $no_handler (i32.ge_u (local.get $i) (call $try_depth)))

                  ;; Read entry fields (depth 0 = top)
                  (local.set $operand1 (call $try_read (local.get $i) (global.get $TRY_CATCH_INDEX)))
                  (local.set $operand2 (call $try_read (local.get $i) (global.get $TRY_FINALLY_INDEX)))
                  (local.set $index (call $try_read (local.get $i) (global.get $TRY_FRAME_DEPTH)))

                  ;; Pop this try entry (consumed before entering handler)
                  (call $try_pop)

                  ;; Check for catch first (if present, exception is handled)
                  ;; After catch completes, it falls through to finally naturally
                  (if (i32.ne (local.get $operand1) (i32.const 0))
                    (then
                      ;; Has catch - unwind frames and jump
                      (call $unwind_frames_to (local.get $index))

                      ;; Push exception value to pending stack for catch to bind
                      (call $restore_completion_value)

                      ;; Clear completion (caught)
                      (call $write_state (global.get $STATE_COMPLETION_TYPE) (global.get $COMPLETION_NORMAL))

                      (local.set $pc (local.get $operand1))
                      (local.set $found (i32.const 1))
                      (br $handler_found)
                    )
                  )

                  ;; No catch - check for finally (will re-throw after running)
                  (if (i32.ne (local.get $operand2) (i32.const 0))
                    (then
                      ;; Has finally - jump to it (will re-throw via FINALLY_END)
                      (call $unwind_frames_to (local.get $index))
                      (local.set $pc (local.get $operand2))
                      (local.set $found (i32.const 1))
                      (br $handler_found)
                    )
                  )

                  ;; No handler in this entry, try next
                  ;; Note: we already popped, so don't increment i (stack shrunk)
                  (br $search_loop)
                )
              )

              ;; No handler found - uncaught exception
              (call $set_error (global.get $ERR_USER_THROW) (i32.const 0))
              (br $exit)
            )

            ;; Handler found, continue execution
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; FINALLY_END: End of finally block, dispatch based on completion type
        ;; If normal: continue to next instruction
        ;; If throw: re-throw (search for next handler)
        ;; If return: complete the return
        (if (i32.eq (local.get $opcode) (global.get $OP_FINALLY_END))
          (then
            (local.set $type1 (call $read_state (global.get $STATE_COMPLETION_TYPE)))

            ;; Normal completion - continue
            (if (i32.eq (local.get $type1) (global.get $COMPLETION_NORMAL))
              (then
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Throw completion - re-throw
            (if (i32.eq (local.get $type1) (global.get $COMPLETION_THROW))
              (then
                ;; Search try stack for next handler
                (local.set $i (i32.const 0))
                (local.set $found (i32.const 0))

                (block $rethrow_found
                  (block $rethrow_no_handler
                    (loop $rethrow_loop
                      (br_if $rethrow_no_handler (i32.ge_u (local.get $i) (call $try_depth)))

                      (local.set $operand1 (call $try_read (local.get $i) (global.get $TRY_CATCH_INDEX)))
                      (local.set $operand2 (call $try_read (local.get $i) (global.get $TRY_FINALLY_INDEX)))
                      (local.set $index (call $try_read (local.get $i) (global.get $TRY_FRAME_DEPTH)))

                      (call $try_pop)

                      ;; Check finally
                      (if (i32.ne (local.get $operand2) (i32.const 0))
                        (then
                          (call $unwind_frames_to (local.get $index))
                          (local.set $pc (local.get $operand2))
                          (local.set $found (i32.const 1))
                          (br $rethrow_found)
                        )
                      )

                      ;; Check catch
                      (if (i32.ne (local.get $operand1) (i32.const 0))
                        (then
                          (call $unwind_frames_to (local.get $index))
                          (call $restore_completion_value)
                          (call $write_state (global.get $STATE_COMPLETION_TYPE) (global.get $COMPLETION_NORMAL))
                          (local.set $pc (local.get $operand1))
                          (local.set $found (i32.const 1))
                          (br $rethrow_found)
                        )
                      )

                      (br $rethrow_loop)
                    )
                  )

                  ;; No handler - uncaught
                  (call $set_error (global.get $ERR_USER_THROW) (i32.const 0))
                  (br $exit)
                )

                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Return completion - complete the return
            (if (i32.eq (local.get $type1) (global.get $COMPLETION_RETURN))
              (then
                ;; Push completion value as return value
                (call $restore_completion_value)

                ;; Clear completion type
                (call $write_state (global.get $STATE_COMPLETION_TYPE) (global.get $COMPLETION_NORMAL))

                ;; Now execute return logic (same as OP_RETURN)
                ;; Pop result from pending stack
                (local.set $val_ptr (call $pending_pop))
                (local.set $result_type (call $value_type (local.get $val_ptr)))
                (local.set $result_flags (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
                (local.set $result_lo (call $value_data_lo (local.get $val_ptr)))
                (local.set $result_hi (call $value_data_hi (local.get $val_ptr)))

                ;; Pop call frame
                (local.set $stack_ptr (call $read_state (global.get $STATE_STACK_POINTER)))

                ;; Check if we have a frame to pop
                (if (i32.le_u (local.get $stack_ptr)
                              (call $read_state (global.get $STATE_STACK_BASE)))
                  (then
                    ;; No frame - top-level return
                    (call $pending_push (local.get $result_type) (local.get $result_flags) (local.get $result_lo) (local.get $result_hi))
                    (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_DONE))
                    (br $exit)
                  )
                )

                ;; Pop frame
                (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
                (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))
                (local.set $frame_ptr (call $abs (local.get $stack_ptr)))

                ;; Restore state
                (local.set $code_block (i32.load (local.get $frame_ptr)))
                (local.set $return_pc (i32.load (i32.add (local.get $frame_ptr) (i32.const 4))))
                (local.set $caller_scope (i32.load (i32.add (local.get $frame_ptr) (i32.const 8))))

                (call $write_state (global.get $STATE_PENDING_POINTER)
                  (i32.load (i32.add (local.get $frame_ptr) (i32.const 12))))

                (global.set $current_scope (local.get $caller_scope))

                (call $pending_push (local.get $result_type) (local.get $result_flags) (local.get $result_lo) (local.get $result_hi))

                (local.set $instr_count (call $read_instr_count (local.get $code_block)))
                (local.set $pc (local.get $return_pc))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Unknown completion type - error
            (call $set_error (global.get $ERR_INVALID_OPERAND) (local.get $type1))
            (br $exit)
          )
        )

        ;; =====================================================================
        ;; FUNCTIONS
        ;; =====================================================================

        ;; MAKE_CLOSURE: operand1 = start_instr, operand2 = end_instr
        ;; Create closure capturing current scope, push to pending
        ;; Closure layout (after GC header): [start_instr:4][end_instr:4][scope:4][flags:4]
        ;; flags = 0 for regular closure (binds `this`)
        (if (i32.eq (local.get $opcode) (global.get $OP_MAKE_CLOSURE))
          (then
            ;; Allocate closure: GC header (8) + data (16) = 24 bytes
            (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))

            ;; GC header: [size | (type << 24)]
            (i32.store (call $abs (local.get $heap_ptr))
              (i32.or (i32.const 24) (i32.shl (global.get $OBJ_CLOSURE) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $heap_ptr) (i32.const 4))) (i32.const 0))

            ;; Closure data (after GC header at +8)
            (i32.store (call $abs (i32.add (local.get $heap_ptr) (i32.const 8))) (local.get $operand1))  ;; start_instr
            (i32.store (call $abs (i32.add (local.get $heap_ptr) (i32.const 12))) (local.get $operand2)) ;; end_instr
            (i32.store (call $abs (i32.add (local.get $heap_ptr) (i32.const 16))) (global.get $current_scope)) ;; scope
            (i32.store (call $abs (i32.add (local.get $heap_ptr) (i32.const 20))) (i32.const 0)) ;; flags = 0 (regular)

            ;; Update heap pointer
            (call $write_state (global.get $STATE_HEAP_POINTER)
              (i32.add (local.get $heap_ptr) (i32.const 24)))

            ;; Push closure value (type=CLOSURE, data_lo=ptr to data after GC header)
            (call $pending_push
              (global.get $TYPE_CLOSURE)
              (i32.const 0)
              (i32.add (local.get $heap_ptr) (global.get $GC_HEADER_SIZE))
              (i32.const 0))

            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; MAKE_ARROW_CLOSURE: operand1 = start_instr, operand2 = end_instr
        ;; Same as MAKE_CLOSURE but sets CLOSURE_FLAG_ARROW (doesn't bind `this`)
        (if (i32.eq (local.get $opcode) (global.get $OP_MAKE_ARROW_CLOSURE))
          (then
            ;; Allocate closure: GC header (8) + data (16) = 24 bytes
            (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))

            ;; GC header: [size | (type << 24)]
            (i32.store (call $abs (local.get $heap_ptr))
              (i32.or (i32.const 24) (i32.shl (global.get $OBJ_CLOSURE) (i32.const 24))))
            (i32.store (call $abs (i32.add (local.get $heap_ptr) (i32.const 4))) (i32.const 0))

            ;; Closure data (after GC header at +8)
            (i32.store (call $abs (i32.add (local.get $heap_ptr) (i32.const 8))) (local.get $operand1))  ;; start_instr
            (i32.store (call $abs (i32.add (local.get $heap_ptr) (i32.const 12))) (local.get $operand2)) ;; end_instr
            (i32.store (call $abs (i32.add (local.get $heap_ptr) (i32.const 16))) (global.get $current_scope)) ;; scope
            (i32.store (call $abs (i32.add (local.get $heap_ptr) (i32.const 20))) (global.get $CLOSURE_FLAG_ARROW)) ;; flags = ARROW

            ;; Update heap pointer
            (call $write_state (global.get $STATE_HEAP_POINTER)
              (i32.add (local.get $heap_ptr) (i32.const 24)))

            ;; Push closure value (type=CLOSURE, data_lo=ptr to data after GC header)
            (call $pending_push
              (global.get $TYPE_CLOSURE)
              (i32.const 0)
              (i32.add (local.get $heap_ptr) (global.get $GC_HEADER_SIZE))
              (i32.const 0))

            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; CALL: operand1 = argc
        ;; Stack before: [closure, arg0, arg1, ..., argN-1] with argN-1 on top
        ;; The closure is argc values below the args.
        ;;
        ;; We read the closure from below the args, save frame state,
        ;; then leave args on stack for function body to consume via LET_VAR.
        ;; On RETURN, pending_ptr is restored to where closure was (discarding closure).
        ;;
        ;; Frame layout: [CODE_BLOCK:4][INSTRUCTION_INDEX:4][SCOPE:4][PENDING_BASE:4]
        ;;               [PENDING_COUNT:4][FLAGS:4][SOURCE_START:4][SOURCE_END:4]
        (if (i32.eq (local.get $opcode) (global.get $OP_CALL))
          (then
            ;; Calculate where the closure is on the pending stack
            ;; closure_position = pending_ptr - (argc + 1) * VALUE_SIZE
            (local.set $pending_ptr (call $read_state (global.get $STATE_PENDING_POINTER)))
            (local.set $val_ptr
              (i32.sub
                (local.get $pending_ptr)
                (i32.mul
                  (i32.add (local.get $operand1) (i32.const 1))
                  (global.get $VALUE_SIZE))))

            (local.set $type1 (call $value_type (local.get $val_ptr)))

            ;; Handle TYPE_BOUND_METHOD (global functions like isNaN, isFinite)
            (if (i32.eq (local.get $type1) (global.get $TYPE_BOUND_METHOD))
              (then
                ;; Extract receiver info from bound method
                ;; Format: [type:i32][flags:i32][receiver_ptr:i32][method_id:i32]
                ;; For arrays/strings: flags = receiver_type, receiver_ptr = pointer
                ;; For msgpack: flags = TYPE_MSGPACK_REF | (length << 8), receiver_ptr = msgpack addr
                ;; For globals: flags = 0, receiver_ptr = 0
                (local.set $receiver_type (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
                (local.set $receiver_ptr (call $value_data_lo (local.get $val_ptr)))
                (local.set $method_id (call $value_data_hi (local.get $val_ptr)))

                ;; Dispatch to builtin method
                (call $dispatch_builtin_method
                  (local.get $receiver_type)
                  (local.get $receiver_ptr)
                  (local.get $method_id)
                  (local.get $operand1)   ;; argc
                  (local.get $val_ptr))   ;; stack_base (where function was)

                ;; Check if method set an error
                (if (i32.eq (call $read_state (global.get $STATE_STATUS)) (global.get $STATUS_ERROR))
                  (then (br $exit)))

                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Handle TYPE_CONSTRUCTOR (String(), Number(), etc. called without new)
            ;; These act as conversion functions
            (if (i32.eq (local.get $type1) (global.get $TYPE_CONSTRUCTOR))
              (then
                ;; Extract method_id from constructor
                ;; Format: [type:i32][flags:i32][object_ptr:i32][method_id:i32]
                (local.set $method_id (call $value_data_hi (local.get $val_ptr)))

                ;; Dispatch conversion based on method_id
                (call $dispatch_builtin_method
                  (global.get $TYPE_CONSTRUCTOR)
                  (i32.const 0)
                  (local.get $method_id)
                  (local.get $operand1)   ;; argc
                  (local.get $val_ptr))   ;; stack_base

                ;; Check if method set an error
                (if (i32.eq (call $read_state (global.get $STATE_STATUS)) (global.get $STATUS_ERROR))
                  (then (br $exit)))

                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Handle TYPE_FFI_METHOD - yield to orchestrator
            ;; Format: [type=FFI_METHOD][flags=0][refId][methodNameOffset]
            ;; FFI_REQUEST layout: [refId:4][methodOffset:4][argsPointer:4][argCount:4]
            (if (i32.eq (local.get $type1) (global.get $TYPE_FFI_METHOD))
              (then
                ;; Write FFI request
                ;; refId = data_lo
                (i32.store (call $abs (global.get $ffi_request_base))
                  (call $value_data_lo (local.get $val_ptr)))
                ;; methodOffset = data_hi
                (i32.store (call $abs (i32.add (global.get $ffi_request_base) (i32.const 4)))
                  (call $value_data_hi (local.get $val_ptr)))
                ;; argsPointer = val_ptr + VALUE_SIZE (first arg is right after the method)
                (i32.store (call $abs (i32.add (global.get $ffi_request_base) (i32.const 8)))
                  (i32.add (local.get $val_ptr) (global.get $VALUE_SIZE)))
                ;; argCount = operand1
                (i32.store (call $abs (i32.add (global.get $ffi_request_base) (i32.const 12)))
                  (local.get $operand1))

                ;; Save state for when we resume (PC after this instruction)
                (call $write_state (global.get $STATE_INSTRUCTION_INDEX)
                  (i32.add (local.get $pc) (i32.const 1)))
                (call $write_state (global.get $STATE_FUEL) (local.get $fuel))

                ;; Set status to FFI_REQUEST and exit
                (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_FFI_REQUEST))
                (br $exit)
              )
            )

            ;; Check it's a closure
            (if (i32.ne (local.get $type1) (global.get $TYPE_CLOSURE))
              (then
                (call $set_error (global.get $ERR_NOT_CALLABLE) (local.get $type1))
                (br $exit)
              )
            )

            ;; Get closure data pointer
            (local.set $closure_ptr (call $value_data_lo (local.get $val_ptr)))

            ;; Read closure fields
            (local.set $start_instr (i32.load (call $abs (local.get $closure_ptr))))
            ;; end_instr at closure_ptr + 4 (not used during call, only for bounds checking)
            (local.set $captured_scope (i32.load (call $abs (i32.add (local.get $closure_ptr) (i32.const 8)))))
            ;; flags at closure_ptr + 12
            (local.set $closure_flags (i32.load (call $abs (i32.add (local.get $closure_ptr) (i32.const 12)))))

            ;; Push call frame
            (local.set $stack_ptr (call $read_state (global.get $STATE_STACK_POINTER)))
            (local.set $frame_ptr (call $abs (local.get $stack_ptr)))

            ;; PENDING_BASE = where the closure was (so on return, closure slot is reclaimed)
            ;; Frame saves the position of the closure, not current pending_ptr
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 0)) (local.get $code_block))          ;; CODE_BLOCK
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 4)) (i32.add (local.get $pc) (i32.const 1))) ;; return INSTRUCTION_INDEX
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 8)) (global.get $current_scope))      ;; caller's SCOPE
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 12)) (local.get $val_ptr))            ;; PENDING_BASE = closure position
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 16)) (i32.const 1))                   ;; PENDING_COUNT (expect 1 return value)
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 20)) (i32.const 0))                   ;; FLAGS
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 24)) (i32.const 0))                   ;; SOURCE_START
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 28)) (i32.const 0))                   ;; SOURCE_END

            ;; Update stack pointer
            (call $write_state (global.get $STATE_STACK_POINTER)
              (i32.add (local.get $stack_ptr) (global.get $FRAME_SIZE)))

            ;; Create new scope with captured scope as parent
            (local.set $new_scope (call $scope_create (local.get $captured_scope)))
            (global.set $current_scope (local.get $new_scope))

            ;; For non-arrow functions, bind `this` to undefined
            ;; (Arrow functions inherit `this` from their lexical scope)
            (if (i32.eqz (i32.and (local.get $closure_flags) (global.get $CLOSURE_FLAG_ARROW)))
              (then
                ;; Read the interned "this" string offset from builtins
                (local.set $key_offset (call $read_builtin (i32.const 0x18))) ;; BUILTIN_NAME.THIS

                ;; Define `this` = undefined in the new scope
                (call $scope_define
                  (local.get $new_scope)
                  (local.get $key_offset)
                  (global.get $TYPE_UNDEFINED)
                  (i32.const 0)  ;; flags
                  (i32.const 0)  ;; data_lo
                  (i32.const 0)) ;; data_hi
              )
            )

            ;; Jump to function body
            ;; Args are still on pending stack for LET_VAR to pop
            (local.set $pc (local.get $start_instr))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; CALL_METHOD: operand1 = argc
        ;; Stack: [receiver, closure/bound_method, arg1, arg2, ..., argN]
        ;; For closures: binds `this` to receiver for non-arrow functions
        ;; For bound methods: dispatches to built-in method implementation
        (if (i32.eq (local.get $opcode) (global.get $OP_CALL_METHOD))
          (then
            ;; Calculate where the closure/method is on the pending stack
            ;; method_position = pending_ptr - (argc + 1) * VALUE_SIZE
            ;; receiver_position = method_position - VALUE_SIZE
            (local.set $pending_ptr (call $read_state (global.get $STATE_PENDING_POINTER)))
            (local.set $val_ptr
              (i32.sub
                (local.get $pending_ptr)
                (i32.mul
                  (i32.add (local.get $operand1) (i32.const 1))
                  (global.get $VALUE_SIZE))))

            ;; receiver is just before closure/method
            (local.set $val_ptr2
              (i32.sub (local.get $val_ptr) (global.get $VALUE_SIZE)))

            (local.set $type1 (call $value_type (local.get $val_ptr)))

            ;; Check if it's a TYPE_BOUND_METHOD
            (if (i32.eq (local.get $type1) (global.get $TYPE_BOUND_METHOD))
              (then
                ;; Extract bound method info
                ;; flags = receiver type (TYPE_ARRAY or TYPE_STRING)
                ;; data_lo = receiver pointer
                ;; data_hi = method ID
                (local.set $receiver_type (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
                (local.set $receiver_ptr (call $value_data_lo (local.get $val_ptr)))
                (local.set $method_id (call $value_data_hi (local.get $val_ptr)))

                ;; Check for callback methods - these need special handling
                (if (call $is_callback_method (local.get $method_id))
                  (then
                    ;; Callback methods work on arrays and msgpack arrays
                    ;; receiver_type for msgpack has TYPE_MSGPACK_REF in low byte, length in upper bytes
                    (if (i32.and
                          (i32.ne (local.get $receiver_type) (global.get $TYPE_ARRAY))
                          (i32.ne (i32.and (local.get $receiver_type) (i32.const 0xff)) (global.get $TYPE_MSGPACK_REF)))
                      (then
                        (call $set_error (global.get $ERR_TYPE_ERROR) (local.get $receiver_type))
                        (br $exit)
                      )
                    )

                    ;; If it's a msgpack array, materialize to heap array first
                    (if (i32.eq (i32.and (local.get $receiver_type) (i32.const 0xff)) (global.get $TYPE_MSGPACK_REF))
                      (then
                        ;; Extract msgpack addr (receiver_ptr) and length (upper bits of receiver_type)
                        (local.set $mp_addr (local.get $receiver_ptr))
                        (local.set $mp_length (i32.shr_u (local.get $receiver_type) (i32.const 8)))
                        (local.set $mp_end (i32.add (local.get $mp_addr) (local.get $mp_length)))

                        ;; Materialize to heap array
                        (local.set $receiver_ptr (call $msgpack_to_array (local.get $mp_addr) (local.get $mp_end)))
                        (if (i32.eqz (local.get $receiver_ptr))
                          (then
                            ;; Error already set by msgpack_to_array
                            (br $exit)
                          )
                        )

                        ;; Now treat as regular array
                        (local.set $receiver_type (global.get $TYPE_ARRAY))
                      )
                    )

                    ;; Need at least 1 argument (the callback)
                    (if (i32.lt_u (local.get $operand1) (i32.const 1))
                      (then
                        (call $set_error (global.get $ERR_ARITY) (i32.const 1))
                        (br $exit)
                      )
                    )

                    ;; Get callback from first arg position
                    ;; arg1 is at val_ptr + VALUE_SIZE (after bound method)
                    (local.set $closure_ptr (i32.add (local.get $val_ptr) (global.get $VALUE_SIZE)))

                    ;; Verify callback is a closure
                    (if (i32.ne (call $value_type (local.get $closure_ptr)) (global.get $TYPE_CLOSURE))
                      (then
                        (call $set_error (global.get $ERR_NOT_CALLABLE) (call $value_type (local.get $closure_ptr)))
                        (br $exit)
                      )
                    )

                    ;; Get array length
                    (local.set $arr_ptr (local.get $receiver_ptr))
                    (local.set $arr_len (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 8)))))

                    ;; Handle empty array - return appropriate default
                    (if (i32.eqz (local.get $arr_len))
                      (then
                        ;; Clean up stack
                        (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $val_ptr2))

                        ;; Return default value based on method
                        (if (i32.or
                              (i32.eq (local.get $method_id) (global.get $METHOD_MAP))
                              (i32.eq (local.get $method_id) (global.get $METHOD_FILTER)))
                          (then
                            ;; Return empty array
                            (local.set $heap_ptr (call $allocate_array (i32.const 0)))
                            (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $heap_ptr) (i32.const 0))
                          )
                          (else
                            (if (i32.eq (local.get $method_id) (global.get $METHOD_REDUCE))
                              (then
                                ;; reduce on empty array without initial value is TypeError
                                ;; With initial value, return initial (check argc >= 2)
                                (if (i32.ge_u (local.get $operand1) (i32.const 2))
                                  (then
                                    ;; Return second argument (initial value)
                                    (local.set $entry_ptr (i32.add (local.get $closure_ptr) (global.get $VALUE_SIZE)))
                                    (call $pending_push
                                      (call $value_type (local.get $entry_ptr))
                                      (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))))
                                      (call $value_data_lo (local.get $entry_ptr))
                                      (call $value_data_hi (local.get $entry_ptr)))
                                  )
                                  (else
                                    ;; reduce on empty with no initial is error
                                    (call $set_error (global.get $ERR_TYPE_ERROR) (i32.const 0))
                                    (br $exit)
                                  )
                                )
                              )
                              (else
                                (if (i32.eq (local.get $method_id) (global.get $METHOD_FOR_EACH))
                                  (then
                                    ;; forEach returns undefined
                                    (call $pending_push (global.get $TYPE_UNDEFINED) (i32.const 0) (i32.const 0) (i32.const 0))
                                  )
                                  (else
                                    (if (i32.or
                                          (i32.eq (local.get $method_id) (global.get $METHOD_FIND))
                                          (i32.eq (local.get $method_id) (global.get $METHOD_SOME)))
                                      (then
                                        ;; find returns undefined, some returns false
                                        (if (i32.eq (local.get $method_id) (global.get $METHOD_FIND))
                                          (then (call $pending_push (global.get $TYPE_UNDEFINED) (i32.const 0) (i32.const 0) (i32.const 0)))
                                          (else (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0)))
                                        )
                                      )
                                      (else
                                        (if (i32.eq (local.get $method_id) (global.get $METHOD_FIND_INDEX))
                                          (then
                                            ;; findIndex returns -1 (as float for consistency)
                                            (call $push_f64 (f64.const -1))
                                          )
                                          (else
                                            ;; every returns true on empty
                                            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))
                                          )
                                        )
                                      )
                                    )
                                  )
                                )
                              )
                            )
                          )
                        )
                        (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                        (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                        (br $loop)
                      )
                    )

                    ;; Array is non-empty - set up iteration
                    ;; Create result array for map/filter (capacity = arr_len)
                    (if (i32.or
                          (i32.eq (local.get $method_id) (global.get $METHOD_MAP))
                          (i32.eq (local.get $method_id) (global.get $METHOD_FILTER)))
                      (then
                        ;; For map, result will be same length; for filter, could be less
                        (local.set $new_data_ptr (call $allocate_array (local.get $arr_len)))
                        ;; For filter, start with length 0; for map, we'll fill it
                        (if (i32.eq (local.get $method_id) (global.get $METHOD_FILTER))
                          (then
                            ;; Set length to 0 for filter (we'll grow it)
                            (i32.store (call $abs (i32.add (local.get $new_data_ptr) (i32.const 8))) (i32.const 0))
                          )
                        )
                        (local.set $result_lo (local.get $new_data_ptr))
                      )
                      (else
                        (if (i32.eq (local.get $method_id) (global.get $METHOD_REDUCE))
                          (then
                            ;; Get initial value from arg2, or use arr[0] if no initial
                            (if (i32.ge_u (local.get $operand1) (i32.const 2))
                              (then
                                ;; Initial value provided
                                (local.set $entry_ptr (i32.add (local.get $closure_ptr) (global.get $VALUE_SIZE)))
                                (local.set $result_type (call $value_type (local.get $entry_ptr)))
                                (local.set $result_lo (call $value_data_lo (local.get $entry_ptr)))
                                (local.set $result_hi (call $value_data_hi (local.get $entry_ptr)))
                                ;; Store accumulator as a value on heap for iteration
                                ;; We'll use a single-element array to hold the accumulator
                                (local.set $new_data_ptr (call $allocate_array (i32.const 1)))
                                ;; Get data pointer of this array (offset 16 = data block including GC header)
                                (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $new_data_ptr) (i32.const 16)))))
                                ;; Write initial value (skip GC header)
                                (call $write_value (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                                  (local.get $result_type)
                                  (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 4))))
                                  (local.get $result_lo)
                                  (local.get $result_hi))
                                (local.set $result_lo (local.get $new_data_ptr))
                                ;; Start from index 0
                                (local.set $index (i32.const 0))
                              )
                              (else
                                ;; No initial value - use arr[0] as initial, start from index 1
                                ;; Get arr[0] element pointer (data_ptr + GC_HEADER_SIZE)
                                (local.set $data_ptr (i32.add
                                  (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 16))))
                                  (global.get $GC_HEADER_SIZE)))
                                (local.set $new_data_ptr (call $allocate_array (i32.const 1)))
                                ;; Get data pointer for accumulator array (offset 16 + GC_HEADER_SIZE)
                                (local.set $dest (i32.add
                                  (i32.load (call $abs (i32.add (local.get $new_data_ptr) (i32.const 16))))
                                  (global.get $GC_HEADER_SIZE)))
                                ;; Copy arr[0] to accumulator
                                (i32.store (call $abs (local.get $dest)) (i32.load (call $abs (local.get $data_ptr))))
                                (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4))) (i32.load (call $abs (i32.add (local.get $data_ptr) (i32.const 4)))))
                                (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8))) (i32.load (call $abs (i32.add (local.get $data_ptr) (i32.const 8)))))
                                (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12))) (i32.load (call $abs (i32.add (local.get $data_ptr) (i32.const 12)))))
                                (local.set $result_lo (local.get $new_data_ptr))
                                ;; Start from index 1
                                (local.set $index (i32.const 1))
                                ;; If array has only 1 element, return it
                                (if (i32.eq (local.get $arr_len) (i32.const 1))
                                  (then
                                    (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $val_ptr2))
                                    ;; data_ptr already points to element (not including GC header)
                                    (call $pending_push
                                      (call $value_type (local.get $data_ptr))
                                      (i32.load (call $abs (i32.add (local.get $data_ptr) (i32.const 4))))
                                      (call $value_data_lo (local.get $data_ptr))
                                      (call $value_data_hi (local.get $data_ptr)))
                                    (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                                    (br $loop)
                                  )
                                )
                              )
                            )
                          )
                          (else
                            ;; forEach, find, findIndex, some, every - no result array needed
                            (local.set $result_lo (i32.const 0))
                            (local.set $index (i32.const 0))
                          )
                        )
                      )
                    )

                    ;; Set default start index for non-reduce methods
                    (if (i32.ne (local.get $method_id) (global.get $METHOD_REDUCE))
                      (then (local.set $index (i32.const 0)))
                    )

                    ;; Push iteration frame
                    (local.set $stack_ptr (call $read_state (global.get $STATE_STACK_POINTER)))
                    (local.set $frame_ptr (call $abs (local.get $stack_ptr)))

                    ;; Save callback closure pointer for iteration
                    (local.set $start_instr (call $value_data_lo (local.get $closure_ptr)))

                    ;; Basic frame fields
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_CODE_BLOCK)) (local.get $code_block))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_INSTRUCTION_INDEX)) (i32.add (local.get $pc) (i32.const 1)))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_SCOPE_POINTER)) (global.get $current_scope))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_PENDING_BASE)) (local.get $val_ptr2))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_PENDING_COUNT)) (i32.const 1))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_FLAGS)) (global.get $FRAME_FLAG_ITERATING))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_SOURCE_START)) (i32.const 0))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_SOURCE_END)) (i32.const 0))

                    ;; Iteration fields
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_INDEX)) (local.get $index))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_LENGTH)) (local.get $arr_len))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_RESULT)) (local.get $result_lo))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_METHOD)) (local.get $method_id))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_RECEIVER)) (local.get $arr_ptr))
                    (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_CALLBACK)) (local.get $start_instr))

                    ;; Update stack pointer
                    (call $write_state (global.get $STATE_STACK_POINTER)
                      (i32.add (local.get $stack_ptr) (global.get $FRAME_SIZE)))

                    ;; Clean pending stack - we'll push callback args
                    (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $val_ptr2))

                    ;; Now call the callback with first element
                    ;; Get callback closure details
                    (local.set $captured_scope (i32.load (call $abs (i32.add (local.get $start_instr) (i32.const 8)))))
                    (local.set $closure_flags (i32.load (call $abs (i32.add (local.get $start_instr) (i32.const 12)))))

                    ;; Get element at index
                    ;; data_ptr points to start of data block (includes GC header), add 8 to skip
                    (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 16)))))
                    (local.set $elem_ptr (i32.add
                      (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $index) (global.get $VALUE_SIZE))))

                    ;; Push callback arguments in REVERSE order (LIFO stack)
                    ;; Callback signature: fn(element, index, array) for non-reduce
                    ;;                     fn(acc, element, index, array) for reduce
                    ;; LET_VAR pops from top, so push in reverse: array, index, element [, acc]

                    ;; Push array reference (will be popped last)
                    (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $arr_ptr) (i32.const 0))

                    ;; Push index (as float for arithmetic compatibility)
                    (call $push_f64 (f64.convert_i32_s (local.get $index)))

                    ;; Push element
                    (call $pending_push
                      (call $value_type (local.get $elem_ptr))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))))
                      (call $value_data_lo (local.get $elem_ptr))
                      (call $value_data_hi (local.get $elem_ptr)))

                    ;; For reduce, push accumulator last (will be popped first)
                    (if (i32.eq (local.get $method_id) (global.get $METHOD_REDUCE))
                      (then
                        ;; result_lo is accumulator array, data_ptr at offset 16 (includes GC header)
                        (local.set $dest (i32.add
                          (i32.load (call $abs (i32.add (local.get $result_lo) (i32.const 16))))
                          (global.get $GC_HEADER_SIZE)))
                        (call $pending_push
                          (call $value_type (local.get $dest))
                          (i32.load (call $abs (i32.add (local.get $dest) (i32.const 4))))
                          (call $value_data_lo (local.get $dest))
                          (call $value_data_hi (local.get $dest)))
                      )
                    )

                    ;; Create new scope for callback with captured scope as parent
                    (local.set $new_scope (call $scope_create (local.get $captured_scope)))
                    (global.set $current_scope (local.get $new_scope))

                    ;; Jump to callback - read start instruction index from closure data
                    ;; Code block stays the same (closures use same code block)
                    (local.set $pc (i32.load (call $abs (local.get $start_instr))))

                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )

                ;; Dispatch to non-callback method implementation
                (call $dispatch_builtin_method
                  (local.get $receiver_type)
                  (local.get $receiver_ptr)
                  (local.get $method_id)
                  (local.get $operand1)          ;; argc
                  (local.get $val_ptr2))         ;; receiver position on stack (for cleanup)

                ;; Check if method set an error
                (if (i32.eq (call $read_state (global.get $STATE_STATUS)) (global.get $STATUS_ERROR))
                  (then (br $exit)))

                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; Handle TYPE_FFI_METHOD - yield to orchestrator
            ;; Stack: [receiver, ffi_method, arg1, ..., argN]
            ;; We need to set up FFI request and yield
            (if (i32.eq (local.get $type1) (global.get $TYPE_FFI_METHOD))
              (then
                ;; Write FFI request
                ;; refId = data_lo
                (i32.store (call $abs (global.get $ffi_request_base))
                  (call $value_data_lo (local.get $val_ptr)))
                ;; methodOffset = data_hi
                (i32.store (call $abs (i32.add (global.get $ffi_request_base) (i32.const 4)))
                  (call $value_data_hi (local.get $val_ptr)))
                ;; argsPointer = val_ptr + VALUE_SIZE (first arg is right after method)
                (i32.store (call $abs (i32.add (global.get $ffi_request_base) (i32.const 8)))
                  (i32.add (local.get $val_ptr) (global.get $VALUE_SIZE)))
                ;; argCount = operand1
                (i32.store (call $abs (i32.add (global.get $ffi_request_base) (i32.const 12)))
                  (local.get $operand1))

                ;; Save state for when we resume
                ;; Note: We save pending stack including receiver for cleanup after FFI returns
                (call $write_state (global.get $STATE_INSTRUCTION_INDEX)
                  (i32.add (local.get $pc) (i32.const 1)))
                (call $write_state (global.get $STATE_FUEL) (local.get $fuel))
                (call $write_state (global.get $STATE_SCOPE) (global.get $current_scope))

                ;; Set status to FFI_REQUEST and exit
                (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_FFI_REQUEST))
                (br $exit)
              )
            )

            ;; Check it's a closure
            (if (i32.ne (local.get $type1) (global.get $TYPE_CLOSURE))
              (then
                (call $set_error (global.get $ERR_NOT_CALLABLE) (local.get $type1))
                (br $exit)
              )
            )

            ;; Get closure data pointer
            (local.set $closure_ptr (call $value_data_lo (local.get $val_ptr)))

            ;; Read closure fields
            (local.set $start_instr (i32.load (call $abs (local.get $closure_ptr))))
            (local.set $captured_scope (i32.load (call $abs (i32.add (local.get $closure_ptr) (i32.const 8)))))
            ;; flags at closure_ptr + 12
            (local.set $closure_flags (i32.load (call $abs (i32.add (local.get $closure_ptr) (i32.const 12)))))

            ;; Read receiver value (we'll need it to bind `this`)
            (local.set $type2 (call $value_type (local.get $val_ptr2)))
            (local.set $data2_lo (call $value_data_lo (local.get $val_ptr2)))
            (local.set $data2_hi (call $value_data_hi (local.get $val_ptr2)))

            ;; Push call frame
            (local.set $stack_ptr (call $read_state (global.get $STATE_STACK_POINTER)))
            (local.set $frame_ptr (call $abs (local.get $stack_ptr)))

            ;; PENDING_BASE = where the receiver was (so on return, receiver + closure slots are reclaimed)
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 0)) (local.get $code_block))          ;; CODE_BLOCK
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 4)) (i32.add (local.get $pc) (i32.const 1))) ;; return INSTRUCTION_INDEX
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 8)) (global.get $current_scope))      ;; caller's SCOPE
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 12)) (local.get $val_ptr2))           ;; PENDING_BASE = receiver position
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 16)) (i32.const 1))                   ;; PENDING_COUNT (expect 1 return value)
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 20)) (i32.const 0))                   ;; FLAGS
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 24)) (i32.const 0))                   ;; SOURCE_START
            (i32.store (i32.add (local.get $frame_ptr) (i32.const 28)) (i32.const 0))                   ;; SOURCE_END

            ;; Update stack pointer
            (call $write_state (global.get $STATE_STACK_POINTER)
              (i32.add (local.get $stack_ptr) (global.get $FRAME_SIZE)))

            ;; Create new scope with captured scope as parent
            (local.set $new_scope (call $scope_create (local.get $captured_scope)))
            (global.set $current_scope (local.get $new_scope))

            ;; For non-arrow functions, bind `this` to receiver
            (if (i32.eqz (i32.and (local.get $closure_flags) (global.get $CLOSURE_FLAG_ARROW)))
              (then
                ;; Read the interned "this" string offset from builtins
                (local.set $key_offset (call $read_builtin (i32.const 0x18))) ;; BUILTIN_NAME.THIS

                ;; Define `this` in the new scope
                (call $scope_define
                  (local.get $new_scope)
                  (local.get $key_offset)
                  (local.get $type2)
                  (i32.const 0)  ;; flags
                  (local.get $data2_lo)
                  (local.get $data2_hi))
              )
            )

            ;; Remove receiver and closure from pending stack (they're consumed)
            ;; The args need to stay for LET_VAR to pop
            ;; Actually, we set PENDING_BASE to receiver position, so on return
            ;; everything from there will be reclaimed. But we need to move args
            ;; down to fill the gap left by receiver+closure.
            ;;
            ;; Stack currently: [receiver, closure, arg1, arg2, ...]
            ;; We want:         [arg1, arg2, ...]
            ;;
            ;; Move args down by 2 VALUE_SIZEs (32 bytes)
            (if (i32.gt_u (local.get $operand1) (i32.const 0))
              (then
                (local.set $i (i32.const 0))
                (block $move_done
                  (loop $move_loop
                    (br_if $move_done (i32.ge_u (local.get $i) (local.get $operand1)))

                    ;; src = val_ptr + VALUE_SIZE + i * VALUE_SIZE (args start after closure)
                    ;; dst = val_ptr2 + i * VALUE_SIZE (where receiver was)
                    (local.set $dest (i32.add (local.get $val_ptr2)
                      (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                    (local.set $entry_ptr (i32.add
                      (i32.add (local.get $val_ptr) (global.get $VALUE_SIZE))
                      (i32.mul (local.get $i) (global.get $VALUE_SIZE))))

                    ;; Copy 16 bytes
                    (i32.store (call $abs (local.get $dest))
                      (i32.load (call $abs (local.get $entry_ptr))))
                    (i32.store (call $abs (i32.add (local.get $dest) (i32.const 4)))
                      (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 4)))))
                    (i32.store (call $abs (i32.add (local.get $dest) (i32.const 8)))
                      (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 8)))))
                    (i32.store (call $abs (i32.add (local.get $dest) (i32.const 12)))
                      (i32.load (call $abs (i32.add (local.get $entry_ptr) (i32.const 12)))))

                    (local.set $i (i32.add (local.get $i) (i32.const 1)))
                    (br $move_loop)
                  )
                )
              )
            )

            ;; Update pending pointer to account for removed receiver+closure
            (call $write_state (global.get $STATE_PENDING_POINTER)
              (i32.sub (local.get $pending_ptr) (i32.mul (global.get $VALUE_SIZE) (i32.const 2))))

            ;; Jump to function body
            (local.set $pc (local.get $start_instr))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; OP_NEW: new Constructor(args)
        ;; operand1 = argc
        ;; Stack: [constructor, arg1, ..., argN]
        ;; Constructor must be TYPE_CONSTRUCTOR
        (if (i32.eq (local.get $opcode) (global.get $OP_NEW))
          (then
            ;; Calculate constructor position: pending_ptr - (argc + 1) * VALUE_SIZE
            (local.set $pending_ptr (call $read_state (global.get $STATE_PENDING_POINTER)))
            (local.set $val_ptr
              (i32.sub
                (local.get $pending_ptr)
                (i32.mul
                  (i32.add (local.get $operand1) (i32.const 1))
                  (global.get $VALUE_SIZE))))

            (local.set $type1 (call $value_type (local.get $val_ptr)))

            ;; Must be TYPE_CONSTRUCTOR
            (if (i32.ne (local.get $type1) (global.get $TYPE_CONSTRUCTOR))
              (then
                (call $set_error (global.get $ERR_NOT_CALLABLE) (local.get $type1))
                (br $exit)
              )
            )

            ;; TYPE_CONSTRUCTOR layout: [type][padding][object_ptr][method_id]
            ;; data_lo = object_ptr (has static methods)
            ;; data_hi = method_id
            (local.set $receiver_ptr (call $value_data_lo (local.get $val_ptr)))
            (local.set $method_id (call $value_data_hi (local.get $val_ptr)))

            ;; Pop constructor and args, move pending pointer
            (call $write_state (global.get $STATE_PENDING_POINTER) (local.get $val_ptr))

            ;; Dispatch based on method_id
            (if (i32.eq (local.get $method_id) (global.get $METHOD_ARRAY_CONSTRUCTOR))
              (then
                ;; new Array() or new Array(length)
                (if (i32.eqz (local.get $operand1))
                  (then
                    ;; new Array() - create empty array
                    (local.set $heap_ptr (call $allocate_array (i32.const 0)))
                    (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $heap_ptr) (i32.const 0))
                  )
                  (else
                    ;; new Array(length) - first arg is length
                    (local.set $val_ptr2 (i32.add (local.get $val_ptr) (global.get $VALUE_SIZE)))
                    (local.set $type2 (call $value_type (local.get $val_ptr2)))
                    (if (i32.eq (local.get $type2) (global.get $TYPE_FLOAT))
                      (then
                        ;; Get length from float
                        (local.set $result_f64 (f64.load (call $abs (i32.add (local.get $val_ptr2) (i32.const 8)))))
                        (local.set $count (i32.trunc_f64_s (local.get $result_f64)))
                        (local.set $heap_ptr (call $allocate_array (local.get $count)))

                        ;; Initialize array elements to undefined
                        ;; Array layout: [GC:8][length:4][capacity:4][data_ptr:4]
                        (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $heap_ptr) (i32.const 16)))))
                        (local.set $i (i32.const 0))
                        (block $init_done
                          (loop $init_loop
                            (br_if $init_done (i32.ge_u (local.get $i) (local.get $count)))
                            ;; elem_ptr = data_ptr + GC_HEADER + i * VALUE_SIZE
                            (local.set $elem_ptr (i32.add
                              (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                              (i32.mul (local.get $i) (global.get $VALUE_SIZE))))
                            ;; Write undefined: [type=undefined][flags=0][data_lo=0][data_hi=0]
                            (i32.store (call $abs (local.get $elem_ptr)) (global.get $TYPE_UNDEFINED))
                            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))) (i32.const 0))
                            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 8))) (i32.const 0))
                            (i32.store (call $abs (i32.add (local.get $elem_ptr) (i32.const 12))) (i32.const 0))
                            (local.set $i (i32.add (local.get $i) (i32.const 1)))
                            (br $init_loop)
                          )
                        )
                        (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $heap_ptr) (i32.const 0))
                      )
                      (else
                        (call $set_error (global.get $ERR_TYPE_ERROR) (local.get $type2))
                        (br $exit)
                      )
                    )
                  )
                )
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            (if (i32.eq (local.get $method_id) (global.get $METHOD_OBJECT_CONSTRUCTOR))
              (then
                ;; new Object() - create empty object
                (local.set $heap_ptr (call $read_state (global.get $STATE_HEAP_POINTER)))
                (local.set $obj_ptr (local.get $heap_ptr))

                ;; Object header: 20 bytes
                (i32.store (call $abs (local.get $obj_ptr))
                  (i32.or (i32.const 20) (i32.shl (global.get $OBJ_OBJECT) (i32.const 24))))
                (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 4))) (i32.const 0))

                ;; Entries block: 8 + 4 * 20 = 88 bytes
                (local.set $entries_ptr (i32.add (local.get $obj_ptr) (i32.const 20)))
                (i32.store (call $abs (local.get $entries_ptr))
                  (i32.or (i32.const 88) (i32.shl (global.get $OBJ_OBJECT_DATA) (i32.const 24))))
                (i32.store (call $abs (i32.add (local.get $entries_ptr) (i32.const 4))) (i32.const 0))

                ;; Object fields: count=0, capacity=4, entries_ptr
                (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 8))) (i32.const 0))
                (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 12))) (i32.const 4))
                (i32.store (call $abs (i32.add (local.get $obj_ptr) (i32.const 16))) (local.get $entries_ptr))

                ;; Update heap pointer
                (call $write_state (global.get $STATE_HEAP_POINTER)
                  (i32.add (local.get $entries_ptr) (i32.const 88)))

                (call $pending_push (global.get $TYPE_OBJECT) (i32.const 0) (local.get $obj_ptr) (i32.const 0))
                (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; new String() or new Number() - NotSupportedError
            (if (i32.or
                  (i32.eq (local.get $method_id) (global.get $METHOD_TO_STRING))
                  (i32.eq (local.get $method_id) (global.get $METHOD_TO_NUMBER)))
              (then
                (call $set_error (global.get $ERR_NOT_SUPPORTED) (local.get $method_id))
                (br $exit)
              )
            )

            ;; Unknown constructor
            (call $set_error (global.get $ERR_NOT_CALLABLE) (local.get $method_id))
            (br $exit)
          )
        )

        ;; RETURN: pop result, check for finally blocks, then return
        ;; Must check try stack for finally blocks at current frame depth
        (if (i32.eq (local.get $opcode) (global.get $OP_RETURN))
          (then
            ;; Pop result from pending stack
            (local.set $val_ptr (call $pending_pop))
            (local.set $result_type (call $value_type (local.get $val_ptr)))
            (local.set $result_flags (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
            (local.set $result_lo (call $value_data_lo (local.get $val_ptr)))
            (local.set $result_hi (call $value_data_hi (local.get $val_ptr)))

            ;; Get current frame depth
            (local.set $count (call $get_frame_depth))

            ;; Search try stack for entry at current frame depth with finally
            (local.set $i (i32.const 0))
            (local.set $found (i32.const 0))

            (block $finally_check_done
              (loop $finally_check_loop
                (br_if $finally_check_done (i32.ge_u (local.get $i) (call $try_depth)))

                ;; Read entry's frame depth
                (local.set $index (call $try_read (local.get $i) (global.get $TRY_FRAME_DEPTH)))

                ;; Only check entries at current frame depth
                (if (i32.eq (local.get $index) (local.get $count))
                  (then
                    ;; Check for finally
                    (local.set $operand2 (call $try_read (local.get $i) (global.get $TRY_FINALLY_INDEX)))
                    (if (i32.ne (local.get $operand2) (i32.const 0))
                      (then
                        ;; Has finally - save return value and jump to finally
                        ;; Push value back so save_completion_value can read it
                        (call $pending_push (local.get $result_type) (local.get $result_flags)
                                            (local.get $result_lo) (local.get $result_hi))
                        (call $save_completion_value)
                        (drop (call $pending_pop))

                        (call $write_state (global.get $STATE_COMPLETION_TYPE) (global.get $COMPLETION_RETURN))

                        ;; Pop the try entry
                        (call $try_pop)

                        ;; Jump to finally
                        (local.set $pc (local.get $operand2))
                        (local.set $found (i32.const 1))
                        (br $finally_check_done)
                      )
                    )
                  )
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $finally_check_loop)
              )
            )

            ;; If finally was found, continue to that
            (if (local.get $found)
              (then
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; No finally intercepted - normal return

            ;; Get current stack pointer
            (local.set $stack_ptr (call $read_state (global.get $STATE_STACK_POINTER)))

            ;; Check if we have a frame
            (if (i32.le_u (local.get $stack_ptr)
                          (call $read_state (global.get $STATE_STACK_BASE)))
              (then
                ;; No frame - this is a top-level return, just set DONE
                (call $pending_push (local.get $result_type) (local.get $result_flags) (local.get $result_lo) (local.get $result_hi))
                (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_DONE))
                (br $exit)
              )
            )

            ;; Peek at current frame (DON'T pop yet - iteration frames stay until complete)
            (local.set $frame_ptr (call $abs (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE))))

            ;; Check if this frame has FRAME_FLAG_ITERATING set
            ;; This means we returned from a callback and need to continue iteration
            (if (i32.and
                  (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_FLAGS)))
                  (global.get $FRAME_FLAG_ITERATING))
              (then
                ;; ============================================
                ;; ITERATION CONTINUATION
                ;; ============================================
                ;; Read iteration state from current frame
                (local.set $index (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_INDEX))))
                (local.set $arr_len (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_LENGTH))))
                (local.set $heap_ptr (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_RESULT))))
                (local.set $method_id (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_METHOD))))
                (local.set $arr_ptr (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_RECEIVER))))
                (local.set $closure_ptr (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_CALLBACK))))

                ;; Process callback result based on method type
                (if (i32.eq (local.get $method_id) (global.get $METHOD_MAP))
                  (then
                    ;; map: store result at current index in result array
                    ;; heap_ptr is result array: [GC header 8][length 4][capacity 4][data_ptr 4]
                    ;; data_ptr points to start of data block (includes GC header), add 8 to skip it
                    (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $heap_ptr) (i32.const 16)))))
                    (local.set $elem_ptr (i32.add
                      (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $index) (global.get $VALUE_SIZE))))
                    (call $write_value (local.get $elem_ptr)
                      (local.get $result_type) (local.get $result_flags)
                      (local.get $result_lo) (local.get $result_hi))
                  )
                )

                (if (i32.eq (local.get $method_id) (global.get $METHOD_FILTER))
                  (then
                    ;; filter: if result is truthy, append current element to result array
                    (if (call $is_truthy_inline (local.get $result_type) (local.get $result_lo) (local.get $result_hi))
                      (then
                        ;; Get current element from source array
                        ;; data_ptr points to start of data block (includes GC header), add 8 to skip it
                        (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 16)))))
                        (local.set $elem_ptr (i32.add
                          (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                          (i32.mul (local.get $index) (global.get $VALUE_SIZE))))

                        ;; Get result array length
                        (local.set $count (i32.load (call $abs (i32.add (local.get $heap_ptr) (i32.const 8)))))

                        ;; Write element to result array at position count
                        (local.set $new_data_ptr (i32.load (call $abs (i32.add (local.get $heap_ptr) (i32.const 16)))))
                        (local.set $dest (i32.add
                          (i32.add (local.get $new_data_ptr) (global.get $GC_HEADER_SIZE))
                          (i32.mul (local.get $count) (global.get $VALUE_SIZE))))
                        (call $write_value (local.get $dest)
                          (call $value_type (local.get $elem_ptr))
                          (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))))
                          (call $value_data_lo (local.get $elem_ptr))
                          (call $value_data_hi (local.get $elem_ptr)))

                        ;; Increment result array length
                        (i32.store (call $abs (i32.add (local.get $heap_ptr) (i32.const 8)))
                          (i32.add (local.get $count) (i32.const 1)))
                      )
                    )
                  )
                )

                (if (i32.eq (local.get $method_id) (global.get $METHOD_REDUCE))
                  (then
                    ;; reduce: store result as new accumulator
                    ;; heap_ptr is accumulator array, data_ptr at offset 16 (includes GC header)
                    (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $heap_ptr) (i32.const 16)))))
                    (call $write_value (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                      (local.get $result_type) (local.get $result_flags)
                      (local.get $result_lo) (local.get $result_hi))
                  )
                )

                ;; forEach: no result processing needed

                (if (i32.eq (local.get $method_id) (global.get $METHOD_FIND))
                  (then
                    ;; find: if result is truthy, return the element
                    (if (call $is_truthy_inline (local.get $result_type) (local.get $result_lo) (local.get $result_hi))
                      (then
                        ;; Get current element (data_ptr includes GC header, add 8 to skip)
                        (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 16)))))
                        (local.set $elem_ptr (i32.add
                          (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                          (i32.mul (local.get $index) (global.get $VALUE_SIZE))))

                        ;; Pop iteration frame
                        (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
                        (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))

                        ;; Restore from iteration frame
                        (local.set $code_block (i32.load (local.get $frame_ptr)))
                        (local.set $return_pc (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_INSTRUCTION_INDEX))))
                        (local.set $caller_scope (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_SCOPE_POINTER))))
                        (call $write_state (global.get $STATE_PENDING_POINTER)
                          (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_PENDING_BASE))))
                        (global.set $current_scope (local.get $caller_scope))

                        ;; Push found element as result
                        (call $pending_push
                          (call $value_type (local.get $elem_ptr))
                          (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))))
                          (call $value_data_lo (local.get $elem_ptr))
                          (call $value_data_hi (local.get $elem_ptr)))

                        (local.set $instr_count (call $read_instr_count (local.get $code_block)))
                        (local.set $pc (local.get $return_pc))
                        (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                        (br $loop)
                      )
                    )
                  )
                )

                (if (i32.eq (local.get $method_id) (global.get $METHOD_FIND_INDEX))
                  (then
                    ;; findIndex: if result is truthy, return the index
                    (if (call $is_truthy_inline (local.get $result_type) (local.get $result_lo) (local.get $result_hi))
                      (then
                        ;; Pop iteration frame
                        (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
                        (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))

                        ;; Restore from iteration frame
                        (local.set $code_block (i32.load (local.get $frame_ptr)))
                        (local.set $return_pc (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_INSTRUCTION_INDEX))))
                        (local.set $caller_scope (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_SCOPE_POINTER))))
                        (call $write_state (global.get $STATE_PENDING_POINTER)
                          (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_PENDING_BASE))))
                        (global.set $current_scope (local.get $caller_scope))

                        ;; Push index as result (as float for consistency)
                        (call $push_f64 (f64.convert_i32_s (local.get $index)))

                        (local.set $instr_count (call $read_instr_count (local.get $code_block)))
                        (local.set $pc (local.get $return_pc))
                        (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                        (br $loop)
                      )
                    )
                  )
                )

                (if (i32.eq (local.get $method_id) (global.get $METHOD_SOME))
                  (then
                    ;; some: if result is truthy, return true
                    (if (call $is_truthy_inline (local.get $result_type) (local.get $result_lo) (local.get $result_hi))
                      (then
                        ;; Pop iteration frame
                        (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
                        (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))

                        ;; Restore from iteration frame
                        (local.set $code_block (i32.load (local.get $frame_ptr)))
                        (local.set $return_pc (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_INSTRUCTION_INDEX))))
                        (local.set $caller_scope (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_SCOPE_POINTER))))
                        (call $write_state (global.get $STATE_PENDING_POINTER)
                          (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_PENDING_BASE))))
                        (global.set $current_scope (local.get $caller_scope))

                        ;; Push true
                        (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))

                        (local.set $instr_count (call $read_instr_count (local.get $code_block)))
                        (local.set $pc (local.get $return_pc))
                        (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                        (br $loop)
                      )
                    )
                  )
                )

                (if (i32.eq (local.get $method_id) (global.get $METHOD_EVERY))
                  (then
                    ;; every: if result is falsy, return false
                    (if (i32.eqz (call $is_truthy_inline (local.get $result_type) (local.get $result_lo) (local.get $result_hi)))
                      (then
                        ;; Pop iteration frame
                        (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
                        (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))

                        ;; Restore from iteration frame
                        (local.set $code_block (i32.load (local.get $frame_ptr)))
                        (local.set $return_pc (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_INSTRUCTION_INDEX))))
                        (local.set $caller_scope (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_SCOPE_POINTER))))
                        (call $write_state (global.get $STATE_PENDING_POINTER)
                          (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_PENDING_BASE))))
                        (global.set $current_scope (local.get $caller_scope))

                        ;; Push false
                        (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))

                        (local.set $instr_count (call $read_instr_count (local.get $code_block)))
                        (local.set $pc (local.get $return_pc))
                        (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                        (br $loop)
                      )
                    )
                  )
                )

                ;; Increment index
                (local.set $index (i32.add (local.get $index) (i32.const 1)))
                (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_INDEX)) (local.get $index))

                ;; Check if more elements
                (if (i32.lt_u (local.get $index) (local.get $arr_len))
                  (then
                    ;; More elements - call callback again
                    ;; Clear pending stack for new args
                    (call $write_state (global.get $STATE_PENDING_POINTER)
                      (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_PENDING_BASE))))

                    ;; Get callback closure details
                    (local.set $captured_scope (i32.load (call $abs (i32.add (local.get $closure_ptr) (i32.const 8)))))
                    (local.set $closure_flags (i32.load (call $abs (i32.add (local.get $closure_ptr) (i32.const 12)))))

                    ;; Get element at new index
                    ;; data_ptr points to start of data block (includes GC header), add 8 to skip
                    (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 16)))))
                    (local.set $elem_ptr (i32.add
                      (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $index) (global.get $VALUE_SIZE))))

                    ;; Push callback arguments in REVERSE order (LIFO stack)
                    ;; Push array reference (will be popped last)
                    (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $arr_ptr) (i32.const 0))

                    ;; Push index (as float for arithmetic compatibility)
                    (call $push_f64 (f64.convert_i32_s (local.get $index)))

                    ;; Push element
                    (call $pending_push
                      (call $value_type (local.get $elem_ptr))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))))
                      (call $value_data_lo (local.get $elem_ptr))
                      (call $value_data_hi (local.get $elem_ptr)))

                    ;; For reduce, push accumulator last (will be popped first)
                    (if (i32.eq (local.get $method_id) (global.get $METHOD_REDUCE))
                      (then
                        ;; heap_ptr is accumulator array, data_ptr at offset 16 (includes GC header)
                        (local.set $dest (i32.add
                          (i32.load (call $abs (i32.add (local.get $heap_ptr) (i32.const 16))))
                          (global.get $GC_HEADER_SIZE)))
                        (call $pending_push
                          (call $value_type (local.get $dest))
                          (i32.load (call $abs (i32.add (local.get $dest) (i32.const 4))))
                          (call $value_data_lo (local.get $dest))
                          (call $value_data_hi (local.get $dest)))
                      )
                    )

                    ;; Create new scope for callback
                    (local.set $new_scope (call $scope_create (local.get $captured_scope)))
                    (global.set $current_scope (local.get $new_scope))

                    ;; Jump to callback - closure_ptr is closure data pointer
                    ;; Read start_instr from offset 0
                    (local.set $pc (i32.load (call $abs (local.get $closure_ptr))))

                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                  (else
                    ;; Iteration complete - return final result

                    ;; Pop iteration frame
                    (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
                    (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))

                    ;; Restore from iteration frame
                    (local.set $code_block (i32.load (local.get $frame_ptr)))
                    (local.set $return_pc (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_INSTRUCTION_INDEX))))
                    (local.set $caller_scope (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_SCOPE_POINTER))))
                    (call $write_state (global.get $STATE_PENDING_POINTER)
                      (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_PENDING_BASE))))
                    (global.set $current_scope (local.get $caller_scope))

                    ;; Push final result based on method
                    (if (i32.or
                          (i32.eq (local.get $method_id) (global.get $METHOD_MAP))
                          (i32.eq (local.get $method_id) (global.get $METHOD_FILTER)))
                      (then
                        ;; Return result array
                        (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $heap_ptr) (i32.const 0))
                      )
                      (else
                        (if (i32.eq (local.get $method_id) (global.get $METHOD_REDUCE))
                          (then
                            ;; Return accumulator value
                            ;; heap_ptr is accumulator array, data_ptr at offset 16 (includes GC header)
                            (local.set $data_ptr (i32.add
                              (i32.load (call $abs (i32.add (local.get $heap_ptr) (i32.const 16))))
                              (global.get $GC_HEADER_SIZE)))
                            (call $pending_push
                              (call $value_type (local.get $data_ptr))
                              (i32.load (call $abs (i32.add (local.get $data_ptr) (i32.const 4))))
                              (call $value_data_lo (local.get $data_ptr))
                              (call $value_data_hi (local.get $data_ptr)))
                          )
                          (else
                            (if (i32.eq (local.get $method_id) (global.get $METHOD_FOR_EACH))
                              (then
                                ;; forEach returns undefined
                                (call $pending_push (global.get $TYPE_UNDEFINED) (i32.const 0) (i32.const 0) (i32.const 0))
                              )
                              (else
                                (if (i32.eq (local.get $method_id) (global.get $METHOD_FIND))
                                  (then
                                    ;; find exhausted - return undefined
                                    (call $pending_push (global.get $TYPE_UNDEFINED) (i32.const 0) (i32.const 0) (i32.const 0))
                                  )
                                  (else
                                    (if (i32.eq (local.get $method_id) (global.get $METHOD_FIND_INDEX))
                                      (then
                                        ;; findIndex exhausted - return -1 (as float for consistency)
                                        (call $push_f64 (f64.const -1))
                                      )
                                      (else
                                        (if (i32.eq (local.get $method_id) (global.get $METHOD_SOME))
                                          (then
                                            ;; some exhausted - return false
                                            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 0) (i32.const 0))
                                          )
                                          (else
                                            ;; every exhausted - return true
                                            (call $pending_push (global.get $TYPE_BOOLEAN) (i32.const 0) (i32.const 1) (i32.const 0))
                                          )
                                        )
                                      )
                                    )
                                  )
                                )
                              )
                            )
                          )
                        )
                      )
                    )

                    (local.set $instr_count (call $read_instr_count (local.get $code_block)))
                    (local.set $pc (local.get $return_pc))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )
              )
            )

            ;; Normal return (non-iteration)

            ;; Check if this is a callback frame (FRAME_FLAG_CALLBACK)
            ;; If so, yield to orchestrator with STATUS_CALLBACK_DONE
            (if (i32.and
                  (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_FLAGS)))
                  (global.get $FRAME_FLAG_CALLBACK))
              (then
                ;; Pop the callback frame
                (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
                (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))

                ;; Restore instruction index from frame (where to resume after callback)
                (call $write_state (global.get $STATE_INSTRUCTION_INDEX)
                  (i32.load (i32.add (local.get $frame_ptr) (i32.const 4))))

                ;; Restore pending pointer from frame
                (call $write_state (global.get $STATE_PENDING_POINTER)
                  (i32.load (i32.add (local.get $frame_ptr) (i32.const 12))))

                ;; Restore scope from frame (both global and state)
                (local.set $caller_scope (i32.load (i32.add (local.get $frame_ptr) (i32.const 8))))
                (global.set $current_scope (local.get $caller_scope))
                (call $write_state (global.get $STATE_SCOPE) (local.get $caller_scope))

                ;; Push result to pending stack (orchestrator will read it)
                (call $pending_push (local.get $result_type) (local.get $result_flags) (local.get $result_lo) (local.get $result_hi))

                ;; Yield to orchestrator
                (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_CALLBACK_DONE))
                (br $exit)
              )
            )

            ;; Pop the frame now
            (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
            (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))

            ;; Restore state from frame
            (local.set $code_block (i32.load (local.get $frame_ptr)))                           ;; CODE_BLOCK
            (local.set $return_pc (i32.load (i32.add (local.get $frame_ptr) (i32.const 4))))    ;; INSTRUCTION_INDEX
            (local.set $caller_scope (i32.load (i32.add (local.get $frame_ptr) (i32.const 8)))) ;; SCOPE

            ;; Restore pending pointer to frame's pending base (discard any leftover args)
            (call $write_state (global.get $STATE_PENDING_POINTER)
              (i32.load (i32.add (local.get $frame_ptr) (i32.const 12))))

            ;; Restore scope
            (global.set $current_scope (local.get $caller_scope))

            ;; Push result to caller's pending stack
            (call $pending_push (local.get $result_type) (local.get $result_flags) (local.get $result_lo) (local.get $result_hi))

            ;; Update instr_count since code_block may have changed (it shouldn't for same code block, but be safe)
            (local.set $instr_count (call $read_instr_count (local.get $code_block)))

            ;; Continue at return address
            (local.set $pc (local.get $return_pc))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; RETURN_UNDEFINED: push undefined then return (same logic as RETURN)
        (if (i32.eq (local.get $opcode) (global.get $OP_RETURN_UNDEFINED))
          (then
            ;; Return value is undefined
            (local.set $result_type (global.get $TYPE_UNDEFINED))
            (local.set $result_flags (i32.const 0))
            (local.set $result_lo (i32.const 0))
            (local.set $result_hi (i32.const 0))

            ;; Get current frame depth
            (local.set $count (call $get_frame_depth))

            ;; Search try stack for entry at current frame depth with finally
            (local.set $i (i32.const 0))
            (local.set $found (i32.const 0))

            (block $finally_check_done
              (loop $finally_check_loop
                (br_if $finally_check_done (i32.ge_u (local.get $i) (call $try_depth)))

                (local.set $index (call $try_read (local.get $i) (global.get $TRY_FRAME_DEPTH)))

                (if (i32.eq (local.get $index) (local.get $count))
                  (then
                    (local.set $operand2 (call $try_read (local.get $i) (global.get $TRY_FINALLY_INDEX)))
                    (if (i32.ne (local.get $operand2) (i32.const 0))
                      (then
                        (call $pending_push (local.get $result_type) (local.get $result_flags)
                                            (local.get $result_lo) (local.get $result_hi))
                        (call $save_completion_value)
                        (drop (call $pending_pop))

                        (call $write_state (global.get $STATE_COMPLETION_TYPE) (global.get $COMPLETION_RETURN))
                        (call $try_pop)

                        (local.set $pc (local.get $operand2))
                        (local.set $found (i32.const 1))
                        (br $finally_check_done)
                      )
                    )
                  )
                )

                (local.set $i (i32.add (local.get $i) (i32.const 1)))
                (br $finally_check_loop)
              )
            )

            (if (local.get $found)
              (then
                (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                (br $loop)
              )
            )

            ;; No finally - normal return

            (local.set $stack_ptr (call $read_state (global.get $STATE_STACK_POINTER)))

            (if (i32.le_u (local.get $stack_ptr)
                          (call $read_state (global.get $STATE_STACK_BASE)))
              (then
                (call $pending_push (local.get $result_type) (local.get $result_flags) (local.get $result_lo) (local.get $result_hi))
                (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_DONE))
                (br $exit)
              )
            )

            ;; Peek at current frame to check for iteration
            (local.set $frame_ptr (call $abs (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE))))

            ;; Check if this frame has FRAME_FLAG_ITERATING set
            (if (i32.and
                  (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_FLAGS)))
                  (global.get $FRAME_FLAG_ITERATING))
              (then
                ;; Iteration frame - continue with next iteration
                ;; Read iteration state from current frame
                (local.set $index (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_INDEX))))
                (local.set $arr_len (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_LENGTH))))
                (local.set $heap_ptr (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_RESULT))))
                (local.set $method_id (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_METHOD))))
                (local.set $arr_ptr (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_RECEIVER))))
                (local.set $closure_ptr (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_CALLBACK))))

                ;; forEach doesn't process return values, just increment index
                ;; Increment index
                (local.set $index (i32.add (local.get $index) (i32.const 1)))
                (i32.store (i32.add (local.get $frame_ptr) (global.get $FRAME_ITER_INDEX)) (local.get $index))

                ;; Check if more elements
                (if (i32.lt_u (local.get $index) (local.get $arr_len))
                  (then
                    ;; More elements - call callback again
                    (call $write_state (global.get $STATE_PENDING_POINTER)
                      (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_PENDING_BASE))))

                    (local.set $captured_scope (i32.load (call $abs (i32.add (local.get $closure_ptr) (i32.const 8)))))
                    (local.set $closure_flags (i32.load (call $abs (i32.add (local.get $closure_ptr) (i32.const 12)))))

                    ;; Get element at new index
                    (local.set $data_ptr (i32.load (call $abs (i32.add (local.get $arr_ptr) (i32.const 16)))))
                    (local.set $elem_ptr (i32.add
                      (i32.add (local.get $data_ptr) (global.get $GC_HEADER_SIZE))
                      (i32.mul (local.get $index) (global.get $VALUE_SIZE))))

                    ;; Push callback arguments: array, index, element
                    (call $pending_push (global.get $TYPE_ARRAY) (i32.const 0) (local.get $arr_ptr) (i32.const 0))
                    (call $push_f64 (f64.convert_i32_s (local.get $index)))
                    (call $pending_push
                      (call $value_type (local.get $elem_ptr))
                      (i32.load (call $abs (i32.add (local.get $elem_ptr) (i32.const 4))))
                      (call $value_data_lo (local.get $elem_ptr))
                      (call $value_data_hi (local.get $elem_ptr)))

                    (local.set $new_scope (call $scope_create (local.get $captured_scope)))
                    (global.set $current_scope (local.get $new_scope))

                    (local.set $pc (i32.load (call $abs (local.get $closure_ptr))))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                  (else
                    ;; Iteration complete - pop frame and return undefined
                    (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
                    (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))

                    (local.set $code_block (i32.load (local.get $frame_ptr)))
                    (local.set $return_pc (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_INSTRUCTION_INDEX))))
                    (local.set $caller_scope (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_SCOPE_POINTER))))
                    (call $write_state (global.get $STATE_PENDING_POINTER)
                      (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_PENDING_BASE))))
                    (global.set $current_scope (local.get $caller_scope))

                    (call $pending_push (global.get $TYPE_UNDEFINED) (i32.const 0) (i32.const 0) (i32.const 0))

                    (local.set $instr_count (call $read_instr_count (local.get $code_block)))
                    (local.set $pc (local.get $return_pc))
                    (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
                    (br $loop)
                  )
                )
              )
            )

            ;; Not an iteration frame - check for callback frame
            ;; frame_ptr was already set before the iteration check

            ;; Check if this is a callback frame (FRAME_FLAG_CALLBACK)
            (if (i32.and
                  (i32.load (i32.add (local.get $frame_ptr) (global.get $FRAME_FLAGS)))
                  (global.get $FRAME_FLAG_CALLBACK))
              (then
                ;; Pop the callback frame
                (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
                (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))

                ;; Restore instruction index from frame (where to resume after callback)
                (call $write_state (global.get $STATE_INSTRUCTION_INDEX)
                  (i32.load (i32.add (local.get $frame_ptr) (i32.const 4))))

                ;; Restore pending pointer from frame
                (call $write_state (global.get $STATE_PENDING_POINTER)
                  (i32.load (i32.add (local.get $frame_ptr) (i32.const 12))))

                ;; Restore scope from frame (both global and state)
                (local.set $caller_scope (i32.load (i32.add (local.get $frame_ptr) (i32.const 8))))
                (global.set $current_scope (local.get $caller_scope))
                (call $write_state (global.get $STATE_SCOPE) (local.get $caller_scope))

                ;; Push result to pending stack (orchestrator will read it)
                (call $pending_push (local.get $result_type) (local.get $result_flags) (local.get $result_lo) (local.get $result_hi))

                ;; Yield to orchestrator
                (call $write_state (global.get $STATE_STATUS) (global.get $STATUS_CALLBACK_DONE))
                (br $exit)
              )
            )

            ;; Normal frame pop
            (local.set $stack_ptr (i32.sub (local.get $stack_ptr) (global.get $FRAME_SIZE)))
            (call $write_state (global.get $STATE_STACK_POINTER) (local.get $stack_ptr))
            (local.set $frame_ptr (call $abs (local.get $stack_ptr)))

            (local.set $code_block (i32.load (local.get $frame_ptr)))
            (local.set $return_pc (i32.load (i32.add (local.get $frame_ptr) (i32.const 4))))
            (local.set $caller_scope (i32.load (i32.add (local.get $frame_ptr) (i32.const 8))))

            (call $write_state (global.get $STATE_PENDING_POINTER)
              (i32.load (i32.add (local.get $frame_ptr) (i32.const 12))))

            (global.set $current_scope (local.get $caller_scope))

            (call $pending_push (local.get $result_type) (local.get $result_flags) (local.get $result_lo) (local.get $result_hi))

            (local.set $instr_count (call $read_instr_count (local.get $code_block)))

            (local.set $pc (local.get $return_pc))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; =====================================================================
        ;; SPECIAL
        ;; =====================================================================
        (if (i32.eq (local.get $opcode) (global.get $OP_POP))
          (then
            (drop (call $pending_pop))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; DUP: duplicate top of pending stack
        (if (i32.eq (local.get $opcode) (global.get $OP_DUP))
          (then
            ;; Peek top value (don't pop)
            (local.set $val_ptr (i32.sub
              (call $read_state (global.get $STATE_PENDING_POINTER))
              (global.get $VALUE_SIZE)))
            ;; Push copy
            (call $pending_push
              (i32.load (call $abs (local.get $val_ptr)))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4))))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 8))))
              (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))))
            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; SWAP: swap top two values on pending stack
        ;; Stack: [a, b] → [b, a]
        (if (i32.eq (local.get $opcode) (global.get $OP_SWAP))
          (then
            ;; Get pointers to top two values
            (local.set $pending_ptr (call $read_state (global.get $STATE_PENDING_POINTER)))
            ;; top value at pending_ptr - 16, second at pending_ptr - 32
            (local.set $val_ptr (i32.sub (local.get $pending_ptr) (global.get $VALUE_SIZE)))
            (local.set $val_ptr2 (i32.sub (local.get $pending_ptr) (i32.mul (global.get $VALUE_SIZE) (i32.const 2))))

            ;; Read top value (b) into temp storage
            (local.set $type1 (i32.load (call $abs (local.get $val_ptr))))
            (local.set $swap_flags (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))))
            (local.set $data1_lo (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 8)))))
            (local.set $data1_hi (i32.load (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))))

            ;; Copy second value (a) to top position
            (i32.store (call $abs (local.get $val_ptr))
              (i32.load (call $abs (local.get $val_ptr2))))
            (i32.store (call $abs (i32.add (local.get $val_ptr) (i32.const 4)))
              (i32.load (call $abs (i32.add (local.get $val_ptr2) (i32.const 4)))))
            (i32.store (call $abs (i32.add (local.get $val_ptr) (i32.const 8)))
              (i32.load (call $abs (i32.add (local.get $val_ptr2) (i32.const 8)))))
            (i32.store (call $abs (i32.add (local.get $val_ptr) (i32.const 12)))
              (i32.load (call $abs (i32.add (local.get $val_ptr2) (i32.const 12)))))

            ;; Write temp (b) to second position
            (i32.store (call $abs (local.get $val_ptr2)) (local.get $type1))
            (i32.store (call $abs (i32.add (local.get $val_ptr2) (i32.const 4))) (local.get $swap_flags))
            (i32.store (call $abs (i32.add (local.get $val_ptr2) (i32.const 8))) (local.get $data1_lo))
            (i32.store (call $abs (i32.add (local.get $val_ptr2) (i32.const 12))) (local.get $data1_hi))

            (local.set $pc (i32.add (local.get $pc) (i32.const 1)))
            (local.set $fuel (i32.sub (local.get $fuel) (i32.const 1)))
            (br $loop)
          )
        )

        ;; Unknown opcode - error
        (call $set_error (global.get $ERR_INVALID_OPERAND) (local.get $opcode))
        (br $exit)
      )
    )

    (call $read_state (global.get $STATE_STATUS))
  )
)
