/**
 * SandScript Fuel-Based Interpreter - Constants
 *
 * All numeric constants shared between JS and WAT.
 */

// =============================================================================
// SANDFUEL Header (16 bytes at segment start)
// =============================================================================

export const HEADER = {
  MAGIC: 0x00,           // 8 bytes: "SANDFUEL"
  LAYOUT_VERSION: 0x08,  // u16
  BYTECODE_VERSION: 0x0A, // u16
  TYPE_VERSION: 0x0C,    // u16
  BUILTIN_VERSION: 0x0E, // u16
};

export const HEADER_SIZE = 0x10; // 16 bytes

// Current version numbers
export const VERSION = {
  LAYOUT: 2,    // Memory region structure (v2: added INVOCATION region)
  BYTECODE: 1,  // Opcode definitions
  TYPE: 1,      // TYPE_* value encodings
  BUILTIN: 1,   // METHOD_* IDs available
};

// =============================================================================
// Status codes (written to state header offset +36)
// =============================================================================

// Status codes (written to state header offset +36)
export const STATUS_RUNNING = 0;
export const STATUS_DONE = 1;
export const STATUS_PAUSED_FUEL = 2;
export const STATUS_FFI_REQUEST = 3;
export const STATUS_ERROR = 4;
export const STATUS_CALLBACK_DONE = 5;
export const STATUS_THROW = 6;

// Error codes (written to error info region)
export const ERR_NONE = 0;
export const ERR_UNDEFINED_VARIABLE = 1;
export const ERR_ASSIGN_UNDEFINED = 2;
export const ERR_NOT_CALLABLE = 3;
export const ERR_NOT_ITERABLE = 4;
export const ERR_PROPERTY_NULL = 5;
export const ERR_INVALID_OPERAND = 6;
export const ERR_STACK_OVERFLOW = 7;
export const ERR_TYPE_ERROR = 8;
export const ERR_ARITY = 9;
export const ERR_FFI_UNKNOWN = 10;
export const ERR_FFI_FAILED = 11;
export const ERR_USER_THROW = 12;
export const ERR_MSGPACK_READONLY = 13;
export const ERR_MSGPACK_INVALID = 14;
export const ERR_NOT_SUPPORTED = 15;   // NotSupportedError: new String(), prototype chain, etc.
export const ERR_OUT_OF_MEMORY = 16;   // Out of memory (heap, string table, or stacks)
export const ERR_MISSING_REGION = 17;  // Required memory region not found

// Completion types (for try/catch/finally)
export const COMPLETION_NORMAL = 0;
export const COMPLETION_THROW = 1;
export const COMPLETION_RETURN = 2;

// State header offsets (relative to baseOffset, after HEADER_SIZE)
// Note: typeof strings moved to BUILTINS segment
export const STATE = {
  HEAP_POINTER: HEADER_SIZE + 0x00,
  STRING_POINTER: HEADER_SIZE + 0x04,
  SCOPE: HEADER_SIZE + 0x08,  // current scope pointer (root scope has parent == 0)
  RESULT_POINTER: HEADER_SIZE + 0x0C,
  HEAP_START: HEADER_SIZE + 0x10,
  HEAP_END: HEADER_SIZE + 0x14,
  STRING_START: HEADER_SIZE + 0x18,
  SEGMENT_SIZE: HEADER_SIZE + 0x1C,
  FUEL: HEADER_SIZE + 0x20,
  STATUS: HEADER_SIZE + 0x24,
  STACK_POINTER: HEADER_SIZE + 0x28,
  STACK_BASE: HEADER_SIZE + 0x2C,
  PENDING_POINTER: HEADER_SIZE + 0x30,
  PENDING_BASE: HEADER_SIZE + 0x34,
  BASE_OFFSET: HEADER_SIZE + 0x38,
  // Reverse-growing code block
  CODE_POINTER: HEADER_SIZE + 0x3C,    // current write position for code, decrements as code grows
  // New fields for flat stream model
  CODE_BLOCK: HEADER_SIZE + 0x40,      // current CodeBlock being executed
  INSTRUCTION_INDEX: HEADER_SIZE + 0x44, // current instruction index
  // Try/catch support
  TRY_POINTER: HEADER_SIZE + 0x48,     // try stack pointer
  TRY_BASE: HEADER_SIZE + 0x4C,        // try stack base (constant)
  // Completion state (for finally blocks)
  COMPLETION_TYPE: HEADER_SIZE + 0x50,   // 0=normal, 1=throw, 2=return
  COMPLETION_VALUE: HEADER_SIZE + 0x54,  // pointer to 16-byte value stored at PENDING_BASE - 16
  // Region base offsets (computed at init, no longer hardcoded)
  FFI_REQUEST_BASE: HEADER_SIZE + 0x58,
  ERROR_INFO_BASE: HEADER_SIZE + 0x5C,
  SCRATCH_BASE: HEADER_SIZE + 0x60,
  CALL_STACK_BASE: HEADER_SIZE + 0x64,
  PENDING_STACK_BASE: HEADER_SIZE + 0x68,
  TRY_STACK_BASE: HEADER_SIZE + 0x6C,
  BUILTINS_BASE: HEADER_SIZE + 0x70,
};

// State header size (0x74 bytes = last field at 0x70 + 4, padded to 0x80 for alignment)
export const STATE_SIZE = 0x80;

// =============================================================================
// Invocation region (fixed size, for callable images)
// =============================================================================

// Invocation region offsets (relative to LAYOUT.INVOCATION)
export const INVOCATION = {
  INPUT_COUNT: 0x00,
  OUTPUT_COUNT: 0x04,
  // reserved: 0x08-0x0F
  INPUT_SLOTS: 0x10,   // 8 × 8 bytes = 64 bytes
  OUTPUT_SLOTS: 0x50,  // 8 × 8 bytes = 64 bytes
};

export const INVOCATION_SIZE = 0x90;  // 144 bytes
export const MAX_INPUT_SLOTS = 8;
export const MAX_OUTPUT_SLOTS = 8;

// Slot descriptor offsets (each slot is 8 bytes)
export const SLOT = {
  VALUE_PTR: 0x00,  // Pointer to value region in scope entry
  FLAGS: 0x04,      // Reserved for future use
};

// Default region sizes (used by initialize() to compute layout)
export const REGION_SIZE = {
  FFI_REQUEST: 16,
  ERROR_INFO: 16,
  INVOCATION: INVOCATION_SIZE,  // 144 bytes, fixed
  SCRATCH: 256,
  CALL_STACK: 4096,      // 128 frames × 32 bytes
  PENDING_STACK: 4096,   // 256 values × 16 bytes
  TRY_STACK: 1024,       // 64 entries × 16 bytes
  BUILTINS: 4096,
};

// Legacy LAYOUT for backwards compatibility during transition
// TODO: Remove once all code uses STATE region base fields
export const LAYOUT = {
  HEADER: 0x0000,
  STATE_HEADER: HEADER_SIZE,
  FFI_REQUEST: HEADER_SIZE + STATE_SIZE,                           // 0x90
  ERROR_INFO: HEADER_SIZE + STATE_SIZE + 0x10,                     // 0xA0
  INVOCATION: HEADER_SIZE + STATE_SIZE + 0x20,                     // 0xB0 (NEW)
  SCRATCH: HEADER_SIZE + STATE_SIZE + 0x20 + INVOCATION_SIZE,      // 0x140 (shifted +0x90)
  CALL_STACK: HEADER_SIZE + STATE_SIZE + 0x120 + INVOCATION_SIZE,  // 0x240 (shifted)
  PENDING_STACK: HEADER_SIZE + STATE_SIZE + 0x1120 + INVOCATION_SIZE, // 0x1240 (shifted)
  TRY_STACK: HEADER_SIZE + STATE_SIZE + 0x2120 + INVOCATION_SIZE,  // 0x2240 (shifted)
  BUILTINS: HEADER_SIZE + STATE_SIZE + 0x2520 + INVOCATION_SIZE,   // 0x2640 (shifted)
  HEAP_START: HEADER_SIZE + STATE_SIZE + 0x3520 + INVOCATION_SIZE, // 0x3640 (shifted)
};

export const SCRATCH_SIZE = 256;
export const CALL_STACK_SIZE = 4096;
export const PENDING_STACK_SIZE = 4096;
export const TRY_STACK_SIZE = 1024;
export const BUILTINS_SIZE = 4096;

// BUILTINS segment offsets (relative to LAYOUT.BUILTINS)
// Pre-interned string pointers for built-in names
export const BUILTIN_NAME = {
  // Typeof strings (moved from STATE)
  TYPEOF_UNDEFINED: 0x00,
  TYPEOF_BOOLEAN: 0x04,
  TYPEOF_NUMBER: 0x08,
  TYPEOF_STRING: 0x0C,
  TYPEOF_OBJECT: 0x10,
  TYPEOF_FUNCTION: 0x14,

  // Special keywords
  THIS: 0x18,
  LENGTH: 0x1C,

  // Array methods (0x20 - 0x6F)
  PUSH: 0x20,
  POP: 0x24,
  SHIFT: 0x28,
  UNSHIFT: 0x2C,
  SLICE: 0x30,
  CONCAT: 0x34,
  JOIN: 0x38,
  REVERSE: 0x3C,
  INDEX_OF: 0x40,
  INCLUDES: 0x44,
  MAP: 0x48,
  FILTER: 0x4C,
  REDUCE: 0x50,
  FOR_EACH: 0x54,
  FIND: 0x58,
  FIND_INDEX: 0x5C,
  SOME: 0x60,
  EVERY: 0x64,

  // String methods (0x80 - 0xDF)
  CHAR_AT: 0x80,
  CHAR_CODE_AT: 0x84,
  STRING_INDEX_OF: 0x88,
  STRING_INCLUDES: 0x8C,
  STRING_SLICE: 0x90,
  SUBSTRING: 0x94,
  SPLIT: 0x98,
  TRIM: 0x9C,
  TO_LOWER_CASE: 0xA0,
  TO_UPPER_CASE: 0xA4,
  STARTS_WITH: 0xA8,
  ENDS_WITH: 0xAC,
  REPEAT: 0xB0,
  PAD_START: 0xB4,
  PAD_END: 0xB8,
  REPLACE: 0xBC,

  // Literal strings for type conversion (0xC0 - 0xDF)
  LIT_NULL: 0xC0,              // "null"
  LIT_TRUE: 0xC4,              // "true"
  LIT_FALSE: 0xC8,             // "false"
  LIT_OBJECT_OBJECT: 0xCC,     // "[object Object]"
  LIT_NAN: 0xD0,               // "NaN"
  LIT_INFINITY: 0xD4,          // "Infinity"
  LIT_NEG_INFINITY: 0xD8,      // "-Infinity"
  LIT_OBJECT_FUNCTION: 0xDC,   // "[object Function]"
  LIT_EMPTY_STRING: 0xE0,      // ""

  // Math methods (0x100 - 0x15F) - for future use
  MATH_ABS: 0x100,
  MATH_FLOOR: 0x104,
  MATH_CEIL: 0x108,
  MATH_ROUND: 0x10C,
  MATH_MIN: 0x110,
  MATH_MAX: 0x114,
  MATH_POW: 0x118,
  MATH_SQRT: 0x11C,
  MATH_RANDOM: 0x120,
};

// Frame layout (56 bytes)
// Base frame fields (32 bytes):
export const FRAME = {
  CODE_BLOCK: 0x00,      // which CodeBlock
  INSTRUCTION_INDEX: 0x04, // return instruction index
  SCOPE_POINTER: 0x08,   // scope for this frame
  PENDING_BASE: 0x0C,    // pending stack base
  PENDING_COUNT: 0x10,   // expected return values
  FLAGS: 0x14,           // TAIL_CALL, ITERATING, etc.
  SOURCE_START: 0x18,    // for stack traces
  SOURCE_END: 0x1C,      // for stack traces
  // Iteration fields (24 bytes, used when FRAME_FLAG_ITERATING is set):
  ITER_INDEX: 0x20,      // current index in array
  ITER_LENGTH: 0x24,     // array length
  ITER_RESULT: 0x28,     // accumulator (array ptr for map/filter, value for reduce)
  ITER_METHOD: 0x2C,     // which method (METHOD_MAP, METHOD_FILTER, etc.)
  ITER_RECEIVER: 0x30,   // array pointer
  ITER_CALLBACK: 0x34,   // closure pointer
};
export const FRAME_SIZE = 56;

// Try stack entry layout (16 bytes)
export const TRY_ENTRY = {
  CODE_BLOCK: 0x00,      // which CodeBlock the handler is in
  CATCH_INDEX: 0x04,     // instruction index for catch, 0 if none
  FINALLY_INDEX: 0x08,   // instruction index for finally, 0 if none
  FRAME_DEPTH: 0x0C,     // call stack depth when try was entered
};
export const TRY_ENTRY_SIZE = 16;

// Frame flags
export const FRAME_FLAG_CALLBACK = 1;
export const FRAME_FLAG_TAIL = 2;
export const FRAME_FLAG_ITERATING = 4;

// Closure flags
export const CLOSURE_FLAG_ARROW = 1;

// Value size
export const VALUE_SIZE = 16;

// Type tags
export const TYPE = {
  NULL: 0x00,
  UNDEFINED: 0x01,
  BOOLEAN: 0x02,
  INTEGER: 0x03,
  FLOAT: 0x04,
  STRING: 0x05,
  ARRAY: 0x06,
  OBJECT: 0x07,
  CLOSURE: 0x08,
  SCOPE: 0x09,
  FFI_REF: 0x0a,
  FFI_NAMESPACE: 0x0b,
  FFI_METHOD: 0x0c,
  MSGPACK_REF: 0x0d,
  BOUND_METHOD: 0x0e,
  CONSTRUCTOR: 0x0f,  // callable object (Array, Object, String, Number)
};

// GC object types
export const OBJ = {
  ARRAY: 0,
  OBJECT: 1,
  SCOPE: 2,
  CLOSURE: 3,
  PARAM_LIST: 4,
  ARRAY_DATA: 5,
  CODE_BLOCK: 6,
  OBJECT_DATA: 7,  // Entries block for objects (like ARRAY_DATA for arrays)
};

export const GC_HEADER_SIZE = 8;

// Opcodes for flat instruction stream
export const OP = {
  // Literals (push to pending stack)
  LIT_INT: 0x01,         // operand1 = i32 value
  LIT_FLOAT: 0x02,       // operand1,2 = f64 bits (little-endian)
  LIT_STRING: 0x03,      // operand1 = string table offset
  LIT_NULL: 0x04,
  LIT_UNDEFINED: 0x05,
  LIT_TRUE: 0x06,
  LIT_FALSE: 0x07,

  // Variables
  GET_VAR: 0x10,         // operand1 = name (string offset) → push value
  SET_VAR: 0x11,         // operand1 = name, pop value → assign
  LET_VAR: 0x12,         // operand1 = name, pop value → define in scope
  BIND_PARAM: 0x13,      // operand1 = name, pop arg → define in scope

  // Arithmetic (pop operands, push result)
  ADD: 0x20,             // pop 2, push sum (polymorphic: number or string)
  SUB: 0x21,             // pop 2, push difference
  MUL: 0x22,             // pop 2, push product
  DIV: 0x23,             // pop 2, push quotient
  MOD: 0x24,             // pop 2, push remainder
  NEG: 0x25,             // pop 1, push negation
  POW: 0x26,             // pop 2, push power

  // Comparison (pop 2, push bool)
  EQ: 0x30,              // ===
  NEQ: 0x31,             // !==
  LT: 0x32,              // <
  GT: 0x33,              // >
  LTE: 0x34,             // <=
  GTE: 0x35,             // >=

  // Logic
  NOT: 0x40,             // pop 1, push !value
  AND: 0x41,             // peek top; if falsy jump to operand1; else pop and continue
  OR: 0x42,              // peek top; if truthy jump to operand1; else pop and continue
  NULLISH: 0x43,         // peek top; if nullish (null/undefined) pop and jump to operand1; else keep

  // Bitwise
  BAND: 0x27,            // pop 2, push bitwise AND
  BOR: 0x28,             // pop 2, push bitwise OR
  BXOR: 0x29,            // pop 2, push bitwise XOR
  BNOT: 0x2A,            // pop 1, push bitwise NOT
  SHL: 0x2B,             // pop 2, push left shift
  SHR: 0x2C,             // pop 2, push signed right shift
  USHR: 0x2D,            // pop 2, push unsigned right shift

  // Control Flow
  JUMP: 0x50,            // operand1 = target instruction index (or CodeBlock ptr)
  JUMP_IF_FALSE: 0x51,   // pop condition, jump if falsy
  JUMP_IF_TRUE: 0x52,    // pop condition, jump if truthy

  // Scope
  SCOPE_PUSH: 0x60,      // create child scope
  SCOPE_POP: 0x61,       // restore parent scope

  // Functions
  MAKE_CLOSURE: 0x70,    // operand1 = start instr, operand2 = end instr → push closure (binds this)
  CALL: 0x71,            // operand1 = argc, pop closure + args, push frame, jump (this = undefined)
  RETURN: 0x72,          // pop result, pop frame, push result to caller's pending
  RETURN_UNDEFINED: 0x73, // push undefined then return
  MAKE_ARROW_CLOSURE: 0x74, // same as MAKE_CLOSURE but sets CLOSURE_FLAG_ARROW (no this binding)
  CALL_METHOD: 0x75,     // operand1 = argc, pop closure + args + receiver, push frame (this = receiver)
  NEW: 0x76,             // operand1 = argc, pop constructor + args, construct object

  // Objects/Arrays
  MAKE_ARRAY: 0x80,      // operand1 = count, pop N → push array
  MAKE_OBJECT: 0x81,     // operand1 = count, pop N key-value pairs → push object
  GET_PROP: 0x82,        // operand1 = name, pop obj → push value
  SET_PROP: 0x83,        // operand1 = name, pop val, pop obj → assign
  GET_INDEX: 0x84,       // pop index, pop obj → push value
  SET_INDEX: 0x85,       // pop val, pop idx, pop obj → assign

  // Builtins
  BUILTIN: 0x90,         // operand1 = builtin ID, operand2 = argc
  TYPEOF: 0x91,          // pop 1, push type string

  // Iteration
  ITER_INIT: 0xA0,       // pop iterable → push iterator
  ITER_NEXT: 0xA1,       // peek iterator → push value, push done (bool)
  ITER_CLOSE: 0xA2,      // pop iterator

  // Error Handling
  THROW: 0xB0,           // pop value, unwind to handler or halt
  TRY_PUSH: 0xB1,        // operand1 = catchPC, operand2 = finallyPC
  TRY_POP: 0xB2,         // normal exit from try, run finally if present
  FINALLY_END: 0xB3,     // end of finally: re-throw, complete return, or continue

  // Special
  POP: 0xF0,             // discard top of pending stack
  DUP: 0xF1,             // duplicate top of pending stack
  SWAP: 0xF2,            // swap top two values on pending stack
  NOP: 0xFF,             // no operation
};

// Method IDs (for bound method dispatch in CALL_METHOD)
export const METHOD = {
  // Array instance methods (0x01 - 0x1F)
  PUSH: 0x01,
  POP: 0x02,
  SHIFT: 0x03,
  UNSHIFT: 0x04,
  SLICE: 0x05,
  CONCAT: 0x06,
  JOIN: 0x07,
  REVERSE: 0x08,
  INDEX_OF: 0x09,
  INCLUDES: 0x0A,
  MAP: 0x0B,
  FILTER: 0x0C,
  REDUCE: 0x0D,
  FOR_EACH: 0x0E,
  FIND: 0x0F,
  FIND_INDEX: 0x10,
  SOME: 0x11,
  EVERY: 0x12,

  // String instance methods (0x20 - 0x3F)
  CHAR_AT: 0x20,
  CHAR_CODE_AT: 0x21,
  STRING_INDEX_OF: 0x22,
  STRING_INCLUDES: 0x23,
  STRING_SLICE: 0x24,
  SUBSTRING: 0x25,
  SPLIT: 0x26,
  TRIM: 0x27,
  TO_LOWER_CASE: 0x28,
  TO_UPPER_CASE: 0x29,
  STARTS_WITH: 0x2A,
  ENDS_WITH: 0x2B,
  REPEAT: 0x2C,
  PAD_START: 0x2D,
  PAD_END: 0x2E,
  REPLACE: 0x2F,

  // Math static methods (0x40 - 0x5F)
  MATH_ABS: 0x40,
  MATH_FLOOR: 0x41,
  MATH_CEIL: 0x42,
  MATH_ROUND: 0x43,
  MATH_TRUNC: 0x44,
  MATH_SIGN: 0x45,
  MATH_MIN: 0x46,
  MATH_MAX: 0x47,
  MATH_POW: 0x48,
  MATH_SQRT: 0x49,
  MATH_CBRT: 0x4A,
  MATH_HYPOT: 0x4B,
  MATH_SIN: 0x4C,
  MATH_COS: 0x4D,
  MATH_TAN: 0x4E,
  MATH_ASIN: 0x4F,
  MATH_ACOS: 0x50,
  MATH_ATAN: 0x51,
  MATH_ATAN2: 0x52,
  MATH_SINH: 0x53,
  MATH_COSH: 0x54,
  MATH_TANH: 0x55,
  MATH_ASINH: 0x56,
  MATH_ACOSH: 0x57,
  MATH_ATANH: 0x58,
  MATH_LOG: 0x59,
  MATH_LOG10: 0x5A,
  MATH_LOG2: 0x5B,
  MATH_LOG1P: 0x5C,
  MATH_EXP: 0x5D,
  MATH_EXPM1: 0x5E,
  MATH_CLZ32: 0x5F,
  MATH_IMUL: 0x60,
  MATH_FROUND: 0x61,

  // Object static methods (0x70 - 0x7F)
  OBJECT_KEYS: 0x70,
  OBJECT_VALUES: 0x71,
  OBJECT_ENTRIES: 0x72,
  OBJECT_ASSIGN: 0x73,

  // Array static methods (0x80 - 0x8F)
  ARRAY_IS_ARRAY: 0x80,
  ARRAY_FROM: 0x81,

  // String static methods (0x90 - 0x9F)
  STRING_FROM_CHAR_CODE: 0x90,
  STRING_FROM_CODE_POINT: 0x91,

  // Number static methods (0xA0 - 0xAF)
  NUMBER_IS_NAN: 0xA0,
  NUMBER_IS_FINITE: 0xA1,
  NUMBER_IS_INTEGER: 0xA2,
  NUMBER_PARSE_INT: 0xA3,
  NUMBER_PARSE_FLOAT: 0xA4,

  // Global conversion functions (0xB0 - 0xBF)
  TO_STRING: 0xB0,     // String(x)
  TO_NUMBER: 0xB1,     // Number(x)
  TO_BOOLEAN: 0xB2,    // Boolean(x)
  PARSE_INT: 0xB3,     // parseInt(str, radix)
  PARSE_FLOAT: 0xB4,   // parseFloat(str)
  IS_NAN: 0xB5,        // isNaN(x) - global version with coercion
  IS_FINITE: 0xB6,     // isFinite(x) - global version with coercion

  // Constructor method IDs (0xC0 - 0xCF)
  ARRAY_CONSTRUCTOR: 0xC0,
  OBJECT_CONSTRUCTOR: 0xC1,
};

// Note: BUILTIN IDs have been merged into METHOD for unified dispatch.
// OP.BUILTIN is no longer used; static methods use TYPE_BOUND_METHOD with METHOD IDs.

// CodeBlock flags
export const CODE_BLOCK_FLAG = {
  COMPLETE: 0x01,        // All backpatches resolved
  HAS_ERROR: 0x02,       // Parse error occurred
  IS_FUNCTION: 0x04,     // This is a function body (not top-level)
};

// Instruction size
export const INSTRUCTION_SIZE = 16;

// =============================================================================
// Reverse Lookup Helpers
// =============================================================================

// Status code to string
const STATUS_NAMES = {
  [STATUS_RUNNING]: 'running',
  [STATUS_DONE]: 'done',
  [STATUS_PAUSED_FUEL]: 'paused',
  [STATUS_FFI_REQUEST]: 'ffi',
  [STATUS_ERROR]: 'error',
  [STATUS_CALLBACK_DONE]: 'callback_done',
  [STATUS_THROW]: 'throw',
};

export function statusToString(code) {
  return STATUS_NAMES[code] ?? `unknown_${code}`;
}

// Error code to string
const ERROR_NAMES = {
  [ERR_NONE]: 'NONE',
  [ERR_UNDEFINED_VARIABLE]: 'UNDEFINED_VARIABLE',
  [ERR_ASSIGN_UNDEFINED]: 'ASSIGN_UNDEFINED',
  [ERR_NOT_CALLABLE]: 'NOT_CALLABLE',
  [ERR_NOT_ITERABLE]: 'NOT_ITERABLE',
  [ERR_PROPERTY_NULL]: 'PROPERTY_NULL',
  [ERR_INVALID_OPERAND]: 'INVALID_OPERAND',
  [ERR_STACK_OVERFLOW]: 'STACK_OVERFLOW',
  [ERR_TYPE_ERROR]: 'TYPE_ERROR',
  [ERR_ARITY]: 'ARITY',
  [ERR_FFI_UNKNOWN]: 'FFI_UNKNOWN',
  [ERR_FFI_FAILED]: 'FFI_FAILED',
  [ERR_USER_THROW]: 'USER_THROW',
  [ERR_MSGPACK_READONLY]: 'MSGPACK_READONLY',
  [ERR_MSGPACK_INVALID]: 'MSGPACK_INVALID',
  [ERR_NOT_SUPPORTED]: 'NOT_SUPPORTED',
  [ERR_OUT_OF_MEMORY]: 'OUT_OF_MEMORY',
  [ERR_MISSING_REGION]: 'MISSING_REGION',
};

export function errorCodeToString(code) {
  return ERROR_NAMES[code] ?? `UNKNOWN_${code}`;
}

// Opcode to string (reverse lookup from OP object)
const OP_NAMES = Object.fromEntries(
  Object.entries(OP).map(([name, code]) => [code, name])
);

export function opcodeToString(code) {
  return OP_NAMES[code] ?? `OP_${code.toString(16)}`;
}
