/**
 * SandScript Fuel Interpreter - Garbage Collector
 *
 * Mark-compact collector for the linear memory heap.
 * Host-triggered (no automatic GC).
 *
 * IMPORTANT: Pointer Convention
 * - All pointers stored in state/values/objects are DATA pointers
 *   (pointing to the data area AFTER the 8-byte GC header)
 * - GC header is at (dataPtr - GC_HEADER_SIZE)
 * - When walking the heap, we iterate by HEADER pointers
 *
 * Phases:
 *   1. Mark - traverse from roots, mark reachable objects and strings
 *   2. Compute - calculate forwarding addresses for live objects
 *   3. Update - update all pointers to new addresses
 *   4. Compact - move objects to new addresses
 *   5. String compaction - same for string table, rebuild hash map
 */

import {
  STATUS_FFI_REQUEST,
  STATUS_RUNNING,
  STATE,
  LAYOUT,
  FRAME,
  FRAME_SIZE,
  FRAME_FLAG_ITERATING,
  VALUE_SIZE,
  TYPE,
  OBJ,
  GC_HEADER_SIZE,
  OP,
  INSTRUCTION_SIZE,
  INVOCATION,
  SLOT,
} from './constants.js';

// Hash table size for string interning (must match interpreter.wat)
const HASH_TABLE_SIZE = 2048 * 8; // 16KB

/**
 * Garbage collector for fuel interpreter.
 */
export class Collector {
  /**
   * @param {MemoryManipulator} manipulator - Memory manipulator instance
   */
  constructor(manipulator) {
    this.manipulator = manipulator;
    this.memory = manipulator.memory;
    this.baseOffset = manipulator.baseOffset;
    this._refreshViews();

    // String mark bitmap - allocated during mark phase
    this.stringMarks = null;

    // String data starts after the hash table
    this.stringDataStart = null;

    // External roots (set by collect(), default empty for direct method calls)
    this.externalRoots = [];
  }

  /**
   * Refresh typed array views (call after memory growth).
   */
  _refreshViews() {
    this.buffer = this.memory.buffer;
    this.view = new DataView(this.buffer);
    this.bytes = new Uint8Array(this.buffer);
  }

  /**
   * Convert segment-relative offset to absolute memory offset.
   */
  abs(offset) {
    const result = this.baseOffset + offset;
    if (result < 0 || result >= this.view.byteLength) {
      const stack = new Error().stack.split('\n').slice(1, 5).join('\n');
      throw new Error(`abs(${offset}): out of bounds\n${stack}`);
    }
    return result;
  }

  /**
   * Run garbage collection.
   *
   * @param {number[]} externalRoots - Optional array of DATA pointers to treat as roots
   *   (e.g., closure pointers held by JS). These will be kept alive and their
   *   forwarding info returned so the caller can update external references.
   * @returns {object} - { heapCollected, heapRetained, stringsCollected, stringsRetained, forwarding }
   *   forwarding is a Map<oldDataPtr, newDataPtr> for external roots that moved
   */
  collect(externalRoots = []) {
    // Check status - GC forbidden during FFI or while running
    const status = this.manipulator.getStatus();
    if (status === STATUS_RUNNING) {
      throw new Error('Cannot GC while interpreter is running');
    }
    if (status === STATUS_FFI_REQUEST) {
      throw new Error('Cannot GC during FFI request');
    }

    this._refreshViews();

    // Store external roots for mark phase
    this.externalRoots = externalRoots;

    const startHeapPointer = this.manipulator.getHeapPointer();
    const startStringPointer = this.manipulator.getStringPointer();

    // Phase 1: Mark
    this.markPhase();

    // Phase 2-4: Compact heap (returns forwarding map for external roots)
    const { heapRetained, heapCollected, forwarding } = this.compactHeap();

    // Phase 5: Compact strings
    const { stringsRetained, stringsCollected } = this.compactStrings();

    // Clean up
    this.externalRoots = [];

    return {
      heapCollected,
      heapRetained,
      stringsCollected,
      stringsRetained,
      forwarding,
    };
  }

  // ===========================================================================
  // GC Header Operations
  //
  // These methods work with HEADER pointers (pointing to the GC header).
  // To get header from data pointer: headerPtr = dataPtr - GC_HEADER_SIZE
  // ===========================================================================

  /**
   * Read object type from GC header.
   * @param {number} headerPtr - Pointer to GC header
   */
  readHeaderType(headerPtr) {
    const headerWord = this.view.getUint32(this.abs(headerPtr), true);
    return (headerWord >> 24) & 0x7f;
  }

  /**
   * Read object size from GC header.
   * @param {number} headerPtr - Pointer to GC header
   */
  readHeaderSize(headerPtr) {
    const headerWord = this.view.getUint32(this.abs(headerPtr), true);
    return headerWord & 0x00ffffff;
  }

  /**
   * Check if object is marked.
   * @param {number} headerPtr - Pointer to GC header
   */
  isMarked(headerPtr) {
    const headerWord = this.view.getUint32(this.abs(headerPtr), true);
    return (headerWord & 0x80000000) !== 0;
  }

  /**
   * Set mark bit on object.
   * @param {number} headerPtr - Pointer to GC header
   */
  setMark(headerPtr) {
    const absPtr = this.abs(headerPtr);
    const headerWord = this.view.getUint32(absPtr, true);
    this.view.setUint32(absPtr, headerWord | 0x80000000, true);
  }

  /**
   * Clear mark bit on object.
   * @param {number} headerPtr - Pointer to GC header
   */
  clearMark(headerPtr) {
    const absPtr = this.abs(headerPtr);
    const headerWord = this.view.getUint32(absPtr, true);
    this.view.setUint32(absPtr, headerWord & 0x7fffffff, true);
  }

  /**
   * Set forwarding pointer (new HEADER pointer after compaction).
   * @param {number} headerPtr - Pointer to GC header
   * @param {number} newHeaderPtr - New header pointer
   */
  setForwardingPtr(headerPtr, newHeaderPtr) {
    this.view.setUint32(this.abs(headerPtr) + 4, newHeaderPtr, true);
  }

  /**
   * Get forwarding pointer.
   * @param {number} headerPtr - Pointer to GC header
   * @returns {number} - New header pointer (0 if not set)
   */
  getForwardingPtr(headerPtr) {
    return this.view.getUint32(this.abs(headerPtr) + 4, true);
  }

  /**
   * Align size to 16 bytes.
   */
  alignSize(size) {
    return (size + 15) & ~15;
  }

  // ===========================================================================
  // Phase 1: Mark
  // ===========================================================================

  /**
   * Mark phase - traverse from roots and mark all reachable objects.
   */
  markPhase() {
    // Clear all marks first
    this.clearAllMarks();

    // Initialize string mark bitmap
    this.initStringMarks();

    // Mark from roots

    // 1. Global scope (stored as DATA pointer)
    const globalScopeData = this.manipulator.getScope();
    if (globalScopeData !== 0) {
      this.markObjectByDataPtr(globalScopeData);
    }

    // 2. Result pointer (it's an address of a value slot, not a data pointer)
    const resultPointer = this.manipulator.getResultPointer();
    if (resultPointer !== 0) {
      this.markValue(resultPointer);
    }

    // 3. Pending stack entries
    this.markPendingStack();

    // 4. Call stack frames (scopes + iteration state)
    this.markCallStack();

    // 5. Completion value (if in finally block)
    const completionType = this.manipulator.getCompletionType();
    if (completionType !== 0) {
      const completionValue = this.manipulator.getCompletionValue();
      if (completionValue !== 0) {
        this.markValue(completionValue);
      }
    }

    // 6. External roots (closure pointers held by JS)
    for (const dataPtr of this.externalRoots) {
      if (dataPtr !== 0) {
        this.markObjectByDataPtr(dataPtr);
      }
    }

    // 7. Mark strings used in the code block (so they're not compacted away)
    this.markCodeBlockStrings();

    // 8. Mark strings used in builtins table (method names like "push", "length", etc.)
    this.markBuiltinStrings();
  }

  /**
   * Mark all strings referenced by code instructions.
   * This prevents string compaction from invalidating code operands.
   *
   * Opcodes that use string offsets in operand1:
   * - LIT_STRING (0x03): push string literal
   * - GET_VAR (0x10): get variable by name
   * - SET_VAR (0x11): set variable by name
   * - LET_VAR (0x12): define variable by name
   * - BIND_PARAM (0x13): bind parameter by name
   * - GET_PROP (0x82): get property by name
   * - SET_PROP (0x83): set property by name
   */
  markCodeBlockStrings() {
    const codeBlock = this.manipulator.getCodeBlock();
    if (codeBlock === 0) return;

    const instrCount = this.manipulator.codeBlockInstructionCount(codeBlock);
    const codeStart = this.manipulator.getCodeStart();

    // Opcodes that use string offsets in operand1
    const stringOpcodes = new Set([
      OP.LIT_STRING,  // 0x03
      OP.GET_VAR,     // 0x10
      OP.SET_VAR,     // 0x11
      OP.LET_VAR,     // 0x12
      OP.BIND_PARAM,  // 0x13
      OP.GET_PROP,    // 0x82
      OP.SET_PROP,    // 0x83
    ]);

    // Instructions grow downward from code_start
    // instr_ptr = code_start - (index + 1) * INSTRUCTION_SIZE
    for (let i = 0; i < instrCount; i++) {
      const instrAddr = codeStart - (i + 1) * INSTRUCTION_SIZE;
      const absAddr = this.abs(instrAddr);
      const opcode = this.view.getUint8(absAddr);

      if (stringOpcodes.has(opcode)) {
        // operand1 is at offset 8 within the instruction
        const stringOffset = this.view.getUint32(absAddr + 8, true);
        if (stringOffset !== 0) {
          this.markString(stringOffset);
        }
      }
    }
  }

  /**
   * Mark all strings in the builtins table.
   * The builtins table stores string offsets for built-in method names.
   */
  markBuiltinStrings() {
    const builtinsBase = this.manipulator.getBuiltinsBase();

    // The builtins table has slots from 0x00 to 0xDC (56 slots of 4 bytes each)
    // Each slot stores a string table offset
    const BUILTINS_SLOT_COUNT = 56;

    for (let i = 0; i < BUILTINS_SLOT_COUNT; i++) {
      const slotAddr = builtinsBase + i * 4;
      const absSlotAddr = this.abs(slotAddr);
      const stringOffset = this.view.getUint32(absSlotAddr, true);

      if (stringOffset !== 0) {
        this.markString(stringOffset);
      }
    }
  }

  /**
   * Clear mark bits on all heap objects.
   */
  clearAllMarks() {
    const heapStart = this.manipulator.getHeapStart();
    const heapPointer = this.manipulator.getHeapPointer();

    // Walk by header pointers
    // Note: sizes in headers are actual sizes, not aligned - allocator places
    // objects consecutively without padding
    let headerPtr = heapStart;
    while (headerPtr < heapPointer) {
      const size = this.readHeaderSize(headerPtr);
      if (size === 0) break;

      this.clearMark(headerPtr);
      this.setForwardingPtr(headerPtr, 0);
      headerPtr += size;
    }
  }

  /**
   * Initialize string mark bitmap.
   */
  initStringMarks() {
    const stringStart = this.manipulator.getStringStart();
    const stringEnd = this.manipulator.getStringEnd();

    this.stringDataStart = stringStart + HASH_TABLE_SIZE;

    // One bit per 4 bytes in string data area
    const stringDataSize = stringEnd - this.stringDataStart;
    const bitmapSize = Math.ceil(stringDataSize / 4 / 8);
    this.stringMarks = new Uint8Array(bitmapSize);
  }

  /**
   * Mark a string offset as live.
   */
  markString(offset) {
    if (offset === 0 || offset < this.stringDataStart) return;

    const relativeOffset = offset - this.stringDataStart;
    const bitIndex = Math.floor(relativeOffset / 4);
    const byteIndex = Math.floor(bitIndex / 8);
    const bitPos = bitIndex % 8;

    if (byteIndex < this.stringMarks.length) {
      this.stringMarks[byteIndex] |= (1 << bitPos);
    }
  }

  /**
   * Check if a string offset is marked.
   */
  isStringMarked(offset) {
    if (offset === 0) return true;
    if (offset < this.stringDataStart) return true; // In hash table, not data

    const relativeOffset = offset - this.stringDataStart;
    const bitIndex = Math.floor(relativeOffset / 4);
    const byteIndex = Math.floor(bitIndex / 8);
    const bitPos = bitIndex % 8;

    if (byteIndex >= this.stringMarks.length) return false;

    return (this.stringMarks[byteIndex] & (1 << bitPos)) !== 0;
  }

  /**
   * Mark a value (16-byte slot) at the given address.
   * @param {number} valueAddr - Address of the value slot
   */
  markValue(valueAddr) {
    const absPtr = this.abs(valueAddr);
    const type = this.view.getUint32(absPtr, true);

    switch (type) {
      case TYPE.ARRAY:
      case TYPE.OBJECT: {
        // Arrays and objects store HEADER pointers in data_lo
        const headerPtr = this.view.getUint32(absPtr + 8, true);
        if (headerPtr !== 0) {
          this.markObjectByHeaderPtr(headerPtr);
        }
        break;
      }
      case TYPE.CLOSURE:
      case TYPE.SCOPE: {
        // Closures and scopes store DATA pointers in data_lo
        const dataPtr = this.view.getUint32(absPtr + 8, true);
        if (dataPtr !== 0) {
          this.markObjectByDataPtr(dataPtr);
        }
        break;
      }

      case TYPE.STRING: {
        const strOffset = this.view.getUint32(absPtr + 8, true);
        this.markString(strOffset);
        break;
      }

      case TYPE.BOUND_METHOD: {
        // BOUND_METHOD layout: [type:4][flags:4][data_lo:4][data_hi:4]
        // data_lo (offset +8) = receiver pointer (HEADER pointer to array/object)
        const receiver = this.view.getUint32(absPtr + 8, true);
        if (receiver !== 0) {
          this.markObjectByHeaderPtr(receiver);
        }
        break;
      }

      case TYPE.CONSTRUCTOR: {
        // object pointer is in bytes 8-11 (HEADER pointer from allocateObject)
        const headerPtr = this.view.getUint32(absPtr + 8, true);
        if (headerPtr !== 0) {
          this.markObjectByHeaderPtr(headerPtr);
        }
        break;
      }

      // Other types don't reference heap
    }
  }

  /**
   * Mark a heap object given its HEADER pointer.
   * @param {number} headerPtr - Pointer to GC header
   */
  markObjectByHeaderPtr(headerPtr) {
    // Sanity check
    const heapStart = this.manipulator.getHeapStart();
    const heapPointer = this.manipulator.getHeapPointer();
    if (headerPtr < heapStart || headerPtr >= heapPointer) {
      return;
    }

    // Already marked? Skip to avoid cycles
    if (this.isMarked(headerPtr)) return;

    // Mark this object
    this.setMark(headerPtr);

    // Data is after header
    const dataPtr = headerPtr + GC_HEADER_SIZE;

    // Traverse children based on object type
    const objType = this.readHeaderType(headerPtr);

    switch (objType) {
      case OBJ.ARRAY:
        this.markArray(dataPtr);
        break;

      case OBJ.OBJECT:
        this.markObjectEntries(dataPtr);
        break;

      case OBJ.SCOPE:
        this.markScope(dataPtr);
        break;

      case OBJ.CLOSURE:
        this.markClosure(dataPtr);
        break;

      case OBJ.ARRAY_DATA:
        // Data blocks are marked via their parent array
        // But we still need to mark elements
        this.markArrayDataElements(headerPtr);
        break;

      case OBJ.PARAM_LIST:
        this.markParamList(dataPtr);
        break;
    }
  }

  /**
   * Mark a heap object given its DATA pointer.
   * @param {number} dataPtr - Pointer to object data (after GC header)
   */
  markObjectByDataPtr(dataPtr) {
    this.markObjectByHeaderPtr(dataPtr - GC_HEADER_SIZE);
  }

  /**
   * Mark array elements and data block.
   * @param {number} dataPtr - Pointer to array data
   */
  markArray(dataPtr) {
    const absDataPtr = this.abs(dataPtr);

    // Array layout: [length:4][capacity:4][data_ptr:4]
    const length = this.view.getUint32(absDataPtr, true);
    const dataBlockHeaderPtr = this.view.getUint32(absDataPtr + 8, true);

    // Sanity check length (arrays shouldn't have millions of elements)
    if (length > 10000) {
      return;
    }

    // Mark the data block (data_ptr is a HEADER pointer)
    if (dataBlockHeaderPtr !== 0) {
      // Sanity check: data_ptr should be in heap
      const heapStart = this.manipulator.getHeapStart();
      const heapPointer = this.manipulator.getHeapPointer();
      if (dataBlockHeaderPtr < heapStart || dataBlockHeaderPtr >= heapPointer) {
        return;
      }

      this.setMark(dataBlockHeaderPtr);

      // Data starts after GC header
      const dataBlockDataPtr = dataBlockHeaderPtr + GC_HEADER_SIZE;

      // Mark each element
      for (let i = 0; i < length; i++) {
        const elemAddr = dataBlockDataPtr + i * VALUE_SIZE;
        this.markValue(elemAddr);
      }
    }
  }

  /**
   * Mark array data block elements (when encountering data block directly).
   * @param {number} headerPtr - Pointer to data block GC header
   */
  markArrayDataElements(headerPtr) {
    const size = this.readHeaderSize(headerPtr);
    const dataSize = size - GC_HEADER_SIZE;
    const count = Math.floor(dataSize / VALUE_SIZE);

    const dataPtr = headerPtr + GC_HEADER_SIZE;
    for (let i = 0; i < count; i++) {
      const elemAddr = dataPtr + i * VALUE_SIZE;
      this.markValue(elemAddr);
    }
  }

  /**
   * Mark object entries.
   * @param {number} dataPtr - Pointer to object data
   */
  markObjectEntries(dataPtr) {
    const absDataPtr = this.abs(dataPtr);

    // Object layout: [count:4][capacity:4][entries_ptr:4]
    const count = this.view.getUint32(absDataPtr, true);
    const entriesHeaderPtr = this.view.getUint32(absDataPtr + 8, true);

    if (entriesHeaderPtr === 0) return;

    // Mark entries block (entries_ptr is a HEADER pointer)
    this.setMark(entriesHeaderPtr);

    // Entry data starts after GC header
    const entriesDataPtr = entriesHeaderPtr + GC_HEADER_SIZE;

    // Entry layout: [key_offset:4][type:4][flags:4][data_lo:4][data_hi:4] = 20 bytes
    for (let i = 0; i < count; i++) {
      const entryAddr = entriesDataPtr + i * 20;
      const absEntryAddr = this.abs(entryAddr);

      // Key is a string offset
      const keyOffset = this.view.getUint32(absEntryAddr, true);
      this.markString(keyOffset);

      // Value starts at entry + 4
      this.markValue(entryAddr + 4);
    }
  }

  /**
   * Mark scope.
   * @param {number} dataPtr - Pointer to scope data
   */
  markScope(dataPtr) {
    const absDataPtr = this.abs(dataPtr);

    // Scope layout: [parent:4][count:4][capacity:4][entries_ptr:4]
    const parentDataPtr = this.view.getUint32(absDataPtr, true);
    const count = this.view.getUint32(absDataPtr + 4, true);
    const entriesHeaderPtr = this.view.getUint32(absDataPtr + 12, true);

    // Mark parent scope
    if (parentDataPtr !== 0) {
      this.markObjectByDataPtr(parentDataPtr);
    }

    // Mark entries block (entries_ptr is a HEADER pointer)
    if (entriesHeaderPtr !== 0) {
      this.setMark(entriesHeaderPtr);

      // Entry data starts after GC header
      const entriesDataPtr = entriesHeaderPtr + GC_HEADER_SIZE;

      // Entry layout: [key:4][type:4][flags:4][data_lo:4][data_hi:4] = 20 bytes
      for (let i = 0; i < count; i++) {
        const entryAddr = entriesDataPtr + i * 20;
        const absEntryAddr = this.abs(entryAddr);

        // Key is a string offset
        const keyOffset = this.view.getUint32(absEntryAddr, true);
        this.markString(keyOffset);

        // Value starts at entry + 4
        this.markValue(entryAddr + 4);
      }
    }
  }

  /**
   * Mark closure.
   * @param {number} dataPtr - Pointer to closure data
   */
  markClosure(dataPtr) {
    const absDataPtr = this.abs(dataPtr);

    // Closure layout: [start:4][end:4][scope:4][flags:4]
    const scopeDataPtr = this.view.getUint32(absDataPtr + 8, true);

    if (scopeDataPtr !== 0) {
      this.markObjectByDataPtr(scopeDataPtr);
    }
  }

  /**
   * Mark param list.
   * @param {number} dataPtr - Pointer to param list data
   */
  markParamList(dataPtr) {
    const absDataPtr = this.abs(dataPtr);

    // Param list layout: [count:4][offset:4][offset:4]...
    const count = this.view.getUint32(absDataPtr, true);

    for (let i = 0; i < count; i++) {
      const offset = this.view.getUint32(absDataPtr + 4 + i * 4, true);
      this.markString(offset);
    }
  }

  /**
   * Mark all values on the pending stack.
   */
  markPendingStack() {
    const pendingBase = this.manipulator.getPendingBase();
    const pendingPointer = this.manipulator.getPendingPointer();

    for (let addr = pendingBase; addr < pendingPointer; addr += VALUE_SIZE) {
      this.markValue(addr);
    }
  }

  /**
   * Mark scopes and iteration state from call stack frames.
   */
  markCallStack() {
    const stackBase = this.manipulator.getStackBase();
    const stackPointer = this.manipulator.getStackPointer();

    for (let frameAddr = stackBase; frameAddr < stackPointer; frameAddr += FRAME_SIZE) {
      const absFrameAddr = this.abs(frameAddr);

      // Mark frame's scope (DATA pointer)
      const scopeDataPtr = this.view.getUint32(absFrameAddr + FRAME.SCOPE_POINTER, true);
      if (scopeDataPtr !== 0) {
        this.markObjectByDataPtr(scopeDataPtr);
      }

      // Check if frame is iterating
      const flags = this.view.getUint32(absFrameAddr + FRAME.FLAGS, true);
      if (flags & FRAME_FLAG_ITERATING) {
        // Mark iteration state (DATA pointers)
        const iterResult = this.view.getUint32(absFrameAddr + FRAME.ITER_RESULT, true);
        const iterReceiver = this.view.getUint32(absFrameAddr + FRAME.ITER_RECEIVER, true);

        if (iterResult !== 0) this.markObjectByDataPtr(iterResult);
        if (iterReceiver !== 0) this.markObjectByDataPtr(iterReceiver);
        // iterCallback stores instruction index, not a heap pointer
      }
    }
  }

  // ===========================================================================
  // Phase 2-4: Heap Compaction
  // ===========================================================================

  /**
   * Compact the heap by moving live objects to the beginning.
   */
  compactHeap() {
    const heapStart = this.manipulator.getHeapStart();
    const oldHeapPointer = this.manipulator.getHeapPointer();

    // Phase 2: Compute forwarding addresses
    this.computeForwardingAddresses();

    // Build forwarding map for external roots BEFORE moving objects
    // (moveObjects clears forwarding pointers)
    const forwarding = this.buildExternalForwarding();

    // Phase 3: Update all pointers
    this.updatePointers();

    // Phase 4: Move objects
    const newHeapPointer = this.moveObjects();

    const heapRetained = newHeapPointer - heapStart;
    const heapCollected = oldHeapPointer - newHeapPointer;

    return { heapRetained, heapCollected, forwarding };
  }

  /**
   * Build forwarding map for external roots.
   * Must be called after computeForwardingAddresses but before moveObjects.
   *
   * @returns {Map<number, number>} - Map from old DATA pointer to new DATA pointer
   */
  buildExternalForwarding() {
    const forwarding = new Map();

    for (const oldDataPtr of this.externalRoots) {
      if (oldDataPtr === 0) continue;

      const newDataPtr = this.getNewDataPtr(oldDataPtr);
      if (newDataPtr !== oldDataPtr) {
        forwarding.set(oldDataPtr, newDataPtr);
      }
    }

    return forwarding;
  }

  /**
   * Compute forwarding addresses for all live objects.
   * Forwarding addresses are HEADER pointers.
   */
  computeForwardingAddresses() {
    const heapStart = this.manipulator.getHeapStart();
    const heapPointer = this.manipulator.getHeapPointer();

    let readHeaderPtr = heapStart;
    let writeHeaderPtr = heapStart;

    while (readHeaderPtr < heapPointer) {
      const size = this.readHeaderSize(readHeaderPtr);
      if (size === 0) break;

      if (this.isMarked(readHeaderPtr)) {
        // Live object - record where it will move to
        this.setForwardingPtr(readHeaderPtr, writeHeaderPtr);
        writeHeaderPtr += size;
      }

      readHeaderPtr += size;
    }
  }

  /**
   * Update all pointers to use forwarding addresses.
   */
  updatePointers() {
    const heapStart = this.manipulator.getHeapStart();
    const heapPointer = this.manipulator.getHeapPointer();

    // Update pointers in heap objects
    let headerPtr = heapStart;
    while (headerPtr < heapPointer) {
      const size = this.readHeaderSize(headerPtr);
      if (size === 0) break;

      if (this.isMarked(headerPtr)) {
        this.updateObjectPointers(headerPtr);
      }

      headerPtr += size;
    }

    // Update root pointers

    // Global scope (DATA pointer)
    const globalScopeData = this.manipulator.getScope();
    if (globalScopeData !== 0) {
      const newDataPtr = this.getNewDataPtr(globalScopeData);
      if (newDataPtr !== globalScopeData) {
        this.manipulator.setScope(newDataPtr);
      }
    }

    // Result pointer - update the value inside
    const resultPointer = this.manipulator.getResultPointer();
    if (resultPointer !== 0) {
      this.updateValuePointer(resultPointer);
    }

    // Pending stack
    this.updatePendingStackPointers();

    // Call stack
    this.updateCallStackPointers();

    // Completion value
    const completionType = this.manipulator.getCompletionType();
    if (completionType !== 0) {
      const completionValue = this.manipulator.getCompletionValue();
      if (completionValue !== 0) {
        this.updateValuePointer(completionValue);
      }
    }

    // Invocation slots (point into scope entries which may have moved)
    this.updateInvocationSlots();
  }

  /**
   * Update invocation slot pointers after heap compaction.
   *
   * Slot pointers point to value regions within scope entry blocks.
   * When the entries block moves, we need to update these pointers
   * by the same delta.
   */
  updateInvocationSlots() {
    const inputCount = this.manipulator.getInputCount();
    const outputCount = this.manipulator.getOutputCount();

    if (inputCount === 0 && outputCount === 0) {
      return; // No slots configured
    }

    // Update input slots
    for (let i = 0; i < inputCount; i++) {
      const slotAddr = LAYOUT.INVOCATION + INVOCATION.INPUT_SLOTS + i * 8;
      this.updateSlotPointer(slotAddr);
    }

    // Update output slots
    for (let i = 0; i < outputCount; i++) {
      const slotAddr = LAYOUT.INVOCATION + INVOCATION.OUTPUT_SLOTS + i * 8;
      this.updateSlotPointer(slotAddr);
    }
  }

  /**
   * Update a single slot's value pointer.
   *
   * The slot pointer points into an OBJ_ENTRIES block (scope entries).
   * We find the block's header by scanning backwards from the pointer,
   * then use the forwarding address to compute the new pointer.
   *
   * @param {number} slotAddr - Address of slot descriptor
   */
  updateSlotPointer(slotAddr) {
    const absSlotAddr = this.abs(slotAddr);
    const oldValuePtr = this.view.getUint32(absSlotAddr + SLOT.VALUE_PTR, true);

    if (oldValuePtr === 0) {
      return; // Slot not configured
    }

    // The value pointer is: entriesHeaderPtr + GC_HEADER_SIZE + entryIndex * 20 + 4
    // We need to find the entries block header that contains this pointer.
    // The pointer is inside the entries data region, so we scan backwards
    // to find a valid GC header.

    const heapStart = this.manipulator.getHeapStart();

    // The entries data starts at headerPtr + GC_HEADER_SIZE.
    // The value ptr is at some offset within that data.
    // We compute: entriesHeaderPtr = oldValuePtr - offsetWithinData
    // where offsetWithinData = (oldValuePtr - entriesHeaderPtr - GC_HEADER_SIZE)
    //
    // Since entry size is 20 and value is at +4 within entry:
    // offsetWithinData = GC_HEADER_SIZE + entryIndex * 20 + 4
    //
    // We can find the header by: for each candidate header,
    // check if (oldValuePtr - candidate - GC_HEADER_SIZE - 4) % 20 == 0
    // But that's expensive. Instead, we use a simpler approach:
    //
    // The GC header type at (ptr - GC_HEADER_SIZE) would be OBJ_ENTRIES.
    // We scan backwards in 4-byte steps to find it.

    // Actually, simpler: compute offset relative to heap start,
    // then find which object contains this pointer by walking the heap.

    // Walk heap to find the containing object
    let headerPtr = heapStart;
    const heapPointer = this.manipulator.getHeapPointer();
    let foundHeaderPtr = 0;

    while (headerPtr < heapPointer) {
      const size = this.readHeaderSize(headerPtr);
      if (size === 0) break;

      const dataStart = headerPtr + GC_HEADER_SIZE;
      const dataEnd = headerPtr + size;

      if (oldValuePtr >= dataStart && oldValuePtr < dataEnd) {
        foundHeaderPtr = headerPtr;
        break;
      }

      headerPtr += size;
    }

    if (foundHeaderPtr === 0) {
      // Pointer not in heap - might be invalid or already freed
      return;
    }

    // Get the new header pointer
    const newHeaderPtr = this.getNewHeaderPtr(foundHeaderPtr);
    if (newHeaderPtr === foundHeaderPtr) {
      return; // Object didn't move
    }

    // Compute offset within the object and apply to new location
    const offsetWithinObject = oldValuePtr - foundHeaderPtr;
    const newValuePtr = newHeaderPtr + offsetWithinObject;

    this.view.setUint32(absSlotAddr + SLOT.VALUE_PTR, newValuePtr, true);
  }

  /**
   * Given an old DATA pointer, return the new DATA pointer after compaction.
   * @param {number} oldDataPtr - Old data pointer
   * @returns {number} - New data pointer
   */
  getNewDataPtr(oldDataPtr) {
    // Validate pointer is within heap bounds
    const heapStart = this.manipulator.getHeapStart();
    const heapPointer = this.manipulator.getHeapPointer();
    if (oldDataPtr < heapStart || oldDataPtr >= heapPointer) {
      // Not a valid heap pointer - likely a corrupted or misinterpreted value
      // Return unchanged to avoid crash
      return oldDataPtr;
    }

    const oldHeaderPtr = oldDataPtr - GC_HEADER_SIZE;
    const newHeaderPtr = this.getForwardingPtr(oldHeaderPtr);
    if (newHeaderPtr === 0 || newHeaderPtr === oldHeaderPtr) {
      return oldDataPtr; // Not moving
    }
    return newHeaderPtr + GC_HEADER_SIZE;
  }

  /**
   * Get new HEADER pointer for old HEADER pointer.
   * Used for entries_ptr and data_ptr which store HEADER pointers.
   * @param {number} oldHeaderPtr - Old header pointer
   * @returns {number} - New header pointer
   */
  getNewHeaderPtr(oldHeaderPtr) {
    // Validate pointer is within heap bounds
    const heapStart = this.manipulator.getHeapStart();
    const heapPointer = this.manipulator.getHeapPointer();
    if (oldHeaderPtr < heapStart || oldHeaderPtr >= heapPointer) {
      // Not a valid heap pointer - likely a corrupted or misinterpreted value
      // Return unchanged to avoid crash
      return oldHeaderPtr;
    }

    const newHeaderPtr = this.getForwardingPtr(oldHeaderPtr);
    if (newHeaderPtr === 0 || newHeaderPtr === oldHeaderPtr) {
      return oldHeaderPtr; // Not moving
    }
    return newHeaderPtr;
  }

  /**
   * Update pointers within an object.
   * @param {number} headerPtr - Pointer to GC header
   */
  updateObjectPointers(headerPtr) {
    const objType = this.readHeaderType(headerPtr);
    const dataPtr = headerPtr + GC_HEADER_SIZE;

    switch (objType) {
      case OBJ.ARRAY:
        this.updateArrayPointers(dataPtr);
        break;

      case OBJ.OBJECT:
        this.updateObjectEntryPointers(dataPtr);
        break;

      case OBJ.SCOPE:
        this.updateScopePointers(dataPtr);
        break;

      case OBJ.CLOSURE:
        this.updateClosurePointers(dataPtr);
        break;

      case OBJ.ARRAY_DATA:
        this.updateArrayDataPointers(headerPtr);
        break;
    }
  }

  /**
   * Update a value's heap pointer if it references a heap object.
   * @param {number} valueAddr - Address of value slot
   */
  updateValuePointer(valueAddr) {
    const absAddr = this.abs(valueAddr);
    const type = this.view.getUint32(absAddr, true);

    if (type === TYPE.ARRAY || type === TYPE.OBJECT) {
      // Arrays and objects store HEADER pointers
      const oldHeaderPtr = this.view.getUint32(absAddr + 8, true);
      if (oldHeaderPtr !== 0) {
        const newHeaderPtr = this.getNewHeaderPtr(oldHeaderPtr);
        if (newHeaderPtr !== oldHeaderPtr) {
          this.view.setUint32(absAddr + 8, newHeaderPtr, true);
        }
      }
    } else if (type === TYPE.CLOSURE || type === TYPE.SCOPE) {
      // Closures and scopes store DATA pointers
      const oldDataPtr = this.view.getUint32(absAddr + 8, true);
      if (oldDataPtr !== 0) {
        const newDataPtr = this.getNewDataPtr(oldDataPtr);
        if (newDataPtr !== oldDataPtr) {
          this.view.setUint32(absAddr + 8, newDataPtr, true);
        }
      }
    } else if (type === TYPE.BOUND_METHOD) {
      // BOUND_METHOD layout: [type:4][flags:4][data_lo:4][data_hi:4]
      // data_lo (offset +8) = receiver pointer (HEADER pointer to array/object)
      // flags (offset +4) = receiver type (e.g., TYPE_ARRAY)
      const oldReceiver = this.view.getUint32(absAddr + 8, true);
      if (oldReceiver !== 0) {
        const newReceiver = this.getNewHeaderPtr(oldReceiver);
        if (newReceiver !== oldReceiver) {
          this.view.setUint32(absAddr + 8, newReceiver, true);
        }
      }
    } else if (type === TYPE.CONSTRUCTOR) {
      // CONSTRUCTOR stores a HEADER pointer to the prototype object
      const oldHeaderPtr = this.view.getUint32(absAddr + 8, true);
      if (oldHeaderPtr !== 0) {
        const newHeaderPtr = this.getNewHeaderPtr(oldHeaderPtr);
        if (newHeaderPtr !== oldHeaderPtr) {
          this.view.setUint32(absAddr + 8, newHeaderPtr, true);
        }
      }
    }
  }

  /**
   * Update pointers in array.
   * @param {number} dataPtr - Array data pointer
   */
  updateArrayPointers(dataPtr) {
    const absDataPtr = this.abs(dataPtr);

    // data_ptr stores a HEADER pointer
    const oldDataBlockHeaderPtr = this.view.getUint32(absDataPtr + 8, true);
    if (oldDataBlockHeaderPtr !== 0) {
      const newDataBlockHeaderPtr = this.getNewHeaderPtr(oldDataBlockHeaderPtr);
      if (newDataBlockHeaderPtr !== oldDataBlockHeaderPtr) {
        this.view.setUint32(absDataPtr + 8, newDataBlockHeaderPtr, true);
      }
    }
  }

  /**
   * Update pointers in array data block elements.
   * @param {number} headerPtr - Header pointer of data block
   */
  updateArrayDataPointers(headerPtr) {
    const size = this.readHeaderSize(headerPtr);
    const dataSize = size - GC_HEADER_SIZE;
    const count = Math.floor(dataSize / VALUE_SIZE);

    const dataPtr = headerPtr + GC_HEADER_SIZE;
    for (let i = 0; i < count; i++) {
      const elemAddr = dataPtr + i * VALUE_SIZE;
      this.updateValuePointer(elemAddr);
    }
  }

  /**
   * Update pointers in object entries.
   * @param {number} dataPtr - Object data pointer
   */
  updateObjectEntryPointers(dataPtr) {
    const absDataPtr = this.abs(dataPtr);

    const count = this.view.getUint32(absDataPtr, true);
    // entries_ptr stores a HEADER pointer
    const oldEntriesHeaderPtr = this.view.getUint32(absDataPtr + 8, true);

    if (oldEntriesHeaderPtr !== 0) {
      // Update entries_ptr
      const newEntriesHeaderPtr = this.getNewHeaderPtr(oldEntriesHeaderPtr);
      if (newEntriesHeaderPtr !== oldEntriesHeaderPtr) {
        this.view.setUint32(absDataPtr + 8, newEntriesHeaderPtr, true);
      }

      // Update values in entries (using OLD pointer - objects haven't moved yet)
      // Entry data starts after GC header
      const entriesDataPtr = oldEntriesHeaderPtr + GC_HEADER_SIZE;
      for (let i = 0; i < count; i++) {
        const entryAddr = entriesDataPtr + i * 20;
        this.updateValuePointer(entryAddr + 4);
      }
    }
  }

  /**
   * Update pointers in scope.
   * @param {number} dataPtr - Scope data pointer
   */
  updateScopePointers(dataPtr) {
    const absDataPtr = this.abs(dataPtr);

    // Update parent pointer
    const oldParentDataPtr = this.view.getUint32(absDataPtr, true);
    if (oldParentDataPtr !== 0) {
      const newParentDataPtr = this.getNewDataPtr(oldParentDataPtr);
      if (newParentDataPtr !== oldParentDataPtr) {
        this.view.setUint32(absDataPtr, newParentDataPtr, true);
      }
    }

    // Update entries_ptr (stores a HEADER pointer)
    const count = this.view.getUint32(absDataPtr + 4, true);
    const oldEntriesHeaderPtr = this.view.getUint32(absDataPtr + 12, true);

    if (oldEntriesHeaderPtr !== 0) {
      const newEntriesHeaderPtr = this.getNewHeaderPtr(oldEntriesHeaderPtr);
      if (newEntriesHeaderPtr !== oldEntriesHeaderPtr) {
        this.view.setUint32(absDataPtr + 12, newEntriesHeaderPtr, true);
      }

      // Update values in entries (using OLD pointer - objects haven't moved yet)
      // Entry data starts after GC header
      const entriesDataPtr = oldEntriesHeaderPtr + GC_HEADER_SIZE;
      for (let i = 0; i < count; i++) {
        const entryAddr = entriesDataPtr + i * 20;
        this.updateValuePointer(entryAddr + 4);
      }
    }
  }

  /**
   * Update pointers in closure.
   * @param {number} dataPtr - Closure data pointer
   */
  updateClosurePointers(dataPtr) {
    const absDataPtr = this.abs(dataPtr);

    const oldScopeDataPtr = this.view.getUint32(absDataPtr + 8, true);
    if (oldScopeDataPtr !== 0) {
      const newScopeDataPtr = this.getNewDataPtr(oldScopeDataPtr);
      if (newScopeDataPtr !== oldScopeDataPtr) {
        this.view.setUint32(absDataPtr + 8, newScopeDataPtr, true);
      }
    }
  }

  /**
   * Update pointers in pending stack values.
   */
  updatePendingStackPointers() {
    const pendingBase = this.manipulator.getPendingBase();
    const pendingPointer = this.manipulator.getPendingPointer();

    for (let addr = pendingBase; addr < pendingPointer; addr += VALUE_SIZE) {
      this.updateValuePointer(addr);
    }
  }

  /**
   * Update pointers in call stack frames.
   */
  updateCallStackPointers() {
    const stackBase = this.manipulator.getStackBase();
    const stackPointer = this.manipulator.getStackPointer();

    for (let frameAddr = stackBase; frameAddr < stackPointer; frameAddr += FRAME_SIZE) {
      const absFrameAddr = this.abs(frameAddr);

      // Update scope pointer
      const oldScopeDataPtr = this.view.getUint32(absFrameAddr + FRAME.SCOPE_POINTER, true);
      if (oldScopeDataPtr !== 0) {
        const newScopeDataPtr = this.getNewDataPtr(oldScopeDataPtr);
        if (newScopeDataPtr !== oldScopeDataPtr) {
          this.view.setUint32(absFrameAddr + FRAME.SCOPE_POINTER, newScopeDataPtr, true);
        }
      }

      // Update iteration state if iterating
      const flags = this.view.getUint32(absFrameAddr + FRAME.FLAGS, true);
      if (flags & FRAME_FLAG_ITERATING) {
        const oldIterResult = this.view.getUint32(absFrameAddr + FRAME.ITER_RESULT, true);
        if (oldIterResult !== 0) {
          const newIterResult = this.getNewDataPtr(oldIterResult);
          if (newIterResult !== oldIterResult) {
            this.view.setUint32(absFrameAddr + FRAME.ITER_RESULT, newIterResult, true);
          }
        }

        const oldIterReceiver = this.view.getUint32(absFrameAddr + FRAME.ITER_RECEIVER, true);
        if (oldIterReceiver !== 0) {
          const newIterReceiver = this.getNewDataPtr(oldIterReceiver);
          if (newIterReceiver !== oldIterReceiver) {
            this.view.setUint32(absFrameAddr + FRAME.ITER_RECEIVER, newIterReceiver, true);
          }
        }
      }
    }
  }

  /**
   * Move live objects to their new locations.
   */
  moveObjects() {
    const heapStart = this.manipulator.getHeapStart();
    const heapPointer = this.manipulator.getHeapPointer();

    let readHeaderPtr = heapStart;
    let newHeapPointer = heapStart;

    while (readHeaderPtr < heapPointer) {
      const size = this.readHeaderSize(readHeaderPtr);
      if (size === 0) break;

      if (this.isMarked(readHeaderPtr)) {
        const newHeaderPtr = this.getForwardingPtr(readHeaderPtr);

        if (newHeaderPtr !== readHeaderPtr) {
          // Move the object using copyWithin
          const absSrc = this.abs(readHeaderPtr);
          const absDst = this.abs(newHeaderPtr);
          this.bytes.copyWithin(absDst, absSrc, absSrc + size);
        }

        // Clear mark and forwarding pointer in new location
        this.clearMark(newHeaderPtr);
        this.setForwardingPtr(newHeaderPtr, 0);

        newHeapPointer = newHeaderPtr + size;
      }

      readHeaderPtr += size;
    }

    this.manipulator.setHeapPointer(newHeapPointer);
    return newHeapPointer;
  }

  // ===========================================================================
  // Phase 5: String Compaction
  // ===========================================================================

  /**
   * Compact the string table by removing unmarked strings.
   */
  compactStrings() {
    const stringStart = this.manipulator.getStringStart();
    const oldStringPointer = this.manipulator.getStringPointer();

    // Build forwarding table for strings
    const stringForwarding = this.computeStringForwarding();

    // If no strings were collected, skip the rest
    if (stringForwarding.size === 0) {
      const stringsRetained = oldStringPointer - this.stringDataStart;
      return { stringsRetained, stringsCollected: 0 };
    }

    // Update all string offsets in the heap
    this.updateStringOffsets(stringForwarding);

    // Update string offsets in pending stack
    this.updatePendingStackStrings(stringForwarding);

    // Update string offsets in code block
    this.updateCodeBlockStrings(stringForwarding);

    // Update string offsets in builtins table
    this.updateBuiltinsStrings(stringForwarding);

    // Move strings to new locations
    const newStringPointer = this.moveStrings(stringForwarding);

    // Rebuild hash table
    this.rebuildHashTable();

    const stringsRetained = newStringPointer - this.stringDataStart;
    const stringsCollected = oldStringPointer - newStringPointer;

    return { stringsRetained, stringsCollected };
  }

  /**
   * Compute forwarding offsets for live strings.
   */
  computeStringForwarding() {
    const forwarding = new Map();
    const stringPointer = this.manipulator.getStringPointer();

    let readPtr = this.stringDataStart;
    let writePtr = this.stringDataStart;

    while (readPtr < stringPointer) {
      const absReadPtr = this.abs(readPtr);
      const length = this.view.getUint32(absReadPtr, true);
      const entrySize = 4 + length;
      const alignedSize = (entrySize + 3) & ~3;

      if (this.isStringMarked(readPtr)) {
        // Live string
        if (readPtr !== writePtr) {
          forwarding.set(readPtr, writePtr);
        }
        writePtr += alignedSize;
      } else {
        // Dead string - mark for removal
        forwarding.set(readPtr, 0);
      }

      readPtr += alignedSize;
    }

    return forwarding;
  }

  /**
   * Update all string offsets in the heap.
   */
  updateStringOffsets(forwarding) {
    const heapStart = this.manipulator.getHeapStart();
    const heapPointer = this.manipulator.getHeapPointer();

    let headerPtr = heapStart;
    while (headerPtr < heapPointer) {
      const size = this.readHeaderSize(headerPtr);
      if (size === 0) break;

      const objType = this.readHeaderType(headerPtr);
      const dataPtr = headerPtr + GC_HEADER_SIZE;
      this.updateObjectStringOffsets(dataPtr, objType, forwarding);

      headerPtr += size;
    }
  }

  /**
   * Update string offsets within a heap object.
   */
  updateObjectStringOffsets(dataPtr, objType, forwarding) {
    switch (objType) {
      case OBJ.ARRAY:
        this.updateArrayStringOffsets(dataPtr, forwarding);
        break;

      case OBJ.OBJECT:
        this.updateObjectEntryStringOffsets(dataPtr, forwarding);
        break;

      case OBJ.SCOPE:
        this.updateScopeStringOffsets(dataPtr, forwarding);
        break;

      case OBJ.PARAM_LIST:
        this.updateParamListStringOffsets(dataPtr, forwarding);
        break;

      case OBJ.ARRAY_DATA:
        this.updateArrayDataStringOffsets(dataPtr - GC_HEADER_SIZE, forwarding);
        break;
    }
  }

  /**
   * Update string offsets in array elements.
   */
  updateArrayStringOffsets(dataPtr, forwarding) {
    const absDataPtr = this.abs(dataPtr);

    const length = this.view.getUint32(absDataPtr, true);
    const dataBlockDataPtr = this.view.getUint32(absDataPtr + 8, true);

    if (dataBlockDataPtr !== 0) {
      for (let i = 0; i < length; i++) {
        const elemAddr = dataBlockDataPtr + i * VALUE_SIZE;
        this.updateValueStringOffset(elemAddr, forwarding);
      }
    }
  }

  /**
   * Update string offsets in array data elements.
   */
  updateArrayDataStringOffsets(headerPtr, forwarding) {
    const size = this.readHeaderSize(headerPtr);
    const dataSize = size - GC_HEADER_SIZE;
    const count = Math.floor(dataSize / VALUE_SIZE);

    const dataPtr = headerPtr + GC_HEADER_SIZE;
    for (let i = 0; i < count; i++) {
      const elemAddr = dataPtr + i * VALUE_SIZE;
      this.updateValueStringOffset(elemAddr, forwarding);
    }
  }

  /**
   * Update string offsets in object entries.
   */
  updateObjectEntryStringOffsets(dataPtr, forwarding) {
    const absDataPtr = this.abs(dataPtr);

    const count = this.view.getUint32(absDataPtr, true);
    // entries_ptr stores a HEADER pointer
    const entriesHeaderPtr = this.view.getUint32(absDataPtr + 8, true);

    if (entriesHeaderPtr === 0) return;

    // Entry data starts after GC header
    const entriesDataPtr = entriesHeaderPtr + GC_HEADER_SIZE;

    for (let i = 0; i < count; i++) {
      const entryAddr = entriesDataPtr + i * 20;
      const absEntryAddr = this.abs(entryAddr);

      // Key is a string offset
      const oldKey = this.view.getUint32(absEntryAddr, true);
      const newKey = forwarding.get(oldKey);
      if (newKey !== undefined && newKey !== 0 && newKey !== oldKey) {
        this.view.setUint32(absEntryAddr, newKey, true);
      }

      // Value might be a string
      this.updateValueStringOffset(entryAddr + 4, forwarding);
    }
  }

  /**
   * Update string offsets in scope bindings.
   */
  updateScopeStringOffsets(dataPtr, forwarding) {
    const absDataPtr = this.abs(dataPtr);

    const count = this.view.getUint32(absDataPtr + 4, true);
    // entries_ptr stores a HEADER pointer
    const entriesHeaderPtr = this.view.getUint32(absDataPtr + 12, true);

    if (entriesHeaderPtr === 0) return;

    // Sanity check
    const heapStart = this.manipulator.getHeapStart();
    const heapPointer = this.manipulator.getHeapPointer();
    if (entriesHeaderPtr < heapStart || entriesHeaderPtr >= heapPointer) {
      return;
    }

    // Entry data starts after GC header
    const entriesDataPtr = entriesHeaderPtr + GC_HEADER_SIZE;

    for (let i = 0; i < count; i++) {
      const entryAddr = entriesDataPtr + i * 20;
      const absEntryAddr = this.abs(entryAddr);

      // Key is a string offset
      const oldKey = this.view.getUint32(absEntryAddr, true);
      const newKey = forwarding.get(oldKey);
      if (newKey !== undefined && newKey !== 0 && newKey !== oldKey) {
        this.view.setUint32(absEntryAddr, newKey, true);
      }

      // Value might be a string
      this.updateValueStringOffset(entryAddr + 4, forwarding);
    }
  }

  /**
   * Update string offsets in param list.
   */
  updateParamListStringOffsets(dataPtr, forwarding) {
    const absDataPtr = this.abs(dataPtr);

    const count = this.view.getUint32(absDataPtr, true);

    for (let i = 0; i < count; i++) {
      const offsetAddr = absDataPtr + 4 + i * 4;
      const oldOffset = this.view.getUint32(offsetAddr, true);
      const newOffset = forwarding.get(oldOffset);
      if (newOffset !== undefined && newOffset !== 0 && newOffset !== oldOffset) {
        this.view.setUint32(offsetAddr, newOffset, true);
      }
    }
  }

  /**
   * Update a value's string offset if it's a string type.
   */
  updateValueStringOffset(valueAddr, forwarding) {
    const absAddr = this.abs(valueAddr);
    const type = this.view.getUint32(absAddr, true);

    if (type === TYPE.STRING) {
      const oldOffset = this.view.getUint32(absAddr + 8, true);
      const newOffset = forwarding.get(oldOffset);
      if (newOffset !== undefined && newOffset !== 0 && newOffset !== oldOffset) {
        this.view.setUint32(absAddr + 8, newOffset, true);
      }
    }
  }

  /**
   * Update string offsets in pending stack values.
   */
  updatePendingStackStrings(forwarding) {
    const pendingBase = this.manipulator.getPendingBase();
    const pendingPointer = this.manipulator.getPendingPointer();

    for (let addr = pendingBase; addr < pendingPointer; addr += VALUE_SIZE) {
      this.updateValueStringOffset(addr, forwarding);
    }
  }

  /**
   * Update string offsets in code block instructions.
   * Same opcodes as markCodeBlockStrings.
   */
  updateCodeBlockStrings(forwarding) {
    const codeBlock = this.manipulator.getCodeBlock();
    if (codeBlock === 0) return;

    const instrCount = this.manipulator.codeBlockInstructionCount(codeBlock);
    const codeStart = this.manipulator.getCodeStart();

    // Opcodes that use string offsets in operand1
    const stringOpcodes = new Set([
      OP.LIT_STRING,  // 0x03
      OP.GET_VAR,     // 0x10
      OP.SET_VAR,     // 0x11
      OP.LET_VAR,     // 0x12
      OP.BIND_PARAM,  // 0x13
      OP.GET_PROP,    // 0x82
      OP.SET_PROP,    // 0x83
    ]);

    // Instructions grow downward from code_start
    for (let i = 0; i < instrCount; i++) {
      const instrAddr = codeStart - (i + 1) * INSTRUCTION_SIZE;
      const absAddr = this.abs(instrAddr);
      const opcode = this.view.getUint8(absAddr);

      if (stringOpcodes.has(opcode)) {
        const oldOffset = this.view.getUint32(absAddr + 8, true);
        if (oldOffset !== 0) {
          const newOffset = forwarding.get(oldOffset);
          if (newOffset !== undefined && newOffset !== 0 && newOffset !== oldOffset) {
            this.view.setUint32(absAddr + 8, newOffset, true);
          }
        }
      }
    }
  }

  /**
   * Update string offsets in the builtins table.
   * The builtins table stores string offsets for built-in method names.
   */
  updateBuiltinsStrings(forwarding) {
    const builtinsBase = this.manipulator.getBuiltinsBase();

    // The builtins table has slots from 0x00 to 0xDC (56 slots of 4 bytes each)
    // Each slot stores a string table offset
    const BUILTINS_SLOT_COUNT = 56;

    for (let i = 0; i < BUILTINS_SLOT_COUNT; i++) {
      const slotAddr = builtinsBase + i * 4;
      const absSlotAddr = this.abs(slotAddr);
      const oldOffset = this.view.getUint32(absSlotAddr, true);

      if (oldOffset !== 0) {
        const newOffset = forwarding.get(oldOffset);
        if (newOffset !== undefined && newOffset !== 0 && newOffset !== oldOffset) {
          this.view.setUint32(absSlotAddr, newOffset, true);
        }
      }
    }
  }

  /**
   * Move live strings to their new locations.
   */
  moveStrings(forwarding) {
    const stringPointer = this.manipulator.getStringPointer();

    let readPtr = this.stringDataStart;
    let newStringPointer = this.stringDataStart;

    while (readPtr < stringPointer) {
      const absReadPtr = this.abs(readPtr);
      const length = this.view.getUint32(absReadPtr, true);
      const entrySize = 4 + length;
      const alignedSize = (entrySize + 3) & ~3;

      if (this.isStringMarked(readPtr)) {
        const newPtr = forwarding.get(readPtr);
        const targetPtr = (newPtr !== undefined && newPtr !== 0) ? newPtr : readPtr;

        if (targetPtr !== readPtr) {
          // Move the string
          const absSrc = this.abs(readPtr);
          const absDst = this.abs(targetPtr);
          this.bytes.copyWithin(absDst, absSrc, absSrc + alignedSize);
        }

        newStringPointer = targetPtr + alignedSize;
      }

      readPtr += alignedSize;
    }

    this.manipulator.setStringPointer(newStringPointer);
    return newStringPointer;
  }

  /**
   * Rebuild the hash table after string compaction.
   */
  rebuildHashTable() {
    const stringStart = this.manipulator.getStringStart();
    const stringPointer = this.manipulator.getStringPointer();

    // Clear hash table
    const hashTableStart = this.abs(stringStart);
    for (let i = 0; i < HASH_TABLE_SIZE; i += 4) {
      this.view.setUint32(hashTableStart + i, 0, true);
    }

    // Re-insert each live string
    let ptr = this.stringDataStart;
    while (ptr < stringPointer) {
      const absPtr = this.abs(ptr);
      const length = this.view.getUint32(absPtr, true);
      const entrySize = 4 + length;
      const alignedSize = (entrySize + 3) & ~3;

      // Compute hash of string bytes
      const hash = this.computeStringHash(ptr + 4, length);

      // Insert into hash table with linear probing
      const bucketCount = HASH_TABLE_SIZE / 8;
      let bucketIndex = hash % bucketCount;

      for (let probe = 0; probe < bucketCount; probe++) {
        const bucketPtr = hashTableStart + bucketIndex * 8;
        const storedHash = this.view.getUint32(bucketPtr, true);

        if (storedHash === 0) {
          // Empty bucket - insert here
          this.view.setUint32(bucketPtr, hash, true);
          this.view.setUint32(bucketPtr + 4, ptr, true);
          break;
        }

        bucketIndex = (bucketIndex + 1) % bucketCount;
      }

      ptr += alignedSize;
    }
  }

  /**
   * Compute FNV-1a hash of string bytes.
   */
  computeStringHash(ptr, length) {
    const FNV_OFFSET_BASIS = 0x811c9dc5;
    const FNV_PRIME = 0x01000193;

    let hash = FNV_OFFSET_BASIS;
    const absPtr = this.abs(ptr);

    for (let i = 0; i < length; i++) {
      const byte = this.bytes[absPtr + i];
      hash ^= byte;
      hash = Math.imul(hash, FNV_PRIME) >>> 0;
    }

    return hash;
  }
}
